/* *##%
 * Copyright (C) 2005 - 2010 Ifremer, Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package fr.ifremer.isisfish;

import static org.nuiton.i18n.I18n._;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Image;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.imageio.ImageIO;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.UIManager;
import javax.swing.UnsupportedLookAndFeelException;

import org.apache.commons.beanutils.ConvertUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;
import org.nuiton.math.matrix.DoubleBigVector;
import org.nuiton.math.matrix.MatrixFactory;
import org.nuiton.topia.TopiaException;
import org.nuiton.util.FileUtil;
import org.nuiton.util.LocaleConverter;
import org.nuiton.util.Version;
import org.nuiton.widget.SwingUtil;

import fr.ifremer.isisfish.cron.CronService;
import fr.ifremer.isisfish.datastore.AnalysePlanStorage;
import fr.ifremer.isisfish.datastore.ExportStorage;
import fr.ifremer.isisfish.datastore.FormuleStorage;
import fr.ifremer.isisfish.datastore.RegionStorage;
import fr.ifremer.isisfish.datastore.RuleStorage;
import fr.ifremer.isisfish.datastore.ScriptStorage;
import fr.ifremer.isisfish.datastore.SensitivityExportStorage;
import fr.ifremer.isisfish.datastore.SensitivityStorage;
import fr.ifremer.isisfish.datastore.SimulationStorage;
import fr.ifremer.isisfish.datastore.SimulatorStorage;
import fr.ifremer.isisfish.simulator.launcher.SimulationService;
import fr.ifremer.isisfish.types.Date;
import fr.ifremer.isisfish.types.Month;
import fr.ifremer.isisfish.types.RangeOfValues;
import fr.ifremer.isisfish.types.TimeUnit;
import fr.ifremer.isisfish.ui.WelcomeUI;
import fr.ifremer.isisfish.ui.util.ErrorHelper;
import fr.ifremer.isisfish.util.ConverterUtil;
import fr.ifremer.isisfish.util.DateConverter;
import fr.ifremer.isisfish.util.MonthConverter;
import fr.ifremer.isisfish.util.RangeOfValuesConverter;
import fr.ifremer.isisfish.util.StringConverter;
import fr.ifremer.isisfish.util.TimeUnitConverter;
import fr.ifremer.isisfish.vcs.VCS;
import fr.ifremer.isisfish.vcs.VCSActionEvent;
import fr.ifremer.isisfish.vcs.VCSException;
import fr.ifremer.isisfish.vcs.VCSFactory;
import fr.ifremer.isisfish.vcs.VetoableActionListener;

/**
 * This is the main class of <code>IsisFish</code> application.
 * 
 * Created: 1 aout 2005 18:37:25 CEST
 *
 * @author Benjamin POUSSIN <poussin@codelutin.com>
 * @author chemit
 * @version $Revision: 3070 $
 *
 * Last update: $Date: 2010-06-18 18:17:03 +0200 (ven., 18 juin 2010) $
 * by : $Author: chatellier $
 */
public class IsisFish { // IsisFish

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(IsisFish.class);

    static public IsisConfig config = null;

    static public VCS vcs = null;

    /**
     * ask for application quit
     */
    static public void quit() {
        System.exit(0);
    }

    public static void main(String... args) {

        try {
            // permet de faire fonctionner la compilation en webstart
            System.setSecurityManager(null);

            // initialisation de l'application
            IsisFish.init(args);

            if (log.isDebugEnabled()) {
                log.debug(_("isisfish.launch.init.done", config.getElapsedTimeAsString()));
            }

            // action after init
            config.doAction(IsisConfig.STEP_AFTER_INIT);

            // initVCS ask for passphrase, ui must be set before
            initLookAndFeel();
            
            // static vcs init (needed for some actions)
            try {
                initVCS();
            } catch (Exception eee) {
                log.warn(_("Error during vcs initialisation"), eee);
            }

            if (log.isInfoEnabled()) {
                log.info(_("isisfish.launching", config.getElapsedTimeAsString()));
            }
            
            // after init vcs and local data
            config.doAction(IsisConfig.STEP_AFTER_INIT_VCS);
            
            doNuitonMigration();

            launchUI();

            // action after ui launched
            config.doAction(IsisConfig.STEP_AFTER_UI);
            
            startCronService();
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Global IsisFish exception", e);
            }
            quit();
        } catch (LinkageError e) {
            
            if (log.isFatalEnabled()) {
                log.fatal("Linkage error detected", e);
            }
            
            // a real java.lang.Error sometimes happen when
            // using non recompiled script due to dependency changes
            // this is the better solution found :
            // remove isis build directory
            FileUtil.deleteRecursively(IsisFish.config.getCompileDirectory());
            // display message
            JOptionPane.showMessageDialog(null, _("isisfish.error.linkageerror.message"),
                    _("isisfish.error.linkageerror.title"), JOptionPane.ERROR_MESSAGE);
            // restart isis
            quit();
        }
    }

    /**
     * Start cron service (if enabled).
     */
    protected static void startCronService() {
        if (config.isPerformCron()) {
            // start cron service
            CronService cronService = new CronService();
            cronService.start();
        }
    }

    /**
     * Move user script to nuiton dependants packages.
     * 
     * Find all files containing "org.codelutin." into "org.nuiton."
     */
    protected static void doNuitonMigration() {
        if (config.isPerformMigration()) {
            // Search in a set of directories
            // Searching throw simulations/* can take a huge time...
            File[] folders = new File[]{
                    AnalysePlanStorage.getAnalysePlanDirectory(),
                    ExportStorage.getExportDirectory(),
                    FormuleStorage.getFormuleDirectory(),
                    RuleStorage.getRuleDirectory(),
                    ScriptStorage.getScriptDirectory(),
                    SensitivityStorage.getSensitivityDirectory(),
                    SensitivityExportStorage.getSensitivityExportDirectory(),
                    SimulatorStorage.getSimulatorDirectory()
            };
            
            try {
                Map<File, List<CharSequence>> filesToMigrate = new HashMap<File, List<CharSequence>>();
                for (File folder : folders) {
                    filesToMigrate.putAll(FileUtil.grep("org\\.codelutin\\.", folder, ".*\\.java", "ISO-8859-1"));
                }
    
                if (filesToMigrate != null && !filesToMigrate.isEmpty()) {
    
                    // yes by defaut (for tests)
                    // but ask user if UI launch
                    boolean migrationOption = true;
                    if (IsisFish.config.isLaunchUI()) {
                        // display a user frame
                        String migratedFiles = "";
                        String separator = "";
                        for (File fileToMigrate : filesToMigrate.keySet()) {
                            migratedFiles += separator + fileToMigrate.toString();
                            separator = "\n";
                        }
                        JLabel labelModifiedFiles = new JLabel(_("isisfish.misc.nuitonmigration"));
                        JTextArea areaModifiedFiles = new JTextArea(migratedFiles);
                        areaModifiedFiles.setEditable(false);
                        areaModifiedFiles.setAutoscrolls(true);
                        JScrollPane sp = new JScrollPane(areaModifiedFiles);
                        sp.setPreferredSize(new Dimension(500, 100)); // don't remove popup is huge
                        migrationOption = ask(new Component[] { labelModifiedFiles, sp} );
                    }
                    
                    // if migration has to be done
                    if (migrationOption) {
                        if (log.isInfoEnabled()) {
                            log.info("Starting nuiton.org migration for user scripts :");
                        }
                        
                        for (File fileToMigrate : filesToMigrate.keySet()) {
                            if (log.isInfoEnabled()) {
                                log.info(" migrate file : " + fileToMigrate.getAbsolutePath());
                            }
                            FileUtil.sed("org\\.codelutin\\.", "org.nuiton.", fileToMigrate, "ISO-8859-1");
                        }
                    }
                }
            }
            catch(IOException eee) {
                if (log.isErrorEnabled()) {
                    log.error("Can't to nuiton migration", eee);
                }
            }
        }
        else {
            if (log.isInfoEnabled()) {
                log.info(_("Skip data migration (disabled)"));
            }
        }
    }

    /**
     * Install "Nimbus" LookAndFeel if available.
     * 
     * @throws InstantiationException
     * @throws IllegalAccessException
     * @throws UnsupportedLookAndFeelException
     */
    private static void initLookAndFeel() throws InstantiationException, IllegalAccessException, UnsupportedLookAndFeelException {
        for (UIManager.LookAndFeelInfo laf : UIManager.getInstalledLookAndFeels()) {
            if ("Nimbus".equals(laf.getName())) {
                try {
                    UIManager.setLookAndFeel(laf.getClassName());
                } catch (ClassNotFoundException e) {
                    // could not fin nimbus look-and-feel
                    log.warn("Can't install nimbus");
                }
            }
        }
    }

    /**
     * Start daemon that monitor quit value, if value is true, quit isis. Ce
     * mecanisme permet de faire des actions avant de quitter l'application
     */
    static public class IsisQuitHook extends Thread {

        @Override
        public void run() {
            try {
                config.doAction(IsisConfig.STEP_BEFORE_EXIT);
            } catch (Exception eee) {
                log.info("Error in quit daemon", eee);
            }
        }
    }

    /**
     * All main in other class must call this method, parse arguments,
     * load configuration file, init language and load converter.
     * 
     * @param args main args
     * @throws Exception if any exception while build configuration
     */
    public static void init(String... args) throws Exception {

        // i18n is not inited here
        if (log.isInfoEnabled()) {
            log.info("Starting Isis-Fish with args : " + java.util.Arrays.toString(args));
            log.info("Date: " + SimpleDateFormat.getInstance().format(new java.util.Date()));
            log.info("Java version: " + System.getProperty("java.runtime.version"));
            log.info("Java VM: " + System.getProperty("java.vm.name"));
            log.info("System arch: " + System.getProperty("os.arch"));
            log.info("R_HOME: " + System.getenv("R_HOME"));
            log.info("R.type: " + System.getProperty("R.type"));
            log.info("PATH: " + System.getenv("PATH"));
        }

        // first load converter and matrixFactory
        initConvertersAndMatrixFactory();
        
        // after init shutdown hook
        Runtime.getRuntime().addShutdownHook(new IsisQuitHook());

        // parsing des options à partir des arguments passés
        config = new IsisConfig();
        config.parse(args);

        // init i18n
        I18n.init(config.getLocale());
    }

    /**
     * Veto utilise lors de l'init du vcs pour permettre a l'utilisateur de
     * refuser certaine action. S'il refuse les switchs le vcs est passe en 
     * readonly.
     */
    static class VCSActionAsker implements VetoableActionListener {

        public boolean canDoAction(VCS vcs, VCSActionEvent action, File... files) {
            boolean result = true;
            if (action == VCSActionEvent.SWITCH_PROTOCOL) {
                result = ask(_("isisfish.vcs.switchprotocol.confirm"));
                if (!result) {
                    // l'utilisateur ne souhaite pas changer de protocol, 
                    // on force le repo en read-only pour eviter les erreurs
                    vcs.setWriteable(false);
                }
            } else if (action == VCSActionEvent.SWITCH) {
                result = ask(_("isisfish.vcs.switchversion.confirm", IsisConfig.getVersion()));
                if (!result) {
                    // l'utilisateur ne souhaite pas changer de branche,
                    // on force le repo en read-only pour eviter les erreurs
                    vcs.setWriteable(false);
                }
            } else if (action == VCSActionEvent.UPDATE_REPOSITORY) {
                
                // construit une chaine plutot qu'un Arrays.toString() qui
                // est illisible
                String modifiedFiles = "";
                String separator = "";
                for(File file : files) {
                    modifiedFiles += separator + file.toString();
                    separator = "\n";
                }
                
                // FIXME maybe make a JAXX UI ?
                JLabel labelModifiedFiles = new JLabel(_("isisfish.vcs.updaterepository.confirm"));
                JTextArea areaModifiedFiles = new JTextArea(modifiedFiles);
                areaModifiedFiles.setEditable(false);
                areaModifiedFiles.setAutoscrolls(true);
                JScrollPane sp = new JScrollPane(areaModifiedFiles);
                sp.setPreferredSize(new Dimension(500, 100)); // don't remove popup is huge
                result = ask(new Component[] { labelModifiedFiles, sp} );
            }
            return result;
        }
        
    }
    
    /**
     * Permet de faire une demande a l'utilisateur. S'il repond annuler, on 
     * quit l'application
     * 
     * @param msg question to ask
     * @return true if user confirm question
     */
    protected static boolean ask(Object msg) {
        boolean result = true;
        int value = JOptionPane.showConfirmDialog(null, msg);
        if (value == JOptionPane.CANCEL_OPTION) {
            quit();
        }
        result = value == JOptionPane.YES_OPTION;
        return result;
    }

    /**
     * Switch le vcs vers VCSNone et le sauvegarde pour le prochain lancement
     */
    protected static void switchToNoneVCS() {
        log.info(_("Switch repository type to none"));
        config.setOption(IsisConfig.Option.VCS_TYPE.key, VCS.TYPE_NONE);
        config.saveForUser();
        vcs = VCSFactory.createVCS(config);
    }

    /**
     * Initialise le VCS et check s'il y a des mises à jour pour
     * prevenir l'utilisateur.
     * 
     * @throws VCSException 
     */
    static public void initVCS() throws VCSException {

        // vcs must be done is ui is enabled too
        if (config.isLaunchUI() && config.isPerformVcsUpdate()) {
            
            // init vcs
            // in graphical mode, real VCS
            vcs = VCSFactory.createVCS(config);

            VCSActionAsker asker = new VCSActionAsker();
            vcs.addVetoableActionListener(asker);

            // Si le repo local exist mais n'est pas du bon type, on renome ce repertoire
            File local = config.getDatabaseDirectory();
            if (log.isInfoEnabled()) {
                log.info(_("Check state of local repository: %s", local));
            }

            if (local.exists()) {
                if (!vcs.isValidLocalRepository()) {
                    if (log.isInfoEnabled()) {
                        log.info(_("Local repository exists but it's not valide for current vcs: %s",
                                config.getOption(VCS.VCS_TYPE)));
                    }
                    if (ask(_("isisfish.vcs.init.wrongprotocol", local))) {
                        File localBackup = new File(local.getParentFile(),
                                local.getName() + "-" +
                                new SimpleDateFormat("yyyy-mm-dd-HH-mm-ss").format(new java.util.Date()));
                        if (log.isInfoEnabled()) {
                            log.info(_("Rename data directory to %s", localBackup));
                        }
                        if (!local.renameTo(localBackup)) {
                            throw new IsisFishRuntimeException(
                                    "Can't rename local repository that don't use svn");
                        }
                    } else {
                        switchToNoneVCS();
                    }
                }
            }

            // Si le repo local n'existe pas on fait un check out complet
            if (!local.exists()) {
                if (log.isInfoEnabled()) {
                    log.info(_("Local repository don't exist"));
                }
                if (!vcs.isConnected()) {
                    ErrorHelper.showErrorDialog(_(
                            "isisfish.vcs.init.notfoundcantdownload",
                            IsisConfig.getApiVersion()), null);
                } else {
                    // Si on utilise pas le bon tag on change de tag
                    Version tag = IsisConfig.getApiVersion();
                    if (!vcs.isTag(tag)) {
                        // pas de tag pour cette version, on checkout le trunk
                        tag = null;
                    }

                    // initialise le repo local
                    vcs.checkout(tag, false);

                    // ajoute les repertoires qu'il faut
                    AnalysePlanStorage.checkout();
                    ExportStorage.checkout();
                    FormuleStorage.checkout();
                    RuleStorage.checkout();
                    ScriptStorage.checkout();
                    SimulatorStorage.checkout();
                    SensitivityStorage.checkout();
                    SensitivityExportStorage.checkout();

                    // on ne prend pas toutes les simu ni toutes les regions
                    vcs.update(new File(local, SimulationStorage.SIMULATION_PATH), false);
                    vcs.update(new File(local, RegionStorage.REGION_PATH), false);
                    try {
                        RegionStorage.checkout("DemoRegion");
                    } catch (TopiaException eee) {
                        log.warn("Can't checkout DemoRegion", eee);
                    }
                }
            }

            if (!local.exists()) {
                // arrive ici le repo devrait exister
                throw new IsisFishRuntimeException("Can't find local repository");
            }

            // on s'arrete la si on est pas connecte
            if (vcs.isConnected()) {

                // cleanup
                vcs.cleanup(null);
                
                // check protocol, user, host
                vcs.checkProtocol();

                // Suivant la version du logiciel et les versions de base disponible
                // il est possiblement obligatoire de ne plus etre sur le trunk, ou
                // de migrer sur un autre tag

                List<File> filesInClonflict = null;
                
                // si on est sur une branche, on est en developpement, on ne fait donc rien
                if (vcs.getTag().startsWith("branches")) {
                    log.info(_("Use branches, switch not needed"));
                } else {
                    // Si on utilise pas le bon tag on change de tag
                    Version tag = IsisConfig.getApiVersion();

                    if (vcs.isTag(tag)) {
                        // un tag dispo, on a donc pas la derniere version, on switch
                        filesInClonflict = vcs.setTag(tag);
                    } else {
                        // pas de tag dispo on retourne sur le trunk
                        filesInClonflict = vcs.setTag(null);
                    }
                    
                    // TODO change code is case of conflict
                    // display a beautiful frame to manage each case
                    // for now, display a simple warning message
                    // si refus de l'utilisateur, c'est null aussi
                    if (filesInClonflict != null && !filesInClonflict.isEmpty()) {
                        warnFileListDialog(_("isisfish.error.warning.title"), _("isisfish.vcs.switchtag.warningconflict"), filesInClonflict);
                    }
                }

                // check file status
                // WARNING : this do the real svn update
                filesInClonflict = vcs.checkFileStatus();
                if (filesInClonflict != null && !filesInClonflict.isEmpty()) {
                    warnFileListDialog(_("isisfish.error.warning.title"), _("isisfish.vcs.update.warningconflict"), filesInClonflict);
                }
            }
            
            // fin de l'init on supprime le vetoable du vcs
            vcs.remoteVetoableActionListener(asker);
        }
        else {
            // VCS can't be null
            // set none if ui isn't launched
            config.setOption(VCS.VCS_TYPE, "none"); // to make him happy
            vcs = VCSFactory.createVCS(config);
        }
    }

    /**
     * Display dialog with files list, and specifique label.
     * 
     * @param dialogTitle dialog title
     * @param labelTitle labelTitle
     * @param conflictFiles conflict files
     */
    protected static void warnFileListDialog(String dialogTitle, String labelTitle, List<File> conflictFiles) {
        // construit une chaine plutot qu'un Arrays.toString() qui
        // est illisible
        String conflictFilesString = "";
        String separator = "";
        for(File file : conflictFiles) {
            conflictFilesString += separator + file.toString();
            separator = "\n";
        }
        
        JLabel labelModifiedFiles = new JLabel(labelTitle);
        JTextArea areaModifiedFiles = new JTextArea(conflictFilesString);
        areaModifiedFiles.setEditable(false);
        areaModifiedFiles.setAutoscrolls(true);
        JScrollPane sp = new JScrollPane(areaModifiedFiles);
        sp.setPreferredSize(new Dimension(500, 100)); // don't remove popup is huge
        
        JOptionPane.showMessageDialog(null, new Component[] { labelModifiedFiles, sp},
                dialogTitle,
                JOptionPane.WARNING_MESSAGE);
    }

    /**
     * Initialise et lance l'interface graphique si elle a demandé a être lancée.
     */
    public static void launchUI() {
        if (config.isLaunchUI()) {
            
            // migration must be done in UI envirronement and
            // must nerver be done in caparmor
            // TODO set it here for now, ui is not displayed on caparmor
            //doNuitonMigration();
            
            // init simulater manager
            SimulationService.getService();

            // init IsisTray
            IsisTray.getInstance();
            // lauch first UI (welcomeUI)
            WelcomeUI welcome = new WelcomeUI();
            // Set to exit on close
            welcome.setDefaultCloseOperation(WelcomeUI.DO_NOTHING_ON_CLOSE);
            welcome.addWindowListener(new WindowAdapter() {
                @Override
                public void windowClosing(WindowEvent e) {
                    ((WelcomeUI)e.getSource()).close();
                }
            });
            
            try {
                InputStream imageStream = WelcomeUI.class.getResourceAsStream("/images/simulation.gif");
                Image image = ImageIO.read(imageStream);
                welcome.setIconImage(image);
            }
            catch (IOException ex) {
                if (log.isErrorEnabled()) {
                    log.error("Can't set frame icon", ex);
                }
            }
            SwingUtil.center(welcome);
            welcome.setVisible(true);
        }
        else {
            if (log.isInfoEnabled()) {
                log.info(_("isisfish.message.launchui.notlaunch"));
            }
        }
    }

    /**
     * Initialise les convertiseurs utilisé par commons-beansutils.
     * 
     * @deprecated since 3.2.0.5 duplicated with {@link ConverterUtil#getConverter(org.nuiton.topia.TopiaContext)}
     */
    protected static void initConvertersAndMatrixFactory() {

        ConvertUtils.register(new LocaleConverter(), Locale.class);

        // init converters
        ConvertUtils.register(new DateConverter(), Date.class);
        ConvertUtils.register(new MonthConverter(), Month.class);
        ConvertUtils.register(new TimeUnitConverter(), TimeUnit.class);
        ConvertUtils.register(new RangeOfValuesConverter(), RangeOfValues.class);
        // ... et inversement
        ConvertUtils.register(new StringConverter(), String.class);

        // par defaut on utilise des doubles pour les matrices
        MatrixFactory.setDefaultVectorClass(DoubleBigVector.class);
    }
    
} // IsisFish
