/*
 * #%L
 * EchoBase :: UI
 * 
 * $Id: EchoBaseApplicationContext.java 882 2013-11-04 07:45:35Z tchemit $
 * $HeadURL: https://forge.codelutin.com/svn/echobase/tags/echobase-2.5/echobase-ui/src/main/java/fr/ifremer/echobase/ui/EchoBaseApplicationContext.java $
 * %%
 * Copyright (C) 2011 Ifremer, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.ifremer.echobase.ui;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import com.google.common.io.Files;
import com.opensymphony.xwork2.ActionContext;
import fr.ifremer.echobase.config.EchoBaseConfiguration;
import fr.ifremer.echobase.entities.DriverType;
import fr.ifremer.echobase.entities.EchoBaseUser;
import fr.ifremer.echobase.entities.TopiaEchoBaseInternalPersistenceContext;
import fr.ifremer.echobase.entities.TopiaEchoBasePersistenceContext;
import fr.ifremer.echobase.persistence.EchoBaseDbMeta;
import fr.ifremer.echobase.persistence.EchoBaseEntityHelper;
import fr.ifremer.echobase.persistence.EchobaseTopiaContexts;
import fr.ifremer.echobase.services.DefaultEchoBaseServiceContext;
import fr.ifremer.echobase.services.EchoBaseServiceContext;
import fr.ifremer.echobase.services.service.UserService;
import fr.ifremer.echobase.services.service.embeddedapplication.EmbeddedApplicationService;
import fr.ifremer.echobase.services.service.workingDb.WorkingDbConfigurationService;
import fr.ird.converter.FloatConverter;
import org.apache.commons.beanutils.ConvertUtils;
import org.apache.commons.beanutils.Converter;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.log4j.LogManager;
import org.apache.log4j.PropertyConfigurator;
import org.nuiton.i18n.I18n;
import org.nuiton.i18n.init.DefaultI18nInitializer;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.util.RecursiveProperties;
import org.nuiton.util.SortedProperties;
import org.nuiton.util.converter.ConverterUtil;

import javax.servlet.ServletContext;
import java.beans.Introspector;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class EchoBaseApplicationContext {

    /** Logger. */
    private static Log log =
            LogFactory.getLog(EchoBaseApplicationContext.class);

    /** Key to store the single instance of the application context */
    private static final String APPLICATION_CONTEXT_PARAMETER =
            "echobaseApplicationContext";

    /** Set of all loggued user sessions to be close at shutdown time. */
    protected Set<EchoBaseSession> sessions;

    public static EchoBaseApplicationContext getApplicationContext(ActionContext actionContext) {
        Map<String, Object> application = actionContext.getApplication();
        EchoBaseApplicationContext result =
                (EchoBaseApplicationContext) application.get(
                        APPLICATION_CONTEXT_PARAMETER);
        return result;
    }

    public static EchoBaseApplicationContext getApplicationContext(ServletContext servletContext) {
        EchoBaseApplicationContext result =
                (EchoBaseApplicationContext) servletContext.getAttribute(
                        APPLICATION_CONTEXT_PARAMETER);
        return result;
    }

    public static void setApplicationContext(ServletContext servletContext,
                                             EchoBaseApplicationContext applicationContext) {
        servletContext.setAttribute(APPLICATION_CONTEXT_PARAMETER,
                                    applicationContext);
    }

    public static void removeApplicationContext(ServletContext servletContext) {
        servletContext.removeAttribute(APPLICATION_CONTEXT_PARAMETER);
    }

    protected EchoBaseConfiguration configuration;

    protected EchoBaseDbMeta dbMeta;

    /** Root context for the internal database. */
    protected TopiaContext internalRootContext;

    /**
     * Flag setted to true when internal db was just created (should then
     * display in ui created user password).
     *
     * @since 1.1
     */
    protected boolean defaultUsersCreated;

//    /**
//     * A simple cache of spatial data.
//     *
//     * @since 2.2
//     */
//    protected final SpatialDataCache spatialDataCache;

    public EchoBaseApplicationContext() {

//        spatialDataCache = new SpatialDataCache();
    }

    public Set<EchoBaseSession> getEchoBaseSessions() {
        return sessions;
    }

    public synchronized void registerEchoBaseSession(EchoBaseSession session) {
        Preconditions.checkNotNull(session);
        Preconditions.checkNotNull(session.getUser());
        if (sessions == null) {
            sessions = Sets.newHashSet();
        }
        if (log.isInfoEnabled()) {
            log.info("Register user session for [" +
                     session.getUser().getEmail() + "]");
        }
        sessions.add(session);
    }

    public synchronized void destroyEchoBaseSession(EchoBaseSession session) {
        Preconditions.checkNotNull(session);
        Preconditions.checkNotNull(session.getUser());
        Preconditions.checkNotNull(sessions);
        if (log.isInfoEnabled()) {
            log.info("Destroy user session for [" +
                     session.getUser().getEmail() + "]");
        }
        // remove session from active ones
        sessions.remove(session);
        // close session
        session.close();
    }

    public void init() {

        // init I18n
        DefaultI18nInitializer i18nInitializer =
                new DefaultI18nInitializer("echobase-i18n");
        i18nInitializer.setMissingKeyReturnNull(true);
        I18n.init(i18nInitializer, Locale.getDefault());

        // init converters
        Converter converter = ConverterUtil.getConverter(Float.class);
        if (converter != null) {
            ConvertUtils.deregister(Float.class);
        }
        ConvertUtils.register(new FloatConverter(), Float.class);

        // initialize configuration
        EchoBaseConfiguration configuration = new EchoBaseConfiguration();

        try {
            initLog(configuration);
        } catch (IOException e) {
            Logger.getAnonymousLogger().log(Level.ALL,
                                            "Could not init logger.", e);
        }

        if (log.isInfoEnabled()) {
            log.info(configuration.printConfig());
        }

        // initialize internal root context
        TopiaContext internalRootContext =
                EchobaseTopiaContexts.newInternalDb(
                        configuration.getInternalDbDirectory());

        setConfiguration(configuration);
        setDbMeta(EchoBaseDbMeta.newDbMeta());
        setInternalRootContext(internalRootContext);

        // create a service context
        EchoBaseServiceContext serviceContext =
                DefaultEchoBaseServiceContext.newContext(
                        Locale.getDefault(),
                        getConfiguration(),
                        getDbMeta());

        // init database (and create minimal admin user if required)
        initInternalDatabase(serviceContext);

        // extract files to library directory if required
        try {
            extractFiles(serviceContext);
        } catch (IOException e) {
            throw new TopiaException("Could not extract files (drivers + embedded war)", e);
        }
    }

    public EchoBaseConfiguration getConfiguration() {
        return configuration;
    }

    public EchoBaseDbMeta getDbMeta() {
        return dbMeta;
    }

    public TopiaContext getInternalRootContext() {
        return internalRootContext;
    }

    public boolean isDefaultUsersCreated() {
        return defaultUsersCreated;
    }

    public EchoBaseServiceContext newServiceContext(Locale locale,
                                                    TopiaContext topiaInternalContext,
                                                    TopiaContext topiaContext) {

        EchoBaseServiceContext newServiceContext =
                DefaultEchoBaseServiceContext.newContext(
                        locale,
                        configuration,
                        dbMeta);

        TopiaEchoBaseInternalPersistenceContext internalPersistenceContext =
                new TopiaEchoBaseInternalPersistenceContext(topiaInternalContext);

        newServiceContext.setEchoBaseInternalPersistenceContext(
                internalPersistenceContext);

        TopiaEchoBasePersistenceContext persistenceContext =
                new TopiaEchoBasePersistenceContext(topiaContext);

        newServiceContext.setEchoBasePersistenceContext(persistenceContext);

        return newServiceContext;
    }

    public void close() {

        try {
            if (internalRootContext != null) {
                // release internal db
                EchoBaseEntityHelper.releaseRootContext(internalRootContext);
            }
        } finally {

            try {
                // release all user sessions
                if (CollectionUtils.isNotEmpty(sessions)) {
                    for (EchoBaseSession session : sessions) {
                        destroyEchoBaseSession(session);
                    }
                }
            } finally {
                // see http://wiki.apache.org/commons/Logging/FrequentlyAskedQuestions#A_memory_leak_occurs_when_undeploying.2Fredeploying_a_webapp_that_uses_Commons_Logging._How_do_I_fix_this.3F
                ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
                LogFactory.release(contextClassLoader);

                Introspector.flushCaches();
            }
        }
    }

    public void setDbMeta(EchoBaseDbMeta dbMeta) {
        this.dbMeta = dbMeta;
    }

    public void setInternalRootContext(TopiaContext internalRootContext) {
        this.internalRootContext = internalRootContext;
    }

    public void setDefaultUsersCreated(boolean defaultUsersCreated) {
        this.defaultUsersCreated = defaultUsersCreated;
    }

    public void setConfiguration(EchoBaseConfiguration configuration) {
        this.configuration = configuration;
    }


    /**
     * Init the internal database, says :
     * <ul>
     * <li>If no schema found or if asked to updateSchema using the
     * {@code updateSchema} configuration option is on.</li>
     * <li>If no user found is db, create a administrator user
     * {@code admin/admin}</li>
     * </ul>
     */
    protected void initInternalDatabase(EchoBaseServiceContext serviceContext) throws TopiaException {

        EchoBaseConfiguration configuration = this.getConfiguration();
        Preconditions.checkNotNull(configuration);

        EchoBaseDbMeta dbMeta = getDbMeta();
        Preconditions.checkNotNull(dbMeta);

        TopiaContext rootContext = this.getInternalRootContext();
        Preconditions.checkNotNull(rootContext);

        if (configuration.isUpdateSchema()) {
            if (log.isInfoEnabled()) {
                log.info("Will update schema...");
            }
            rootContext.updateSchema();
        }

        TopiaContext tx = rootContext.beginTransaction();
        try {
            serviceContext.setEchoBaseInternalPersistenceContext(new TopiaEchoBaseInternalPersistenceContext(tx));

            UserService service = serviceContext.newService(UserService.class);

            List<EchoBaseUser> users = service.getUsers();

            if (CollectionUtils.isEmpty(users)) {

                // no users in database create the admin user
                if (log.isInfoEnabled()) {
                    log.info("No user in database, will create default " +
                             "users.");
                }

                service.createDefaultUsers();
            }

            if (configuration.isEmbedded()) {

                if (log.isInfoEnabled()) {
                    log.info("Will try to create default working db " +
                             "configuration for internal db.");
                }
                // try to create a default embedded working db configuration
                serviceContext.newService(WorkingDbConfigurationService.class).
                        createEmbeddedWorkingDbConfiguration();
            }
        } finally {
            serviceContext.setEchoBaseInternalPersistenceContext(null);
            EchoBaseEntityHelper.closeConnection(tx);
        }
    }

    protected void initLog(EchoBaseConfiguration configuration) throws IOException {

        // get log configuration
        File logFile = configuration.getLogConfigFile();

        if (!logFile.exists()) {
            EmbeddedApplicationService.copyEmbeddedBinaryFile(
                    logFile.getName(),
                    logFile.getParentFile());

            RecursiveProperties properties = new RecursiveProperties();

            BufferedReader reader = Files.newReader(logFile, Charsets.UTF_8);
            try {
                properties.load(reader);
                reader.close();
            } finally {
                IOUtils.closeQuietly(reader);
            }

            // set default log directory
            properties.setProperty(
                    "echobase.log.dir",
                    configuration.getDefaultLogDirectory().getAbsolutePath());
            // copy configuration (with substitutions)
            Properties p2 = new SortedProperties();
            for (Map.Entry<Object, Object> entry : properties.entrySet()) {
                String key = String.valueOf(entry.getKey());

                p2.setProperty(key, properties.getProperty(key));
            }
            p2.remove("echobase.log.dir");
            // write back log configuration file
            BufferedWriter writer = Files.newWriter(logFile, Charsets.UTF_8);
            try {
                p2.store(writer, "Generated by " + getClass().getName());
                writer.close();
            } finally {
                IOUtils.closeQuietly(writer);
            }
        }
        // reste logger configuration
        LogManager.resetConfiguration();
        // use generate log config file
        PropertyConfigurator.configure(logFile.getAbsolutePath());
        log = LogFactory.getLog(EchoBaseApplicationContext.class);
        if (log.isInfoEnabled()) {
            log.info("Use now logFile: " + logFile);
        }
    }

    protected void extractFiles(EchoBaseServiceContext serviceContext) throws IOException {

        EchoBaseConfiguration configuration = serviceContext.getConfiguration();

        // copy drivers
        File libDirectory = configuration.getLibDirectory();
        for (DriverType driverType : DriverType.values()) {
            String pilotFileName = driverType.getPilotFileName(configuration);

            // copy it from class-path
            if (log.isInfoEnabled()) {
                log.info("Copy embedded resource " + pilotFileName +
                         " to directory " + libDirectory);
            }
            EmbeddedApplicationService.copyEmbeddedBinaryFile(pilotFileName, libDirectory);
        }

        if (!getConfiguration().isEmbedded()) {
            // copy embedded war
            File warLocation = configuration.getWarLocation();
            File embeddedWarDirectory = warLocation.getParentFile();
            String embeddedWarFileName = warLocation.getName();

            // copy it from class-path
            if (log.isInfoEnabled()) {
                log.info("Copy embedded war " + embeddedWarFileName +
                         " to directory " + embeddedWarDirectory);
            }
            EmbeddedApplicationService.copyEmbeddedBinaryFile(embeddedWarFileName, embeddedWarDirectory);
        }
    }
}
