package fr.ifremer.echobase.ui.interceptors;

/*
 * #%L
 * EchoBase :: UI
 * $Id: EchoBaseInjectInterceptor.java 843 2013-08-20 06:07:41Z tchemit $
 * $HeadURL: https://forge.codelutin.com/svn/echobase/tags/echobase-2.4/echobase-ui/src/main/java/fr/ifremer/echobase/ui/interceptors/EchoBaseInjectInterceptor.java $
 * %%
 * Copyright (C) 2011 - 2013 Ifremer, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.opensymphony.xwork2.ActionInvocation;
import com.opensymphony.xwork2.interceptor.Interceptor;
import fr.ifremer.echobase.services.EchoBaseService;
import fr.ifremer.echobase.services.EchoBaseServiceContext;
import fr.ifremer.echobase.ui.EchoBaseApplicationContext;
import fr.ifremer.echobase.ui.EchoBaseInternalDbTransactionFilter;
import fr.ifremer.echobase.ui.EchoBaseSession;
import fr.ifremer.echobase.ui.EchoBaseWorkingDbTransactionFilter;
import fr.ifremer.echobase.ui.actions.EchoBaseActionSupport;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.util.beans.BeanUtil;

import java.beans.PropertyDescriptor;
import java.util.Locale;
import java.util.Set;

public class EchoBaseInjectInterceptor implements Interceptor {

    private static final Log log =
            LogFactory.getLog(EchoBaseInjectInterceptor.class);

    private static final long serialVersionUID = 1L;

    @Override
    public void init() {

        if (log.isInfoEnabled()) {
            log.info("init " + this);
        }

    }

    @Override
    public String intercept(ActionInvocation invocation) throws Exception {

        Object action = invocation.getAction();

        if (action instanceof EchoBaseActionSupport) {
            EchoBaseActionSupport echoBaseActionSupport = (EchoBaseActionSupport) action;

            EchoBaseSession userSession = getEchoBaseSession(invocation);

            EchoBaseServiceContext serviceContext = newServiceContext(
                    invocation,
                    echoBaseActionSupport.getLocale());

            Set<PropertyDescriptor> descriptors =
                    BeanUtil.getDescriptors(
                            action.getClass(),
                            BeanUtil.IS_WRITE_DESCRIPTOR);

            for (PropertyDescriptor propertyDescriptor : descriptors) {

                Class<?> propertyType = propertyDescriptor.getPropertyType();

                Object toInject = null;

                if (EchoBaseServiceContext.class.isAssignableFrom(propertyType)) {
                    toInject = serviceContext;
                } else if (EchoBaseService.class.isAssignableFrom(propertyType)) {

                    Class<? extends EchoBaseService> serviceClass =
                            (Class<? extends EchoBaseService>) propertyType;

                    toInject = serviceContext.newService(serviceClass);

                } else if (EchoBaseSession.class.isAssignableFrom(propertyType)) {

                    toInject = userSession;

                } else if (EchoBaseApplicationContext.class.isAssignableFrom(propertyType)) {

                    toInject = getEchoBaseApplicationContext(invocation);

                }

                if (toInject != null) {

                    if (log.isTraceEnabled()) {
                        log.trace("injecting " + toInject + " in action " + action);
                    }

                    propertyDescriptor.getWriteMethod().invoke(action, toInject);

                }
            }

            try {

                return invocation.invoke();

            } finally {

                // serviceContext.getPersistenceContext().getEntityTransaction().rollback();

            }

        } else {

            // not an action, just process

            return invocation.invoke();

        }

    }

    protected EchoBaseSession getEchoBaseSession(ActionInvocation invocation) {

        EchoBaseSession session = EchoBaseSession.getEchoBaseSession(
                invocation.getInvocationContext());

//        if (session == null) {
//
//            session = new EchoBaseSession();
//
//            invocation.getInvocationContext().getSession().put(EchoBaseSession.SESSION_PARAMETER, session);
//
//        }

        return session;

    }

    protected EchoBaseApplicationContext getEchoBaseApplicationContext(ActionInvocation invocation) {

        EchoBaseApplicationContext applicationContext =
                EchoBaseApplicationContext.getApplicationContext(
                        invocation.getInvocationContext());

        Preconditions.checkNotNull(
                "application context must be initialized before calling an action",
                applicationContext);

        return applicationContext;

    }

    protected EchoBaseServiceContext newServiceContext(ActionInvocation invocation,
                                                       Locale locale) {


        TopiaContext topiaInternalContext =
                EchoBaseInternalDbTransactionFilter.getTransaction(
                        invocation.getInvocationContext());

        TopiaContext topiaContext =
                EchoBaseWorkingDbTransactionFilter.getTransaction(
                        invocation.getInvocationContext());

        EchoBaseApplicationContext applicationContext =
                getEchoBaseApplicationContext(invocation);

        EchoBaseServiceContext serviceContext =
                applicationContext.newServiceContext(locale,
                                                     topiaInternalContext,
                                                     topiaContext);

        return serviceContext;

    }

    @Override
    public void destroy() {

        if (log.isInfoEnabled()) {
            log.info("destroy " + this);
        }

    }

}
