package fr.ifremer.coselmar.services.v1;

/*
 * #%L
 * Coselmar :: Rest Services
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2014 Ifremer, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import fr.ifremer.coselmar.beans.DocumentBean;
import fr.ifremer.coselmar.beans.UserWebToken;
import fr.ifremer.coselmar.converter.BeanEntityConverter;
import fr.ifremer.coselmar.persistence.entity.CoselmarUser;
import fr.ifremer.coselmar.persistence.entity.CoselmarUserRole;
import fr.ifremer.coselmar.persistence.entity.Document;
import fr.ifremer.coselmar.persistence.entity.Privacy;
import fr.ifremer.coselmar.services.CoselmarTechnicalException;
import fr.ifremer.coselmar.services.CoselmarWebServiceSupport;
import fr.ifremer.coselmar.services.errors.InvalidCredentialException;
import fr.ifremer.coselmar.services.errors.NoResultException;
import fr.ifremer.coselmar.services.errors.UnauthorizedException;
import fr.ifremer.coselmar.services.indexation.DocumentsIndexationService;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.logging.Log;
import org.apache.lucene.queryparser.classic.ParseException;
import org.debux.webmotion.server.call.UploadFile;
import org.debux.webmotion.server.render.Render;
import org.nuiton.topia.persistence.TopiaNoResultException;
import org.nuiton.util.DateUtil;

import static org.apache.commons.logging.LogFactory.getLog;

/**
 * @author ymartel <martel@codelutin.com>
 */
public class DocumentsWebService extends CoselmarWebServiceSupport {

    private static final Log log = getLog(DocumentsWebService.class);

    public static final List<String> DOCUMENT_EDIT_ALLOWED_USER_ROLES =
        Lists.newArrayList(CoselmarUserRole.EXPERT.name(), CoselmarUserRole.SUPERVISOR.name());

    public static final List<String> DOCUMENT_VIEW_ALLOWED_USER_ROLES =
        Lists.newArrayList(
            CoselmarUserRole.ADMIN.name(),
            CoselmarUserRole.SUPERVISOR.name(),
            CoselmarUserRole.EXPERT.name()
        );

    public DocumentBean getDocument(String documentId) throws InvalidCredentialException, UnauthorizedException {

        // Check authentication
        String authorization = getContext().getHeader("Authorization");
        UserWebToken userWebToken = checkAuthentication(authorization);

        // reconstitute full id
        String fullId = getDocumentFullId(documentId);

        Document document = getDocumentDao().forTopiaIdEquals(fullId).findUnique();

        // If document if public, only admin, supervisor and expert could view it
        // If document if private, only admin, supervisor and owner could view it
        if (!isAllowedToAccessDocument(userWebToken, document)) {

            String message = String.format("User %s %s ('%s') try to access to document '%s'",
                userWebToken.getFirstName(), userWebToken.getLastName(), userWebToken.getUserId(), documentId);
            if (log.isWarnEnabled()) {
                log.warn(message);
            }
            throw new UnauthorizedException(message);

        }

        DocumentBean documentBean = BeanEntityConverter.toBean(documentId, document);
        return documentBean;
    }

    public List<DocumentBean> getDocuments(List<String> searchKeywords) throws InvalidCredentialException {

        // Check authentication
        String authorization = getContext().getHeader("Authorization");
        UserWebToken userWebToken = checkAuthentication(authorization);

        // Retrieve current user
        String fullCurrentUserId = getFullUserIdFromShort(userWebToken.getUserId());
        CoselmarUser currentUser = getCoselmarUserDao().forTopiaIdEquals(fullCurrentUserId).findAnyOrNull();

        String currentUserRole = userWebToken.getRole().toUpperCase();

        List<Document> documentList;

        // Admin and Supervisor can see all documents (public, private and restricted)
        if (Lists.newArrayList(CoselmarUserRole.ADMIN.name(), CoselmarUserRole.SUPERVISOR.name()).contains(currentUserRole)) {
            if (searchKeywords != null && !searchKeywords.isEmpty()) {
                DocumentsIndexationService documentsIndexationService = getServicesContext().newService(DocumentsIndexationService.class);

                try {
                    List<String> documentIds = documentsIndexationService.searchDocuments(searchKeywords);
                    List<String> documentFullIds = getDocumentsFullId(documentIds);

                    documentList = getDocumentDao().forTopiaIdIn(documentFullIds).findAll();

                } catch (IOException | ParseException e) {
                    if (log.isErrorEnabled()) {
                        log.error("Unable to search by lucene, make search directly in database", e);
                    }
                    documentList = getDocumentDao().findAllContainingAllKeywords(searchKeywords);

                }
            } else {
                documentList = getDocumentDao().findAll();
            }

        } else {
            //Other can only see public, his own private and restricted for which he is allowed
            documentList = getDocumentDao().findAllFilterByUser(currentUser, searchKeywords);;
        }

        List<DocumentBean> result = new ArrayList<>(documentList.size());

        for (Document document : documentList) {
            String lightId = getPersistenceContext().getTopiaIdFactory().getRandomPart(document.getTopiaId());
            DocumentBean documentBean = BeanEntityConverter.toBean(lightId, document);
            result.add(documentBean);
        }

        return result;
    }

    public DocumentBean addDocument(DocumentBean document, UploadFile uploadFile) throws InvalidCredentialException, UnauthorizedException {

        // Check authentication
        String authorization = getContext().getHeader("Authorization");
        UserWebToken userWebToken = checkAuthentication(authorization);

        // Only Expert or Supervisor can add document
        String userRole = userWebToken.getRole();
        if (!DOCUMENT_EDIT_ALLOWED_USER_ROLES.contains(userRole.toUpperCase())) {
            String message = String.format("User %s %s ('%s') is not allowed to add document",
                userWebToken.getFirstName(), userWebToken.getLastName(), userWebToken.getUserId());
            if (log.isWarnEnabled()) {
                log.warn(message);
            }
            throw new UnauthorizedException(message);
        }

        Preconditions.checkNotNull(document);

        // retrieve user who will be assigned as document owner
        String fullId = getFullUserIdFromShort(userWebToken.getUserId());

        CoselmarUser owner;
        try {
            owner = getCoselmarUserDao().forTopiaIdEquals(fullId).findUnique();
        } catch (TopiaNoResultException tnre) {
            // Should not happened, cause user are not really deleted
            String message = String.format("Seems that logged user ('%s') does not exist anymore.", fullId);
            if (log.isErrorEnabled()) {
                log.equals(message);
            }
            throw new InvalidCredentialException(message);
        }

        String documentName = document.getName();
        String contentType = null;
        String filePath = null;

        // If document has a file, manager it !
        if (uploadFile != null) {
            Pair<String, String> pathAndContentType = managerDocumentFile(uploadFile, owner);
            filePath = pathAndContentType.getLeft();
            contentType = pathAndContentType.getRight();
        }

        // Document Metadata
        Document documentEntity = getDocumentDao().create();

        documentEntity.setOwner(owner);

        documentEntity.setName(documentName);
        documentEntity.setPrivacy(Privacy.valueOf(document.getPrivacy().toUpperCase()));
        documentEntity.addAllKeywords(document.getKeywords());

        Date depositDate = document.getDepositDate();
        if (depositDate != null) {
            documentEntity.setDepositDate(new Date(depositDate.getTime()));
        } else {
            documentEntity.setDepositDate(new Date());
        }

        documentEntity.setType(document.getType());
        documentEntity.setSummary(document.getSummary());
        documentEntity.setLanguage(document.getLanguage());
        documentEntity.setPublicationDate(document.getPublicationDate());


        // Legal / copyright part
        documentEntity.setAuthors(document.getAuthors());
        documentEntity.setCopyright(document.getCopyright());
        documentEntity.setLicense(document.getLicense());

        // Document resource part
        if (uploadFile != null) {
            documentEntity.setWithFile(true);
            documentEntity.setMimeType(contentType);
            documentEntity.setFilePath(filePath);
        } else {
            documentEntity.setWithFile(false);
        }
        documentEntity.setExternalUrl(document.getExternalUrl());

        documentEntity.setComment(document.getComment());

        commit();
        String lightId = getPersistenceContext().getTopiaIdFactory().getRandomPart(documentEntity.getTopiaId());
        DocumentBean result = BeanEntityConverter.toBean(lightId, documentEntity);

        DocumentsIndexationService documentsIndexationService = getServicesContext().newService(DocumentsIndexationService.class);
        try {
            documentsIndexationService.indexDocument(result);
            if (log.isDebugEnabled()) {
                String message = String.format("Document '%s' added to index", documentName);
                log.debug(message);
            }
        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error("Unable to index new document", e);
            }
        }

        return result;

    }

    public Render getDocumentFile(String documentId) throws NoResultException {

        // reconstitute full id
        String fullId =getDocumentFullId(documentId);

        Document document = getDocumentDao().forTopiaIdEquals(fullId).findUnique();
        //TODO ymartel 20141103 : manage file owner ?

        // Get file attached to document
        String filePath = document.getFilePath();
        if (StringUtils.isBlank(filePath)) {
            throw new NoResultException("No File");
        }
        File documentFile = new File(filePath);

        String fileName = document.getName();
        String fileMimeType = document.getMimeType();
        try {
            InputStream fileStream = new FileInputStream(documentFile);
            return renderDownload(fileStream, fileName, fileMimeType);

        } catch (FileNotFoundException e) {
            if (log.isErrorEnabled()) {
                String message = String.format("Unable to retrieve file %s", fileName);
                log.error(message);
            }
            throw new NoResultException("File does not exist");
        }

    }

    public void saveDocument(DocumentBean documentBean) {
        throw new CoselmarTechnicalException("not yet implemented");
    }

    public void deleteDocument(String documentId) throws InvalidCredentialException, UnauthorizedException {

        // Check authentication
        String authorization = getContext().getHeader("Authorization");
        UserWebToken userWebToken = checkAuthentication(authorization);

        // reconstitute full id
        String fullId = getDocumentFullId(documentId);

        Document document = getDocumentDao().forTopiaIdEquals(fullId).findUnique();

        if (!isAllowedToAccessDocument(userWebToken, document)) {

            String message = String.format("User %s %s ('%s') try to delete document '%s'",
                userWebToken.getFirstName(), userWebToken.getLastName(), userWebToken.getUserId(), documentId);
            if (log.isWarnEnabled()) {
                log.warn(message);
            }
            throw new UnauthorizedException(message);

        }

        // Delete physical file
        if (StringUtils.isNotBlank(document.getFilePath())) {
            File documentFile = new File(document.getFilePath());
            FileUtils.deleteQuietly(documentFile);
        }

        getDocumentDao().delete(document);

        commit();

        //Remove index entry
        DocumentsIndexationService documentsIndexationService = getServicesContext().newService(DocumentsIndexationService.class);
        try {
            documentsIndexationService.deleteDocument(documentId);
            if (log.isDebugEnabled()) {
                String message = String.format("Document '%s' removed from index", documentId);
                log.debug(message);
            }
        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error("Unable to remove document entry from index", e);
            }
        }
    }

    public List<String> getKeywords() throws InvalidCredentialException, UnauthorizedException {

        List<String> themes = getDocumentDao().findAllKeywords();

        return themes;
    }

    public List<String> getTypes() throws InvalidCredentialException, UnauthorizedException {

        List<String> types = getDocumentDao().findAllTypes();

        return types;
    }


    ////////////////////////////////////////////////////////////////////////////
    ///////////////////////     Internal Parts     /////////////////////////////
    ////////////////////////////////////////////////////////////////////////////

    /**
     * When a Document is sent, it could have a File part : this manage the
     * upload file. The file is stored in the user specific directory, and the
     * contentType of document is returned, cause need in Document Metadata.
     *
     * @return the upload file Metadata
     */
    protected Pair<String, String> managerDocumentFile(UploadFile uploadFile, CoselmarUser owner) {
        Preconditions.checkNotNull(uploadFile);

        // Document File
        String fileName = uploadFile.getName();
        File uploadedFile = uploadFile.getFile();
        String contentType = uploadFile.getContentType();

        if (log.isInfoEnabled()) {
            String message = String.format("File name : %s, content-type : %s", fileName, contentType);
            log.info(message);
        }

        // put the document in the good directory
        String userPath = getUserDocumentPath(owner);

        Date now = getNow();
        String formattedDay = DateUtil.formatDate(now, "yyyyMMddHHmm");
        String prefix = formattedDay + "-";

        File destFile = new File(userPath + File.separator + prefix + fileName);
        try {
            FileUtils.moveFile(uploadedFile, destFile);
        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error("error during File transfer", e);
            }
            throw new CoselmarTechnicalException("Internal error during file transfer");
        }
        return Pair.of(destFile.getAbsolutePath(), contentType);
    }

    protected String getUserDocumentPath(CoselmarUser user) {
        File dataDirectory = getCoselmarServicesConfig().getDataDirectory();
        String absolutePath = dataDirectory.getAbsolutePath();
        String userFolder = StringUtils.replaceChars(user.getFirstname() + "-" + user.getName(), " ", "_");
        String userPath = absolutePath + File.separator + userFolder;
        return userPath;
    }

    /**
     * Check if an user can access to a document metadata, according to its
     * privacy settings and the user grant.
     *
     * @param userWebToken  :   Session data for current user,
     *                      containing {@link fr.ifremer.coselmar.persistence.entity.CoselmarUserRole} as String
     * @param document  :   the document user trying to access.
     *
     * @return true is user can access, false else.
     */
    protected boolean isAllowedToAccessDocument(UserWebToken userWebToken, Document document) {
        boolean isAuthorized = false;

        String viewerRole = userWebToken.getRole().toUpperCase();

        // For public : only admin/supervisor/expert can access
        if (document.getPrivacy() == Privacy.PUBLIC) {
            isAuthorized = DOCUMENT_VIEW_ALLOWED_USER_ROLES.contains(viewerRole);

        // For Private : only admin/supervisor/owner can access
        } else if (document.getPrivacy() == Privacy.PRIVATE) {
            CoselmarUser documentOwner = document.getOwner();
            boolean isOwner = StringUtils.equals(documentOwner.getTopiaId(), getFullUserIdFromShort(userWebToken.getUserId()));
            isAuthorized = isOwner || Lists.newArrayList(CoselmarUserRole.ADMIN.name(), CoselmarUserRole.SUPERVISOR.name()).contains(viewerRole);

        } else {
            // Restricted : Not yet implemented
        }

        return isAuthorized;
    }

    protected String getDocumentFullId(String documentId) {
        return Document.class.getCanonicalName() + getPersistenceContext().getTopiaIdFactory().getSeparator() + documentId;
    }

    protected List<String> getDocumentsFullId(List<String> documentShortIds) {

        Function<String, String> getFullIds = new Function<String, String>() {
            @Override
            public String apply(String shortId) {
                return getDocumentFullId(shortId);
            }
        };

        List<String> fullIds = Lists.transform(documentShortIds, getFullIds);
        return fullIds;
    }

    protected List<Document> findAllDocuments(List<String> searchKeywords) {
        List<Document> documentList;
        if (searchKeywords != null && !searchKeywords.isEmpty()) {
            documentList = getDocumentDao().findAllContainingAllKeywords(searchKeywords);
        } else {
            documentList = getDocumentDao().findAll();
        }
        return documentList;
    }

}
