package fr.ifremer.common.synchro.service;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.*;
import fr.ifremer.common.synchro.intercept.SynchroOperationRepository;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * <p>SynchroTableOperation class.</p>
 *
 */
public class SynchroTableOperation implements SynchroOperationRepository {

	private Map<String, Map<String, Object>> missingUpdates = Maps.newHashMap();

	private Map<String, Object[]> missingUpdatesByPks = Maps.newHashMap();

	/**
	 * Entities (list of pks) to delete
	 */
	private List<List<Object>> missingDeletes = Lists.newArrayList();

	/**
	 * Entities (list of pks) to detach from server (e.g. reset the
	 * remote_id...)
	 */
	private List<List<Object>> missingDetachs = Lists.newArrayList();

	private Map<String, Map<Set<String>, List<List<Object>>>> childrenToUpdate = Maps
			.newHashMap();

	private Map<String, Map<Set<String>, List<List<Object>>>> childrenToDelete = Maps
			.newHashMap();

	private Map<String, Map<Set<String>, List<List<Object>>>> childrenToDetach = Maps
			.newHashMap();

	private final String tableName;

	private final SynchroContext context;

	/**
	 * Say if progressionModel should be increment, when processing the
	 * operation.<br>
	 * Usually, enableProgress is set to <code>true</true> for "root" tables.
	 */
	private boolean enableProgress = false;

	/**
	 * This is used to disable add into missing deletes, when a deletes has
	 * already be tried.<br>
	 * {@see SynchroServiceImpl#deleteRows()}
	 */
	private boolean allowMissingDeletes = true;

	/**
	 * <p>Constructor for SynchroTableOperation.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 */
	public SynchroTableOperation(String tableName, SynchroContext context) {
		this.tableName = tableName;
		this.context = context;
	}

	/**
	 * <p>Constructor for SynchroTableOperation.</p>
	 *
	 * @param parentTableName a {@link java.lang.String} object.
	 * @param tableName a {@link java.lang.String} object.
	 * @param filterColumName a {@link java.lang.String} object.
	 * @param filterColumValues a {@link java.util.List} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 */
	public SynchroTableOperation(String parentTableName, String tableName,
			String filterColumName, List<List<Object>> filterColumValues,
			SynchroContext context) {
		this.tableName = parentTableName;
		this.context = context;
		addChildrenToUpdateFromManyColumns(tableName,
				Sets.newHashSet(filterColumName), filterColumValues);
	}

	/**
	 * <p>Constructor for SynchroTableOperation.</p>
	 *
	 * @param parentTableName a {@link java.lang.String} object.
	 * @param tableName a {@link java.lang.String} object.
	 * @param filterColumNames a {@link java.util.Set} object.
	 * @param filterColumValues a {@link java.util.List} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 */
	public SynchroTableOperation(String parentTableName, String tableName,
			Set<String> filterColumNames, List<List<Object>> filterColumValues,
			SynchroContext context) {
		this.tableName = parentTableName;
		this.context = context;
		addChildrenToUpdateFromManyColumns(tableName, filterColumNames,
				filterColumValues);
	}

	/**
	 * <p>Getter for the field <code>tableName</code>.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getTableName() {
		return tableName;
	}

	/**
	 * <p>isAllowMissingDeletes.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isAllowMissingDeletes() {
		return allowMissingDeletes;
	}

	/**
	 * <p>Setter for the field <code>allowMissingDeletes</code>.</p>
	 *
	 * @param allowMissingDeletes a boolean.
	 */
	public void setAllowMissingDeletes(boolean allowMissingDeletes) {
		this.allowMissingDeletes = allowMissingDeletes;
	}

	/** {@inheritDoc} */
	@Override
	public SynchroContext getSynchroContext() {
		return context;
	}

	/**
	 * <p>isEnableProgress.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isEnableProgress() {
		return enableProgress;
	}

	/**
	 * <p>Setter for the field <code>enableProgress</code>.</p>
	 *
	 * @param enableProgress a boolean.
	 */
	public void setEnableProgress(boolean enableProgress) {
		this.enableProgress = enableProgress;
	}

	/** {@inheritDoc} */
	@Override
	public void addMissingColumnUpdate(String columnName, List<Object> pk,
			Object columnValue) {
		Preconditions.checkNotNull(pk);
		addMissingColumnUpdate(columnName, SynchroTableMetadata.toPkStr(pk),
				columnValue);
	}

	/** {@inheritDoc} */
	@Override
	public void addMissingColumnUpdate(String columnName, String pkStr,
			Object columnValue) {
		Preconditions.checkNotNull(pkStr);
		Map<String, Object> destRows = missingUpdates.get(columnName);
		if (destRows == null) {
			destRows = Maps.newIdentityHashMap();
			missingUpdates.put(columnName, destRows);
		}
		destRows.put(pkStr, columnValue);
	}

	/** {@inheritDoc} */
	@Override
	public void addAllMissingDelete(List<List<Object>> pks) {
		Preconditions.checkNotNull(pks);
		missingDeletes.addAll(pks);
	}

	/** {@inheritDoc} */
	@Override
	public void addMissingDelete(List<Object> pk) {
		Preconditions.checkNotNull(pk);
		missingDeletes.add(pk);
	}

	/** {@inheritDoc} */
	@Override
	public void addMissingDelete(String pkStr) {
		Preconditions.checkNotNull(pkStr);
		List<Object> pk = SynchroTableMetadata.fromPkStr(pkStr);
		missingDeletes.add(pk);
	}

	/** {@inheritDoc} */
	@Override
	public void addAllMissingDetach(List<List<Object>> pks) {
		Preconditions.checkNotNull(pks);
		missingDetachs.addAll(pks);
	}

	/** {@inheritDoc} */
	@Override
	public void addMissingDetach(List<Object> pk) {
		Preconditions.checkNotNull(pk);
		missingDetachs.add(pk);
	}

	/** {@inheritDoc} */
	@Override
	public void addMissingDetach(String pkStr) {
		Preconditions.checkNotNull(pkStr);
		List<Object> pk = SynchroTableMetadata.fromPkStr(pkStr);
		missingDetachs.add(pk);
	}

	/** {@inheritDoc} */
	@Override
	public void addAllMissingColumnUpdates(
			Map<String, Map<String, Object>> missingUpdates) {
		this.missingUpdates.putAll(missingUpdates);
	}

	/** {@inheritDoc} */
	@Override
	public void addAllMissingUpdatesByPks(
			Map<String, Object[]> missingUpdatesByPkStr) {
		this.missingUpdatesByPks.putAll(missingUpdatesByPkStr);
	}


	/** {@inheritDoc} */
	@Override
	public void addChildToUpdateFromOneColumn(String childTablename,
			String columnName, Object columnValue) {
		addChildToUpdateFromManyColumns(childTablename,
				ImmutableSet.of(columnName), ImmutableList.of(columnValue));
	}

	/** {@inheritDoc} */
	@Override
	public void addChildrenToUpdateFromOneColumn(String childTablename,
			String columnName, List<Object> columnValues) {
		List<List<Object>> values = Lists.newArrayListWithCapacity(columnValues
				.size());
		for (Object singleValue : columnValues) {
			values.add(ImmutableList.of(singleValue));
		}
		addChildrenToUpdateFromManyColumns(childTablename,
				ImmutableSet.of(columnName), values);
	}

	/** {@inheritDoc} */
	@Override
	public void addChildToUpdateFromManyColumns(String childTablename,
			Set<String> columnNames, List<Object> columnValues) {
		Map<Set<String>, List<List<Object>>> childTable = childrenToUpdate
				.get(childTablename);
		if (childTable == null) {
			childTable = Maps.newHashMap();
			childrenToUpdate.put(childTablename, childTable);
		}
		List<List<Object>> childColumnValues = childTable.get(columnNames);
		if (childColumnValues == null) {
			childColumnValues = Lists.newArrayList();
			childTable.put(columnNames, childColumnValues);
		}
		childColumnValues.add(columnValues);
	}

	/** {@inheritDoc} */
	@Override
	public void addChildrenToUpdateFromManyColumns(String childTablename,
			Set<String> columnNames, List<List<Object>> columnValues) {
		Map<Set<String>, List<List<Object>>> childTable = childrenToUpdate
				.get(childTablename);
		if (childTable == null) {
			childTable = Maps.newHashMap();
			childrenToUpdate.put(childTablename, childTable);
		}
		List<List<Object>> childColumnValues = childTable.get(columnNames);
		if (childColumnValues == null) {
			childColumnValues = Lists.newArrayList();
			childTable.put(columnNames, childColumnValues);
		}
		childColumnValues.addAll(columnValues);
	}

	/** {@inheritDoc} */
	@Override
	public void addChildToDeleteFromOneColumn(String childTablename,
			String columnName, Object columnValue) {
		addChildToDeleteFromManyColumns(childTablename,
				ImmutableSet.of(columnName), ImmutableList.of(columnValue));
	}

	/** {@inheritDoc} */
	@Override
	public void addChildrenToDeleteFromOneColumn(String childTablename,
			String columnName, List<Object> columnValues) {
		List<List<Object>> values = Lists.newArrayListWithCapacity(columnValues
				.size());
		for (Object singleValue : columnValues) {
			values.add(ImmutableList.of(singleValue));
		}
		addChildrenToDeleteFromManyColumns(childTablename,
				ImmutableSet.of(columnName), values);
	}

	/** {@inheritDoc} */
	@Override
	public void addChildToDeleteFromManyColumns(String childTablename,
			Set<String> columnNames, List<Object> columnValue) {
		Map<Set<String>, List<List<Object>>> childTable = childrenToDelete
				.get(childTablename);
		if (childTable == null) {
			childTable = Maps.newHashMap();
			childrenToDelete.put(childTablename, childTable);
		}
		List<List<Object>> childColumnValues = childTable.get(columnNames);
		if (childColumnValues == null) {
			childColumnValues = Lists.newArrayList();
			childTable.put(columnNames, childColumnValues);
		}
		childColumnValues.add(columnValue);
	}

	/** {@inheritDoc} */
	@Override
	public void addChildrenToDeleteFromManyColumns(String childTablename,
			Set<String> columnNames, List<List<Object>> columnValues) {
		Map<Set<String>, List<List<Object>>> childTable = childrenToDelete
				.get(childTablename);
		if (childTable == null) {
			childTable = Maps.newHashMap();
			childrenToDelete.put(childTablename, childTable);
		}
		List<List<Object>> childColumnValues = childTable.get(columnNames);
		if (childColumnValues == null) {
			childColumnValues = Lists.newArrayList();
			childTable.put(columnNames, childColumnValues);
		}
		childColumnValues.addAll(columnValues);
	}

	/** {@inheritDoc} */
	@Override
	public void addChildToDetachFromOneColumn(String childTablename,
			String columnName, Object columnValue) {
		addChildToDetachFromManyColumn(childTablename,
				ImmutableSet.of(columnName), ImmutableList.of(columnValue));
	}

	/** {@inheritDoc} */
	@Override
	public void addChildToDetachFromManyColumn(String childTablename,
			Set<String> columnNames, List<Object> columnValue) {
		Map<Set<String>, List<List<Object>>> childTable = childrenToDetach
				.get(childTablename);
		if (childTable == null) {
			childTable = Maps.newHashMap();
			childrenToDetach.put(childTablename, childTable);
		}
		List<List<Object>> childColumnValues = childTable.get(columnNames);
		if (childColumnValues == null) {
			childColumnValues = Lists.newArrayList();
			childTable.put(columnNames, childColumnValues);
		}
		childColumnValues.add(columnValue);
	}

	/** {@inheritDoc} */
	@Override
	public void addChildrenToDetachFromManyColumns(String childTablename,
			Set<String> columnNames, List<List<Object>> columnValues) {
		Map<Set<String>, List<List<Object>>> childTable = childrenToDetach
				.get(childTablename);
		if (childTable == null) {
			childTable = Maps.newHashMap();
			childrenToDetach.put(childTablename, childTable);
		}
		List<List<Object>> childColumnValues = childTable.get(columnNames);
		if (childColumnValues == null) {
			childColumnValues = Lists.newArrayList();
			childTable.put(columnNames, childColumnValues);
		}
		childColumnValues.addAll(columnValues);
	}

	/**
	 * <p>Getter for the field <code>childrenToUpdate</code>.</p>
	 *
	 * @return a {@link java.util.Map} object.
	 */
	public Map<String, Map<Set<String>, List<List<Object>>>> getChildrenToUpdate() {
		return ImmutableMap.copyOf(childrenToUpdate);
	}

	/**
	 * <p>hasChildrenToUpdate.</p>
	 *
	 * @return a boolean.
	 */
	public boolean hasChildrenToUpdate() {
		return MapUtils.isNotEmpty(childrenToUpdate);
	}

	/**
	 * <p>Getter for the field <code>childrenToDelete</code>.</p>
	 *
	 * @return a {@link java.util.Map} object.
	 */
	public Map<String, Map<Set<String>, List<List<Object>>>> getChildrenToDelete() {
		return ImmutableMap.copyOf(childrenToDelete);
	}

	/**
	 * <p>Getter for the field <code>childrenToDetach</code>.</p>
	 *
	 * @return a {@link java.util.Map} object.
	 */
	public Map<String, Map<Set<String>, List<List<Object>>>> getChildrenToDetach() {
		return ImmutableMap.copyOf(childrenToDetach);
	}

	/**
	 * <p>hasChildrenToDelete.</p>
	 *
	 * @return a boolean.
	 */
	public boolean hasChildrenToDelete() {
		return MapUtils.isNotEmpty(childrenToDelete);
	}

	/**
	 * <p>hasChildrenToDetach.</p>
	 *
	 * @return a boolean.
	 */
	public boolean hasChildrenToDetach() {
		return MapUtils.isNotEmpty(childrenToDetach);
	}

	/**
	 * <p>Getter for the field <code>missingUpdates</code>.</p>
	 *
	 * @return a {@link java.util.Map} object.
	 */
	public Map<String, Map<String, Object>> getMissingUpdates() {
		return ImmutableMap.copyOf(missingUpdates);
	}

	/**
	 * <p>Getter for the field <code>missingUpdatesByPks</code>.</p>
	 *
	 * @return a {@link java.util.Map} object.
	 */
	public Map<String, Object[]> getMissingUpdatesByPks() {
		return ImmutableMap.copyOf(missingUpdatesByPks);
	}

	/**
	 * <p>Getter for the field <code>missingDeletes</code>.</p>
	 *
	 * @return a {@link java.util.List} object.
	 */
	public List<List<Object>> getMissingDeletes() {
		return ImmutableList.copyOf(missingDeletes);
	}

	/**
	 * <p>Getter for the field <code>missingDetachs</code>.</p>
	 *
	 * @return a {@link java.util.List} object.
	 */
	public List<List<Object>> getMissingDetachs() {
		return ImmutableList.copyOf(missingDetachs);
	}

	/**
	 * <p>isEmpty.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isEmpty() {
		return MapUtils.isEmpty(missingUpdates)
				&& MapUtils.isEmpty(missingUpdatesByPks)
				&& CollectionUtils.isEmpty(missingDeletes)
				&& CollectionUtils.isEmpty(missingDetachs)
				&& MapUtils.isEmpty(childrenToUpdate)
				&& MapUtils.isEmpty(childrenToDelete)
				&& MapUtils.isEmpty(childrenToDetach);
	}

	/**
	 * <p>clearMissingUpdates.</p>
	 */
	public void clearMissingUpdates() {
		missingUpdates.clear();
	}

	/**
	 * <p>clearMissingUpdatesByPks.</p>
	 */
	public void clearMissingUpdatesByPks() {
		missingUpdatesByPks.clear();
	}

	/**
	 * <p>clearMissingDeletes.</p>
	 */
	public void clearMissingDeletes() {
		missingDeletes.clear();
	}

	/**
	 * <p>clearMissingDetachs.</p>
	 */
	public void clearMissingDetachs() {
		missingDetachs.clear();
	}

	/**
	 * <p>clearChildrenToUpdate.</p>
	 */
	public void clearChildrenToUpdate() {
		childrenToUpdate.clear();
	}

	/**
	 * <p>clearChildrenToDelete.</p>
	 */
	public void clearChildrenToDelete() {
		childrenToDelete.clear();
	}

	/**
	 * <p>clearChildrenToDetach.</p>
	 */
	public void clearChildrenToDetach() {
		childrenToDetach.clear();
	}
}
