package fr.ifremer.common.synchro.service;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.*;
import fr.ifremer.common.synchro.SynchroTechnicalException;
import fr.ifremer.common.synchro.config.SynchroConfiguration;
import fr.ifremer.common.synchro.dao.*;
import fr.ifremer.common.synchro.intercept.SynchroBadUpdateDateRowException;
import fr.ifremer.common.synchro.intercept.SynchroDeletedRowException;
import fr.ifremer.common.synchro.intercept.SynchroDuplicateRowException;
import fr.ifremer.common.synchro.intercept.SynchroRejectRowException;
import fr.ifremer.common.synchro.meta.*;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata.DuplicateKeyStrategy;
import fr.ifremer.common.synchro.type.ProgressionModel;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.cfg.Environment;
import org.nuiton.util.TimeLog;
import org.springframework.dao.DataRetrievalFailureException;
import org.springframework.jdbc.datasource.DataSourceUtils;
import org.springframework.transaction.interceptor.TransactionInterceptor;
import org.springframework.transaction.support.TransactionSynchronizationAdapter;
import org.springframework.transaction.support.TransactionSynchronizationManager;

import javax.sql.DataSource;
import java.io.File;
import java.sql.*;
import java.util.*;
import java.util.Date;
import java.util.Map.Entry;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import static org.nuiton.i18n.I18n.t;

/**
 * <p>SynchroServiceImpl class.</p>
 *
 */
public class SynchroServiceImpl implements SynchroService {

	/** Logger. */
	private static final Log log = LogFactory.getLog(SynchroServiceImpl.class);

	/** Constant <code>TIME</code> */
	protected static final TimeLog TIME = new TimeLog(SynchroServiceImpl.class);

	/* A timer, with longer time (FK check are always long !) */
	/** Constant <code>TIME_DELETION</code> */
	protected static final TimeLog TIME_DELETION = new TimeLog(
			SynchroServiceImpl.class, 3000l * 1000000l /* =3s */,
			5000l * 1000000l /* =5s */);

	protected final SynchroConfiguration config;

	protected final int MAX_ROW_COUNT_FOR_PK_PRELOADING = 50000;

	protected final int batchSize;

	protected final boolean debug;

	protected final DataSource dataSource;

	private final boolean disableIntegrityConstraints;

	private final boolean allowMissingOptionalColumn;

	private final boolean allowAdditionalMandatoryColumnInSourceSchema;

	private final boolean keepWhereClauseOnQueriesByFks;

	private int daoCacheSize = -1;

	private int statementCacheSize = -1;

	private final LoadingCache<SynchroContext, Map<String, Object>> defaultBindingCache;

	/**
	 * <p>Constructor for SynchroServiceImpl.</p>
	 *
	 * @param dataSource a {@link javax.sql.DataSource} object.
	 * @param config a {@link fr.ifremer.common.synchro.config.SynchroConfiguration} object.
	 * @param disableIntegrityConstraints a boolean.
	 * @param allowMissingOptionalColumn a boolean.
	 * @param allowAdditionalMandatoryColumnInSourceSchema a boolean.
	 * @param keepWhereClauseOnQueriesByFks a boolean.
	 */
	public SynchroServiceImpl(DataSource dataSource,
			SynchroConfiguration config, boolean disableIntegrityConstraints,
			boolean allowMissingOptionalColumn,
			boolean allowAdditionalMandatoryColumnInSourceSchema,
			boolean keepWhereClauseOnQueriesByFks) {
		Preconditions.checkNotNull(config);

		this.dataSource = dataSource;
		this.config = config;
		this.batchSize = config.getImportJdbcBatchSize();
		this.defaultBindingCache = initBindingCache(5);
		this.disableIntegrityConstraints = disableIntegrityConstraints;
		this.allowMissingOptionalColumn = allowMissingOptionalColumn;
		this.allowAdditionalMandatoryColumnInSourceSchema = allowAdditionalMandatoryColumnInSourceSchema;
		this.keepWhereClauseOnQueriesByFks = keepWhereClauseOnQueriesByFks;
		this.debug = log.isTraceEnabled();
	}

	/**
	 * <p>Constructor for SynchroServiceImpl.</p>
	 *
	 * @param couldDisableIntegrityConstraints a boolean.
	 * @param allowMissingOptionalColumn a boolean.
	 * @param allowAdditionalMandatoryColumnInSourceSchema a boolean.
	 * @param keepWhereClauseOnQueriesByFks a boolean.
	 */
	public SynchroServiceImpl(boolean couldDisableIntegrityConstraints,
			boolean allowMissingOptionalColumn,
			boolean allowAdditionalMandatoryColumnInSourceSchema,
			boolean keepWhereClauseOnQueriesByFks) {
		this(null /* no dataSource */, SynchroConfiguration.getInstance(),
				couldDisableIntegrityConstraints, allowMissingOptionalColumn,
				allowAdditionalMandatoryColumnInSourceSchema,
				keepWhereClauseOnQueriesByFks);
	}

	/** {@inheritDoc} */
	public SynchroContext createSynchroContext(File sourceDbDirectory,
			Set<String> tableToIncludes) {

		String dbName = config.getDbName();
		Properties targetConnectionProperties = config
				.getConnectionProperties();

		Properties sourceConnectionProperties = new Properties();
		sourceConnectionProperties.putAll(targetConnectionProperties);
		sourceConnectionProperties.setProperty(Environment.URL,
				Daos.getJdbcUrl(sourceDbDirectory, dbName));

		return this.createSynchroContext(sourceConnectionProperties,
				tableToIncludes);
	}

	/** {@inheritDoc} */
	public SynchroContext createSynchroContext(
			Properties sourceConnectionProperties, Set<String> tableToIncludes) {
		Preconditions.checkNotNull(sourceConnectionProperties);

		Properties targetConnectionProperties = config
				.getConnectionProperties();

		SynchroContext context = SynchroContext.newContext(tableToIncludes,
				sourceConnectionProperties, targetConnectionProperties,
				new SynchroResult());

		// Apply some constructor options
		context.getTarget().setKeepWhereClauseOnQueriesByFks(
				keepWhereClauseOnQueriesByFks);
		context.getSource().setKeepWhereClauseOnQueriesByFks(
				keepWhereClauseOnQueriesByFks);

		return context;
	}

	/** {@inheritDoc} */
	@Override
	public void prepare(SynchroContext context) {
		Preconditions.checkNotNull(context);
		if (log.isDebugEnabled()) {
			log.debug("Preparing for synchronization - " + context.toString());
		}

		SynchroDatabaseConfiguration source = context.getSource();
		Preconditions.checkNotNull(source);
		source.setReadOnly(true);

		SynchroDatabaseConfiguration target = context.getTarget();
		Preconditions.checkNotNull(target);
		target.setReadOnly(true);

		// Make sure one DB has metadata enable
		boolean useTargetMetadata = target.isFullMetadataEnable();
		source.setFullMetadataEnable(!useTargetMetadata);

		Set<String> tableToIncludes = context.getTableNames();
		if (CollectionUtils.isEmpty(tableToIncludes)) {
			log.info(t("synchro.prepare.noTableFilter"));
		}

		SynchroResult result = context.getResult();
		Preconditions.checkNotNull(result);

		result.setLocalUrl(target.getJdbcUrl());
		result.setRemoteUrl(source.getJdbcUrl());

		Connection targetConnection = null;
		Connection sourceConnection = null;
		DaoFactory sourceDaoFactory = null;
		DaoFactory targetDaoFactory = null;
		SynchroDatabaseMetadata dbMeta = null;

		try {

			ProgressionModel progressionModel = result.getProgressionModel();

			// create target connection
			progressionModel.setMessage(t("synchro.prepare.step1"));
			targetConnection = createConnection(target);

			// create source Connection
			sourceConnection = createConnection(source);

			// load metas and check it
			{
				// load metas
				progressionModel.setMessage(t("synchro.prepare.step2"));
				SynchroDatabaseMetadata targetMeta = loadDatabaseMetadata(
						targetConnection, target, tableToIncludes);
				SynchroDatabaseMetadata sourceMeta = loadDatabaseMetadata(
						sourceConnection, source, tableToIncludes);

				progressionModel.setMessage(t("synchro.prepare.step3"));

				// check schema
				checkSchemasAndExcludeMissingColumns(source, target,
						sourceMeta, targetMeta, allowMissingOptionalColumn,
						allowAdditionalMandatoryColumnInSourceSchema, result);
				if (!result.isSuccess()) {
					return;
				}

				// Retrieve the meta to use
				dbMeta = useTargetMetadata ? targetMeta : sourceMeta;
				if (useTargetMetadata) {
					closeSilently(sourceMeta);
				} else {
					closeSilently(targetMeta);
				}
			}

			// Create DAO factories
			sourceDaoFactory = newDaoFactory(sourceConnection, source, dbMeta);
			targetDaoFactory = newDaoFactory(targetConnection, target, dbMeta);

			// Clean temp query parameter table
			sourceDaoFactory.getDao().cleanTempQueryParameter();
			targetDaoFactory.getDao().cleanTempQueryParameter();

			// prepare model (compute update date, count rows to update,...)
			Set<String> rootTableNames = dbMeta.getLoadedRootTableNames();
			if (CollectionUtils.isEmpty(rootTableNames)) {
				log.warn(t("synchro.prepare.noRootTable"));
			}

			for (String tableName : rootTableNames) {

				long t0 = TimeLog.getTime();

				progressionModel.setMessage(t("synchro.prepare.step4",
						tableName));

				SynchroTableMetadata table = dbMeta.getLoadedTable(tableName);

				if (log.isDebugEnabled()) {
					log.debug("Prepare table: " + tableName);
				}
				prepareRootTable(sourceDaoFactory, targetDaoFactory, table,
						context, result);

				TIME.log(t0, "prepare table " + tableName);
			}

			long totalRows = result.getTotalRows();
			if (log.isInfoEnabled()) {
				log.info("Total root rows to update: " + totalRows);
			}

			// Do a rollback at end (only if need - cf mantis #26388)
			rollbackIfNewTransaction(targetConnection);

		} catch (SQLException e) {
			rollbackSilently(targetConnection);
			result.setError(e);

		} finally {
			IOUtils.closeQuietly(sourceDaoFactory);
			IOUtils.closeQuietly(targetDaoFactory);
			closeSilently(dbMeta);
			closeSilently(sourceConnection);
			closeSilently(targetConnection);
			releaseContext(context);
		}
	}

	/** {@inheritDoc} */
	@Override
	public void synchronize(SynchroContext context) {
		Preconditions.checkNotNull(context);
		if (log.isDebugEnabled()) {
			log.debug("Starting synchronization - " + context.toString());
		}

		SynchroDatabaseConfiguration source = context.getSource();
		Preconditions.checkNotNull(source);
		source.setReadOnly(true); // no writes in the source db

		SynchroDatabaseConfiguration target = context.getTarget();
		Preconditions.checkNotNull(target);
		target.setReadOnly(false); // enable writes on target

		// Make sure one DB has metadata enable
		boolean useTargetMetadata = target.isFullMetadataEnable();
		source.setFullMetadataEnable(!useTargetMetadata);

		Set<String> tableNames = context.getTableNames();

		SynchroResult result = context.getResult();
		Preconditions.checkNotNull(result);

		Connection targetConnection = null;
		Connection sourceConnection = null;
		DaoFactory sourceDaoFactory = null;
		DaoFactory targetDaoFactory = null;
		SynchroDatabaseMetadata dbMeta = null;

		try {

			// create source Connection
			sourceConnection = createConnection(source);

			// create target connection
			targetConnection = createConnection(target);

			// load metas
			if (useTargetMetadata) {
				log.debug("Loading target database metadata...");
				dbMeta = loadDatabaseMetadata(targetConnection, target,
						tableNames);
			} else {
				log.debug("Loading source database metadata...");
				dbMeta = loadDatabaseMetadata(sourceConnection, source,
						tableNames);
			}

			// Create DAO factories
			sourceDaoFactory = newDaoFactory(sourceConnection, source, dbMeta);
			targetDaoFactory = newDaoFactory(targetConnection, target, dbMeta);

			// set total in progression model
			ProgressionModel progressionModel = result.getProgressionModel();
			progressionModel.setTotal(result.getTotalRows() + 1);
			progressionModel.setCurrent(0);
			progressionModel.setMessage(t("synchro.synchronize.step0"));

			// prepare target connection (e.g. disable constraints if need)
			prepareConnection(targetConnection, context);

			// Clean temp query parameter table
			sourceDaoFactory.getDao().cleanTempQueryParameter();
			targetDaoFactory.getDao().cleanTempQueryParameter();

			// Create a operation stack with all root tables
			Deque<SynchroTableOperation> pendingOperations = Queues
					.newArrayDeque(getRootOperations(sourceDaoFactory,
							targetDaoFactory, dbMeta, context));
			progressionModel.increments(1);

			// Process table operation, while some exists
			while (!pendingOperations.isEmpty()) {
				// process the first operation (on top of the stack)
				SynchroTableOperation operation = pendingOperations.pop();

				// Execute synchronization operation
				synchronizeOperation(operation, dbMeta, sourceDaoFactory,
						targetDaoFactory, context, result, pendingOperations);
			}

			if (log.isInfoEnabled()) {
				long totalInserts = result.getTotalInserts();
				long totalUpdates = result.getTotalUpdates();
				long totalDeletes = result.getTotalDeletes();
				long totalRejects = result.getTotalRejects();
				log.info("Total root rows to treat: " + result.getTotalRows());
				log.info("Total rows inserted: " + totalInserts);
				log.info("Total rows  updated: " + totalUpdates);
				log.info("Total rows  deleted: " + totalDeletes);
				log.info("Total rows rejected: " + totalRejects);
				log.info("Total rows  treated: "
						+ (totalInserts + totalUpdates + totalDeletes + totalRejects));
				if (totalRejects > 0) {
					log.warn(String.format(
							"Some rows has been rejected (%s rows)",
							totalRejects));
				}
			}

			progressionModel.setMessage(t("synchro.synchronize.step2"));
			progressionModel.setCurrent(progressionModel.getTotal());

			// Rollback if need
			if (target.isReadOnly() || target.isRollbackOnly()) {
				rollback(targetConnection);
				// Release connection (e.g. restore integrity constraints)
				releaseConnection(targetConnection, context);
			}

			// Or commit
			else {

				// Release connection (e.g. restore integrity constraints)
				releaseConnection(targetConnection, context);

				// Final commit
				commit(targetConnection);
			}

		} catch (Exception e) {
			rollbackSilently(targetConnection);

			// Release the connection
			// WARN : must be done AFTER the rollback (because enable integrity
			// constraints
			// will do a commit, on HsqlDB 2.3.x
			releaseConnectionSilently(targetConnection, context);

			updateResultOnSynchronizeError(e, result);
			if (log.isDebugEnabled()) {
				log.debug(e);
			}
		} finally {
			IOUtils.closeQuietly(sourceDaoFactory);
			IOUtils.closeQuietly(targetDaoFactory);
			closeSilently(dbMeta);
			closeSilently(sourceConnection);
			closeSilently(targetConnection);
		}
	}

	/** {@inheritDoc} */
	@Override
	public Timestamp getSourceLastUpdateDate(SynchroContext context) {
		if (log.isInfoEnabled()) {
			log.info("Read max(update_date) on referential tables...");
		}

		SynchroDatabaseConfiguration source = context.getSource();
		source.setFullMetadataEnable(true);
		source.setReadOnly(true);

		Set<String> tableNames = context.getTableNames();
		if (CollectionUtils.isEmpty(tableNames)) {
			log.info(t("synchro.prepare.noTableFilter"));
		}

		// set total in progression model
		Preconditions.checkNotNull(context.getResult());
		ProgressionModel progressionModel = context.getResult()
				.getProgressionModel();
		progressionModel.setTotal(tableNames.size());

		Connection connection = null;
		SynchroDatabaseMetadata dbMeta = null;
		try {

			progressionModel
					.setMessage(t("synchro.referential.lastUpdateDate.step1"));
			if (log.isDebugEnabled()) {
				log.debug(t("synchro.referential.lastUpdateDate.step1"));
			}

			long t0 = TimeLog.getTime();

			// create source Connection
			connection = createConnection(source);

			// load metas
			dbMeta = loadDatabaseMetadata(connection, source, tableNames);

			// Update tableNames with table found, and set progression model
			// total
			Set<String> rootTableNames = dbMeta.getLoadedRootTableNames();
			progressionModel.setTotal(rootTableNames.size());

			Timestamp result = null;

			// For each table
			for (String tableName : rootTableNames) {

				progressionModel.setMessage(t(
						"synchro.referential.lastUpdateDate.step2",
						tableName));
				if (log.isDebugEnabled()) {
					log.debug(t(
							"synchro.referential.lastUpdateDate.step2",
							tableName));
				}

				SynchroTableMetadata table = dbMeta.getTable(tableName);

				// get last updateDate used by db
				Timestamp updateDate = SynchroTableDaoUtils.getLastUpdateDate(
						table, connection);

				if (Daos.compareUpdateDates(result, updateDate) < 0) {
					result = updateDate;
				}

				progressionModel.increments(1);
			}

			if (log.isInfoEnabled()) {
				log.info(String.format(
						"Read last update_date on referential tables [%s] ",
						result));
				TIME.log(t0, "Read last update_date on referential tables");
			}

			return result;

		} catch (Exception e) {
			log.error(
					"Error while reading last update_date on referential tables",
					e);
			throw new DataRetrievalFailureException(
					"Error while reading last update_date on referential tables",
					e);
		} finally {
			closeSilently(dbMeta);
			closeSilently(connection);
		}
	}

	/** {@inheritDoc} */
	@Override
	public void finish(SynchroContext synchroContext,
			SynchroResult resultWithPendingOperation,
			Map<RejectedRow.Cause, RejectedRow.ResolveStrategy> rejectStrategies) {
		Preconditions.checkNotNull(synchroContext);
		Preconditions.checkNotNull(synchroContext.getResult());
		Preconditions.checkNotNull(resultWithPendingOperation);
		Preconditions
				.checkArgument(synchroContext.getResult() != resultWithPendingOperation);
		Preconditions.checkArgument(synchroContext.getSource() == null,
				"synchroContext.source is not need for finish()");
		Preconditions.checkArgument(MapUtils.isNotEmpty(rejectStrategies));

		if (log.isDebugEnabled()) {
			log.debug(String
					.format("Finish (execute missing updates + rejects resolution with strategies %s): %s",
							rejectStrategies, synchroContext.toString()));
		}

		SynchroResult result = synchroContext.getResult();
		ProgressionModel progressionModel = result.getProgressionModel();

		// Configure the target db configuration
		SynchroDatabaseConfiguration target = synchroContext.getTarget();
		target.setReadOnly(false);
		target.setFullMetadataEnable(true);
		target.setIsTarget(true);

		Set<String> tableNames = synchroContext.getTableNames();
		if (CollectionUtils.isEmpty(tableNames)) {
			log.info(t("synchro.prepare.noTableFilter"));
		}

		Connection targetConnection = null;
		DaoFactory targetDaoFactory = null;

		try {
			// create source Connection
			targetConnection = createConnection(target);

			// load metas
			SynchroDatabaseMetadata dbMeta = loadDatabaseMetadata(
					targetConnection, target, tableNames);

			// Create dao factory
			targetDaoFactory = newDaoFactory(targetConnection, target, dbMeta);

			// Create a operation stack with all operations
			Deque<SynchroTableOperation> pendingOperations = Queues
					.newArrayDeque(getSourceMissingOperations(
							resultWithPendingOperation, synchroContext));

			// Resolve rejects (will add operations to queue if necessary)
			if (resultWithPendingOperation.getRejectedRows() != null
					&& !resultWithPendingOperation.getRejectedRows().isEmpty()) {
				resolveRejects(targetConnection, dbMeta, targetDaoFactory,
						synchroContext,
						resultWithPendingOperation.getRejectedRows(),
						rejectStrategies, pendingOperations);
			}

			// Compute the progression total
			progressionModel.setCurrent(0);
			progressionModel.setTotal(pendingOperations.size());

			// Process table operation, while some exists
			while (!pendingOperations.isEmpty()) {
				// process the first operation (on top of the stack)
				SynchroTableOperation operation = pendingOperations.pop();

				// Execute synchronization operation
				synchronizeOperation(operation, dbMeta, null, targetDaoFactory,
						synchroContext, result, pendingOperations);

			}

			// Final commit
			commit(targetConnection);

		} catch (Exception e) {
			rollbackSilently(targetConnection);
			result.setError(e);
		} finally {
			IOUtils.closeQuietly(targetDaoFactory);
			closeSilently(targetConnection);
		}

	}

	/**
	 * <p>Setter for the field <code>daoCacheSize</code>.</p>
	 *
	 * @param daoFactoryCacheSize a int.
	 */
	protected void setDaoCacheSize(int daoFactoryCacheSize) {
		this.daoCacheSize = daoFactoryCacheSize;
	}

	/**
	 * <p>Setter for the field <code>statementCacheSize</code>.</p>
	 *
	 * @param daoFactoryStatementCacheSize a int.
	 */
	protected void setStatementCacheSize(int daoFactoryStatementCacheSize) {
		this.statementCacheSize = daoFactoryStatementCacheSize;
	}

	/* -- Internal methods -- */

	/**
	 * <p>checkSchemasAndExcludeMissingColumns.</p>
	 *
	 * @param source a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 * @param target a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 * @param sourceMeta a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 * @param targetMeta a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 * @param allowMissingOptionalColumn a boolean.
	 * @param allowAdditionalMandatoryColumnInSourceSchema a boolean.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 */
	protected void checkSchemasAndExcludeMissingColumns(
			SynchroDatabaseConfiguration source,
			SynchroDatabaseConfiguration target,
			SynchroDatabaseMetadata sourceMeta,
			SynchroDatabaseMetadata targetMeta,
			boolean allowMissingOptionalColumn,
			boolean allowAdditionalMandatoryColumnInSourceSchema,
			SynchroResult result) {
		try {
			Set<String> columnExcludes = SynchroMetadataUtils.checkSchemas(
					sourceMeta, targetMeta, allowMissingOptionalColumn,
					allowAdditionalMandatoryColumnInSourceSchema);

			// Add column to exclude to configuration
			if (CollectionUtils.isNotEmpty(columnExcludes)) {
				if (log.isDebugEnabled()) {
					log.debug(String
							.format("Some missing optional columns will be skipped: %s",
									columnExcludes));
				}
				source.addColumnExcludes(columnExcludes);
				target.addColumnExcludes(columnExcludes);
			}

		} catch (SynchroTechnicalException e) {
			result.setError(e);
		} catch (SynchroSchemaValidationException e) {
			log.error(e.getMessage());
			result.setError(e);
		}
	}

	/**
	 * <p>reportProgress.</p>
	 *
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @param dao a {@link fr.ifremer.common.synchro.dao.SynchroTableDao} object.
	 * @param countR a int.
	 * @param tablePrefix a {@link java.lang.String} object.
	 */
	protected void reportProgress(SynchroResult result, SynchroTableDao dao,
			int countR, String tablePrefix) {
		if (dao.getCurrentOperation().isEnableProgress()) {
			if (countR % batchSize == 0) {
				result.getProgressionModel().increments(batchSize);
			}
		}

		if (countR % (batchSize * 10) == 0) {
			if (log.isInfoEnabled()) {
				log.info(String.format(
						"%s Done: %s (inserts: %s, updates: %s deletes: %s)",
						tablePrefix, countR, dao.getInsertCount(),
						dao.getUpdateCount(), dao.getDeleteCount()));
			}
		}
	}

	/**
	 * <p>createConnection.</p>
	 *
	 * @param databaseConfiguration a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 * @return a {@link java.sql.Connection} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected Connection createConnection(
			SynchroDatabaseConfiguration databaseConfiguration)
			throws SQLException {
		return createConnection(databaseConfiguration.getJdbcUrl(),
				databaseConfiguration.getJdbcUser(),
				databaseConfiguration.getJdbcPassword());
	}

	/**
	 * <p>createConnection.</p>
	 *
	 * @param jdbcUrl a {@link java.lang.String} object.
	 * @param user a {@link java.lang.String} object.
	 * @param password a {@link java.lang.String} object.
	 * @return a {@link java.sql.Connection} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected Connection createConnection(String jdbcUrl, String user,
			String password) throws SQLException {
		Preconditions.checkArgument(StringUtils.isNotBlank(jdbcUrl));

		// If same URL as datasource, use the dataSource
		Connection connection;
		if (isManagedByDataSource(jdbcUrl) && this.dataSource != null) {
			connection = DataSourceUtils.getConnection(this.dataSource);
		} else {
			connection = DriverManager.getConnection(jdbcUrl, user, password);
		}
		connection.setAutoCommit(false);
		return connection;
	}

	/**
	 * <p>closeSilently.</p>
	 *
	 * @param connection a {@link java.sql.Connection} object.
	 */
	protected void closeSilently(Connection connection) {
		if (connection == null) {
			return;
		}

		// If connection use the datasource: use it
		if (isManagedByDataSource(connection)) {
			DataSourceUtils.releaseConnection(connection, this.dataSource);
		} else {
			Daos.closeSilently(connection);
		}
	}

	/**
	 * <p>closeSilently.</p>
	 *
	 * @param dbMeta a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 */
	protected void closeSilently(SynchroDatabaseMetadata dbMeta) {
		if (dbMeta == null) {
			return;
		}
		// If same URL as datasource, use the dataSource
		if (!dbMeta.isClosed()) {
			IOUtils.closeQuietly(dbMeta);
		}
	}

	/**
	 * <p>isManagedByDataSource.</p>
	 *
	 * @param connection a {@link java.sql.Connection} object.
	 * @return a boolean.
	 */
	protected boolean isManagedByDataSource(Connection connection) {
		Preconditions.checkNotNull(connection);

		String jdbcUrl = Daos.getUrl(connection);
		// If same URL as datasource, use the dataSource
		return this.dataSource != null && isManagedByDataSource(jdbcUrl);
	}

	/**
	 * <p>isTransactional.</p>
	 *
	 * @param connection a {@link java.sql.Connection} object.
	 * @return a boolean.
	 */
	protected boolean isTransactional(Connection connection) {
		return isManagedByDataSource(connection)
				&& DataSourceUtils.isConnectionTransactional(connection,
						dataSource);
	}

	/**
	 * Apply a commit on transaction, only if there is no Spring transaction
	 * managment enable.
	 *
	 * @param connection a {@link java.sql.Connection} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected void commit(Connection connection) throws SQLException {
		if (connection == null || isTransactional(connection)) {
			return;
		}

		// No transaction: do a normal commit
		connection.commit();
	}

	/**
	 * Apply a rollback on transaction. If there is a Spring transaction
	 * managment enable, will mark the transaction as rollback only.
	 *
	 * @param connection a {@link java.sql.Connection} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected void rollback(Connection connection) throws SQLException {
		if (connection == null) {
			return;
		}

		// If transactional: mark transaction as rollback only
		if (isTransactional(connection)) {
			if (log.isDebugEnabled()) {
				log.debug(String.format(
						"Mark transaction as 'rollbackOnly' on [%s]",
						Daos.getUrl(connection)));
			}

			TransactionInterceptor.currentTransactionStatus().setRollbackOnly();
		}

		// else : do a normal rollback
		else {
			if (log.isDebugEnabled()) {
				log.debug(String.format("Rollback on [%s]",
						Daos.getUrl(connection)));
			}
			connection.rollback();
		}
	}

	/**
	 * Apply a rollback on transaction, only if there is no Spring transaction
	 * managment enable.
	 *
	 * @param connection a {@link java.sql.Connection} object.
	 */
	protected void rollbackSilently(Connection connection) {
		try {
			rollback(connection);
		} catch (SQLException e) {
			// silent !
		}
	}

	/**
	 * Apply a rollback on transaction, only if there is NOT transaction, or if
	 * the transaction is a new one (Propagation.REQUIRES_NEW)
	 *
	 * @param connection a {@link java.sql.Connection} object.
	 */
	protected void rollbackIfNewTransaction(Connection connection) {
		if (connection == null) {
			return;
		}

		// Is not transactional: do a normal rollback
		if (!isTransactional(connection)) {
			try {
				connection.rollback();
			} catch (SQLException e) {
				// silent !
			}
		}

		// If transaction AND new transaction: mark this transaction as rollback
		// only
		else if (TransactionInterceptor.currentTransactionStatus()
				.isNewTransaction()) {
			TransactionInterceptor.currentTransactionStatus().setRollbackOnly();
		}

		// If transaction has been (or will be) reused:
		else {
			// DO NOT rollback (see mantis #26388)
		}
	}

	/**
	 * Getting the default binding to use for select queries
	 *
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @return a {@link java.util.Map} object.
	 */
	protected final Map<String, Object> getSelectBindings(SynchroContext context) {
		try {
			return defaultBindingCache.get(context);
		} catch (ExecutionException e) {
			return ImmutableMap.copyOf(createDefaultSelectBindings(context));
		}
	}

	/**
	 * Could be override to put some values need by overidden select queries
	 *
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @return a {@link java.util.Map} object.
	 */
	protected Map<String, Object> createDefaultSelectBindings(
			SynchroContext context) {

		Map<String, Object> defaultBinding = Maps.newHashMap();
		if (context.getLastSynchronizationDate() != null) {
			defaultBinding.put(SynchroTableMetadata.UPDATE_DATE_BINDPARAM,
					context.getLastSynchronizationDate());
		}

		return defaultBinding;
	}

	/**
	 * Could be override to put wome values need for select queries
	 *
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @param tableName a {@link java.lang.String} object.
	 * @return a {@link java.util.Map} object.
	 */
	protected Map<String, Object> createSelectBindingsForTable(
			SynchroContext context, String tableName) {
		// Get defaults binding from context
		Map<String, Object> bindings = Maps
				.newHashMap(getSelectBindings(context));

		Date tableUpdateDate = context.getResult().getUpdateDate(tableName);

		// if update date exists for the table (=some rows exists on table)
		if (tableUpdateDate != null || context.getTarget().isMirrorDatabase()) {

			Timestamp lastSynchronizationDate = context
					.getLastSynchronizationDate();

			// If context has a date, use it instead
			if (lastSynchronizationDate != null) {
				bindings.put(SynchroTableMetadata.UPDATE_DATE_BINDPARAM,
						lastSynchronizationDate);
			} else if (tableUpdateDate != null) {
				bindings.put(SynchroTableMetadata.UPDATE_DATE_BINDPARAM,
						tableUpdateDate);
			}
		}

		return bindings;
	}

	/**
	 * Could be used by subclasses
	 *
	 * @param connectionProperties a {@link java.util.Properties} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 */
	protected void disableIntegrityConstraints(Properties connectionProperties,
			SynchroResult result) {
		result.getProgressionModel().setMessage(
				t("synchro.synchronize.disableIntegrityConstraints"));
		if (log.isDebugEnabled()) {
			log.debug(t("synchro.synchronize.disableIntegrityConstraints"));
		}
		try {
			Daos.setIntegrityConstraints(connectionProperties, false);
		} catch (SynchroTechnicalException e) {
			result.setError(e);
		} catch (SQLException e) {
			result.setError(e);
		}
	}

	/**
	 * Utility method to known if the connection could be manage by the
	 * dataSource bean.
	 * <br>
	 * Could be subclasses if dataSource configuration change (i.e. in
	 * core-allegro-ui-wicket)
	 *
	 * @param jdbcUrl a {@link java.lang.String} object.
	 * @return true
	 */
	protected boolean isManagedByDataSource(String jdbcUrl) {
		return Objects.equals(config.getJdbcURL(), jdbcUrl);
	}

	/**
	 * Load (or create) a database metadata, for the given configuration and
	 * connection. By default, a new object is return (no cache). <br> Could be
	 * override by subclasses (i.e. to use cache, ...)
	 *
	 * @param connection a {@link java.sql.Connection} object.
	 * @param dbConfiguration a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 * @param tableNames a {@link java.util.Set} object.
	 * @return a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 */
	protected SynchroDatabaseMetadata loadDatabaseMetadata(
			Connection connection,
			SynchroDatabaseConfiguration dbConfiguration, Set<String> tableNames) {
		if (log.isDebugEnabled()) {
			log.debug(String.format("Loading database metadata... [%s]",
					dbConfiguration.getJdbcUrl()));
		}

		SynchroDatabaseMetadata result = SynchroDatabaseMetadata
				.loadDatabaseMetadata(connection, dbConfiguration, tableNames);

		return result;
	}

	/**
	 * <p>newDaoFactory.</p>
	 *
	 * @param connection a {@link java.sql.Connection} object.
	 * @param dbConfig a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 * @param dbMeta a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 * @return a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 */
	protected DaoFactory newDaoFactory(Connection connection,
			SynchroDatabaseConfiguration dbConfig,
			SynchroDatabaseMetadata dbMeta) {
		return new DaoFactoryImpl(connection, dbConfig, dbMeta, daoCacheSize,
				statementCacheSize);
	}

	/**
	 * <p>initBindingCache.</p>
	 *
	 * @param maximumSize a int.
	 * @return a {@link com.google.common.cache.LoadingCache} object.
	 */
	protected LoadingCache<SynchroContext, Map<String, Object>> initBindingCache(
			final int maximumSize) {
		return CacheBuilder.newBuilder().maximumSize(maximumSize)
				.expireAfterAccess(2, TimeUnit.MINUTES)
				.build(new CacheLoader<SynchroContext, Map<String, Object>>() {
					public Map<String, Object> load(SynchroContext context)
							throws SQLException {
						return ImmutableMap
								.copyOf(createDefaultSelectBindings(context));
					}
				});
	}

	/**
	 * <p>synchronizeOperation.</p>
	 *
	 * @param operation a {@link fr.ifremer.common.synchro.service.SynchroTableOperation} object.
	 * @param targetMeta a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 * @param sourceDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param targetDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @param pendingOperations a {@link java.util.Deque} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected final void synchronizeOperation(SynchroTableOperation operation,
			SynchroDatabaseMetadata targetMeta, DaoFactory sourceDaoFactory,
			DaoFactory targetDaoFactory, SynchroContext context,
			SynchroResult result, Deque<SynchroTableOperation> pendingOperations)
			throws SQLException {

		boolean hasMissingUpdates = MapUtils.isNotEmpty(operation
				.getMissingUpdates());
		boolean hasMissingUpdatesByPks = MapUtils.isNotEmpty(operation.getMissingUpdatesByPks());
		boolean hasMissingDeletes = CollectionUtils.isNotEmpty(operation
				.getMissingDeletes());
		boolean hasMissingDetachs = CollectionUtils.isNotEmpty(operation
				.getMissingDetachs());
		boolean hasChildrenToUpdate = operation.hasChildrenToUpdate();
		boolean hasChildrenToDelete = operation.hasChildrenToDelete();
		boolean hasChildrenToDetach = operation.hasChildrenToDetach();

		boolean rootTableOperation = !hasChildrenToUpdate
				&& !hasChildrenToDelete && !hasMissingUpdates
				&& !hasMissingUpdatesByPks
				&& !hasMissingDeletes && !hasMissingDetachs
				&& !hasChildrenToDetach;

		// Root table
		if (rootTableOperation) {
			String tableName = operation.getTableName();
			SynchroTableMetadata table = targetMeta.getTable(tableName);
			long t0 = TimeLog.getTime();

			if (log.isDebugEnabled()) {
				log.debug("Synchronize root table: " + tableName);
			}
			long countToUpdate = result.getNbRows(tableName);

			if (countToUpdate > 0) {

				synchronizeRootTable(table, sourceDaoFactory, targetDaoFactory,
						context, result, operation, pendingOperations);

				TIME.log(t0, "synchronize root table " + tableName);

			}

			// Add to pending operations (if not empty)
			addToPendingOperationsIfNotEmpty(operation, pendingOperations,
					context);
		}

		// missing delete
		else if (hasMissingDeletes) {
			String tableName = operation.getTableName();
			SynchroTableMetadata table = targetMeta.getTable(tableName);

			// Get child to update and clean operation
			List<List<Object>> pksToDeletes = operation.getMissingDeletes();
			operation.clearMissingDeletes();

			long t0 = TimeLog.getTime();

			if (log.isDebugEnabled()) {
				log.debug("Execute missing deletes on table: " + tableName);
			}

			synchronizeDeletes(operation, table, pksToDeletes,
					sourceDaoFactory, targetDaoFactory, context, result,
					pendingOperations);

			TIME_DELETION.log(t0, "Execute update deletes reference on table "
					+ tableName);

			// Add to pending operations (if not empty)
			addToPendingOperationsIfNotEmpty(operation, pendingOperations,
					context);
		}

		// missing update :
		else if (hasMissingUpdates) {

			String tableName = operation.getTableName();
			SynchroTableMetadata table = targetMeta.getTable(tableName);

			// Get child to update and clean operation
			Map<String, Map<String, Object>> missingUpdates = operation
					.getMissingUpdates();
			operation.clearMissingUpdates();

			long t0 = TimeLog.getTime();

			if (log.isDebugEnabled()) {
				log.debug("Update missing references on table: " + tableName);
			}

			synchronizeColumnUpdates(operation, table, missingUpdates,
					sourceDaoFactory, targetDaoFactory, context, result,
					pendingOperations);

			TIME.log(t0, "Update missing reference on table " + tableName);

			// Add to pending operations (if not empty)
			addToPendingOperationsIfNotEmpty(operation, pendingOperations,
					context);
		}

		// missing update by pk :
		else if (hasMissingUpdatesByPks) {

			String tableName = operation.getTableName();
			SynchroTableMetadata table = targetMeta.getTable(tableName);

			// Get child to update and clean operation
			Map<String, Object[]> missingUpdates = operation.getMissingUpdatesByPks();
			operation.clearMissingUpdatesByPks();

			long t0 = TimeLog.getTime();

			if (log.isDebugEnabled()) {
				log.debug("Update missing data on table: " + tableName);
			}

			synchronizeColumnUpdatesByPks(operation,
					table,
					missingUpdates,
					sourceDaoFactory,
					targetDaoFactory,
					context,
					result,
					pendingOperations);

			TIME.log(t0, "Update missing data on table " + tableName);

			// Add to pending operations (if not empty)
			addToPendingOperationsIfNotEmpty(operation, pendingOperations, context);
		}

		// missing detachment
		else if (hasMissingDetachs) {
			String tableName = operation.getTableName();
			SynchroTableMetadata table = targetMeta.getTable(tableName);

			// Get detachment infos and clean operation
			List<List<Object>> missingDetachs = operation.getMissingDetachs();
			operation.clearMissingDetachs();

			long t0 = TimeLog.getTime();

			if (log.isDebugEnabled()) {
				log.debug("Detachs rows on table: " + tableName);
			}

			synchronizeDetachs(operation, table, missingDetachs,
					sourceDaoFactory, targetDaoFactory, context, result,
					pendingOperations);

			TIME.log(t0, "Detachs rows on table " + tableName);

			// Add to pending operations (if not empty)
			addToPendingOperationsIfNotEmpty(operation, pendingOperations,
					context);
		}

		// Update child tables (selected by one column)
		else if (hasChildrenToUpdate) {

			// Get child to update and clean operation
			Map<String, Map<Set<String>, List<List<Object>>>> childrenToUpdate = operation
					.getChildrenToUpdate();
			operation.clearChildrenToUpdate();

			for (Entry<String, Map<Set<String>, List<List<Object>>>> entry : childrenToUpdate
					.entrySet()) {
				String tableName = entry.getKey();
				Map<Set<String>, List<List<Object>>> childToUpdate = entry
						.getValue();

				SynchroTableOperation childOperation = new SynchroTableOperation(
						tableName, context);
				SynchroTableMetadata table = targetMeta
						.getLoadedTable(tableName);

				long t0 = TimeLog.getTime();

				if (table != null) {
					if (log.isDebugEnabled()) {
						log.debug(String.format(
								"Synchronize child table: %s (child of %s)",
								tableName, operation.getTableName()));
					}

					for (Set<String> columnNames : childToUpdate.keySet()) {
						List<List<Object>> columnsValues = childToUpdate
								.get(columnNames);

						synchronizeChildrenByFks(childOperation, table,
								columnNames, columnsValues, sourceDaoFactory,
								targetDaoFactory, context, result,
								pendingOperations);

					}
					TIME.log(t0, "synchronize child table " + tableName);

					// Add to pending operations (if not empty)
					addToPendingOperationsIfNotEmpty(childOperation,
							pendingOperations, context);
				}
			}
		}

		// Delete child tables (selected by one column)
		else if (hasChildrenToDelete) {

			// Get child to update and clean operation
			Map<String, Map<Set<String>, List<List<Object>>>> childrenTablesToDelete = operation
					.getChildrenToDelete();
			operation.clearChildrenToDelete();

			for (Entry<String, Map<Set<String>, List<List<Object>>>> entry : childrenTablesToDelete
					.entrySet()) {
				String tableName = entry.getKey();
				Map<Set<String>, List<List<Object>>> childrenToDelete = entry
						.getValue();

				SynchroTableOperation childOperation = new SynchroTableOperation(
						tableName, context);
				SynchroTableMetadata table = targetMeta
						.getLoadedTable(tableName);

				long t0 = TimeLog.getTime();

				if (table != null) {
					if (log.isDebugEnabled()) {
						log.debug(String.format(
								"Delete from child table: %s (child of %s)",
								tableName, operation.getTableName()));
					}

					for (Set<String> columnNames : childrenToDelete.keySet()) {
						List<List<Object>> columnsValues = childrenToDelete
								.get(columnNames);

						synchronizeChildrenToDeletes(childOperation, table,
								columnNames, columnsValues, sourceDaoFactory,
								targetDaoFactory, context, result,
								pendingOperations);

					}
					TIME_DELETION.log(t0, "delete child table " + tableName);

					// Add to pending operations (if not empty)
					// DO NOT add addToPending here, because this is done in
					// method synchronizeChildrenToDeletes()
					// (special queue management for deletion)
					// addToPendingOperationsIfNotEmpty(childOperation,
					// pendingOperations, context);

				}
			}
		}

		// Detach child tables (selected by one column)
		else if (hasChildrenToDetach) {

			// Get child to update and clean operation
			Map<String, Map<Set<String>, List<List<Object>>>> childrenTablesToDetach = operation
					.getChildrenToDetach();
			operation.clearChildrenToDetach();

			for (Entry<String, Map<Set<String>, List<List<Object>>>> entry : childrenTablesToDetach
					.entrySet()) {
				String tableName = entry.getKey();
				Map<Set<String>, List<List<Object>>> childrenToDetach = entry
						.getValue();

				SynchroTableOperation childOperation = new SynchroTableOperation(
						tableName, context);
				SynchroTableMetadata table = targetMeta
						.getLoadedTable(tableName);

				long t0 = TimeLog.getTime();

				if (table != null) {
					if (log.isDebugEnabled()) {
						log.debug(String.format(
								"Detach from child table: %s (child of %s)",
								tableName, operation.getTableName()));
					}

					for (Set<String> columnNames : childrenToDetach.keySet()) {
						List<List<Object>> columnsValues = childrenToDetach
								.get(columnNames);

						synchronizeChildrenToDetach(childOperation, table,
								columnNames, columnsValues, sourceDaoFactory,
								targetDaoFactory, context, result,
								pendingOperations);

					}
					TIME.log(t0, "detach child table " + tableName);

					addToPendingOperationsIfNotEmpty(childOperation,
							pendingOperations, context);
				}
			}
		}
	}

	/**
	 * <p>addToPendingOperationsIfNotEmpty.</p>
	 *
	 * @param operation a {@link fr.ifremer.common.synchro.service.SynchroTableOperation} object.
	 * @param pendingOperations a {@link java.util.Deque} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 */
	protected final void addToPendingOperationsIfNotEmpty(
			SynchroTableOperation operation,
			Deque<SynchroTableOperation> pendingOperations,
			SynchroContext context) {
		boolean hasChildToUpdate = MapUtils.isNotEmpty(operation
				.getChildrenToUpdate());
		boolean hasChildToDelete = MapUtils.isNotEmpty(operation
				.getChildrenToDelete());
		boolean hasChildToDetach = MapUtils.isNotEmpty(operation
				.getChildrenToDetach());
		boolean hasMissingUpdates = MapUtils.isNotEmpty(operation
				.getMissingUpdates());
		boolean hasMissingUpdatesByPks = MapUtils.isNotEmpty(operation.getMissingUpdatesByPks());
		boolean hasMissingDeletes = CollectionUtils.isNotEmpty(operation
				.getMissingDeletes());
		boolean hasMissingDetach = CollectionUtils.isNotEmpty(operation
				.getMissingDetachs());
		boolean isEmpty = !hasChildToUpdate && !hasChildToDelete
				&& !hasMissingUpdates && !hasMissingUpdatesByPks && !hasMissingDeletes
				&& !hasMissingDetach & !hasChildToDetach;

		if (isEmpty) {
			return;
		}

		// Disable progression if reused
		operation.setEnableProgress(false);

		if (hasChildToUpdate || hasChildToDelete) {
			// if only child to update : put in the begin
			if (!hasMissingUpdates && !hasMissingDeletes && !hasMissingUpdatesByPks) {
				pendingOperations.addFirst(operation);
			}

			// If has child to update AND (missing updates or deletes) : split
			// in many operations
			else {
				if (hasMissingUpdates) {
					// Put missing updates at the end
					SynchroTableOperation newOperation = new SynchroTableOperation(
							operation.getTableName(), context);
					newOperation.addAllMissingColumnUpdates(operation
							.getMissingUpdates());
					pendingOperations.add(newOperation);
					operation.clearMissingUpdates();
				}
				if (hasMissingUpdatesByPks) {
					// Put missing updates by pk at the end
					SynchroTableOperation newOperation = new SynchroTableOperation(operation.getTableName(), context);
					newOperation.addAllMissingUpdatesByPks(operation.getMissingUpdatesByPks());
					pendingOperations.add(newOperation);
					operation.clearMissingUpdatesByPks();
				}
				if (hasMissingDeletes) {
					// Put missing deletes at the end
					SynchroTableOperation newOperation = new SynchroTableOperation(
							operation.getTableName(), context);
					newOperation.setAllowMissingDeletes(operation.isAllowMissingDeletes());
					newOperation.addAllMissingDelete(operation
							.getMissingDeletes());
					pendingOperations.add(newOperation);
					operation.clearMissingDeletes();
				}

				// Put child to update/delete on the top
				pendingOperations.addFirst(operation);
			}
		}

		// If has only missing updates or deletes
		else {
			// Put at the end
			pendingOperations.add(operation);
		}
	}

	/**
	 * <p>synchronizeRootTable.</p>
	 *
	 * @param table a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param sourceDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param targetDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @param operation a {@link fr.ifremer.common.synchro.service.SynchroTableOperation} object.
	 * @param pendingOperations a {@link java.util.Deque} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected final void synchronizeRootTable(SynchroTableMetadata table,
			DaoFactory sourceDaoFactory, DaoFactory targetDaoFactory,
			SynchroContext context, SynchroResult result,
			SynchroTableOperation operation,
			Deque<SynchroTableOperation> pendingOperations) throws SQLException {

		String tableName = table.getName();
		boolean enableUpdateDate = table.isWithUpdateDateColumn();

		result.getProgressionModel().setMessage(
				t("synchro.synchronize.step1", tableName));

		SynchroTableDao sourceDao = sourceDaoFactory.getSourceDao(table);
		SynchroTableDao targetDao = targetDaoFactory.getTargetDao(table,
				sourceDao, operation);

		ResultSet dataToUpdate = null;
		try {
			// Get existing Pks and updateDates from the target db
			long existingRowCount = targetDao.countAll(true);
			boolean doGetExistingPks = existingRowCount > 0
					&& existingRowCount <= MAX_ROW_COUNT_FOR_PK_PRELOADING;

			Set<String> existingPks = null;
			Map<String, Timestamp> existingUpdateDates = null;

			// Getting existing Pks only for not big table
			if (doGetExistingPks) {
				if (enableUpdateDate) {
					existingUpdateDates = targetDao.getPksStrWithUpdateDate();
					existingPks = existingUpdateDates.keySet();
				} else {
					existingPks = targetDao.getPksStr();
				}
			}

			else if (existingRowCount == 0) {
				// Avoid a new call of countAll() in updateTablexxx methods
				existingPks = Sets.newHashSet();
			}

			// Create a binding for the table, then get data to update from
			// source db
			Map<String, Object> bindings = createSelectBindingsForTable(
					context, table.getName());
			dataToUpdate = sourceDao.getData(bindings);

			if (table.hasUniqueConstraints()) {
				updateTableWithUniqueConstraints(sourceDao, targetDao,
						dataToUpdate, existingPks, false, existingUpdateDates,
						context, result, pendingOperations);
			} else {
				updateTable(targetDao, dataToUpdate, existingPks, false,
						existingUpdateDates, context, result, pendingOperations);
			}
		} finally {
			Daos.closeSilently(dataToUpdate);
		}
	}

	/**
	 * Récupération des données d'une table à partir de valeurs de FKs
	 * <br>
	 * Sert aussi bien quand les tables parentes ont une clef simple ou un clé
	 * composite
	 *
	 * @param operation a {@link fr.ifremer.common.synchro.service.SynchroTableOperation} object.
	 * @param table a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param fkColumnNames a {@link java.util.Set} object.
	 * @param fkSourceColumnValues a {@link java.util.List} object.
	 * @param sourceDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param targetDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @param pendingOperations a {@link java.util.Deque} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected void synchronizeChildrenByFks(SynchroTableOperation operation,
			SynchroTableMetadata table, Set<String> fkColumnNames,
			List<List<Object>> fkSourceColumnValues,
			DaoFactory sourceDaoFactory, DaoFactory targetDaoFactory,
			SynchroContext context, SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations) throws SQLException {

		String tableName = table.getName();
		boolean enableUpdateDate = table.isWithUpdateDateColumn()
				&& table.isRoot();

		result.getProgressionModel().setMessage(
				t("synchro.synchronize.step1", tableName));

		SynchroTableDao sourceDao = sourceDaoFactory.getSourceDao(table);
		SynchroTableDao targetDao = targetDaoFactory.getTargetDao(table,
				sourceDao, operation);

		Map<String, Object> bindings = createSelectBindingsForTable(context,
				tableName);

		ResultSet dataToUpdate = null;
		try {
			// Get existing Pks and updateDates from the target db
			long existingRowCount = targetDao.countAll(true);
			Set<String> existingPks = null;
			Map<String, Timestamp> existingUpdateDates = null;

			if (existingRowCount > 0) {
				Set<String> fkTargetColumnNames = targetDao
						.transformColumnNames(fkColumnNames);
				List<List<Object>> fkTargetColumnValues = targetDao
						.transformOnRead(fkColumnNames, fkSourceColumnValues);
				if (enableUpdateDate) {
					existingUpdateDates = targetDao
							.getPksStrWithUpdateDateByFks(fkTargetColumnNames,
									fkTargetColumnValues, bindings);
					existingPks = existingUpdateDates.keySet();
				} else {
					existingPks = targetDao.getPksStrByFks(fkTargetColumnNames,
							fkTargetColumnValues, bindings);
				}
			}
			// Make sure the list is never null
			// and avoid a new call of countAll() in updateTablexxx methods
			if (existingPks == null) {
				existingPks = Sets.newHashSet();
			}

			// Create a binding for the table, then get data to update from
			// source db
			dataToUpdate = sourceDao.getDataByFks(fkColumnNames,
					fkSourceColumnValues, bindings);

			// Table with a REMOTE_ID (and ID), or other unique constraint
			if (table.hasUniqueConstraints()) {
				updateTableWithUniqueConstraints(sourceDao, targetDao,
						dataToUpdate, existingPks, true, // manage deletion
						existingUpdateDates, context, result, pendingOperations);
			}

			// Association tables, ...
			else {
				updateTable(targetDao, dataToUpdate, existingPks, true, // manage
																		// deletion
						existingUpdateDates, context, result, pendingOperations);
			}
		} finally {
			Daos.closeSilently(dataToUpdate);
		}
	}

	/**
	 * Delete all child rows, by FK on parent table
	 *
	 * @param operation a {@link fr.ifremer.common.synchro.service.SynchroTableOperation} object.
	 * @param table a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param fkColumnNames a {@link java.util.Set} object.
	 * @param fkColumnValues a {@link java.util.List} object.
	 * @param sourceDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param targetDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @param pendingOperations a {@link java.util.Deque} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected final void synchronizeChildrenToDeletes(
			SynchroTableOperation operation, SynchroTableMetadata table,
			Set<String> fkColumnNames, List<List<Object>> fkColumnValues,
			DaoFactory sourceDaoFactory, DaoFactory targetDaoFactory,
			SynchroContext context, SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations) throws SQLException {

		String tableName = table.getName();
        String tablePrefix = table.getTableLogPrefix() + " - "
                + result.getNbRows(tableName);
		boolean hasChildTables = table.hasChildJoins();
		boolean checkPkNotUsed = isCheckPkNotUsedBeforeDelete(context);

		result.getProgressionModel().setMessage(
				t("synchro.synchronize.step5", tableName));

		SynchroTableDao sourceDao = sourceDaoFactory.getSourceDao(table);
		SynchroTableDao targetDao = targetDaoFactory.getTargetDao(table,
				sourceDao, operation);

		Map<String, Object> bindings = createSelectBindingsForTable(context,
				tableName);
		List<List<Object>> pksToDelete = targetDao.getPksByFks(fkColumnNames,
				fkColumnValues, bindings);

		if (CollectionUtils.isNotEmpty(pksToDelete)) {

			// If has children, add deletion to pending operations
			if (hasChildTables) {
				operation.addAllMissingDelete(pksToDelete);

				// Put again on the top of the queue
				pendingOperations.addFirst(operation);

				// Then add children on top
				addDeleteChildrenToDeque(table, pksToDelete, pendingOperations,
						context);
			}

			// If no children: do deletion
			else {
				deleteRows(targetDao, pksToDelete, checkPkNotUsed, context,
						result, pendingOperations);

                targetDao.flush();

                int deleteCount = targetDao.getDeleteCount();

                result.addDeletes(tableName, deleteCount);

                if (log.isInfoEnabled()) {
                    log.info(String.format("%s done: %s (deletes)", tablePrefix,
                            deleteCount));
                }

				addToPendingOperationsIfNotEmpty(operation, pendingOperations,
						context);
			}
		}
	}

	/**
	 * Delete all child rows, by FK on parent table
	 *
	 * @param operation a {@link fr.ifremer.common.synchro.service.SynchroTableOperation} object.
	 * @param table a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param fkColumnNames a {@link java.util.Set} object.
	 * @param fkColumnValues a {@link java.util.List} object.
	 * @param sourceDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param targetDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @param pendingOperations a {@link java.util.Deque} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected final void synchronizeChildrenToDetach(
			SynchroTableOperation operation, SynchroTableMetadata table,
			Set<String> fkColumnNames, List<List<Object>> fkColumnValues,
			DaoFactory sourceDaoFactory, DaoFactory targetDaoFactory,
			SynchroContext context, SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations) throws SQLException {

		String tableName = table.getName();
		boolean hasChildTables = table.hasChildJoins();

		result.getProgressionModel().setMessage(
				t("synchro.synchronize.step7", tableName));

		SynchroTableDao sourceDao = null;
		if (sourceDaoFactory != null) {
			sourceDao = sourceDaoFactory.getSourceDao(table);
		}
		SynchroTableDao targetDao = targetDaoFactory.getTargetDao(table,
				sourceDao, operation);

		Map<String, Object> bindings = createSelectBindingsForTable(context,
				tableName);
		List<List<Object>> pksToDetach = targetDao.getPksByFks(fkColumnNames,
				fkColumnValues, bindings);

		if (CollectionUtils.isNotEmpty(pksToDetach)) {

			detachRows(targetDao, pksToDetach, context, result,
					pendingOperations);

			// If has children, add detachment to pending operations
			if (hasChildTables) {
				// Then add children on top
				addDetachChildrenToDeque(table, pksToDetach, pendingOperations,
						context);
			}
		}
	}

	/**
	 * <p>synchronizeColumnUpdates.</p>
	 *
	 * @param operation a {@link fr.ifremer.common.synchro.service.SynchroTableOperation} object.
	 * @param table a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param columnUpdatesByPkStr a {@link java.util.Map} object.
	 * @param sourceDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param targetDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @param pendingOperations a {@link java.util.Deque} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected void synchronizeColumnUpdates(SynchroTableOperation operation,
			SynchroTableMetadata table,
			Map<String, Map<String, Object>> columnUpdatesByPkStr,
			DaoFactory sourceDaoFactory, DaoFactory targetDaoFactory,
			SynchroContext context, SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations) throws SQLException {

		String tableName = table.getName();

		result.getProgressionModel().setMessage(
				t("synchro.synchronize.step4", tableName));

		SynchroTableDao sourceDao = null;
		if (sourceDaoFactory != null) {
			sourceDao = sourceDaoFactory.getSourceDao(table);
		}
		SynchroTableDao targetDao = targetDaoFactory.getTargetDao(table,
				sourceDao, operation);

		// For each column to update
		for (String columnName : columnUpdatesByPkStr.keySet()) {
			Map<String, Object> valuesByPkStr = columnUpdatesByPkStr
					.get(columnName);

			// Update the column
			updateColumn(targetDao, columnName, valuesByPkStr, result,
					pendingOperations);

            targetDao.prepare();
		}
	}

	/**
	 * <p>synchronizeColumnUpdatesByPks.</p>
	 *
	 * @param operation a {@link fr.ifremer.common.synchro.service.SynchroTableOperation} object.
	 * @param table a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param columnUpdatesByPks a {@link java.util.Map} object.
	 * @param sourceDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param targetDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @param pendingOperations a {@link java.util.Deque} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected void synchronizeColumnUpdatesByPks(
			SynchroTableOperation operation,
			SynchroTableMetadata table,
			Map<String, Object[]> columnUpdatesByPks,
			DaoFactory sourceDaoFactory,
			DaoFactory targetDaoFactory,
			SynchroContext context,
			SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations
	) throws SQLException {

		String tableName = table.getName();

		result.getProgressionModel().setMessage(t("synchro.synchronize.step4", tableName));

		SynchroTableDao sourceDao = null;
		if (sourceDaoFactory != null) {
			sourceDao = sourceDaoFactory.getSourceDao(table);
		}
		SynchroTableDao targetDao = targetDaoFactory.getTargetDao(table, sourceDao, operation);

		// Update the columns for each rows
		updateColumns(targetDao,
				columnUpdatesByPks,
				result,
				pendingOperations);

	}

	/**
	 * <p>synchronizeDetachs.</p>
	 *
	 * @param operation a {@link fr.ifremer.common.synchro.service.SynchroTableOperation} object.
	 * @param table a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param pksToDetach a {@link java.util.List} object.
	 * @param sourceDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param targetDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @param pendingOperations a {@link java.util.Deque} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected void synchronizeDetachs(SynchroTableOperation operation,
			SynchroTableMetadata table, List<List<Object>> pksToDetach,
			DaoFactory sourceDaoFactory, DaoFactory targetDaoFactory,
			SynchroContext context, SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations) throws SQLException {

		String tableName = table.getName();
		boolean hasChildJoin = table.hasChildJoins();

		result.getProgressionModel().setMessage(
				t("synchro.synchronize.step6", tableName));

		SynchroTableDao sourceDao = null;
		if (sourceDaoFactory != null) {
			sourceDao = sourceDaoFactory.getSourceDao(table);
		}
		SynchroTableDao targetDao = targetDaoFactory.getTargetDao(table,
				sourceDao, operation);

		detachRows(targetDao, pksToDetach, context, result, pendingOperations);

		if (hasChildJoin) {
			addDetachChildrenToDeque(table, pksToDetach, pendingOperations,
					context);
		}
	}

	/**
	 * <p>synchronizeDeletes.</p>
	 *
	 * @param operation a {@link fr.ifremer.common.synchro.service.SynchroTableOperation} object.
	 * @param table a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param pksToDelete a {@link java.util.List} object.
	 * @param sourceDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param targetDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @param pendingOperations a {@link java.util.Deque} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected final void synchronizeDeletes(SynchroTableOperation operation,
			SynchroTableMetadata table, List<List<Object>> pksToDelete,
			DaoFactory sourceDaoFactory, DaoFactory targetDaoFactory,
			SynchroContext context, SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations) throws SQLException {

		String tableName = table.getName();
        String tablePrefix = table.getTableLogPrefix() + " - "
                + result.getNbRows(tableName);
		boolean checkPkNotUsed = isCheckPkNotUsedBeforeDelete(context);

		result.getProgressionModel().setMessage(
				t("synchro.synchronize.step3", tableName));

		SynchroTableDao sourceDao = null;
		if (sourceDaoFactory != null) {
			sourceDao = sourceDaoFactory.getSourceDao(table);
		}
		SynchroTableDao targetDao = targetDaoFactory.getTargetDao(table,
				sourceDao, operation);

		deleteRows(targetDao, pksToDelete, checkPkNotUsed, context, result,
				pendingOperations);

        targetDao.flush();

        int deleteCount = targetDao.getDeleteCount();

        result.addDeletes(tableName, deleteCount);

        if (log.isInfoEnabled()) {
            log.info(String.format("%s done: %s (deletes)", tablePrefix,
                    deleteCount));
        }

        if (targetDao.getCurrentOperation().isEnableProgress()) {
            result.getProgressionModel().increments(deleteCount % batchSize);
        }
	}

	/**
	 * To update the content of the given {@code table} on the target db, from
	 * the given {@code incomingData} of the source db.
	 * <br>
	 * The algorithm use remote_id : for each row of the {@code incomingData},
	 * if exists on target table, then do an update, otherwise do a insert.
	 * <br>
	 *
	 * @param context
	 *            Synchronization context
	 * @param targetDao
	 *            connection on the target db
	 * @param incomingData
	 *            data to update from the source db
	 * @param result
	 *            where to store operation results
	 * @throws java.sql.SQLException
	 *             if any sql errors
	 * @param sourceDao a {@link fr.ifremer.common.synchro.dao.SynchroTableDao} object.
	 * @param existingPks a {@link java.util.Set} object.
	 * @param deleteExtraRows a boolean.
	 * @param existingUpdateDates a {@link java.util.Map} object.
	 * @param pendingOperations a {@link java.util.Deque} object.
	 */
	protected final void updateTableWithUniqueConstraints(
			SynchroTableDao sourceDao, SynchroTableDao targetDao,
			ResultSet incomingData, Set<String> existingPks,
			boolean deleteExtraRows,
			Map<String, Timestamp> existingUpdateDates, SynchroContext context,
			SynchroResult result, Deque<SynchroTableOperation> pendingOperations)
			throws SQLException {

		SynchroTableMetadata table = targetDao.getTable();
		Preconditions.checkArgument(MapUtils.isNotEmpty(table
				.getUniqueConstraints()));

		boolean hasChildTables = table.hasChildJoins();
		boolean checkUpdateDate = table.isWithUpdateDateColumn()
				&& table.isRoot();
		boolean allowSkipRow = targetDao.getCurrentOperation()
				.isEnableProgress();

		String tableName = table.getName();
		String tablePrefix = table.getTableLogPrefix() + " - "
				+ result.getNbRows(tableName);
		Map<String, DuplicateKeyStrategy> duplicateKeyStrategies = table
				.getDuplicatKeyStrategies();
		LinkedHashMap<String, Object[]> dataWithManyDuplicationKeys = Maps.newLinkedHashMap();

		boolean isTableEmpty = targetDao.countAll(true) == 0;
		boolean skipGetDuplicateKeys = isTableEmpty
				&& !context.getSource()
						.isCheckUniqueConstraintBetweenInputRows();

		List<List<Object>> processedSourcePks = null;
		if (hasChildTables) {
			processedSourcePks = Lists.newArrayList();
		}
		Set<String> targetPksStrToRemove = Sets.newHashSet();
		if (deleteExtraRows && existingPks != null) {
			targetPksStrToRemove.addAll(existingPks);
		}

		result.addTableName(tableName);
		int countR = 0;

		// For each incoming row
		while (incomingData.next()) {

			List<Object> pk = null;
			boolean skipRow = false;
			boolean hasManyDuplicatedKeys = false;

			// Retrieve duplicate keys (if table is not empty)
			Map<String, List<Object>> duplicatedKeys = null;
			if (!skipGetDuplicateKeys) {
				duplicatedKeys = targetDao
						.getPkFromUniqueConstraints(incomingData);
			}

			boolean hasDuplicateKey = MapUtils.isNotEmpty(duplicatedKeys);
			List<Object> targetDuplicatePk = null;
			String uniqueConstraintName = null;
			boolean targetDuplicatePkRemap = false;

			// Retrieve pk from duplicate keys
			if (hasDuplicateKey) {

				// Check each duplicate key found
				Map<String, List<Object>> duplicatedKeysRejected = null;
				for (Entry<String, List<Object>> entry : duplicatedKeys
						.entrySet()) {
					DuplicateKeyStrategy strategy = duplicateKeyStrategies
							.get(entry.getKey());

					// Check if there is a overwrite
					List<Object> rowPk = entry.getValue();
					switch (strategy) {
						case WARN :
							log.warn(String
									.format("%s Duplicate key (%s): [%s]. Will continue to process this row.",
											tablePrefix, entry.getKey(), rowPk));
							break;
						case REPLACE_AND_REMAP :
							targetDuplicatePkRemap = true;
							// Continue (no break)
						case REPLACE :
							if (debug) {
								log.trace(String
										.format("%s Duplicate key (%s): [%s] - will try to update this row",
												tablePrefix, entry.getKey(),
												rowPk));
							}
                            // If many duplicated keys, on NOT equal pks
                            if (pk != null && !SynchroTableMetadata.equals(pk, rowPk)) {
                                hasManyDuplicatedKeys = true;
                            }
							pk = rowPk;
							break;
                        case REPLACE_LOW_PRIORITY:
                            // Apply ONLY if no other replacement applied
                            if (pk == null) {
                                if (debug) {
                                    log.trace(String.format("%s Duplicate key (%s): [%s] - will try to update this row (if no other replacement asked)",
                                            tablePrefix, entry.getKey(), rowPk));
                                }
                                pk = rowPk;
                            }
                            else if (!SynchroTableMetadata.equals(pk, rowPk)) {
                                hasManyDuplicatedKeys = true;
                            }
                            break;
						case DUPLICATE :
							if (debug) {
								log.trace(String
										.format("%s Duplicate key (%s): [%s] - will insert as new row (duplication allowed)",
												tablePrefix, entry.getKey(),
												rowPk));
							}
							break;
						case REJECT_AND_REMAP :
							targetDuplicatePkRemap = true;
							// Continue (no break)
						case REJECT :
							if (duplicatedKeysRejected == null) {
								duplicatedKeysRejected = Maps.newHashMap();
							}
							duplicatedKeysRejected.put(entry.getKey(), rowPk);
							break;
					}
				}

				// Check each duplicate key found, with the strategy REJECT
				if (MapUtils.isNotEmpty(duplicatedKeysRejected)) {
					for (Entry<String, List<Object>> entry : duplicatedKeysRejected
							.entrySet()) {
						uniqueConstraintName = entry.getKey();
						List<Object> rowPk = entry.getValue();
						if (pk == null || !pk.equals(rowPk)) {
							if (debug) {
								log.trace(String
										.format("%s Duplicate key (%s): [%s] - will reject this row",
												tablePrefix,
												uniqueConstraintName, rowPk));
							}
							String rowPkStr = SynchroTableMetadata
									.toPkStr(rowPk);
							targetPksStrToRemove.remove(rowPkStr);
							skipRow = true;
							targetDuplicatePk = rowPk;
							break;
						}
					}
				}
			}

			// Duplicate key : add row to rejects
			if (skipRow) {
				rejectDuplicatedRow(tableName, sourceDao, incomingData, result,
						targetDuplicatePk, uniqueConstraintName);
				if (!allowSkipRow) {
					// Retrieve the original PK (not transformed)
					List<Object> sourcePk = targetDao.getPk(incomingData);
					throw new SynchroDuplicateRowException(tableName, sourcePk,
							pk);
				}
				if (targetDuplicatePkRemap) {
					List<Object> sourcePk = targetDao.getPk(incomingData);
					// Retrieve the original PK (not transformed)
					context.getTarget().addPkRemap(tableName,
							SynchroTableMetadata.toPkStr(sourcePk),
							SynchroTableMetadata.toPkStr(targetDuplicatePk));
				}
			} else {
				// Will update if a pk has been found
				boolean doUpdate = (pk != null);

				// apply the update, using the pk found
				if (doUpdate) {

					if (targetDuplicatePkRemap) {
						List<Object> sourcePk = targetDao.getPk(incomingData);
						// Retrieve the original PK (not transformed)
						context.getTarget().addPkRemap(tableName,
								SynchroTableMetadata.toPkStr(sourcePk),
								SynchroTableMetadata.toPkStr(pk));
					}

					String pkStr = SynchroTableMetadata.toPkStr(pk);
					targetPksStrToRemove.remove(pkStr);

					// Try to lock
					if (!targetDao.lock(pk)) {
						// lock failed: add row to rejects, and skip
						rejectLockedRow(tableName, sourceDao, incomingData,
								pkStr,
								result);
						skipRow = true;
					}

					else if (checkUpdateDate) {
						Timestamp transformIncomingUpdateDate = targetDao
								.getUpdateDate(incomingData, true);
						Timestamp existingUpdateDate;
						if (existingUpdateDates != null) {
							existingUpdateDate = existingUpdateDates.get(pkStr);
						} else {
							existingUpdateDate = targetDao
									.getUpdateDateByPk(pk);
						}
						int updateDateCompare = Daos
								.compareUpdateDates(existingUpdateDate,
										transformIncomingUpdateDate);

						// up to date : skip
						if (updateDateCompare == 0) {
							if (debug) {
								log.trace(String
										.format("%s row is up to date: [%s] - will skip this row",
												tablePrefix, pkStr));
							}
							skipRow = true;
						}

						else {
							Timestamp incomingUpdateDate = targetDao
									.getUpdateDate(incomingData, false);
							updateDateCompare = Daos.compareUpdateDates(
									existingUpdateDate, incomingUpdateDate);

							// existing date > incoming date: reject row (row
							// has been updated !)
							if (updateDateCompare > 0) {
								rejectBadUpdateDateRow(tableName, sourceDao,
										incomingData, existingUpdateDate,
										pkStr, result);
								if (debug) {
									log.trace(String
											.format("%s row has incorrect update_date: [%s] - will reject this row (expected update_date: '%s' but found: '%s')",
													tablePrefix, pkStr,
													existingUpdateDate,
													incomingUpdateDate));
								}
								skipRow = true;
							} else if (debug) {
								log.trace(String
										.format("%s row is older in target DB: [%s] - row will be updated (new update_date: '%s')",
												tablePrefix, pkStr,
												incomingUpdateDate));
							}
						}
					}

					if (!skipRow) {
						if (!hasManyDuplicatedKeys) {
							// Execute the update
                            try {
                                targetDao.executeUpdate(pk, incomingData);
                            } catch (SynchroRejectRowException re) {
                                if (!allowSkipRow) {
                                    throw re;
                                }
                                // Skip the row, and continue
                                rejectRow(re, result);
                                skipRow = true;
                            }
                        }
						else {
							// Store data, to apply update AFTER deletion
							dataWithManyDuplicationKeys.put(pkStr, targetDao.getDataAsArray(incomingData));
				        }
					}
				}

				// Insert (if no pk found)
				else {

					try {
						targetDao.executeInsert(incomingData);
					} catch (SynchroRejectRowException re) {
						if (!allowSkipRow) {
							throw re;
						}
						// Skip the row, and continue
						rejectRow(re, result);
						skipRow = true;
					}
				}
			}

			if (!hasManyDuplicatedKeys) {
                if (!skipRow) {
                    // Mark has updated
                    if (hasChildTables) {
                        // Retrieve the original PK (not transformed)
                        List<Object> sourcePk = targetDao.getPk(incomingData);
                        processedSourcePks.add(sourcePk);
                    }
                }

                countR++;

                reportProgress(result, targetDao, countR, tablePrefix);
            }
		}

		// Process row to delete
		if (CollectionUtils.isNotEmpty(targetPksStrToRemove)) {
			List<List<Object>> targetPksToRemove = SynchroTableMetadata
					.fromPksStr(targetPksStrToRemove);

			// If has children, add deletion to pending operations
			if (hasChildTables) {
				targetDao.getCurrentOperation().addAllMissingDelete(
						targetPksToRemove);
				addDeleteChildrenToDeque(table, targetPksToRemove,
						pendingOperations, context);
			}

			// If no children: do deletion
			else {
				// Use a new operation, to have a valid flag allowMissingDeletes (mantis #29629)
				SynchroTableOperation previousOperation = targetDao.getCurrentOperation();
				SynchroTableOperation operation = new SynchroTableOperation(tableName, context);
				targetDao.setCurrentOperation(operation);

				deleteRows(targetDao,
						targetPksToRemove,
						isCheckPkNotUsedBeforeDelete(context), context, result,
						pendingOperations);

				addToPendingOperationsIfNotEmpty(operation, pendingOperations, context);
				targetDao.setCurrentOperation(previousOperation); // Restore operation

			}
		}

        // flush all operations
		targetDao.flush();

        int insertCount = targetDao.getInsertCount();
        int updateCount = targetDao.getUpdateCount();
        int deleteCount = targetDao.getDeleteCount();

		// Add child table to pending operation
		if (hasChildTables && CollectionUtils.isNotEmpty(processedSourcePks)) {
			addChildrenToDeque(table, processedSourcePks, pendingOperations, context);
		}

        // Apply update on data with many duplication keys (more than one pk).
		// Executed AFTER deletes, because duplication could has been resolved by a deletion...
		// (e.g. mantis #29693 on GEAR_PHYSICAL_FEATURES)
		if (MapUtils.isNotEmpty(dataWithManyDuplicationKeys)) {

			// If has children, add updates to pending operations
			if (hasChildTables) {

				targetDao.getCurrentOperation().addAllMissingUpdatesByPks(dataWithManyDuplicationKeys);
			}

			// If no children: do updates
			else {

				targetDao.prepare();

				for (Entry<String, Object[]> dataEntry : dataWithManyDuplicationKeys.entrySet()) {
					Object[] data = dataEntry.getValue();

					// Retrieve the original PK (not transformed)
					List<Object> sourcePk = targetDao.getPk(data);

					List<Object> pk = SynchroTableMetadata.fromPkStr(dataEntry.getKey());
					boolean skipRow = false;

					try {
						targetDao.executeUpdate(pk, data);
					} catch (SynchroRejectRowException re) {
						if (!allowSkipRow) {
							throw re;
						}
						// Skip the row, and continue
						rejectRow(re, result);
						skipRow = true;
					}

					if (!skipRow) {
						// Mark has updated
						if (hasChildTables) {
							processedSourcePks.add(sourcePk);
						}
					}

					countR++;
					reportProgress(result, targetDao, countR, tablePrefix);
				}

				targetDao.flush();
				updateCount += targetDao.getUpdateCount();
			}
		}

		result.addInserts(tableName, insertCount);
		result.addUpdates(tableName, updateCount);
		result.addDeletes(tableName, deleteCount);

		if (log.isInfoEnabled()) {
			log.info(String.format(
					"%s done: %s (inserts: %s, updates: %s, deletes: %s)",
					tablePrefix, insertCount + updateCount + deleteCount,
					insertCount, updateCount, deleteCount));
		}

		if (targetDao.getCurrentOperation().isEnableProgress()) {
			result.getProgressionModel().increments(countR % batchSize);
		}

	}

	/**
	 * To update the content of the given {@code table} on the target db, from
	 * the given {@code incomingData} of the source db. The algorithm use a
	 * standard update strategy, using primary key.
	 *
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @param targetDao a {@link fr.ifremer.common.synchro.dao.SynchroTableDao} object.
	 * @param incomingData a {@link java.sql.ResultSet} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @throws java.sql.SQLException if any.
	 * @param existingPks a {@link java.util.Set} object.
	 * @param deleteExtraRows a boolean.
	 * @param existingUpdateDates a {@link java.util.Map} object.
	 * @param pendingOperations a {@link java.util.Deque} object.
	 */
	protected final void updateTable(SynchroTableDao targetDao,
			ResultSet incomingData, Set<String> existingPks,
			boolean deleteExtraRows,
			Map<String, Timestamp> existingUpdateDates, SynchroContext context,
			SynchroResult result, Deque<SynchroTableOperation> pendingOperations)
			throws SQLException {
		SynchroTableMetadata table = targetDao.getTable();
		Preconditions.checkArgument(MapUtils.isEmpty(table
				.getUniqueConstraints()));

		String tableName = table.getName();
		String tablePrefix = table.getTableLogPrefix() + " - "
				+ result.getNbRows(tableName);

		boolean enableUpdateDateCheck = table.isWithUpdateDateColumn()
				&& table.isRoot();
		boolean hasChildTables = table.hasChildJoins();

		List<List<Object>> processedTargetPks = Lists.newArrayList();

		long existingRowCount = existingPks != null
				? existingPks.size()
				: targetDao.countAll(true);
		boolean isTableEmpty = existingRowCount == 0;
		if (log.isDebugEnabled()) {
			log.debug(tablePrefix + " existing rows: " + existingRowCount);
		}

		List<List<Object>> updatedPks = null;
		if (hasChildTables) {
			updatedPks = Lists.newArrayList();
		}
		Set<String> targetPksStrToRemove = Sets.newHashSet();
		if (deleteExtraRows && existingPks != null) {
			targetPksStrToRemove.addAll(existingPks);
		}

		result.addTableName(tableName);
		int countR = 0;

		// For each incoming row
		while (incomingData.next()) {
			List<Object> pk = targetDao.getPk(incomingData, true);
			String pkStr = SynchroTableMetadata.toPkStr(pk);

			boolean doUpdate = false;
			if (!isTableEmpty) {
				if (existingPks != null) {
					doUpdate = existingPks.contains(pkStr);
				} else {
					doUpdate = targetDao.exists(pk);
				}
			}
			boolean skipRow = false;

			if (doUpdate) {
				targetPksStrToRemove.remove(pkStr);

				if (enableUpdateDateCheck) {
					Timestamp transformedIncomingUpdateDate = targetDao
							.getUpdateDate(incomingData, true);
					Timestamp existingUpdateDate = null;
					if (existingUpdateDates != null) {
						existingUpdateDate = existingUpdateDates.get(pkStr);
					} else {
						existingUpdateDate = targetDao.getUpdateDateByPk(pk);
					}
					int updateDateCompare = Daos.compareUpdateDates(
							existingUpdateDate, transformedIncomingUpdateDate);

					// up to date : skip
					if (updateDateCompare == 0) {
						skipRow = true;
					}

					else {
						Timestamp incomingUpdateDate = targetDao.getUpdateDate(
								incomingData, false);
						updateDateCompare = Daos.compareUpdateDates(
								existingUpdateDate, incomingUpdateDate);

						// existing date > incoming date: reject row (row has
						// been updated !)
						if (updateDateCompare > 0) {
							rejectBadUpdateDateRow(tableName, targetDao,
									incomingData, existingUpdateDate, pkStr,
									result);
							skipRow = true;
						}

						else if (debug) {
							log.trace(String
									.format("%s row is older in target DB: [%s] - row will be updated (new update_date: '%s')",
											tablePrefix, pkStr,
											incomingUpdateDate));
						}
					}
				}
				if (!skipRow) {
					targetDao.executeUpdate(pk, incomingData);
				}

			} else {

				targetDao.executeInsert(incomingData);
			}

			if (!skipRow) {
				if (hasChildTables) {
					// Retrieve the original PK (not transformed)
					List<Object> incomingPk = targetDao.getPk(incomingData);
					updatedPks.add(incomingPk);
				}

				if (!isTableEmpty && pk != null) {
					processedTargetPks.add(pk);
				}
			}

			countR++;

			reportProgress(result, targetDao, countR, tablePrefix);
		}

		// Process row to delete
		if (CollectionUtils.isNotEmpty(targetPksStrToRemove)) {
			List<List<Object>> targetPksToRemove = SynchroTableMetadata
					.fromPksStr(targetPksStrToRemove);

			// If has children, add deletion to pending operations
			if (hasChildTables) {
				targetDao.getCurrentOperation().addAllMissingDelete(
						targetPksToRemove);
				addDeleteChildrenToDeque(table, targetPksToRemove,
						pendingOperations, context);
			}

			// If no children: do deletion
			else {
				// Do deletion now using deleteRow, to enable retry later mode (mantis #29629)
				// And use a new operation, to have a valid flag allowMissingDeletes
				SynchroTableOperation previousOperation = targetDao.getCurrentOperation();
				SynchroTableOperation operation = new SynchroTableOperation(tableName, context);
				targetDao.setCurrentOperation(operation);

				deleteRows(targetDao,
						targetPksToRemove,
						isCheckPkNotUsedBeforeDelete(context),
						context,
						result,
						pendingOperations);

				addToPendingOperationsIfNotEmpty(operation, pendingOperations, context);
				targetDao.setCurrentOperation(previousOperation); // restore operation
			}
		}

        // flush all operations
		targetDao.flush();

		// Put in context (to be used by child join tables)
		if (hasChildTables && CollectionUtils.isNotEmpty(updatedPks)) {
			addChildrenToDeque(table, updatedPks, pendingOperations, context);
		}

		int insertCount = targetDao.getInsertCount();
		int updateCount = targetDao.getUpdateCount();
		int deleteCount = targetDao.getDeleteCount();

		result.addInserts(tableName, insertCount);
		result.addUpdates(tableName, updateCount);
		result.addDeletes(tableName, deleteCount);

		if (log.isInfoEnabled()) {
			log.info(String.format(
					"%s done: %s (inserts: %s, updates: %s, delete: %s)",
					tablePrefix, insertCount + updateCount + deleteCount,
					insertCount, updateCount, deleteCount));
		}

		if (targetDao.getCurrentOperation().isEnableProgress()) {
			result.getProgressionModel().increments(countR % batchSize);
		}
	}

	/**
	 * To delete some rows of the given {@code table} on the target db, from the
	 * given {@code pks}.
	 *
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @param targetDao a {@link fr.ifremer.common.synchro.dao.SynchroTableDao} object.
	 * @param pks
	 *            PKs of the row to delete
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @throws java.sql.SQLException if any.
	 * @param checkPkNotUsedBeforeDelete a boolean.
	 * @param pendingOperations a {@link java.util.Deque} object.
	 */
	protected void deleteRows(SynchroTableDao targetDao,
			List<List<Object>> pks, boolean checkPkNotUsedBeforeDelete,
			SynchroContext context, SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations) throws SQLException {
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(pks));

		SynchroTableMetadata table = targetDao.getTable();
		String tableName = table.getName();
		String tablePrefix = table.getTableLogPrefix() + " - "
				+ result.getNbRows(tableName);

		SynchroTableOperation operation = targetDao.getCurrentOperation();
		boolean dataIntegrityExceptionOccured = false;

		result.addTableName(tableName);

		// For each pk to delete
		int countR = 0;
		for (List<Object> pk : pks) {
			try {

				targetDao.executeDelete(pk, checkPkNotUsedBeforeDelete);
				countR++;
				reportProgress(result, targetDao, countR, tablePrefix);

			}

			// If check failed (e.g. PK still referenced by tables)
			catch (DataIntegrityViolationOnDeleteException e) {

				// If the deletion could be done later: add to missing delete
				// (this is allowed only once)
				boolean retryLater = operation.isAllowMissingDeletes()
						&& hasProcessedTable(e.getExistingFkTableNames(),
								context);

				// If missing deletes are NOT allowed: throw error
				if (!retryLater) {
					throw e;
				}
				// Else, do deletion later (mantis #23383)
				else {
					dataIntegrityExceptionOccured = true;
					operation.addMissingDelete(pk);
				}
			}
		}

		if (dataIntegrityExceptionOccured) {
			// Mark the operation has not allow to retry
			// (if done inside the previous catch, will not allow to have more than one error)
			// cf mantis 29629: ne pas marquer trop tôt l'opération avec setAllowMissingDeletes(false),
			// mais attendre que toutes les PKs soient bien traitées par la boucle for.
			operation.setAllowMissingDeletes(false);
		}

	}

	/**
	 * <p>updateColumn.</p>
	 *
	 * @param targetDao a {@link fr.ifremer.common.synchro.dao.SynchroTableDao} object.
	 * @param columnName a {@link java.lang.String} object.
	 * @param valuesByPkStr a {@link java.util.Map} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @param pendingOperations a {@link java.util.Deque} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected void updateColumn(SynchroTableDao targetDao, String columnName,
			Map<String, Object> valuesByPkStr, SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations) throws SQLException {
		Preconditions.checkArgument(MapUtils.isNotEmpty(valuesByPkStr));
		Preconditions.checkNotNull(columnName);

		SynchroTableMetadata table = targetDao.getTable();
		String tableName = table.getName();
		String tablePrefix = table.getTableLogPrefix() + " - "
				+ result.getNbRows(tableName);

		result.addTableName(tableName);

		int countR = 0;

		// For each pk's row to update
		for (Entry<String, Object> entry : valuesByPkStr.entrySet()) {
			if (entry.getKey() != null) {

				List<Object> pk = SynchroTableMetadata
						.fromPkStr(entry.getKey());
				Object columnValue = entry.getValue();

				// Execute update on the pk's row
				targetDao.executeUpdateColumn(columnName, pk, columnValue);

				countR++;
				reportProgress(result, targetDao, countR, tablePrefix);
			}
		}

		int updateCount = targetDao.getUpdateColumnCount(columnName);
		result.addUpdates(tableName, updateCount);

		targetDao.flush();

		if (log.isInfoEnabled()) {
			log.info(String.format("%s done: %s (updates)", tablePrefix, updateCount));
		}

		if (targetDao.getCurrentOperation().isEnableProgress()) {
			result.getProgressionModel().increments(countR % batchSize);
		}
	}

	/**
	 * <p>updateColumns.</p>
	 *
	 * @param targetDao a {@link fr.ifremer.common.synchro.dao.SynchroTableDao} object.
	 * @param columnUpdatesByPks a {@link java.util.Map} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @param pendingOperations a {@link java.util.Deque} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected void updateColumns(SynchroTableDao targetDao,
								 Map<String, Object[]> columnUpdatesByPks,
								 SynchroResult result,
								 Deque<SynchroTableOperation> pendingOperations) throws SQLException {
		Preconditions.checkArgument(MapUtils.isNotEmpty(columnUpdatesByPks));

		SynchroTableMetadata table = targetDao.getTable();
		String tableName = table.getName();
		String tablePrefix = table.getTableLogPrefix() + " - " + result.getNbRows(tableName);

		result.addTableName(tableName);

		int countR = 0;

		// For each pk's row to update
		for (Entry<String, Object[]> entry : columnUpdatesByPks.entrySet()) {
			String pkValue = entry.getKey();

			Object[] data = entry.getValue();

			if (pkValue != null) {

				// Execute update for the pk's row
				targetDao.executeUpdate(SynchroTableMetadata.fromPkStr(pkValue), data);

				countR++;
				reportProgress(result, targetDao, countR, tablePrefix);
			}
		}

		int updateCount = targetDao.getUpdateCount();
		result.addUpdates(tableName, updateCount);

		targetDao.flush();

		if (log.isInfoEnabled()) {
			log.info(String.format("%s done: %s (updates)", tablePrefix,
					updateCount));
		}

		if (targetDao.getCurrentOperation().isEnableProgress()) {
			result.getProgressionModel().increments(countR % batchSize);
		}
	}

	/**
	 * To detach some rows of the given {@code targetDao.getTable()} on the
	 * target db, from the given {@code pks}.<br>
	 *
	 * Should be override (e.g. to reset remote_id...)
	 *
	 * @param targetDao a {@link fr.ifremer.common.synchro.dao.SynchroTableDao} object.
	 * @param pks a {@link java.util.List} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @param pendingOperations a {@link java.util.Deque} object.
	 * @throws java.sql.SQLException if any.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 */
	protected void detachRows(SynchroTableDao targetDao,
			List<List<Object>> pks, SynchroContext context,
			SynchroResult result, Deque<SynchroTableOperation> pendingOperations)
			throws SQLException {

		SynchroTableMetadata table = targetDao.getTable();
		String tableName = table.getName();
		String tablePrefix = table.getTableLogPrefix() + " - "
				+ result.getNbRows(tableName);

		result.addTableName(tableName);

		// For each pk to detach
		int countR = 0;
		for (List<Object> pk : pks) {
			targetDao.executeDetach(pk);

			countR++;
			reportProgress(result, targetDao, countR, tablePrefix);
		}

		targetDao.flush();

		result.addUpdates(tableName, countR);

		if (log.isInfoEnabled()) {
			log.info(String.format("%s done: %s (detach)", tablePrefix,
					countR));
		}

		if (targetDao.getCurrentOperation().isEnableProgress()) {
			result.getProgressionModel().increments(countR % batchSize);
		}

	}

	/**
	 * <p>rejectDuplicatedRow.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @param sourceDao a {@link fr.ifremer.common.synchro.dao.SynchroTableDao} object.
	 * @param incomingData a {@link java.sql.ResultSet} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @param targetPk a {@link java.util.List} object.
	 * @param constraintName a {@link java.lang.String} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected final void rejectDuplicatedRow(String tableName,
			SynchroTableDao sourceDao, ResultSet incomingData,
			SynchroResult result, List<Object> targetPk, String constraintName)
			throws SQLException {

		List<Object> sourcePk = sourceDao.getPk(incomingData);
		String sourcePkStr = SynchroTableMetadata.toPkStr(sourcePk);
		String targetPkStr = targetPk == null ? null : SynchroTableMetadata
				.toPkStr(targetPk);
		result.addReject(tableName, sourcePkStr,
				RejectedRow.Cause.DUPLICATE_KEY.name(), targetPkStr,
				constraintName);
	}

	/**
	 * <p>rejectLockedRow.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @param sourceDao a {@link fr.ifremer.common.synchro.dao.SynchroTableDao} object.
	 * @param incomingData a {@link java.sql.ResultSet} object.
	 * @param targetPkStr a {@link java.lang.String} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected final void rejectLockedRow(String tableName,
			SynchroTableDao sourceDao, ResultSet incomingData,
			String targetPkStr,
			SynchroResult result
			) throws SQLException {

		List<Object> sourcePk = sourceDao.getPk(incomingData);
		String sourcePkStr = SynchroTableMetadata.toPkStr(sourcePk);
		result.addReject(tableName, sourcePkStr, RejectedRow.Cause.LOCKED.name(), targetPkStr);
	}

	/**
	 * <p>rejectBadUpdateDateRow.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @param sourceDao a {@link fr.ifremer.common.synchro.dao.SynchroTableDao} object.
	 * @param incomingData a {@link java.sql.ResultSet} object.
	 * @param validUpdateDate a {@link java.sql.Timestamp} object.
	 * @param targetPkStr a {@link java.lang.String} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected final void rejectBadUpdateDateRow(String tableName,
			SynchroTableDao sourceDao, ResultSet incomingData,
			Timestamp validUpdateDate, String targetPkStr, SynchroResult result)
			throws SQLException {

		// Retrieve source Pk
		List<Object> sourcePk = sourceDao.getPk(incomingData);
		String sourcePkStr = SynchroTableMetadata.toPkStr(sourcePk);

		rejectBadUpdateDateRow(tableName, sourcePkStr, validUpdateDate,
				targetPkStr, result);
	}

	/**
	 * <p>rejectBadUpdateDateRow.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @param sourcePkStr a {@link java.lang.String} object.
	 * @param validUpdateDate a {@link java.sql.Timestamp} object.
	 * @param targetPkStr a {@link java.lang.String} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected final void rejectBadUpdateDateRow(String tableName,
			String sourcePkStr, Timestamp validUpdateDate, String targetPkStr,
			SynchroResult result) throws SQLException {

		result.addReject(tableName, sourcePkStr,
				RejectedRow.Cause.BAD_UPDATE_DATE.name(),
				validUpdateDate.toString(), targetPkStr);
	}

	/**
	 * <p>rejectDeletedRow.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @param sourcePkStr a {@link java.lang.String} object.
	 * @param targetPkStr a {@link java.lang.String} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected final void rejectDeletedRow(String tableName, String sourcePkStr,
			String targetPkStr, SynchroResult result) throws SQLException {

		result.addReject(tableName, sourcePkStr,
				RejectedRow.Cause.DELETED.name(), targetPkStr);
	}

	/**
	 * <p>rejectRow.</p>
	 *
	 * @param error a {@link fr.ifremer.common.synchro.intercept.SynchroRejectRowException} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected final void rejectRow(SynchroRejectRowException error,
			SynchroResult result) throws SQLException {
		Preconditions.checkNotNull(error);

		if (error instanceof SynchroDeletedRowException) {
			SynchroDeletedRowException e = (SynchroDeletedRowException) error;
			rejectDeletedRow(e.getTableName(), e.getSourcePkStr(),
					e.getTargetPkStr(), result);
			return;
		}

		if (error instanceof SynchroBadUpdateDateRowException) {
			SynchroBadUpdateDateRowException e = (SynchroBadUpdateDateRowException) error;
			rejectBadUpdateDateRow(e.getTableName(), e.getSourcePkStr(),
					e.getValidUpdateDate(), e.getTargetPkStr(), result);
			return;
		}

		// Not known exception class
		throw new SynchroTechnicalException(String.format(
				"Unknown reject exception class: %s", error.getClass()
						.getName()));
	}

	/**
	 * <p>getRootOperations.</p>
	 *
	 * @param sourceDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param targetDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param dbMeta a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @return a {@link java.util.List} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected List<SynchroTableOperation> getRootOperations(
			DaoFactory sourceDaoFactory,
			DaoFactory targetDaoFactory,
			SynchroDatabaseMetadata dbMeta,
			SynchroContext context) throws SQLException {
		Set<String> rootTableNames = dbMeta.getLoadedRootTableNames();

		List<SynchroTableOperation> result = Lists
				.newArrayListWithCapacity(rootTableNames.size());
		for (String rootTableName : rootTableNames) {
			SynchroTableOperation operation = new SynchroTableOperation(
					rootTableName, context);
			operation.setEnableProgress(true);
			result.add(operation);
		}
		return result;
	}

	/**
	 * <p>prepareRootTable.</p>
	 *
	 * @param sourceDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param targetDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param table a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected void prepareRootTable(
			DaoFactory sourceDaoFactory,
			DaoFactory targetDaoFactory,
			SynchroTableMetadata table,
			SynchroContext context,
			SynchroResult result) throws SQLException {

		String tablePrefix = table.getTableLogPrefix();
		String tableName = table.getName();

		SynchroTableDao sourceDao = sourceDaoFactory.getSourceDao(table);
		SynchroTableDao targetDao = targetDaoFactory.getTargetDao(table,
				sourceDao, null);

		// Skip get last update if:
		// - target DB is a mirror database (=empty database)
		// - OR the table is too big (e.g. a data table on server)
		boolean skipGetLastUpdateDate = context.getTarget().isMirrorDatabase()
				|| targetDao.countAll(true) > MAX_ROW_COUNT_FOR_PK_PRELOADING;

		Timestamp updateDate = null;
		if (!skipGetLastUpdateDate) {
			// get last updateDate used by target db
			updateDate = targetDao.getLastUpdateDate();

			if (updateDate != null) {

				// just inscrements of 1 second to not having same
				updateDate = new Timestamp(DateUtils.setMilliseconds(
						updateDate, 0).getTime());
				updateDate = new Timestamp(DateUtils.addSeconds(updateDate, 1)
						.getTime());
			}
		}
		result.setUpdateDate(tableName, updateDate);

		Map<String, Object> bindings = createSelectBindingsForTable(context,
				tableName);
		long countToUpdate = sourceDao.countData(bindings);
		result.addRows(tableName, (int) countToUpdate);
		if (log.isInfoEnabled()) {
			log.info(String.format("%s nb rows to update: %s", tablePrefix,
					countToUpdate));
		}
	}

	/**
	 * <p>releaseContext.</p>
	 *
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 */
	protected void releaseContext(SynchroContext context) {
		context.setSourceMeta(null);
		context.setTargetMeta(null);
	}

	/**
	 * <p>prepareConnection.</p>
	 *
	 * @param targetConnection a {@link java.sql.Connection} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected void prepareConnection(Connection targetConnection,
			SynchroContext context) throws SQLException {
		prepareIntegrityConstraints(targetConnection, context);
	}

	/**
	 * <p>releaseConnection.</p>
	 *
	 * @param targetConnection a {@link java.sql.Connection} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected void releaseConnection(final Connection targetConnection,
			SynchroContext context) throws SQLException {
		restoreIntegrityConstraints(targetConnection, context);
	}

	/**
	 * <p>releaseConnectionSilently.</p>
	 *
	 * @param targetConnection a {@link java.sql.Connection} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 */
	protected void releaseConnectionSilently(final Connection targetConnection,
			SynchroContext context) {
		try {
			releaseConnection(targetConnection, context);
		} catch (SQLException e) {
			// nothing to do
		}
	}

	/**
	 * <p>prepareIntegrityConstraints.</p>
	 *
	 * @param targetConnection a {@link java.sql.Connection} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected void prepareIntegrityConstraints(Connection targetConnection,
			SynchroContext context) throws SQLException {
		if (disableIntegrityConstraints
				|| context.getTarget().isMirrorDatabase()) {
			Daos.setIntegrityConstraints(targetConnection, false);
		}
	}

	/**
	 * <p>restoreIntegrityConstraints.</p>
	 *
	 * @param connection a {@link java.sql.Connection} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected void restoreIntegrityConstraints(final Connection connection,
			SynchroContext context) throws SQLException {
		if (disableIntegrityConstraints) {
			// If a last synchronization date has been set
			// do not try a enable integrity, because content is maybe partial
			if (context.getLastSynchronizationDate() != null
					&& context.getTarget().isMirrorDatabase()) {
				return;
			}

			boolean isTransactionalAndRollback = isTransactional(connection)
					&& TransactionInterceptor.currentTransactionStatus()
							.isRollbackOnly();

			// If transaction exists, and mark as rollback only:
			// Make sure to do it AFTER the rollback
			if (isTransactionalAndRollback) {
				TransactionSynchronizationManager
						.registerSynchronization(new TransactionSynchronizationAdapter() {
							@Override
							public void afterCompletion(int status) {
								if (log.isDebugEnabled()) {
									log.debug(String
											.format("Enabling integrity constraints, after rollback on [%s]",
													Daos.getUrl(connection)));
								}
								try {
									Daos.setIntegrityConstraints(connection,
											true);
								} catch (SQLException e) {
									log.warn(String
											.format("Error while trying to enable integrity constraints on [%s]: %s",
													Daos.getUrl(connection),
													e.getMessage(), e));
								}
							}
						});
			}

			// If not transactional connection
			// or if transactional BUT not mark as rollbackOnly
			// => enable integrity constraints
			else {
				if (log.isDebugEnabled()) {
					log.debug(String.format(
							"Enabling integrity constraints on [%s]",
							Daos.getUrl(connection)));
				}
				Daos.setIntegrityConstraints(connection, true);
			}

		}
	}

	/**
	 * <p>restoreIntegrityConstraintsSilently.</p>
	 *
	 * @param targetConnection a {@link java.sql.Connection} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 */
	protected void restoreIntegrityConstraintsSilently(
			final Connection targetConnection, final SynchroContext context) {
		try {
			restoreIntegrityConstraints(targetConnection, context);
		} catch (SQLException e) {
			// nothing to do
		}
	}

	/**
	 * <p>addChildrenToDeque.</p>
	 *
	 * @param parentTable a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param parentPks a {@link java.util.List} object.
	 * @param pendingOperations a {@link java.util.Deque} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 */
	protected final void addChildrenToDeque(SynchroTableMetadata parentTable,
			List<List<Object>> parentPks,
			Deque<SynchroTableOperation> pendingOperations,
			SynchroContext context) {

		Set<String> pkNames = parentTable.getPkNames();

		// More than one PK: not implement yet
		if (pkNames.size() > 1) {
			throw new UnsupportedOperationException(
					"Not sure of this implementation: please check before comment out this exception !");
		}

		// Retrieve childs join
		for (SynchroJoinMetadata join : parentTable.getChildJoins()) {
			SynchroTableMetadata childTable = join.getTargetTable();
			SynchroColumnMetadata childTableColumn = join.getTargetColumn();

			// Add child to update into operation
			SynchroTableOperation operation = new SynchroTableOperation(
					parentTable.getName(), childTable.getName(),
					childTableColumn.getName(), parentPks, context);
			pendingOperations.addFirst(operation);
		}
	}

	/**
	 * <p>addDeleteChildrenToDeque.</p>
	 *
	 * @param parentTable a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param parentPks a {@link java.util.List} object.
	 * @param pendingOperations a {@link java.util.Deque} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 */
	protected final void addDeleteChildrenToDeque(
			SynchroTableMetadata parentTable, List<List<Object>> parentPks,
			Deque<SynchroTableOperation> pendingOperations,
			SynchroContext context) {

		Set<String> pkNames = parentTable.getPkNames();

		// More than one PK: not implemented yet
		if (pkNames.size() > 1) {
			throw new UnsupportedOperationException(
					"Not sure of this implementation: please check before comment out this exception !");
		}

		// Retrieve childs join
		for (SynchroJoinMetadata join : parentTable.getChildJoins()) {
			SynchroTableMetadata childTable = join.getTargetTable();
			SynchroColumnMetadata childTableColumn = join.getTargetColumn();

			// Add child to update into operation
			SynchroTableOperation operation = new SynchroTableOperation(
					parentTable.getName(), context);
			operation.addChildrenToDeleteFromManyColumns(childTable.getName(),
					ImmutableSet.of(childTableColumn.getName()), parentPks);
			pendingOperations.addFirst(operation);
		}
	}

	/**
	 * <p>addDetachChildrenToDeque.</p>
	 *
	 * @param parentTable a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param parentPks a {@link java.util.List} object.
	 * @param pendingOperations a {@link java.util.Deque} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 */
	protected final void addDetachChildrenToDeque(
			SynchroTableMetadata parentTable, List<List<Object>> parentPks,
			Deque<SynchroTableOperation> pendingOperations,
			SynchroContext context) {

		Set<String> pkNames = parentTable.getPkNames();

		// More than one PK: not implemented yet
		if (pkNames.size() > 1) {
			throw new UnsupportedOperationException(
					"Not sure of this implementation: please check before comment out this exception !");
		}

		// Retrieve childs join
		for (SynchroJoinMetadata join : parentTable.getChildJoins()) {
			SynchroTableMetadata childTable = join.getTargetTable();
			SynchroColumnMetadata childTableColumn = join.getTargetColumn();

			// Add child to update into operation
			SynchroTableOperation operation = new SynchroTableOperation(
					parentTable.getName(), context);
			operation.addChildrenToDetachFromManyColumns(childTable.getName(),
					ImmutableSet.of(childTableColumn.getName()), parentPks);
			pendingOperations.addFirst(operation);
		}
	}

	/**
	 * <p>getSourceMissingOperations.</p>
	 *
	 * @param synchroResult a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @return a {@link java.util.List} object.
	 */
	protected List<SynchroTableOperation> getSourceMissingOperations(
			SynchroResult synchroResult, SynchroContext context) {

		List<SynchroTableOperation> result = Lists.newArrayList();

		// Add source missing updates
		if (MapUtils.isNotEmpty(synchroResult.getSourceMissingUpdates())) {
			for (Entry<String, Map<String, Map<String, Object>>> entry : synchroResult
					.getSourceMissingUpdates().entrySet()) {
				String tableName = entry.getKey();
				Map<String, Map<String, Object>> tableMissingUpdates = entry
						.getValue();

				SynchroTableOperation operation = new SynchroTableOperation(
						tableName, context);
				operation.addAllMissingColumnUpdates(tableMissingUpdates);
				result.add(operation);
			}
		}

		// Add source missing deletes
		if (synchroResult.getSourceMissingDeletes() != null
				&& !synchroResult.getSourceMissingDeletes().isEmpty()) {

			for (String tableName : synchroResult.getSourceMissingDeletes()
					.keySet()) {
				Collection<String> pkStrs = synchroResult
						.getSourceMissingDeletes().get(tableName);

				List<List<Object>> pks = Lists.newArrayList();
				for (String pkStr : pkStrs) {
					List<Object> pk = SynchroTableMetadata.fromPkStr(pkStr);
					pks.add(pk);
				}
				SynchroTableOperation operation = new SynchroTableOperation(
						tableName, context);
				operation.addAllMissingDelete(pks);
				result.add(operation);
			}
		}

		return result;
	}

	/**
	 * Process rejects
	 *
	 * @param context
	 *            Context of synchronization
	 * @param rejects
	 *            rejects to process
	 * @param rejectStrategies
	 *            strategies to use on rejected row, by reject status
	 * @throws java.sql.SQLException if any.
	 * @param targetConnection a {@link java.sql.Connection} object.
	 * @param dbMeta a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 * @param targetDaoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param pendingOperations a {@link java.util.Deque} object.
	 */
	protected void resolveRejects(
			Connection targetConnection,
			SynchroDatabaseMetadata dbMeta,
			DaoFactory targetDaoFactory,
			SynchroContext context,
			Map<String, String> rejects,
			Map<RejectedRow.Cause, RejectedRow.ResolveStrategy> rejectStrategies,
			Deque<SynchroTableOperation> pendingOperations) throws SQLException {
		Preconditions.checkNotNull(rejects);
		Preconditions.checkNotNull(rejectStrategies);

		if (log.isDebugEnabled()) {
			log.debug(String.format(
					"Resolving rejects with strategies %s - %s",
					rejectStrategies, context.toString()));
		}

		SynchroResult result = context.getResult();

		// Process rejects
		for (Entry<String, String> entry : rejects.entrySet()) {
			String tableName = entry.getKey();
			String tableRejects = entry.getValue();

			SynchroTableMetadata table = dbMeta.getTable(tableName);

			// process the table
			resolveTableRejects(table, tableRejects, rejectStrategies,
					targetDaoFactory, context, result, pendingOperations);
		}
	}

	/**
	 * <p>resolveTableRejects.</p>
	 *
	 * @param table a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param rejectsRows a {@link java.lang.String} object.
	 * @param strategies a {@link java.util.Map} object.
	 * @param daoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @param pendingOperations a {@link java.util.Deque} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected void resolveTableRejects(SynchroTableMetadata table,
			String rejectsRows,
			Map<RejectedRow.Cause, RejectedRow.ResolveStrategy> strategies,
			DaoFactory daoFactory, SynchroContext context,
			SynchroResult result, Deque<SynchroTableOperation> pendingOperations)
			throws SQLException {

		String tableName = table.getName();
		ProgressionModel progressionModel = result.getProgressionModel();

		progressionModel.setMessage(t("synchro.data.finish.step1",
				tableName));
		if (log.isDebugEnabled()) {
			log.debug(t("synchro.data.finish.step1", tableName));
		}

		SynchroTableOperation operation = new SynchroTableOperation(tableName,
				context);

		for (RejectedRow reject : RejectedRow.parseFromString(rejectsRows)) {

			RejectedRow.ResolveStrategy rejectStrategy = strategies
					.get(reject.cause);
			if (rejectStrategy == null) {
				rejectStrategy = RejectedRow.ResolveStrategy.DO_NOTHING;
			}

			resolveTableRow(table, reject, rejectStrategy, daoFactory,
					operation, context, result);
		}

		// If operation is not empty: add it to queue
		addToPendingOperationsIfNotEmpty(operation, pendingOperations, context);
	}

	/**
	 * Process a reject. Could add some missing updates on the current operation
	 *
	 * @param table a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param reject a {@link fr.ifremer.common.synchro.service.RejectedRow} object.
	 * @param rejectStrategy a {@link fr.ifremer.common.synchro.service.RejectedRow.ResolveStrategy} object.
	 * @param operation a {@link fr.ifremer.common.synchro.service.SynchroTableOperation} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @param daoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 */
	protected void resolveTableRow(SynchroTableMetadata table,
			RejectedRow reject, RejectedRow.ResolveStrategy rejectStrategy,
			DaoFactory daoFactory, SynchroTableOperation operation,
			SynchroContext context, SynchroResult result) {
		String tableName = table.getName();

		switch (reject.cause) {
			case BAD_UPDATE_DATE : {
				Timestamp updateDate = reject.validUpdateDate;

				switch (rejectStrategy) {
					case KEEP_LOCAL :
						if (debug) {
					log.debug(String.format("[%s] Bad update date [target pk=%s]: need update_date=%s from [source pk=%s]", tableName,
							reject.targetPkStr, updateDate, reject.pkStr));
						}
				operation.addMissingColumnUpdate(context.getTarget().getColumnUpdateDate(), reject.targetPkStr, updateDate);
						break;

					case UPDATE :
						if (debug) {
					log.debug(String.format("[%s] Bad update date [target pk=%s]: will re-import by [source pk %s]", tableName, reject.targetPkStr,
							reject.pkStr));
						}
				result.addSourceMissingRevert(tableName, reject.pkStr);
						break;

					case DO_NOTHING :
						if (debug) {
							log.debug(String.format("[%s] Bad update date [source pk=%s]: no resolution",
											tableName, reject.pkStr));
						}
						result.addReject(tableName, reject.toString());
						break;

					default :
						if (debug) {
					log.debug(String.format("[%s] Bad update date [source pk=%s]: no resolution (%s strategy not managed)",
							tableName,
							reject.pkStr,
											rejectStrategy.name()));
						}
						result.addReject(tableName, reject.toString());
						break;
				}
				break;
			}
			case DELETED : {
				switch (rejectStrategy) {
					case KEEP_LOCAL :
						if (debug) {
					log.debug(String.format("[%s] Deleted row [target pk=%s] will be keep locally", tableName, reject.targetPkStr));
						}
						operation.addMissingDetach(reject.targetPkStr);
						break;

					case UPDATE :
						if (debug) {
					log.debug(String.format("[%s] Deleted row [target pk=%s] will deleted on local", tableName, reject.targetPkStr));
						}
						operation.addMissingDelete(reject.targetPkStr);
						break;

					case DO_NOTHING :
						if (debug) {
					log.debug(String.format("[%s] Deleted row [target pk=%s]: no resolution", tableName, reject.targetPkStr));
						}
						result.addReject(tableName, reject.toString());
						break;

					default :
						if (debug) {
					log.debug(String.format("[%s] Deleted row [target pk=%s]: no resolution (%s strategy not managed)",
							tableName,
							reject.targetPkStr,
											rejectStrategy.name()));
						}
						result.addReject(tableName, reject.toString());
						break;
				}
				break;
			}
			case DUPLICATE_KEY :
				switch (rejectStrategy) {
				// When importing a file, user could choose to import as new row
					case DUPLICATE :
						if (debug) {
					log.debug(String.format("[%s] Duplicate key found [target pk=%s]: will insert as new row", tableName, reject.targetPkStr));
						}
				table.setUniqueConstraintStrategy(reject.constraintName, DuplicateKeyStrategy.DUPLICATE);
						break;

					// When importing a file, user could choose to remap source
					// DB, to use local PK (e.g. on referential tables)
					case KEEP_LOCAL :
						if (table.isSimpleKey()) {
							if (debug) {
						        log.debug(String.format("[%s] Duplicate key found [target pk=%s]: remap source row to [source pk=%s]", tableName,
								    reject.targetPkStr,
								    reject.pkStr));
							}
					        result.addSourceMissingColumnUpdate(tableName, table.getPkNames().iterator().next(), reject.pkStr, reject.targetPkStr);
							try {
								SynchroTableDao targetDao = daoFactory
										.getTargetDao(table, null, operation);
								Multimap<String, String> pkExportedKeys = targetDao
										.getExportedKeys();
								if (pkExportedKeys != null
										&& !pkExportedKeys.isEmpty()) {
									for (String fkTableName : pkExportedKeys
											.keySet()) {
										for (String fkColumnName : pkExportedKeys
												.get(fkTableName)) {
											result.addSourceMissingColumnUpdate(
													fkTableName, fkColumnName,
													reject.pkStr,
													reject.targetPkStr);
										}
									}
								}
							} catch (SQLException e) {
								throw new SynchroTechnicalException(e);
							}
						} else {
							if (debug) {
						log.debug(String.format("[%s] Duplicate key [target pk=%s]: no resolution (could not remap a composite PK)", tableName,
								reject.targetPkStr));
							}
							result.addReject(tableName, reject.toString());
						}
						break;

					case DO_NOTHING :
						if (debug) {
					log.debug(String.format("[%s] Duplicate key [target pk=%s]: no resolution", tableName, reject.targetPkStr));
						}
						result.addReject(tableName, reject.toString());
						break;

					default :
						if (debug) {
					log.debug(String.format("[%s] Duplicate key [target pk=%s]: no resolution (%s strategy not managed)",
							tableName,
							reject.targetPkStr,
											rejectStrategy.name()));
						}
						result.addReject(tableName, reject.toString());
						break;
				}
				break;

			case LOCKED :
				if (debug) {
				log.debug(String.format("[%s] Unable to get lock [target pk=%s]: no resolution (please wait [source pk=%s] unlocked)", tableName,
						reject.targetPkStr, reject.pkStr));
				}
				result.addReject(tableName, reject.toString());
				break;

			default :
				break;
		}
	}

	/**
	 * Say if deletion must check that PK is not used anymore.<br>Only if NO
	 * integrity constraints AND target DB is not a mirror constraint
	 *
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @return a boolean.
	 */
	protected boolean isCheckPkNotUsedBeforeDelete(SynchroContext context) {
		return this.disableIntegrityConstraints
				&& !context.getTarget().isMirrorDatabase();
	}

	/**
	 * <p>hasProcessedTable.</p>
	 *
	 * @param tableNames a {@link java.util.Set} object.
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @return a boolean.
	 */
	protected boolean hasProcessedTable(Set<String> tableNames,
			SynchroContext context) {
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(tableNames));
		Preconditions.checkNotNull(context);

		Set<String> expectedTableNames = context.getTableNames();
		for (String tableName : tableNames) {
			if (expectedTableNames.contains(tableName)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * <p>updateResultOnSynchronizeError.</p>
	 *
	 * @param e a {@link java.lang.Exception} object.
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 */
	protected void updateResultOnSynchronizeError(Exception e,
			SynchroResult result) {
		// Clear result, but keep reject rows
		if (result.getRejectedRows().isEmpty()) {
			result.clear();
		} else {
			Map<String, String> copiedRejectedRows = ImmutableMap.copyOf(result
					.getRejectedRows());
			result.clear();
			result.getRejectedRows().putAll(copiedRejectedRows);
		}

		// Save the error
		result.setError(e);
	}

	/**
	 * <p>add.</p>
	 *
	 * @param context a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 */
	protected void add(SynchroContext context) {

	}
}
