package fr.ifremer.common.synchro.service;

/*
 * #%L
 * Tutti :: Persistence
 * $Id: ReferentialSynchroService.java 1573 2014-02-04 16:41:40Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/trunk/tutti-persistence/src/main/java/fr/ifremer/adagio/core/service/technical/synchro/ReferentialSynchroService.java $
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.io.File;
import java.sql.Timestamp;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

/**
 * Created on 31/10/14.
 *
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 * @since 3.7.1
 */
public interface SynchroService {

	/**
	 * Create a synchro context, from a source database directory (HSQLDB only).
	 * This use local connection properties as source properties, but change
	 * JDBC URL to use the given directory
	 *
	 * @param sourceDbDirectory
	 *            a directory with a HSQLDB directory.
	 * @param tableToIncludes a {@link java.util.Set} object.
	 * @return a new context for synchronization
	 */
	SynchroContext createSynchroContext(File sourceDbDirectory,
			Set<String> tableToIncludes);

	/**
	 * Create a synchro context, using configuration properties for target and
	 * source properties.
	 *
	 * @param sourceConnectionProperties
	 *            Properties to use as source connection (could be Oracle,
	 *            HSQLDB...)
	 * @param tableToIncludes a {@link java.util.Set} object.
	 * @return a new context for synchronization
	 */
	SynchroContext createSynchroContext(Properties sourceConnectionProperties,
			Set<String> tableToIncludes);

	/**
	 * Prepare the synchronize operation from the local data database supported
	 * by this service, says just compute nb rows to update for each table and
	 * update the result model.
	 *
	 * @param synchroContext
	 *            context of synchro operation
	 */
	void prepare(SynchroContext synchroContext);

	/**
	 * Launch the synchronize operation from the local data database supported
	 * by this service.
	 *
	 * @param synchroContext
	 *            context of synchro operation
	 */
	void synchronize(SynchroContext synchroContext);

	/**
	 * Compute the max(update_date) found on tables
	 *
	 * @param synchroContext
	 *            a context with a valid source
	 * @return a {@link java.sql.Timestamp} object.
	 */
	Timestamp getSourceLastUpdateDate(SynchroContext synchroContext);

	/**
	 * Finish a synchronization (e.g. an exportation). Will check missing
	 * updates on the result, then process them. This method will resolve
	 * reject, using given strategies.
	 *
	 * @param synchroContext a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @param serverExportResult a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @param rejectStrategies a {@link java.util.Map} object.
	 */
	void finish(SynchroContext synchroContext,
			SynchroResult serverExportResult,
			Map<RejectedRow.Cause, RejectedRow.ResolveStrategy> rejectStrategies);

}
