package fr.ifremer.common.synchro.service;

/*
 * #%L
 * Tutti :: Persistence API
 * $Id: ReferentialSynchroResult.java 1486 2014-01-15 08:43:26Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/trunk/tutti-persistence/src/main/java/fr/ifremer/adagio/core/service/technical/synchro/ReferentialSynchroResult.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Objects;
import com.google.common.collect.*;
import fr.ifremer.common.synchro.SynchroTechnicalException;
import fr.ifremer.common.synchro.dao.Daos;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata;
import fr.ifremer.common.synchro.type.ProgressionModel;
import org.apache.commons.lang3.StringUtils;

import java.sql.Timestamp;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

/**
 * Result of a referential synchronize operation.
 *
 * @author tchemit (chemit@codelutin.com)
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 * @since 3.5.2
 */
public class SynchroResult {

	protected Exception error;

	/**
	 * Number of rows detected to update (per table).
	 * 
	 * @since 1.0
	 */
	protected final Map<String, Integer> rowHits = Maps.newTreeMap();

	/**
	 * Number of insert done (per table).
	 * 
	 * @since 1.0
	 */
	protected final Map<String, Integer> insertHits = Maps.newTreeMap();

	/**
	 * Number of update done (per table).
	 * 
	 * @since 1.0
	 */
	protected final Map<String, Integer> updateHits = Maps.newTreeMap();

	/**
	 * Number of delete done (per table).
	 * 
	 * @since 3.7.0
	 */
	protected final Map<String, Integer> deletesHits = Maps.newTreeMap();

	/**
	 * timestamp of last update date (per table).
	 * 
	 * @since 1.0
	 */
	protected final Map<String, Timestamp> updateDateHits = Maps.newTreeMap();

	/**
	 * rejected rows, as a CSV string.
	 * <br>
	 * i.e. : [PK_AS_STR];[REJECT_STATUS];[UPDATE_DATE_NANOS_TIME]
	 * 25364~~~1;BAD_UPDATE_DATE;12437789901254
	 * 
	 * @since 3.7
	 */
	protected final Map<String, String> rejectedRows = Maps.newTreeMap();

	/**
	 * missing updates on source DB, by tableName, and source pkStr
	 * 
	 * @since 3.7
	 */
	protected final Map<String, Map<String, Map<String, Object>>> sourceMissingUpdates = Maps
			.newHashMap();

	/**
	 * PK str that need to be reverted (new importation)
	 * 
	 * @since 3.7.2
	 */
	protected final Multimap<String, String> sourceMissingReverts = ArrayListMultimap
			.create();

	/**
	 * PK str that need to be deleted
	 * 
	 * @since 3.7.2
	 */
	protected final Multimap<String, String> sourceMissingDeletes = ArrayListMultimap
			.create();

	/**
	 * All table treated.
	 * 
	 * @since 1.0
	 */
	protected final Set<String> tableNames = Sets.newHashSet();

	protected String targetUrl;

	protected String sourceUrl;

	/**
	 * Transient, because not need to serialize it
	 */
	protected transient final ProgressionModel progressionModel = new ProgressionModel(
			this);

	/**
	 * <p>Constructor for SynchroResult.</p>
	 */
	public SynchroResult() {
	}

	/**
	 * <p>Constructor for SynchroResult.</p>
	 *
	 * @param targetUrl a {@link java.lang.String} object.
	 * @param sourceUrl a {@link java.lang.String} object.
	 */
	public SynchroResult(String targetUrl, String sourceUrl) {
		this.targetUrl = targetUrl;
		this.sourceUrl = sourceUrl;
	}

	/**
	 * <p>setLocalUrl.</p>
	 *
	 * @param targetUrl a {@link java.lang.String} object.
	 */
	public void setLocalUrl(String targetUrl) {
		this.targetUrl = targetUrl;
	}

	/**
	 * <p>setRemoteUrl.</p>
	 *
	 * @param sourceUrl a {@link java.lang.String} object.
	 */
	public void setRemoteUrl(String sourceUrl) {
		this.sourceUrl = sourceUrl;
	}

	/**
	 * <p>isSuccess.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isSuccess() {
		return error == null;
	}

	/**
	 * <p>Getter for the field <code>error</code>.</p>
	 *
	 * @return a {@link java.lang.Exception} object.
	 */
	public Exception getError() {
		return error;
	}

	/**
	 * <p>Setter for the field <code>error</code>.</p>
	 *
	 * @param error a {@link java.lang.Exception} object.
	 */
	public void setError(Exception error) {
		this.error = error;
	}

	/**
	 * <p>Getter for the field <code>progressionModel</code>.</p>
	 *
	 * @return a {@link fr.ifremer.common.synchro.type.ProgressionModel} object.
	 */
	public ProgressionModel getProgressionModel() {
		return progressionModel;
	}

	/**
	 * <p>Getter for the field <code>tableNames</code>.</p>
	 *
	 * @return a {@link java.util.Set} object.
	 */
	public Set<String> getTableNames() {
		return ImmutableSet.copyOf(tableNames);
	}

	/**
	 * <p>getTotalRows.</p>
	 *
	 * @return a int.
	 */
	public int getTotalRows() {
		int result = 0;
		for (Integer nb : rowHits.values()) {
			result += nb;
		}
		return result;
	}

	/**
	 * <p>getTotalInserts.</p>
	 *
	 * @return a int.
	 */
	public int getTotalInserts() {
		int result = 0;
		for (Integer nb : insertHits.values()) {
			result += nb;
		}
		return result;
	}

	/**
	 * <p>getTotalUpdates.</p>
	 *
	 * @return a int.
	 */
	public int getTotalUpdates() {
		int result = 0;
		for (Integer nb : updateHits.values()) {
			result += nb;
		}
		return result;
	}

	/**
	 * <p>getTotalDeletes.</p>
	 *
	 * @return a int.
	 */
	public int getTotalDeletes() {
		int result = 0;
		for (Integer nb : deletesHits.values()) {
			result += nb;
		}
		return result;
	}

	/**
	 * <p>getTotalRejects.</p>
	 *
	 * @return a int.
	 */
	public int getTotalRejects() {
		int result = 0;
		for (String rows : rejectedRows.values()) {
			result += StringUtils.countMatches(rows, "\n");
		}
		return result;
	}

	/**
	 * <p>getTotalTreated.</p>
	 *
	 * @return a int.
	 */
	public int getTotalTreated() {
		return getTotalInserts() + getTotalUpdates() + getTotalDeletes()
				+ getTotalRejects();
	}

	/**
	 * <p>getNbRows.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @return a int.
	 */
	public int getNbRows(String tableName) {
		Integer result = rowHits.get(tableName);
		if (result == null) {
			result = 0;
		}
		return result;
	}

	/**
	 * <p>getNbInserts.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @return a int.
	 */
	public int getNbInserts(String tableName) {
		Integer result = insertHits.get(tableName);
		if (result == null) {
			result = 0;
		}
		return result;
	}

	/**
	 * <p>getNbUpdates.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @return a int.
	 */
	public int getNbUpdates(String tableName) {
		Integer result = updateHits.get(tableName);
		if (result == null) {
			result = 0;
		}
		return result;
	}

	/**
	 * <p>getNbDeletes.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @return a int.
	 */
	public int getNbDeletes(String tableName) {
		Integer result = deletesHits.get(tableName);
		if (result == null) {
			result = 0;
		}
		return result;
	}

	/**
	 * <p>Getter for the field <code>rejectedRows</code>.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	public String getRejectedRows(String tableName) {
		String result = rejectedRows.get(tableName);
		if (result == null) {
			return "";
		}
		return result;
	}

	/**
	 * <p>addRows.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @param nb a int.
	 */
	public void addRows(String tableName, int nb) {
		if (nb > 0) {
			rowHits.put(tableName, getNbRows(tableName) + nb);
		}
	}

	/**
	 * <p>addUpdates.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @param nb a int.
	 */
	public void addUpdates(String tableName, int nb) {
		if (nb > 0) {
			updateHits.put(tableName, getNbUpdates(tableName) + nb);
		}
	}

	/**
	 * <p>addInserts.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @param nb a int.
	 */
	public void addInserts(String tableName, int nb) {
		if (nb > 0) {
			insertHits.put(tableName, getNbInserts(tableName) + nb);
		}
	}

	/**
	 * <p>addDeletes.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @param nb a int.
	 */
	public void addDeletes(String tableName, int nb) {
		if (nb > 0) {
			deletesHits.put(tableName, getNbDeletes(tableName) + nb);
		}
	}

	/**
	 * <p>addReject.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @param rowInfo a {@link java.lang.String} object.
	 */
	public void addReject(String tableName, String... rowInfo) {
		RejectedRow.appendAsString(rejectedRows, tableName, rowInfo);
	}

	/**
	 * <p>addSourceMissingColumnUpdate.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @param columnName a {@link java.lang.String} object.
	 * @param pk a {@link java.util.List} object.
	 * @param columnValue a {@link java.lang.Object} object.
	 */
	public void addSourceMissingColumnUpdate(String tableName,
			String columnName, List<Object> pk, Object columnValue) {
		addSourceMissingColumnUpdate(tableName, columnName,
				SynchroTableMetadata.toPkStr(pk), columnValue);
	}

	/**
	 * <p>addSourceMissingColumnUpdate.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @param columnName a {@link java.lang.String} object.
	 * @param pkStr a {@link java.lang.String} object.
	 * @param columnValue a {@link java.lang.Object} object.
	 */
	public void addSourceMissingColumnUpdate(String tableName,
			String columnName, String pkStr, Object columnValue) {
		Map<String, Map<String, Object>> tableMissingSourceUpdates = sourceMissingUpdates
				.get(tableName);
		if (tableMissingSourceUpdates == null) {
			tableMissingSourceUpdates = Maps.newHashMap();
			sourceMissingUpdates.put(tableName, tableMissingSourceUpdates);
		}
		Map<String, Object> destRows = tableMissingSourceUpdates
				.get(columnName);
		if (destRows == null) {
			destRows = Maps.newHashMap();
			tableMissingSourceUpdates.put(columnName, destRows);
		}
		if (destRows.containsKey(pkStr)) {
			Object oldValue = destRows.get(pkStr);
			if (!Objects.equal(oldValue, columnValue)) {
				throw new SynchroTechnicalException(
						String.format(
								"Could not update a row column twice, with two differents values: table [%s] pk [%s] - existing %s: [%s] new: [%s]",
								tableName, pkStr, columnName, oldValue,
								columnValue));
			}
		}
		destRows.put(pkStr, columnValue);
	}

	/**
	 * <p>getUpdateDate.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @return a {@link java.sql.Timestamp} object.
	 */
	public Timestamp getUpdateDate(String tableName) {
		return updateDateHits.get(tableName);
	}

	/**
	 * <p>setUpdateDate.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @param t a {@link java.sql.Timestamp} object.
	 */
	public void setUpdateDate(String tableName, Timestamp t) {
		updateDateHits.put(tableName, t);
	}

	/**
	 * <p>Getter for the field <code>updateDateHits</code>.</p>
	 *
	 * @return a {@link java.util.Map} object.
	 */
	public Map<String, Timestamp> getUpdateDateHits() {
		return updateDateHits;
	}

	/**
	 * <p>Getter for the field <code>rejectedRows</code>.</p>
	 *
	 * @return a {@link java.util.Map} object.
	 */
	public Map<String, String> getRejectedRows() {
		return rejectedRows;
	}

	/**
	 * <p>Getter for the field <code>sourceMissingUpdates</code>.</p>
	 *
	 * @return a {@link java.util.Map} object.
	 */
	public Map<String, Map<String, Map<String, Object>>> getSourceMissingUpdates() {
		return sourceMissingUpdates;
	}

	/**
	 * <p>addTableName.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 */
	public void addTableName(String tableName) {
		tableNames.add(tableName);
	}

	/**
	 * <p>getLocalUrl.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getLocalUrl() {
		return targetUrl;
	}

	/**
	 * <p>getRemoteUrl.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getRemoteUrl() {
		return sourceUrl;
	}

	/**
	 * <p>addSourceMissingRevert.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @param pkStr a {@link java.lang.String} object.
	 */
	public void addSourceMissingRevert(String tableName, String pkStr) {
		sourceMissingReverts.put(tableName, pkStr);
	}

	/**
	 * <p>Getter for the field <code>sourceMissingReverts</code>.</p>
	 *
	 * @return a {@link com.google.common.collect.Multimap} object.
	 */
	public Multimap<String, String> getSourceMissingReverts() {
		return sourceMissingReverts;
	}

	/**
	 * <p>addSourceMissingDelete.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @param pkStr a {@link java.lang.String} object.
	 */
	public void addSourceMissingDelete(String tableName, String pkStr) {
		sourceMissingDeletes.put(tableName, pkStr);
	}

	/**
	 * <p>Getter for the field <code>sourceMissingDeletes</code>.</p>
	 *
	 * @return a {@link com.google.common.collect.Multimap} object.
	 */
	public Multimap<String, String> getSourceMissingDeletes() {
		return sourceMissingDeletes;
	}

	/**
	 * Clear every fields
	 */
	public void clear() {
		rowHits.clear();
		insertHits.clear();
		updateHits.clear();
		deletesHits.clear();
		updateDateHits.clear();
		rejectedRows.clear();
		sourceMissingUpdates.clear();
		sourceMissingReverts.clear();
		sourceMissingDeletes.clear();
		error = null;
	}

	/**
	 * <p>addAll.</p>
	 *
	 * @param anotherResult a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 */
	public void addAll(SynchroResult anotherResult) {

		// row hits
		for (Entry<String, Integer> entry : anotherResult.rowHits.entrySet()) {
			addRows(entry.getKey(), entry.getValue());
		}

		for (Entry<String, Integer> entry : anotherResult.insertHits.entrySet()) {
			addInserts(entry.getKey(), entry.getValue());
		}
		for (Entry<String, Integer> entry : anotherResult.updateHits.entrySet()) {
			addUpdates(entry.getKey(), entry.getValue());
		}
		for (Entry<String, Integer> entry : anotherResult.deletesHits
				.entrySet()) {
			addDeletes(entry.getKey(), entry.getValue());
		}
		for (String tableName : anotherResult.updateDateHits.keySet()) {
			Timestamp newUpdateDate = anotherResult.updateDateHits
					.get(tableName);
			Timestamp previousUpdateDate = getUpdateDate(tableName);
			if (Daos.isUpdateDateBefore(previousUpdateDate, newUpdateDate)) {
				setUpdateDate(tableName, newUpdateDate);
			}
		}
		for (Entry<String, String> entry : anotherResult.rejectedRows
				.entrySet()) {
			addReject(entry.getKey(), entry.getValue());
		}
		for (String tableName : anotherResult.sourceMissingUpdates.keySet()) {
			Map<String, Map<String, Object>> tableMissingSourceUpdates = anotherResult.sourceMissingUpdates
					.get(tableName);
			for (String columnName : tableMissingSourceUpdates.keySet()) {
				Map<String, Object> destRows = tableMissingSourceUpdates
						.get(columnName);
				for (String pkStr : destRows.keySet()) {
					Object columnValue = destRows.get(pkStr);
					addSourceMissingColumnUpdate(tableName, columnName, pkStr,
							columnValue);
				}
			}
		}
		sourceMissingReverts.putAll(anotherResult.sourceMissingReverts);
		sourceMissingDeletes.putAll(anotherResult.sourceMissingDeletes);
	}

}
