package fr.ifremer.common.synchro.service;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Joiner;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import fr.ifremer.common.synchro.config.SynchroConfiguration;
import fr.ifremer.common.synchro.dao.Daos;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.cfg.AvailableSettings;
import org.hibernate.cfg.Configuration;
import org.hibernate.cfg.Environment;
import org.hibernate.dialect.Dialect;
import org.hibernate.internal.util.config.ConfigurationHelper;

import java.io.Serializable;
import java.sql.Timestamp;
import java.util.Arrays;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

/**
 * <p>SynchroDatabaseConfiguration class.</p>
 *
 */
public class SynchroDatabaseConfiguration implements Serializable {

	static final long serialVersionUID = -1L;

	/**
	 * is database the targeted database ?
	 */
	public static String IS_TARGET_DATABASE = "synchro.database.isTarget";

	/**
	 * is database in readonly ?
	 */
	public static String READ_ONLY = "synchro.database.readonly";

	/**
	 * Should apply a rollback at the end of synchronization ? (or if managed by
	 * transaction, mark the transaction as rollbackOnly).
	 * <br>
	 * This could be usefull to simulate a synchronization, without applying
	 * changes. (e.g. to get changes throw a interceptor)
	 */
	public static String ROLLBACK_ONLY = "synchro.database.rollbackOnly";

	/**
	 * Enable or not load full metadata.
	 */
	public static String ENABLE_FULL_METADATA = "synchro.database.enableQueryMetadata";

	/**
	 * Is database a mirror database.
	 * <br>
	 * If true, ID will be kept, if false, sequence will be used to generate new ID
	 */
	public static String IS_MIRROR_DATABASE = "synchro.database.isMirror";

	/**
	 * Should queries keep where clause, when getting data (or count() ) by FKs ?
	 * <br>
	 * Should be 'true' for synchronization on referential tables, and 'false'
	 * for data tables
	 */
	public static String KEEP_WHERE_CLAUSE_ON_QUERIES_BY_FKS = "synchro.database.keepWhereClauseOnQueriesByFks";

	/**
	 * Should check is duplicate row exists on the input (source) data ?
	 * <p>
	 * 'true' by default.
	 * </p>
	 * If set to 'true', JDBC batch update will be disable on tables with one or
	 * more unique constraint.<br>
	 * This could affect performance. Should be set to 'false' when source
	 * database is safe.
	 */
	public static String CHECK_UNIQUE_CONSTRAINTS_BETWEEN_INPUT_ROWS = "synchro.database.checkUniqueConstraintOverInputData";

	/**
	 * Column names to excludes : TABLE1.MY_COLUMN,TABLE2.MY_COLUMN;%.MY_COLUMN
	 */
	public static String COLUMN_EXCLUDES = "synchro.database.column.excludes";

	/**
	 * Tables names to includes
	 */
	public static String TABLE_INCLUDES = "synchro.database.table.includes";

	/**
	 * System timestamp
	 */
	public static String SYSTEM_TIMESTAMP = "synchro.database.systimestamp";

	/**
	 * ID column name
	 */
	public static String COLUMN_ID = "synchro.database.column.id";

	/**
	 * UPDATE_DATE column name
	 */
	public static String COLUMN_UPDATE_DATE = "synchro.database.column.updateDate";

	/**
	 * sequence suffix
	 */
	public static String SEQUENCE_SUFFIX = "synchro.database.sequence.suffix";

	/**
	 * max sql name length
	 */
	public static String MAX_SQL_NAME_LENGTH = "synchro.database.maxSqlNameLength";

	/* -- inherited from hibernate settings: -- */

	/**
	 * Names the Hibernate {@literal SQL} {@link org.hibernate.dialect.Dialect}
	 * class
	 */
	public static String DIALECT = Environment.DIALECT;

	/**
	 * Names the {@literal JDBC} driver class
	 */
	public static String DRIVER = Environment.DRIVER;

	/**
	 * Names the {@literal JDBC} connection url.
	 */
	public static String URL = Environment.URL;

	/**
	 * Names the connection user. This might mean one of 2 things in
	 * out-of-the-box Hibernate
	 * {@link org.hibernate.engine.jdbc.connections.spi.ConnectionProvider}:
	 * <ul>
	 * <li>The username used to pass along to creating the JDBC connection</li>
	 * <li>The username used to obtain a JDBC connection from a data source</li>
	 * </ul>
	 */
	public static String USER = Environment.USER;

	/**
	 * Names the connection password. See usage discussion on {@link #USER}
	 */
	public static String PASS = Environment.PASS;

	private Properties properties;

	private Dialect dialect = null;

	private String columnUpdateDate;

	private String columnId;

	private String sequenceSuffix;

	private int maxSqlNameLength;

	/**
	 * A map use to store pk source to target mapping. Used by main synchro
	 * process, to store duplicate rows with REPLACE options Could be used by
	 * interceptors, to store mapping dynamically.
	 * <br>
	 * Use nternal Clearing on each synchro.
	 */
	protected final Map<String, Map<String, String>> remapPks = Maps
			.newHashMap();

	/**
	 * <p>Constructor for SynchroDatabaseConfiguration.</p>
	 *
	 * @param otherBean a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 */
	protected SynchroDatabaseConfiguration(
			SynchroDatabaseConfiguration otherBean) {
		copy(otherBean);
	}

	/**
	 * <p>Constructor for SynchroDatabaseConfiguration.</p>
	 */
	protected SynchroDatabaseConfiguration() {
		// default constructor, need for clone()
		properties = new Properties();
	}

	/**
	 * <p>Constructor for SynchroDatabaseConfiguration.</p>
	 *
	 * @param settings a {@link java.util.Properties} object.
	 * @param isTarget a boolean.
	 */
	public SynchroDatabaseConfiguration(Properties settings, boolean isTarget) {
		super();
		this.properties = settings;
		setIsTarget(isTarget);

		// Default values
		setReadOnly(!isTarget);
		setRollbackOnly(!isTarget);

		// Force to refresh some cached values
		reloadCachedFields();
	}

	/**
	 * <p>getProperty.</p>
	 *
	 * @param propertyName a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	public String getProperty(String propertyName) {
		return properties.getProperty(propertyName);
	}

	/**
	 * put all properties into the database configuration, but <u>always</u>
	 * keep the value of isTarget()
	 *
	 * @param anotherProperties a {@link java.util.Map} object.
	 */
	public void putAllProperties(Map<?, ?> anotherProperties) {
		boolean savedIsTarget = isTarget();
		properties.putAll(anotherProperties);
		setIsTarget(savedIsTarget); // restore previous value

		reloadCachedFields();
	}

	/**
	 * <p>removeProperty.</p>
	 *
	 * @param key a {@link java.lang.String} object.
	 * @return a {@link java.lang.Object} object.
	 */
	public Object removeProperty(String key) {
		return properties.remove(key);
	}

	/**
	 * DO NOT change into public, to avoid the use of getProperties().putAll()
	 * that will break cached fields usage
	 *
	 * @return a {@link java.util.Properties} object.
	 */
	protected Properties getProperties() {
		return properties;
	}

	/**
	 * <p>getConnectionProperties.</p>
	 *
	 * @return a {@link java.util.Properties} object.
	 */
	public Properties getConnectionProperties() {
		return Daos.getConnectionProperties(getJdbcUrl(), getJdbcUser(),
				getJdbcPassword(), getJdbcCatalog(), getJdbcSchema(),
				properties.getProperty(Environment.DIALECT), getJdbcDriver());
	}

	/**
	 * <p>reloadCachedFields.</p>
	 */
	protected void reloadCachedFields() {
		SynchroConfiguration config = SynchroConfiguration.getInstance();
		if (properties.getProperty(Environment.DIALECT) == null) {
			this.dialect = null;
		} else {
			this.dialect = Dialect.getDialect(properties);
		}
		this.columnId = ConfigurationHelper.getString(COLUMN_ID, properties,
				config.getColumnId());
		this.columnUpdateDate = ConfigurationHelper.getString(
				COLUMN_UPDATE_DATE, properties, config.getColumnUpdateDate());
		this.sequenceSuffix = ConfigurationHelper.getString(SEQUENCE_SUFFIX,
				properties, config.getSequenceSuffix());
		this.maxSqlNameLength = ConfigurationHelper.getInt(MAX_SQL_NAME_LENGTH,
				properties, config.getMaxSqlNameLength());
	}

	/**
	 * Set a property value by name
	 *
	 * @param propertyName
	 *            The name of the property to set
	 * @param value
	 *            The new property value
	 * @return this for method chaining
	 */
	public SynchroDatabaseConfiguration setProperty(String propertyName,
			String value) {
		properties.setProperty(propertyName, value);
		return this;
	}

	/**
	 * <p>Getter for the field <code>columnId</code>.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getColumnId() {
		return columnId;
	}

	/**
	 * <p>Setter for the field <code>columnId</code>.</p>
	 *
	 * @param columnName a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 */
	public SynchroDatabaseConfiguration setColumnId(String columnName) {
		this.columnId = columnName;
		setProperty(COLUMN_ID, columnName);
		return this;
	}

	/**
	 * <p>Getter for the field <code>columnUpdateDate</code>.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getColumnUpdateDate() {
		return columnUpdateDate;
	}

	/**
	 * <p>Setter for the field <code>columnUpdateDate</code>.</p>
	 *
	 * @param columnName a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 */
	public SynchroDatabaseConfiguration setColumnUpdateDate(String columnName) {
		this.columnUpdateDate = columnName;
		setProperty(COLUMN_UPDATE_DATE, columnName);
		return this;
	}

	/**
	 * <p>Getter for the field <code>sequenceSuffix</code>.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getSequenceSuffix() {
		return sequenceSuffix;
	}

	/**
	 * <p>Setter for the field <code>sequenceSuffix</code>.</p>
	 *
	 * @param sequenceSuffix a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 */
	public SynchroDatabaseConfiguration setSequenceSuffix(String sequenceSuffix) {
		this.sequenceSuffix = sequenceSuffix;
		setProperty(SEQUENCE_SUFFIX, sequenceSuffix);
		return this;
	}

	/**
	 * <p>Getter for the field <code>maxSqlNameLength</code>.</p>
	 *
	 * @return a int.
	 */
	public int getMaxSqlNameLength() {
		return maxSqlNameLength;
	}

	/**
	 * <p>Setter for the field <code>maxSqlNameLength</code>.</p>
	 *
	 * @param maxSqlNameLength a int.
	 * @return a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 */
	public SynchroDatabaseConfiguration setMaxSqlNameLength(int maxSqlNameLength) {
		this.maxSqlNameLength = maxSqlNameLength;
		setProperty(MAX_SQL_NAME_LENGTH, Integer.toString(maxSqlNameLength));
		return this;
	}

	/**
	 * <p>getJdbcUrl.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getJdbcUrl() {
		return getProperty(URL);
	}

	/**
	 * <p>getJdbcUser.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getJdbcUser() {
		return ConfigurationHelper
				.getString(AvailableSettings.USER, properties);
	}

	/**
	 * <p>getJdbcPassword.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getJdbcPassword() {
		return getProperty(PASS);
	}

	/**
	 * <p>getJdbcSchema.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getJdbcSchema() {
		return ConfigurationHelper.getString(AvailableSettings.DEFAULT_SCHEMA,
				properties);
	}

	/**
	 * <p>getJdbcCatalog.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getJdbcCatalog() {
		return ConfigurationHelper.getString(AvailableSettings.DEFAULT_CATALOG,
				properties);
	}

	/**
	 * <p>getJdbcDriver.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getJdbcDriver() {
		return ConfigurationHelper.getString(AvailableSettings.DRIVER,
				properties);
	}

	/**
	 * <p>Getter for the field <code>dialect</code>.</p>
	 *
	 * @return a {@link org.hibernate.dialect.Dialect} object.
	 */
	public Dialect getDialect() {
		return dialect;
	}

	/**
	 * <p>isTarget.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isTarget() {
		return ConfigurationHelper.getBoolean(IS_TARGET_DATABASE, properties,
				false);
	}

	/**
	 * <p>isSource.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isSource() {
		return !isTarget();
	}

	/**
	 * <p>setIsTarget.</p>
	 *
	 * @param isTarget a boolean.
	 * @return a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 */
	public SynchroDatabaseConfiguration setIsTarget(boolean isTarget) {
		setProperty(IS_TARGET_DATABASE, Boolean.toString(isTarget));
		return this;
	}

	/**
	 * <p>setReadOnly.</p>
	 *
	 * @param readonly a boolean.
	 * @return a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 */
	public SynchroDatabaseConfiguration setReadOnly(boolean readonly) {
		setProperty(READ_ONLY, Boolean.toString(readonly));
		return this;
	}

	/**
	 * <p>isReadOnly.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isReadOnly() {
		return ConfigurationHelper.getBoolean(READ_ONLY, properties, false);
	}

	/**
	 * <p>setRollbackOnly.</p>
	 *
	 * @param rollbackOnly a boolean.
	 * @return a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 */
	public SynchroDatabaseConfiguration setRollbackOnly(boolean rollbackOnly) {
		setProperty(ROLLBACK_ONLY, Boolean.toString(rollbackOnly));
		return this;
	}

	/**
	 * <p>isRollbackOnly.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isRollbackOnly() {
		return ConfigurationHelper.getBoolean(ROLLBACK_ONLY, properties, false);
	}

	/**
	 * <p>setSystemTimestamp.</p>
	 *
	 * @param systimestamp a {@link java.sql.Timestamp} object.
	 */
	public void setSystemTimestamp(Timestamp systimestamp) {
		if (systimestamp == null) {
			properties.remove(SYSTEM_TIMESTAMP);
			return;
		}
		long timeInNanos = (systimestamp.getTime() / 1000) * 1000000000
				+ systimestamp.getNanos();
		setProperty(SYSTEM_TIMESTAMP, Long.toString(timeInNanos));
	}

	/**
	 * <p>getSystemTimestamp.</p>
	 *
	 * @return a {@link java.sql.Timestamp} object.
	 */
	public Timestamp getSystemTimestamp() {
		Long timeInNanos = ConfigurationHelper.getLong(SYSTEM_TIMESTAMP,
				properties, -1);
		if (timeInNanos == -1) {
			return null;
		}
		long time = (timeInNanos / 1000000000) * 1000;
		int nanos = (int) (timeInNanos % 1000000000);
		Timestamp result = new Timestamp(time);
		result.setNanos(nanos);
		return result;
	}

	/**
	 * <p>isFullMetadataEnable.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isFullMetadataEnable() {
		return ConfigurationHelper.getBoolean(ENABLE_FULL_METADATA, properties,
				isTarget());
	}

	/**
	 * <p>setFullMetadataEnable.</p>
	 *
	 * @param enable a boolean.
	 */
	public void setFullMetadataEnable(boolean enable) {
		setProperty(ENABLE_FULL_METADATA, Boolean.toString(enable));
	}

	/**
	 * <p>isMirrorDatabase.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isMirrorDatabase() {
		return ConfigurationHelper.getBoolean(IS_MIRROR_DATABASE, properties,
				isTarget());
	}

	/**
	 * <p>setIsMirrorDatabase.</p>
	 *
	 * @param isMirror a boolean.
	 */
	public void setIsMirrorDatabase(boolean isMirror) {
		setProperty(IS_MIRROR_DATABASE, Boolean.toString(isMirror));
	}

	/**
	 * <p>isKeepWhereClauseOnQueriesByFks.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isKeepWhereClauseOnQueriesByFks() {
		return ConfigurationHelper.getBoolean(
				KEEP_WHERE_CLAUSE_ON_QUERIES_BY_FKS, properties, true);
	}

	/**
	 * <p>setKeepWhereClauseOnQueriesByFks.</p>
	 *
	 * @param keepWhereClauseOnQueriesByFks a boolean.
	 */
	public void setKeepWhereClauseOnQueriesByFks(
			boolean keepWhereClauseOnQueriesByFks) {
		setProperty(KEEP_WHERE_CLAUSE_ON_QUERIES_BY_FKS,
				Boolean.toString(keepWhereClauseOnQueriesByFks));
	}

	/**
	 * <p>isCheckUniqueConstraintBetweenInputRows.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isCheckUniqueConstraintBetweenInputRows() {
		return ConfigurationHelper.getBoolean(
				CHECK_UNIQUE_CONSTRAINTS_BETWEEN_INPUT_ROWS, properties, true);
	}

	/**
	 * <p>setCheckUniqueConstraintBetweenInputRows.</p>
	 *
	 * @param checkUniqueConstraintOverInputData a boolean.
	 */
	public void setCheckUniqueConstraintBetweenInputRows(
			boolean checkUniqueConstraintOverInputData) {
		setProperty(CHECK_UNIQUE_CONSTRAINTS_BETWEEN_INPUT_ROWS,
				Boolean.toString(checkUniqueConstraintOverInputData));
	}

	/**
	 * <p>getColumnExcludes.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getColumnExcludes() {
		return ConfigurationHelper.getString(COLUMN_EXCLUDES, properties, null);
	}

	/**
	 * <p>getColumnExcludesAsSet.</p>
	 *
	 * @return a {@link java.util.Set} object.
	 */
	public Set<String> getColumnExcludesAsSet() {
		String columnExcludesProperty = getColumnExcludes();
		if (StringUtils.isBlank(columnExcludesProperty)) {
			return null;
		}
		Set<String> columnExcludes = Sets.newHashSet(Arrays
				.asList(columnExcludesProperty.split(",")));
		return columnExcludes;
	}

	/**
	 * <p>addColumnExclude.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @param columnName a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 */
	public SynchroDatabaseConfiguration addColumnExclude(String tableName,
			String columnName) {
		Preconditions.checkNotNull(properties);
		Preconditions.checkArgument(StringUtils.isNotBlank(tableName));
		Preconditions.checkArgument(StringUtils.isNotBlank(columnName));

		String excludesProperty = getProperty(COLUMN_EXCLUDES);
		Set<String> excludes = Sets.newHashSet();
		if (excludesProperty != null) {
			excludes.addAll(Arrays.asList(excludesProperty.split(",")));
		}

		excludes.add(tableName.toLowerCase() + "." + columnName.toLowerCase());

		// Store in properties
		excludesProperty = Joiner.on(',').join(excludes);
		setProperty(COLUMN_EXCLUDES, excludesProperty);

		return this;
	}

	/**
	 * <p>addColumnExclude.</p>
	 *
	 * @param columnName a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 */
	public SynchroDatabaseConfiguration addColumnExclude(String columnName) {
		return addColumnExclude("%", columnName);
	}

	/**
	 * <p>addColumnExcludes.</p>
	 *
	 * @param columnExcludes a {@link java.util.Set} object.
	 * @return a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 */
	public SynchroDatabaseConfiguration addColumnExcludes(
			Set<String> columnExcludes) {
		Preconditions.checkNotNull(properties);
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(columnExcludes));

		String excludesProperty = getProperty(COLUMN_EXCLUDES);
		Set<String> excludes = Sets.newHashSet();
		if (excludesProperty != null) {
			excludes.addAll(Splitter.on(',').trimResults()
					.splitToList(excludesProperty));
		}
		for (String exclude : columnExcludes) {
			excludes.add(exclude.toLowerCase());
		}

		// Store in properties
		excludesProperty = Joiner.on(',').join(excludes);
		setProperty(COLUMN_EXCLUDES, excludesProperty);

		return this;
	}

	/**
	 * <p>isSynonymsEnable.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isSynonymsEnable() {
		return ConfigurationHelper.getBoolean(
				AvailableSettings.ENABLE_SYNONYMS, properties, false);
	}

	/**
	 * <p>removeColumnExclude.</p>
	 *
	 * @param columnName a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 */
	public SynchroDatabaseConfiguration removeColumnExclude(String columnName) {
		return removeColumnExclude("%", columnName);
	}

	/**
	 * <p>removeColumnExclude.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @param columnName a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 */
	public SynchroDatabaseConfiguration removeColumnExclude(String tableName,
			String columnName) {
		Preconditions.checkNotNull(properties);
		Preconditions.checkArgument(StringUtils.isNotBlank(tableName));
		Preconditions.checkArgument(StringUtils.isNotBlank(columnName));

		String excludesProperty = getProperty(COLUMN_EXCLUDES);
		Set<String> excludes = Sets.newHashSet();
		if (excludesProperty != null) {
			excludes.addAll(Arrays.asList(excludesProperty.split(",")));
		}

		excludes.remove(tableName.toLowerCase() + "."
				+ columnName.toLowerCase());

		// Store in properties
		excludesProperty = Joiner.on(',').join(excludes);
		setProperty(COLUMN_EXCLUDES, excludesProperty);

		return this;
	}

	/**
	 * <p>Getter for the field <code>remapPks</code>.</p>
	 *
	 * @return a {@link java.util.Map} object.
	 */
	public Map<String, Map<String, String>> getRemapPks() {
		return remapPks;
	}

	/**
	 * <p>clearRemapPks.</p>
	 */
	public void clearRemapPks() {
		remapPks.clear();
	}

	/**
	 * <p>addPkRemap.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @param sourcePkStr a {@link java.lang.String} object.
	 * @param targetPkStr a {@link java.lang.String} object.
	 */
	public void addPkRemap(String tableName, String sourcePkStr,
			String targetPkStr) {
		// Skip if equals
		if (Objects.equal(sourcePkStr, targetPkStr)) {
			return;
		}
		// Add this mapping too this table
		if (remapPks.containsKey(tableName)) {
			remapPks.get(tableName).put(sourcePkStr, targetPkStr);
		} else {
			Map<String, String> map = Maps.newHashMap();
			map.put(sourcePkStr, targetPkStr);
			remapPks.put(tableName, map);
		}
	}

	/**
	 * <p>getPropertyAsBoolean.</p>
	 *
	 * @param key a {@link java.lang.String} object.
	 * @param defaultValue a boolean.
	 * @return a boolean.
	 */
	protected boolean getPropertyAsBoolean(String key, boolean defaultValue) {
		String property = getProperty(key);
		if (StringUtils.isBlank(property)) {
			return defaultValue;
		}
		return "true".equalsIgnoreCase(property);
	}

	/**
	 * <p>toString.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String toString() {
		StringBuilder sb = new StringBuilder();
		if (isTarget()) {
			sb.append("Target ");
		} else {
			sb.append("Source ");
		}
		sb.append("database configuration:");
		sb.append("\n    JDBC URL: ").append(getJdbcUrl());
		sb.append("\n    JDBC User: ").append(getJdbcUser());
		sb.append("\n    JDBC Schema: ").append(getJdbcSchema());
		sb.append("\n    readOnly: ").append(isReadOnly());
		sb.append("\n    enableFullMetadata: ").append(isFullMetadataEnable());
		sb.append("\n    is mirror: ").append(isMirrorDatabase());
		sb.append("\n    check unique constraint between input data: ").append(
				isCheckUniqueConstraintBetweenInputRows());

		// Column excludes
		String columnExcludes = getColumnExcludes();
		sb.append("\n    excludes columns: ");
		if (StringUtils.isNotBlank(columnExcludes)) {
			sb.append(columnExcludes);
		}

		return sb.toString();
	}

	/**
	 * <p>copy.</p>
	 *
	 * @param otherBean a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 */
	public void copy(SynchroDatabaseConfiguration otherBean) {
		// Copy all properties
		properties.putAll(otherBean.properties);

		this.remapPks.putAll(otherBean.remapPks);

		// Reload cache
		reloadCachedFields();
	}

	/**
	 * <p>createHibernateConfiguration.</p>
	 *
	 * @return a {@link org.hibernate.cfg.Configuration} object.
	 */
	public Configuration createHibernateConfiguration() {
		return new Configuration().setProperties(properties);
	}
}
