package fr.ifremer.common.synchro.service;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.sql.Timestamp;
import java.util.Properties;
import java.util.Set;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;

import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;

/**
 * A context class, need for referential and data synchronization Created on
 * 5/05/14.
 *
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 * @since 3.5.3
 */
public class SynchroContext {

	protected SynchroDatabaseConfiguration sourceDatabaseConfiguration;

	protected SynchroDatabaseMetadata sourceMeta;

	protected SynchroDatabaseConfiguration targetDatabaseConfiguration;

	protected SynchroDatabaseMetadata targetMeta;

	protected SynchroResult result;

	protected Set<String> tableNames;

	protected Timestamp lastSynchronizationDate;

	/**
	 * Create a new synchro context
	 *
	 * @param tableNames
	 *            Table set to includes. If null, all tables will be processed
	 *            by table filter
	 * @param sourceConnectionProperties
	 *            Connection properties for source database
	 * @param targetConnectionProperties
	 *            Connection properties for target database
	 * @param result
	 *            Store the synchronization result
	 * @return a new synchro context
	 */
	public static SynchroContext newContext(Set<String> tableNames,
			Properties sourceConnectionProperties,
			Properties targetConnectionProperties, SynchroResult result) {
		Preconditions.checkNotNull(sourceConnectionProperties);
		Preconditions.checkNotNull(targetConnectionProperties);

		return newContext(tableNames, new SynchroDatabaseConfiguration(
				sourceConnectionProperties, false),
				new SynchroDatabaseConfiguration(targetConnectionProperties,
						true), result);
	}

	/**
	 * Create a new synchro context
	 *
	 * @param tableNames
	 *            Table set to includes. If null, all tables will be processed
	 *            by table filter
	 * @param source
	 *            Connection configuration for source database
	 * @param target
	 *            Connection configuration for target database
	 * @param result
	 *            Store the synchronization result
	 * @return a new synchro context
	 */
	public static SynchroContext newContext(Set<String> tableNames,
			SynchroDatabaseConfiguration source,
			SynchroDatabaseConfiguration target, SynchroResult result) {
		Preconditions.checkNotNull(source);
		Preconditions.checkNotNull(target);

		SynchroContext context = new SynchroContext();
		context.setTableNames(tableNames);
		context.setSource(source);
		context.setTarget(target);
		context.setResult(result);
		return context;
	}

	/* -- non static methods-- */

	/**
	 * <p>Constructor for SynchroContext.</p>
	 */
	public SynchroContext() {
	}

	/**
	 * <p>Constructor for SynchroContext.</p>
	 *
	 * @param otherBean a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 */
	public SynchroContext(SynchroContext otherBean) {
		copy(otherBean);
	}

	/**
	 * <p>Getter for the field <code>result</code>.</p>
	 *
	 * @return a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 */
	public SynchroResult getResult() {
		return result;
	}

	/**
	 * <p>Setter for the field <code>result</code>.</p>
	 *
	 * @param result a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 */
	public void setResult(SynchroResult result) {
		this.result = result;
	}

	/**
	 * <p>setSource.</p>
	 *
	 * @param sourceSettings a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 */
	public void setSource(SynchroDatabaseConfiguration sourceSettings) {
		this.sourceDatabaseConfiguration = sourceSettings;
	}

	/**
	 * <p>setTarget.</p>
	 *
	 * @param targetSettings a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 */
	public void setTarget(SynchroDatabaseConfiguration targetSettings) {
		this.targetDatabaseConfiguration = targetSettings;
	}

	/**
	 * <p>Setter for the field <code>tableNames</code>.</p>
	 *
	 * @param tableNames a {@link java.util.Set} object.
	 */
	public void setTableNames(Set<String> tableNames) {
		this.tableNames = tableNames;
	}

	/**
	 * <p>getSource.</p>
	 *
	 * @return a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 */
	public SynchroDatabaseConfiguration getSource() {
		return sourceDatabaseConfiguration;
	}

	/**
	 * <p>getTarget.</p>
	 *
	 * @return a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 */
	public SynchroDatabaseConfiguration getTarget() {
		return targetDatabaseConfiguration;
	}

	/**
	 * <p>Getter for the field <code>tableNames</code>.</p>
	 *
	 * @return a {@link java.util.Set} object.
	 */
	public Set<String> getTableNames() {
		return this.tableNames;
	}

	/**
	 * <p>Getter for the field <code>lastSynchronizationDate</code>.</p>
	 *
	 * @return a {@link java.sql.Timestamp} object.
	 */
	public Timestamp getLastSynchronizationDate() {
		return lastSynchronizationDate;
	}

	/**
	 * <p>Setter for the field <code>lastSynchronizationDate</code>.</p>
	 *
	 * @param lastSynchronizationDate a {@link java.sql.Timestamp} object.
	 */
	public void setLastSynchronizationDate(Timestamp lastSynchronizationDate) {
		this.lastSynchronizationDate = lastSynchronizationDate;
	}

	/**
	 * <p>Setter for the field <code>sourceMeta</code>.</p>
	 *
	 * @param sourceMeta a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 */
	public void setSourceMeta(SynchroDatabaseMetadata sourceMeta) {
		this.sourceMeta = sourceMeta;
	}

	/**
	 * <p>Setter for the field <code>targetMeta</code>.</p>
	 *
	 * @param targetMeta a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 */
	public void setTargetMeta(SynchroDatabaseMetadata targetMeta) {
		this.targetMeta = targetMeta;
	}

	/** {@inheritDoc} */
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder("synchronisation context:");
		if (getSource() != null) {
			sb.append("\n  ").append(getSource().toString());
		}
		if (getTarget() != null) {
			sb.append("\n  ").append(getTarget().toString());
		}
		sb.append("\n  table includes: ");
		if (getTableNames() != null) {
			sb.append(Joiner.on(',').join(getTableNames()));
		}
		sb.append("\n  last synchronization date: ").append(
				getLastSynchronizationDate());
		return sb.toString();
	}

	/**
	 * <p>copy.</p>
	 *
	 * @param otherBean a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 */
	public void copy(SynchroContext otherBean) {
		if (otherBean != null) {
			this.result = otherBean.result;
			this.sourceDatabaseConfiguration = otherBean.sourceDatabaseConfiguration;
			this.sourceMeta = otherBean.sourceMeta;
			this.targetDatabaseConfiguration = otherBean.targetDatabaseConfiguration;
			this.targetMeta = otherBean.targetMeta;
			this.tableNames = otherBean.tableNames;
			this.lastSynchronizationDate = otherBean.lastSynchronizationDate;
		}
	}
}
