package fr.ifremer.common.synchro.service;

/*
 * #%L
 * Ifremer JEE Commons :: JDBC Synchronization
 * %%
 * Copyright (C) 2008 - 2016 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;

import java.sql.Timestamp;
import java.util.List;
import java.util.Map;

/**
 * Helper class for reject infos
 * Created by blavenie on 17/09/15.
 */
public class RejectedRow {

	/**
	 * Reject cause
	 */
	public enum Cause {
		/* duplicated row exists, could not insert */
		DUPLICATE_KEY,
		/* the version of the row has changed. */
		BAD_UPDATE_DATE,
		/* row has been previously deleted, but OK */
		DELETED,
		/* row has been previously deleted, could not update it */
		DELETE_ERROR,
		/* row is locked (at database level) */
		LOCKED
	}

	/**
	 * Resolve reject strategy
	 */
	public enum ResolveStrategy {
		KEEP_LOCAL,
		UPDATE,
		DO_NOTHING,
		DUPLICATE
	}

	public String pkStr;
	public Cause cause;
	public String targetPkStr;
	public Timestamp validUpdateDate;
	public String constraintName;

	private static char INFOS_SEPARATOR = ';';
	private static char REJECT_SEPARATOR = '\n';

	private static Joiner REJECT_INFOS_JOINER = Joiner.on(INFOS_SEPARATOR)
			.skipNulls();
	private static Splitter REJECT_INFOS_SPLITTER = Splitter
			.on(INFOS_SEPARATOR).omitEmptyStrings().trimResults();

	/**
	 * <p>appendAsString.</p>
	 *
	 * @param existingRejectedRows a {@link java.util.Map} object.
	 * @param tableName a {@link java.lang.String} object.
	 * @param rowInfo a {@link java.lang.String} object.
	 */
	public static void appendAsString(Map<String, String> existingRejectedRows,
			String tableName, String... rowInfo) {
		StringBuilder rows = new StringBuilder();
		String existingRejects = existingRejectedRows.get(tableName);
		if (existingRejects != null) {
			rows.append(existingRejects).append(REJECT_SEPARATOR);
		}
		existingRejectedRows.put(tableName,
				REJECT_INFOS_JOINER.appendTo(rows,
						rowInfo)
						.toString());
	}

	/**
	 * <p>parseFromString.</p>
	 *
	 * @param existingRejects a {@link java.lang.String} object.
	 * @return a {@link java.util.List} object.
	 */
	public static List<RejectedRow> parseFromString(String existingRejects) {
		if (StringUtils.isBlank(existingRejects)) {
			return null;
		}

		List<RejectedRow> result = Lists.newArrayList();
		for (String rejectStr : Splitter.on(REJECT_SEPARATOR).split(existingRejects)) {
			RejectedRow reject = parse(rejectStr);
			result.add(reject);
		}
		return result;
	}

	/**
	 * <p>parse.</p>
	 *
	 * @param reject a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.service.RejectedRow} object.
	 */
	public static RejectedRow parse(String reject) {
		if (StringUtils.isBlank(reject)) {
			return null;
		}

		RejectedRow result = new RejectedRow();

		// Split infos
		List<String> infoParts = REJECT_INFOS_SPLITTER.splitToList(reject);

		if (infoParts.size() < 2) {
			throw new IllegalArgumentException(String.format(
					"Bad rejected row format: [%s]", reject));
		}

		// infoParts[0] = source pk str
		result.pkStr = infoParts.get(0);

		// infoParts[1] = reject status
		result.cause = Cause.valueOf(infoParts.get(1));

		switch (result.cause) {
			case BAD_UPDATE_DATE :
			    if (infoParts.size() < 4) {
				    throw new IllegalArgumentException(String.format(
						"Bad rejected row format (timestamp value or targetPkStr is missing, after [%s]): [%s]",
									RejectedRow.Cause.BAD_UPDATE_DATE.name(),
									reject));
				}
				// infoParts[2] = valid update date
			    result.validUpdateDate = java.sql.Timestamp.valueOf(infoParts.get(2));
				// infoParts[3] = target pk str
				result.targetPkStr = infoParts.get(3);
				break;
			case DUPLICATE_KEY :
				if (infoParts.size() >= 3) {
					// infoParts[2] = target pk str
					result.targetPkStr = infoParts.get(2);
					// infoParts[3] = unique constraint name
					result.constraintName = infoParts.get(3);
				}
				break;
			case DELETED :
			case DELETE_ERROR :
				if (infoParts.size() >= 3) {
					// infoParts[2] = target pk str
					result.targetPkStr = infoParts.get(2);
				}
				break;
			case LOCKED :
			if (infoParts.size() >= 3) {
				// infoParts[2] = target pk str
				result.targetPkStr = infoParts.get(2);
			}
			break;
			default :
				// No additional infos
				break;
		}
		return result;
	}

	/**
	 * <p>toString.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String toString() {
		StringBuilder result = new StringBuilder();

		// infoParts[0] = source pk str
		// infoParts[1] = reject status
		result.append(pkStr).append(INFOS_SEPARATOR).append(cause);

		switch (cause) {
			case BAD_UPDATE_DATE :
				// infoParts[2] = valid update date
				result.append(INFOS_SEPARATOR).append(
						validUpdateDate.toString());
				// infoParts[3] = target pk str
				result.append(INFOS_SEPARATOR).append(targetPkStr);
				break;
			case DUPLICATE_KEY :
				if (targetPkStr != null) {
					// infoParts[2] = target pk str
					// infoParts[3] = unique constraint name
					result.append(INFOS_SEPARATOR).append(targetPkStr)
							.append(INFOS_SEPARATOR).append(constraintName);
				}
				break;
			case DELETED :
			case DELETE_ERROR :
				if (targetPkStr != null) {
					// infoParts[2] = target pk str
					result.append(INFOS_SEPARATOR).append(targetPkStr);
				}
				break;
			case LOCKED :
			if (targetPkStr != null) {
				// infoParts[2] = target pk str
				result.append(INFOS_SEPARATOR)
						.append(targetPkStr);
			}
			break;
			default :
				// No additional infos
				break;
		}

		return result.toString();
	}

	/**
	 * Inverse source/target pk String
	 */
	public void inverse() {
		if (targetPkStr != null) {
			String tempPkStr = pkStr;
			pkStr = targetPkStr;
			targetPkStr = tempPkStr;
		}
	}
}
