package fr.ifremer.common.synchro.query.internal;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.util.List;
import java.util.Map;

import com.google.common.collect.Maps;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;

import com.google.common.collect.Lists;

import fr.ifremer.common.synchro.query.SynchroQueryBuilder;
import fr.ifremer.common.synchro.query.SynchroQueryName;

/**
 * SELECT Query
 */
public class SynchroSelectQuery extends SynchroAbstractQuery {
	protected String selectClause;
	protected String fromClause;
	protected List<String> joins = null;
	protected List<String> groupyByColumns = null;
	protected Map<String, String> orderByColumns = null;
	protected boolean columnDistinct;
	protected String havingCondition = null;

	/**
	 * <p>Constructor for SynchroSelectQuery.</p>
	 *
	 * @param queryName a {@link fr.ifremer.common.synchro.query.SynchroQueryName} object.
	 * @param tableName a {@link java.lang.String} object.
	 * @param columnNames a {@link java.util.List} object.
	 * @param fromClause a {@link java.lang.String} object.
	 * @param whereClause a {@link java.lang.String} object.
	 */
	public SynchroSelectQuery(SynchroQueryName queryName, String tableName,
			List<String> columnNames, String fromClause, String whereClause) {
		super(queryName, tableName, columnNames, true);
		setWhereClause(whereClause);
		this.fromClause = fromClause;
		this.columnDistinct = false;
	}

	/**
	 * <p>Constructor for SynchroSelectQuery.</p>
	 *
	 * @param queryName a {@link fr.ifremer.common.synchro.query.SynchroQueryName} object.
	 * @param tableName a {@link java.lang.String} object.
	 * @param columnNames a {@link java.util.List} object.
	 * @param whereClause a {@link java.lang.String} object.
	 */
	public SynchroSelectQuery(SynchroQueryName queryName, String tableName,
			List<String> columnNames, String whereClause) {
		super(queryName, tableName, columnNames, true);
		setWhereClause(whereClause);
		this.fromClause = null;
		this.columnDistinct = false;
	}

	/**
	 * <p>toSql.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String toSql() {
		// Select clause
		StringBuilder selectParams = new StringBuilder();
		if (selectClause != null) {
			selectParams.append("  ") // 2 empty char, for the next substring(2)
					.append(selectClause);
		} else {
			String tableAliasSubstitute = NO_ALIAS_STR;
			if (StringUtils.isNotBlank(tableAlias)) {
				tableAliasSubstitute = tableAlias + ".";
			}
			for (String columnName : columnNames) {
				if (StringUtils.isNotBlank(tableAlias)) {
					columnName = columnName.replace(ALIAS_VAR,
							tableAliasSubstitute);
				}
				selectParams.append(", ").append(columnName);
			}
		}

		// From clause
		StringBuilder fromParams = new StringBuilder();
		if (StringUtils.isNotBlank(fromClause)) {
			fromParams.append(fromClause);
		} else {
			fromParams.append(tableName);
			if (StringUtils.isNotBlank(tableAlias)) {
				fromParams.append(' ').append(tableAlias);
			}
		}

		// From clause > Join
		StringBuilder joinBuffer = new StringBuilder();
		if (CollectionUtils.isNotEmpty(joins)) {
			for (String join : joins) {
				joinBuffer.append(" ").append(join.trim());
			}
		}

		String sql = String.format("SELECT %s%s FROM %s%s%s%s%s%s",
				columnDistinct ? "DISTINCT " : "", selectParams.substring(2),
				fromParams.toString(), joinBuffer.toString(),
				getSqlWhereClause(), getGroupBySql(), getHavingSql(),
				getOrderBySql());
		return sql;
	}

	/** {@inheritDoc} */
	@Override
	public SynchroQueryBuilder setColumnDistinct(boolean columnDistinct) {
		this.columnDistinct = columnDistinct;
		return this;
	}

	/** {@inheritDoc} */
	@Override
	public SynchroQueryBuilder addJoin(String joinClause) {
		if (joins == null) {
			joins = Lists.newArrayList(joinClause);
		} else {
			joins.add(joinClause);
		}
		return this;
	}

	/** {@inheritDoc} */
	@Override
	public SynchroQueryBuilder addGroupByColumn(String columnName) {
		if (groupyByColumns == null) {
			groupyByColumns = Lists.newArrayList(columnName);
		} else {
			groupyByColumns.add(columnName);
		}
		return this;
	}

	/** {@inheritDoc} */
	@Override
	public SynchroQueryBuilder setHavingCondition(String havingCondition) {
		this.havingCondition = havingCondition;
		return this;
	}

	/** {@inheritDoc} */
	@Override
	public SynchroQueryBuilder addOrderByColumn(String columnName,
			boolean ascOrder) {
		if (orderByColumns == null) {
			orderByColumns = Maps.newLinkedHashMap();
		}
		orderByColumns.put(columnName, ascOrder ? "ASC" : "DESC");
		return this;
	}

	/* -- internal columns -- */

	/**
	 * <p>getGroupBySql.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	protected String getGroupBySql() {
		if (CollectionUtils.isEmpty(groupyByColumns)) {
			return "";
		}
		StringBuilder sb = new StringBuilder();
		for (String columnName : groupyByColumns) {
			if (sb.length() > 0) {
				sb.append(", ");
			}
			sb.append(columnName);

		}

		sb.insert(0, " GROUP BY ");

		return sb.toString();
	}

	/**
	 * <p>getHavingSql.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	protected String getHavingSql() {
		if (StringUtils.isBlank(havingCondition)) {
			return "";
		}

		String tableAliasSubstitute = NO_ALIAS_STR;
		if (StringUtils.isNotBlank(tableAlias)) {
			tableAliasSubstitute = tableAlias + ".";
		}
		String sql = havingCondition
				.replaceAll(ALIAS_VAR, tableAliasSubstitute);

		return " HAVING " + sql;
	}

	/**
	 * <p>getOrderBySql.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	protected String getOrderBySql() {
		if (MapUtils.isEmpty(orderByColumns)) {
			return "";
		}
		StringBuilder sb = new StringBuilder();
		for (Map.Entry<String, String> entry : orderByColumns.entrySet()) {
			String columnName = entry.getKey();
			String order = entry.getValue();
			if (sb.length() > 0) {
				sb.append(", ");
			}
			sb.append(columnName).append(' ').append(order);
		}

		sb.insert(0, " ORDER BY ");

		String tableAliasSubstitute = NO_ALIAS_STR;
		if (StringUtils.isNotBlank(tableAlias)) {
			tableAliasSubstitute = tableAlias + ".";
		}

		return sb.toString().replaceAll(ALIAS_VAR, tableAliasSubstitute);
	}
}
