package fr.ifremer.common.synchro.query.internal;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import fr.ifremer.common.synchro.query.SynchroQueryBuilder;
import fr.ifremer.common.synchro.query.SynchroQueryName;

/**
 * INSERT Query
 */
public class SynchroInsertQuery extends SynchroAbstractQuery {

	protected Map<String, String> columnValues;

	/**
	 * <p>Constructor for SynchroInsertQuery.</p>
	 *
	 * @param queryName a {@link fr.ifremer.common.synchro.query.SynchroQueryName} object.
	 * @param tableName a {@link java.lang.String} object.
	 * @param columnNames a {@link java.util.List} object.
	 */
	public SynchroInsertQuery(SynchroQueryName queryName, String tableName,
			List<String> columnNames) {
		super(queryName, tableName, columnNames, false);
		this.columnValues = Maps.newHashMap();
	}

	/**
	 * <p>Constructor for SynchroInsertQuery.</p>
	 *
	 * @param queryName a {@link fr.ifremer.common.synchro.query.SynchroQueryName} object.
	 * @param tableName a {@link java.lang.String} object.
	 * @param columnNames a {@link java.util.List} object.
	 * @param columnValues a {@link java.util.List} object.
	 */
	public SynchroInsertQuery(SynchroQueryName queryName, String tableName,
			List<String> columnNames, List<String> columnValues) {
		super(queryName, tableName, columnNames, false);
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(columnNames));
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(columnValues));
		Preconditions.checkArgument(columnNames.size() == columnValues.size());

		this.columnValues = Maps.newHashMap();
		int i = 0;
		for (String columnValue : columnValues) {
			String columnName = columnNames.get(i++).toLowerCase();
			this.columnValues.put(columnName, columnValue);
		}
	}

	/**
	 * <p>Constructor for SynchroInsertQuery.</p>
	 *
	 * @param queryName a {@link fr.ifremer.common.synchro.query.SynchroQueryName} object.
	 * @param tableName a {@link java.lang.String} object.
	 */
	public SynchroInsertQuery(SynchroQueryName queryName, String tableName) {
		super(queryName, tableName, false);
		this.columnValues = Maps.newHashMap();
	}

	/** {@inheritDoc} */
	@Override
	public SynchroQueryBuilder addColumn(String columnName, String value) {
		super.addColumn(columnName, value);
		columnValues.put(columnName.toLowerCase(), value);
		return this;
	}

	/** {@inheritDoc} */
	@Override
	public SynchroQueryBuilder setColumnValue(String columnName, String value) {
		Preconditions.checkArgument(columnNames.contains(columnName));
		columnValues.put(columnName, value);
		return this;
	}

	/** {@inheritDoc} */
	@Override
	public String getColumnValue(String columnName) {
		return columnValues.get(columnName);
	}

	/** {@inheritDoc} */
	@Override
	public SynchroQueryBuilder deleteColumn(String columnName) {
		super.deleteColumn(columnName);
		columnValues.remove(columnName.toLowerCase());
		return this;
	}

	/** {@inheritDoc} */
	@Override
	public SynchroQueryBuilder deleteColumnIfExists(String columnName) {
		super.deleteColumnIfExists(columnName);
		columnValues.remove(columnName.toLowerCase());
		return this;
	}

	/** {@inheritDoc} */
	public boolean constainsColumn(String columnName) {
		return columnNames.contains(columnName);
	}

	/**
	 * <p>getBindingColumnNames.</p>
	 *
	 * @return a {@link java.util.List} object.
	 */
	public List<String> getBindingColumnNames() {
		List<String> result = Lists.newArrayList();

		for (String columnName : columnNames) {
			String columnValue = columnValues.get(columnName);
			if ("?".equals(columnValue) || columnValue.startsWith(":")) {
				result.add(columnName);
			}
		}
		return ImmutableList.copyOf(result);
	}

	/** {@inheritDoc} */
	@Override
	public String toSql() {
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(columnNames));
		StringBuilder queryParams = new StringBuilder();
		StringBuilder valueParams = new StringBuilder();

		for (String columnName : columnNames) {
			queryParams.append(", ").append(columnName);
			String value = columnValues.get(columnName);
			if (value != null) {
				valueParams.append(", ").append(value);
			} else {
				valueParams.append(", ?");
			}
		}

		String sql = String.format("INSERT INTO %s (%s) VALUES (%s)",
				tableName, queryParams.substring(2), valueParams.substring(2));

		return sql;
	}
}
