package fr.ifremer.common.synchro.query.internal;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map.Entry;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import fr.ifremer.common.synchro.query.SynchroQueryBuilder;
import fr.ifremer.common.synchro.query.SynchroQueryName;
import fr.ifremer.common.synchro.query.SynchroQueryOperator;

/**
 * <p>Abstract SynchroAbstractQuery class.</p>
 *
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 */
public abstract class SynchroAbstractQuery extends SynchroQueryBuilder {

	protected final SynchroQueryName queryName;
	protected String tableName;
	protected String tableAlias;
	protected List<String> columnNames;
	protected LinkedHashMap<String, SynchroQueryOperator> whereClauses = null;
	protected boolean allowWhereClause;

	/**
	 * <p>Constructor for SynchroAbstractQuery.</p>
	 *
	 * @param queryName a {@link fr.ifremer.common.synchro.query.SynchroQueryName} object.
	 * @param tableName a {@link java.lang.String} object.
	 * @param columnNames a {@link java.util.List} object.
	 * @param allowWhereClause a boolean.
	 */
	protected SynchroAbstractQuery(SynchroQueryName queryName,
			String tableName, List<String> columnNames, boolean allowWhereClause) {
		super();
		this.queryName = queryName;
		this.tableName = tableName;
		this.whereClauses = null;
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(columnNames));
		// Convert names in lower case
		this.columnNames = Lists.newArrayListWithExpectedSize(columnNames
				.size());
		for (String columnName : columnNames) {
			this.columnNames.add(columnName.toLowerCase());
		}
		this.allowWhereClause = allowWhereClause;
	}

	/**
	 * <p>Constructor for SynchroAbstractQuery.</p>
	 *
	 * @param queryName a {@link fr.ifremer.common.synchro.query.SynchroQueryName} object.
	 * @param tableName a {@link java.lang.String} object.
	 * @param allowWhereClause a boolean.
	 */
	protected SynchroAbstractQuery(SynchroQueryName queryName,
			String tableName, boolean allowWhereClause) {
		super();
		this.queryName = queryName;
		this.tableName = tableName;
		this.columnNames = Lists.newArrayList();
		this.whereClauses = null;
		this.allowWhereClause = allowWhereClause;
	}

	/**
	 * <p>Constructor for SynchroAbstractQuery.</p>
	 *
	 * @param allowWhereClause a boolean.
	 */
	protected SynchroAbstractQuery(boolean allowWhereClause) {
		super();
		this.queryName = null;
		this.tableName = null;
		this.columnNames = null;
		this.whereClauses = null;
		this.allowWhereClause = allowWhereClause;
	}

	/**
	 * <p>Setter for the field <code>allowWhereClause</code>.</p>
	 *
	 * @param allowWhereClause a boolean.
	 * @return a {@link fr.ifremer.common.synchro.query.SynchroQueryBuilder} object.
	 */
	public SynchroQueryBuilder setAllowWhereClause(boolean allowWhereClause) {
		this.allowWhereClause = allowWhereClause;
		return this;
	}

	/**
	 * <p>Setter for the field <code>tableAlias</code>.</p>
	 *
	 * @param tableAlias a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.query.SynchroQueryBuilder} object.
	 */
	public SynchroQueryBuilder setTableAlias(String tableAlias) {
		this.tableAlias = tableAlias;
		return this;
	}

	/**
	 * <p>Getter for the field <code>tableName</code>.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	protected String getTableName() {
		return this.tableName;
	}

	/** {@inheritDoc} */
	@Override
	public String toString() {
		return String.format("%s [%s]",
				queryName == null ? "" : queryName.name(), toSql());
	}

	/**
	 * <p>getColumnCount.</p>
	 *
	 * @return a int.
	 */
	public int getColumnCount() {
		return columnNames != null ? columnNames.size() : 0;
	}

	/** {@inheritDoc} */
	public SynchroQueryBuilder replaceColumn(String oldColumn, String newColumn) {
		oldColumn = oldColumn.toLowerCase();
		Preconditions.checkArgument(columnNames.contains(oldColumn)
				|| (StringUtils.isNotBlank(tableAlias) && columnNames
						.contains(ALIAS_VAR + oldColumn)));
		Preconditions.checkArgument(!columnNames.contains(newColumn));
		int index = columnNames.indexOf(oldColumn);
		if (index == -1) {
			index = columnNames.indexOf(ALIAS_VAR + oldColumn);
		}
		columnNames.set(index, newColumn);
		return this;
	}

	/** {@inheritDoc} */
	public SynchroQueryBuilder addColumn(String columnName, String value) {
		columnName = columnName.toLowerCase();
		Preconditions.checkArgument(!columnNames.contains(columnName));
		columnNames.add(columnName);
		return this;
	}

	/** {@inheritDoc} */
	@Override
	public List<String> getColumnNames() {
		if (StringUtils.isBlank(tableAlias)) {
			return columnNames != null
					? ImmutableList.copyOf(columnNames)
					: null;
		}

		// Remove alias
		List<String> result = Lists
				.newArrayListWithCapacity(columnNames.size());
		for (String columnName : columnNames) {
			result.add(columnName.replace(ALIAS_VAR, ""));
		}
		return result;
	}

	/** {@inheritDoc} */
	@Override
	public List<String> getColumnNamesWithAlias() {
		return columnNames != null ? ImmutableList.copyOf(columnNames) : null;
	}

	/** {@inheritDoc} */
	public SynchroQueryBuilder deleteColumn(String columnName) {
		columnName = columnName.toLowerCase();
		boolean isRemoved = columnNames.remove(columnName)
				|| (StringUtils.isNotBlank(tableAlias) && columnNames
						.remove(ALIAS_VAR + columnName));
		Preconditions
				.checkArgument(
						isRemoved,
						isRemoved
								? ""
								: String.format(
										"Could not remove column [%s] from query on table [%s]: column not found",
										columnName, tableName));
		return this;
	}

	/** {@inheritDoc} */
	public SynchroQueryBuilder deleteColumnIfExists(String columnName) {
		columnName = columnName.toLowerCase();
		@SuppressWarnings("unused")
		boolean isRemoved = columnNames.remove(columnName)
				|| ((StringUtils.isNotBlank(tableAlias)) && columnNames
						.remove(ALIAS_VAR + columnName));
		return this;
	}

	/** {@inheritDoc} */
	public SynchroQueryBuilder setColumnValue(String columnName, String value) {
		throw new UnsupportedOperationException("not implement for query "
				+ getClass().getSimpleName());
	}

	/** {@inheritDoc} */
	public boolean constainsColumn(String columnName) {
		return columnNames.contains(columnName);
	}

	/** {@inheritDoc} */
	public SynchroQueryBuilder setWhereClause(String whereClause) {
		if (!allowWhereClause) {
			throw new UnsupportedOperationException("not allowed for query "
					+ getClass().getSimpleName());
		}
		if (StringUtils.isBlank(whereClause)) {
			this.whereClauses = null;
		} else {
			this.whereClauses = Maps.newLinkedHashMap();
			this.whereClauses.put(whereClause, SynchroQueryOperator.AND);
		}
		return this;
	}

	/** {@inheritDoc} */
	public SynchroQueryBuilder addWhere(SynchroQueryOperator operator,
			String whereClause) {
		if (!allowWhereClause) {
			throw new UnsupportedOperationException("not allowed for query "
					+ getClass().getSimpleName());
		}
		Preconditions.checkArgument(StringUtils.isNotBlank(whereClause));
		whereClause = whereClause.trim();

		// If whereClause = <COL> = <VALUE> : delegate to next method (to manage
		// tableAlias)
		String[] parts = whereClause.split("[ \t=><]+");
		if (parts.length == 2) {
			String columnName = parts[0];
			String columnValue = parts[1];
			String columnOperator = whereClause.substring(columnName.length());
			columnOperator = columnOperator.substring(0,
					columnOperator.length() - columnValue.length()).trim();
			return addWhere(operator, columnName, columnOperator, columnValue);
		}

		if (this.whereClauses == null) {
			this.whereClauses = Maps.newLinkedHashMap();
		}
		this.whereClauses.put(whereClause, operator);
		return this;

	}

	/** {@inheritDoc} */
	public SynchroQueryBuilder addWhere(SynchroQueryOperator operator,
			String columnName, String columnValue) {
		addWhere(operator, columnName, "=", columnValue);
		return this;

	}

	/** {@inheritDoc} */
	public SynchroQueryBuilder addWhere(SynchroQueryOperator operator,
			String columnName, String columnOperator, String columnValue) {
		if (!allowWhereClause) {
			throw new UnsupportedOperationException("not allowed for query "
					+ getClass().getSimpleName());
		}

		if (this.whereClauses == null) {
			this.whereClauses = Maps.newLinkedHashMap();
		}

		StringBuilder whereParams = new StringBuilder();
		columnName = columnName.toLowerCase();
		// If no alias in column name : insert one tag
		if (columnName.split("[.]").length == 0) {
			whereParams.append(ALIAS_VAR).append(columnName);
		} else {
			whereParams.append(columnName);
		}
		whereParams.append(' ').append(columnOperator).append(' ')
				.append(columnValue);
		this.whereClauses.put(whereParams.toString(), operator);
		return this;

	}

	/** {@inheritDoc} */
	@Override
	public String getColumnValue(String columnName) {
		throw new UnsupportedOperationException("not allowed for query "
				+ getClass().getSimpleName());
	}

	/** {@inheritDoc} */
	public SynchroQueryBuilder setColumnDistinct(boolean columnDistinct) {
		throw new UnsupportedOperationException("not allowed for query "
				+ getClass().getSimpleName());
	}

	/** {@inheritDoc} */
	public SynchroQueryBuilder addJoin(String joinClause) {
		throw new UnsupportedOperationException("not allowed for query "
				+ getClass().getSimpleName());
	}

	/** {@inheritDoc} */
	public SynchroQueryBuilder addGroupByColumn(String groupyByColumn) {
		throw new UnsupportedOperationException("not allowed for query "
				+ getClass().getSimpleName());
	}

	/** {@inheritDoc} */
	public SynchroQueryBuilder setHavingCondition(String havingCondition) {
		throw new UnsupportedOperationException("not allowed for query "
				+ getClass().getSimpleName());
	}

	/** {@inheritDoc} */
	public SynchroQueryBuilder addOrderByColumn(String orderyByColumn,
			boolean ascOrder) {
		throw new UnsupportedOperationException("not allowed for query "
				+ getClass().getSimpleName());
	}

	/**
	 * <p>getSqlWhereClause.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	protected String getSqlWhereClause() {
		if (!allowWhereClause) {
			throw new UnsupportedOperationException("not allowed for query "
					+ getClass().getSimpleName());
		}

		if (MapUtils.isEmpty(whereClauses)) {
			return "";
		}

		StringBuilder whereParams = new StringBuilder();
		for (Entry<String, SynchroQueryOperator> entry : whereClauses
				.entrySet()) {
			if (whereParams.length() == 0) {
				whereParams.append(entry.getKey());
			} else {
				SynchroQueryOperator operator = entry.getValue();
				whereParams.append(" ").append(operator.name()).append(" ")
						.append(entry.getKey());
				// If 'OR', add parenthesis to protected previous condition
				if (operator == SynchroQueryOperator.OR) {
					whereParams.insert(0, "(").append(")");
				}

			}
		}
		whereParams.insert(0, " WHERE ");

		String tableAliasSubstitute = NO_ALIAS_STR;
		if (StringUtils.isNotBlank(tableAlias)) {
			tableAliasSubstitute = tableAlias + ".";
		}
		String sql = whereParams.toString().replaceAll(ALIAS_VAR,
				tableAliasSubstitute);

		return sql;
	}
}
