package fr.ifremer.common.synchro.query;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.util.Arrays;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;

import fr.ifremer.common.synchro.query.internal.SynchroAbstractQuery;
import fr.ifremer.common.synchro.query.internal.SynchroDeleteQuery;
import fr.ifremer.common.synchro.query.internal.SynchroInsertQuery;
import fr.ifremer.common.synchro.query.internal.SynchroSelectQuery;
import fr.ifremer.common.synchro.query.internal.SynchroUpdateQuery;
import org.apache.commons.lang3.StringUtils;

/**
 * <p>Abstract SynchroQueryBuilder class.</p>
 *
 */
public abstract class SynchroQueryBuilder {

	/** Constant <code>COLUMN_VALUE_TABLE_SEQUENCE="~~sequence~~"</code> */
	public static final String COLUMN_VALUE_TABLE_SEQUENCE = "~~sequence~~";

	/** Constant <code>ALIAS_VAR="~~alias~~"</code> */
	protected static final String ALIAS_VAR = "~~alias~~";
	/** Constant <code>NO_ALIAS_STR=""</code> */
	protected static final String NO_ALIAS_STR = "";

	// DO NOT remove the regex part '.*?' (mandatory for query on
	// OBSERVED_LOCATION - see mantis #24103)
	private static Pattern SELECT_FROM_PATTERN = Pattern
			.compile("^[\\s]*SELECT(.*?)FROM(.+)");
	private static Pattern WHERE_PATTERN = Pattern.compile("^(.*?)WHERE(.+)");
	private static Pattern ORDERBY_PATTERN = Pattern
			.compile("^(.*?)ORDER BY(.+)");
	private static Pattern COLUMN_WITH_ALIAS_PATTERN = Pattern
			.compile("([a-zA-Z0-9_]+)[.]([a-zA-Z0-9_]+)");
	private static Pattern COLUMN_WITH_ALIAS_PATTERN_ORDER = Pattern
			.compile("([a-zA-Z0-9_]+)[.]([a-zA-Z0-9_]+)([ ]+ASC|DESC)?");

	/**
	 * <p>newBuilder.</p>
	 *
	 * @param sql a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.query.SynchroQueryBuilder} object.
	 */
	public static SynchroQueryBuilder newBuilder(String sql) {
		return newBuilder(null, sql);
	}

	/**
	 * <p>newBuilder.</p>
	 *
	 * @param queryName a {@link fr.ifremer.common.synchro.query.SynchroQueryName} object.
	 * @param sql a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.query.SynchroQueryBuilder} object.
	 */
	public static SynchroQueryBuilder newBuilder(SynchroQueryName queryName,
			String sql) {
		Preconditions.checkNotNull(sql);

		String upperCaseQuery = sql.trim().toUpperCase();

		SynchroAbstractQuery query = null;
		if (upperCaseQuery.startsWith("SELECT")) {
			query = parseSelectQuery(queryName, sql);
		}
		if (upperCaseQuery.startsWith("INSERT")) {
			query = parseInsertQuery(queryName, sql);
		}
		if (upperCaseQuery.startsWith("UPDATE")) {
			query = parseUpdateQuery(queryName, sql);
		}
		if (upperCaseQuery.startsWith("DELETE")) {
			query = parseDeleteQuery(queryName, sql);
		}
		if (query == null) {
			throw new IllegalArgumentException(String.format(
					"Unable to parse sql: %s", sql));
		}

		return query;
	}

	/**
	 * <p>parseSelectQuery.</p>
	 *
	 * @param queryName a {@link fr.ifremer.common.synchro.query.SynchroQueryName} object.
	 * @param sql a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.query.internal.SynchroAbstractQuery} object.
	 */
	protected static SynchroAbstractQuery parseSelectQuery(
			SynchroQueryName queryName, String sql) {

		Matcher matcher = SELECT_FROM_PATTERN.matcher(sql);
		if (!matcher.matches()) {
			throw new IllegalArgumentException(
					String.format(
							"Bad sql query: [%s]. Expected: SELECT <...> FROM <...> WHERE <...>",
							sql));
		}

		String selectClause = matcher.group(1).trim();
		String fromClause = matcher.group(2).trim();

		// Where clause
		matcher = WHERE_PATTERN.matcher(fromClause);
		String whereClause = null;
		if (matcher.matches()) {
			fromClause = matcher.group(1).trim();
			whereClause = matcher.group(2).trim();
		}

		// Order by
		String orderByClause = null;
		if (StringUtils.isNotBlank(whereClause)) {
			matcher = ORDERBY_PATTERN.matcher(whereClause);
			if (matcher.matches()) {
				whereClause = matcher.group(1).trim();
				orderByClause = matcher.group(2).trim();
			}
		}

		// Select part
		List<String> columnNames = Lists.newArrayList();
		String tableAlias = null;
		boolean hasColumnDistinct = false;
		{
			if (selectClause.startsWith("DISTINCT")) {
				selectClause = selectClause.substring("DISTINCT".length())
						.trim();
				hasColumnDistinct = true;
			}
			Iterable<String> parts = Splitter.on(',').trimResults()
					.omitEmptyStrings().split(selectClause);
			for (String columnName : parts) {
				Matcher columnAliasMatcher = COLUMN_WITH_ALIAS_PATTERN
						.matcher(columnName);
				if (columnAliasMatcher.matches()) {
					// Set table alias only once
					if (tableAlias == null) {
						tableAlias = columnAliasMatcher.group(1);
					}
					columnName = ALIAS_VAR + columnAliasMatcher.group(2);
				}
				columnNames.add(columnName);
			}
		}

		// From part
		String tableName;
		{
			String[] parts = fromClause.split("[ \t]+");
			if (parts.length == 2) {
				tableName = parts[0];
				tableAlias = parts[1];
				fromClause = null;
			} else {
				tableName = null;
			}
		}

		SynchroAbstractQuery selectQuery = new SynchroSelectQuery(queryName,
				tableName, columnNames, fromClause, whereClause);
		selectQuery.setTableAlias(tableAlias);
		selectQuery.setColumnDistinct(hasColumnDistinct);

		if (StringUtils.isNotBlank(orderByClause)) {
			Iterable<String> parts = Splitter.on(',').trimResults()
					.omitEmptyStrings().split(orderByClause);
			for (String columnName : parts) {
				boolean ascOrder = true;
				String[] columnParts = columnName.split("\\s+");
				if (columnParts.length == 2) {
					columnName = columnParts[0];
					String orderSide = columnParts[1];
					ascOrder = !"DESC".equalsIgnoreCase(orderSide.trim());
				}
				Matcher columnAliasMatcher = COLUMN_WITH_ALIAS_PATTERN
						.matcher(columnName);
				if (columnAliasMatcher.matches()) {
					columnName = ALIAS_VAR + columnAliasMatcher.group(2);
				}
				selectQuery.addOrderByColumn(columnName, ascOrder);
			}
		}

		return selectQuery;
	}

	/**
	 * <p>parseInsertQuery.</p>
	 *
	 * @param queryName a {@link fr.ifremer.common.synchro.query.SynchroQueryName} object.
	 * @param sql a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.query.internal.SynchroAbstractQuery} object.
	 */
	protected static SynchroAbstractQuery parseInsertQuery(
			SynchroQueryName queryName, String sql) {
		String[] parts = sql.split("(INTO)|(VALUES[ \t\\(]+)");
		if (parts.length != 2 && parts.length != 3) {
			throw new IllegalArgumentException(
					String.format(
							"Bad sql query: [%s]. Expected: INSERT INTO <...> <...> VALUES <...>",
							sql));
		}
		String[] part1 = parts[1].trim().split("[ \t,\\(\\)]+");
		List<String> columnNames = Lists.newArrayList(Arrays.asList(part1));
		String tableName = columnNames.remove(0);

		// remove last parenthesis
		if (parts[2].lastIndexOf(")") == parts[2].length() - 1) {
			parts[2] = parts[2].substring(0, parts[2].length() - 1);
		}
		String[] part2 = parts[2].trim().split("[\t,\\(\\)]*,[ \t\\(\\)]*");
		List<String> columnvalues = Lists.newArrayList(Arrays.asList(part2));
		if (columnvalues.size() != columnNames.size()) {
			throw new IllegalArgumentException(
					String.format(
							"Bad sql query: [%s]. Not same number of columns and values",
							sql));
		}
		SynchroAbstractQuery query = new SynchroInsertQuery(queryName,
				tableName, columnNames, columnvalues);
		return query;
	}

	/**
	 * <p>parseUpdateQuery.</p>
	 *
	 * @param queryName a {@link fr.ifremer.common.synchro.query.SynchroQueryName} object.
	 * @param sql a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.query.internal.SynchroAbstractQuery} object.
	 */
	protected static SynchroAbstractQuery parseUpdateQuery(
			SynchroQueryName queryName, String sql) {

		String[] parts = sql.split("(UPDATE)|(SET)|(WHERE)");
		if (parts.length != 3 && parts.length != 4) {
			throw new IllegalArgumentException(
					String.format(
							"Bad sql query: [%s]. Expected: UPDATE <...> SET <...> WHERE <...>",
							sql));
		}
		String tableName = parts[1].trim();

		String[] colsAndVal = parts[2].trim().split(",");
		List<String> columnNames = Lists
				.newArrayListWithExpectedSize(colsAndVal.length);
		List<String> columnValues = Lists
				.newArrayListWithExpectedSize(colsAndVal.length);
		for (String colAndVal : colsAndVal) {
			String[] colAndValParts = colAndVal.trim().split("[ \t=]+");
			if (colAndValParts.length != 2) {
				throw new IllegalArgumentException(
						String.format(
								"Bad sql query: [%s]. Expected: UPDATE <...> SET <...> WHERE <...>",
								sql));
			}
			columnNames.add(colAndValParts[0]);
			columnValues.add(colAndValParts[1]);
		}
		String whereClause = null;
		if (parts.length == 4) {
			whereClause = parts[3].trim();
		}
		SynchroUpdateQuery query = new SynchroUpdateQuery(queryName, tableName,
				columnNames, columnValues);
		query.setWhereClause(whereClause);
		return query;
	}

	/**
	 * <p>parseDeleteQuery.</p>
	 *
	 * @param queryName a {@link fr.ifremer.common.synchro.query.SynchroQueryName} object.
	 * @param sql a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.query.internal.SynchroAbstractQuery} object.
	 */
	protected static SynchroAbstractQuery parseDeleteQuery(
			SynchroQueryName queryName, String sql) {

		SynchroAbstractQuery selectQuery = new SynchroDeleteQuery(queryName,
				null);
		return selectQuery;
	}

	/**
	 * <p>build.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String build() {
		return toSql();
	}

	/* -- public abstract method -- */

	// getters :
	/**
	 * <p>toSql.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public abstract String toSql();

	/**
	 * <p>constainsColumn.</p>
	 *
	 * @param columnName a {@link java.lang.String} object.
	 * @return a boolean.
	 */
	public abstract boolean constainsColumn(String columnName);

	/**
	 * <p>getColumnCount.</p>
	 *
	 * @return a int.
	 */
	public abstract int getColumnCount();

	/**
	 * <p>getColumnValue.</p>
	 *
	 * @param columnName a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	public abstract String getColumnValue(String columnName);

	/**
	 * <p>getColumnNames.</p>
	 *
	 * @return a {@link java.util.List} object.
	 */
	public abstract List<String> getColumnNames();

	/**
	 * <p>getColumnNamesWithAlias.</p>
	 *
	 * @return a {@link java.util.List} object.
	 */
	public abstract List<String> getColumnNamesWithAlias();

	// operations :

	/**
	 * <p>replaceColumn.</p>
	 *
	 * @param oldColumn a {@link java.lang.String} object.
	 * @param newColumn a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.query.SynchroQueryBuilder} object.
	 */
	public abstract SynchroQueryBuilder replaceColumn(String oldColumn,
			String newColumn);

	/**
	 * <p>addColumn.</p>
	 *
	 * @param columnName a {@link java.lang.String} object.
	 * @param value a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.query.SynchroQueryBuilder} object.
	 */
	public abstract SynchroQueryBuilder addColumn(String columnName,
			String value);

	/**
	 * <p>deleteColumn.</p>
	 *
	 * @param columnName a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.query.SynchroQueryBuilder} object.
	 */
	public abstract SynchroQueryBuilder deleteColumn(String columnName);

	/**
	 * <p>deleteColumnIfExists.</p>
	 *
	 * @param columnName a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.query.SynchroQueryBuilder} object.
	 */
	public abstract SynchroQueryBuilder deleteColumnIfExists(String columnName);

	/**
	 * <p>setColumnValue.</p>
	 *
	 * @param columnName a {@link java.lang.String} object.
	 * @param value a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.query.SynchroQueryBuilder} object.
	 */
	public abstract SynchroQueryBuilder setColumnValue(String columnName,
			String value);

	/**
	 * <p>setWhereClause.</p>
	 *
	 * @param whereClause a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.query.SynchroQueryBuilder} object.
	 */
	public abstract SynchroQueryBuilder setWhereClause(String whereClause);

	/**
	 * <p>addWhere.</p>
	 *
	 * @param operator a {@link fr.ifremer.common.synchro.query.SynchroQueryOperator} object.
	 * @param whereClause a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.query.SynchroQueryBuilder} object.
	 */
	public abstract SynchroQueryBuilder addWhere(SynchroQueryOperator operator,
			String whereClause);

	/**
	 * <p>addWhere.</p>
	 *
	 * @param operator a {@link fr.ifremer.common.synchro.query.SynchroQueryOperator} object.
	 * @param columnName a {@link java.lang.String} object.
	 * @param columnValue a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.query.SynchroQueryBuilder} object.
	 */
	public abstract SynchroQueryBuilder addWhere(SynchroQueryOperator operator,
			String columnName, String columnValue);

	/**
	 * <p>addWhere.</p>
	 *
	 * @param operator a {@link fr.ifremer.common.synchro.query.SynchroQueryOperator} object.
	 * @param columnName a {@link java.lang.String} object.
	 * @param columnOperator a {@link java.lang.String} object.
	 * @param columnValue a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.query.SynchroQueryBuilder} object.
	 */
	public abstract SynchroQueryBuilder addWhere(SynchroQueryOperator operator,
			String columnName, String columnOperator, String columnValue);

	/**
	 * <p>setColumnDistinct.</p>
	 *
	 * @param columnDistinct a boolean.
	 * @return a {@link fr.ifremer.common.synchro.query.SynchroQueryBuilder} object.
	 */
	public abstract SynchroQueryBuilder setColumnDistinct(boolean columnDistinct);

	/**
	 * <p>addGroupByColumn.</p>
	 *
	 * @param groupyByColumn a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.query.SynchroQueryBuilder} object.
	 */
	public abstract SynchroQueryBuilder addGroupByColumn(String groupyByColumn);

	/**
	 * <p>setHavingCondition.</p>
	 *
	 * @param havingCondition a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.query.SynchroQueryBuilder} object.
	 */
	public abstract SynchroQueryBuilder setHavingCondition(
			String havingCondition);

	/**
	 * <p>addOrderByColumn.</p>
	 *
	 * @param orderyByColumn a {@link java.lang.String} object.
	 * @param ascOrder a boolean.
	 * @return a {@link fr.ifremer.common.synchro.query.SynchroQueryBuilder} object.
	 */
	public abstract SynchroQueryBuilder addOrderByColumn(String orderyByColumn,
			boolean ascOrder);

	/**
	 * Add a join (inner or outer) to the main table (in the 'FROM' clause)
	 *
	 * @param joinClause a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.query.SynchroQueryBuilder} object.
	 */
	public abstract SynchroQueryBuilder addJoin(String joinClause);
}
