package fr.ifremer.common.synchro.meta;

/*
 * #%L
 * Tutti :: Persistence
 * $Id: ReferentialSynchroTableMetadata.java 1573 2014-02-04 16:41:40Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/trunk/tutti-persistence/src/main/java/fr/ifremer/adagio/core/service/technical/synchro/ReferentialSynchroTableMetadata.java $
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.*;
import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.SubscriberExceptionContext;
import com.google.common.eventbus.SubscriberExceptionHandler;
import fr.ifremer.common.synchro.SynchroTechnicalException;
import fr.ifremer.common.synchro.dao.Daos;
import fr.ifremer.common.synchro.intercept.SynchroInterceptor;
import fr.ifremer.common.synchro.meta.event.CreateQueryEvent;
import fr.ifremer.common.synchro.meta.event.LoadJoinEvent;
import fr.ifremer.common.synchro.meta.event.LoadPkEvent;
import fr.ifremer.common.synchro.meta.event.LoadTableEvent;
import fr.ifremer.common.synchro.query.SynchroQueryBuilder;
import fr.ifremer.common.synchro.query.SynchroQueryName;
import fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.LockMode;
import org.hibernate.dialect.Dialect;
import org.hibernate.mapping.ForeignKey;
import org.hibernate.tool.hbm2ddl.ColumnMetadata;
import org.hibernate.tool.hbm2ddl.ForeignKeyMetadata;
import org.hibernate.tool.hbm2ddl.IndexMetadata;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import java.lang.reflect.Field;
import java.sql.*;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import static org.nuiton.i18n.I18n.t;

/**
 * Overrides of the {@link org.hibernate.tool.hbm2ddl.TableMetadata} with some improvements:
 * <ul>
 * <li>Obtains number of columns via {@link #getColumnsCount()}</li>
 * <li>Obtains all columns names available via {@link #getColumnNames()}</li>
 * <li>Obtains number of columns in the select query, via
 * {@link #getSelectColumnsCount()}</li>
 * <li>Obtains all columns names available in the select query via
 * {@link #getSelectColumnNames()}</li>
 * <li>Obtains primary key column names via {@link #getPkNames()}</li>
 * </ul>
 * <br>
 * And others methods used to synchronize referentials:
 * <ul>
 * <li>Obtains query to update a row of the table (column names order is the one
 * introduced by method {@link #getColumnNames()}: {@link #getUpdateQuery()}</li>
 * <li>Obtains query to insert a row in the table (column names order is the one
 * introduced by method {@link #getColumnNames()}: {@link #getInsertQuery()}</li>
 * </ul>
 * Created on 1/14/14.
 *
 * @author Tony Chemit (chemit@codelutin.com)
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 * @since 3.0
 */
public class SynchroTableMetadata {
	private static final Log log = LogFactory
			.getLog(SynchroTableMetadata.class);

	public enum DuplicateKeyStrategy {
		/* Replace existing duplication (useful for REMOTE_ID column managment) */
		REPLACE,

		/*
		 * Replace existing duplication (useful for REMOTE_ID column managment),
		 * but remap all FK to the existing PK - see mantis #29010
		 */
		REPLACE_AND_REMAP,

		/*
		 * Replace existing duplication, only if not other replacement detected (useful for GEAR_PHYSICAL_FEATURES - see
		 * mantis 29693)
		 */
		REPLACE_LOW_PRIORITY,

		/* Skip the row, and save it in rejects */
		REJECT,

		/*
		 * Skip the row, and save it in rejects, but remap all FK to the
		 * existing PK - see mantis #26721
		 */
		REJECT_AND_REMAP,

		/* Continue (will import the row), but display a warning in log */
		WARN,

		/* Allow duplication (need for Reef DB, when importing data from a file) */
		DUPLICATE
	}

	/** Constant <code>PK_SEPARATOR="~~"</code> */
	public static final String PK_SEPARATOR = "~~";

	/** Constant <code>UPDATE_DATE_BINDPARAM="updateDate"</code> */
	public static final String UPDATE_DATE_BINDPARAM = "updateDate";

	private static Field delegateTableMetadataColumnsField = null;

	/**
	 * <p>Getter for the field <code>columns</code>.</p>
	 *
	 * @param delegate a {@link org.hibernate.tool.hbm2ddl.TableMetadata} object.
	 * @return a {@link java.util.Map} object.
	 */
	@SuppressWarnings({"unchecked", "rawtypes"})
	public static Map<String, ColumnMetadata> getColumns(TableMetadata delegate) {
		try {
			if (delegateTableMetadataColumnsField == null) {
				delegateTableMetadataColumnsField = TableMetadata.class
						.getDeclaredField("columns");
				delegateTableMetadataColumnsField.setAccessible(true);
			}
			return Maps
					.<String, ColumnMetadata> newLinkedHashMap((Map) delegateTableMetadataColumnsField
							.get(delegate));
		} catch (Exception e) {
			throw new SynchroTechnicalException(e.getMessage(), e);
		}
	}

	/**
	 * Serialize into a String a list of PK value. Usefull when more than one PK
	 * column in a table
	 *
	 * @param pkList a {@link java.util.Collection} object.
	 * @return a {@link java.lang.String} object.
	 */
	public static String toPkStr(Collection<? extends Object> pkList) {
		StringBuilder sb = new StringBuilder();
		for (Object pk : pkList) {
			sb.append(PK_SEPARATOR).append(pk);
		}
		return sb.substring(PK_SEPARATOR.length());
	}

	/**
	 * Serialize into a String a list of PK value. Usefull when more than one PK
	 * column in a table
	 *
	 * @param pkList an array of {@link java.lang.Object} objects.
	 * @return a {@link java.lang.String} object.
	 */
	public static String toPkStr(Object[] pkList) {
		StringBuilder sb = new StringBuilder();
		for (Object pk : pkList) {
			sb.append(PK_SEPARATOR).append(pk);
		}
		return sb.substring(PK_SEPARATOR.length());
	}

	/**
	 * <p>fromPkStr.</p>
	 *
	 * @param pkStr a {@link java.lang.String} object.
	 * @return a {@link java.util.List} object.
	 */
	public static List<Object> fromPkStr(String pkStr) {
		List<Object> pkList = Lists.newArrayList();
		String[] split = pkStr.split(PK_SEPARATOR);
		for (String s : split) {
			if ("null".equals(s)) {
				s = null;
			}
			pkList.add(s);
		}
		return pkList;
	}

	/**
	 * <p>fromPksStr.</p>
	 *
	 * @param pksStr a {@link java.util.Set} object.
	 * @return a {@link java.util.List} object.
	 */
	public static List<List<Object>> fromPksStr(Set<String> pksStr) {
		List<List<Object>> result = Lists.transform(
				ImmutableList.copyOf(pksStr),
				new Function<String, List<Object>>() {
					@Override
					public List<Object> apply(String pkStr) {
						return SynchroTableMetadata.fromPkStr(pkStr);
					}
				});
		return result;
	}

	/**
	 * <p>equals.</p>
	 *
	 * @param pkList1 a {@link java.util.Collection} object.
	 * @param pkList2 a {@link java.util.Collection} object.
	 * @return a boolean.
	 */
	public static boolean equals(Collection<? extends Object> pkList1, Collection<? extends Object> pkList2) {
		return (pkList1 != null && pkList2 != null)
				&& (pkList1.size() == pkList1.size())
				&& (toPkStr(pkList1).equals(toPkStr(pkList2)));
	}

	protected final String selectPksStrQuery;

	protected final String selectPksQuery;

	protected final String selectMaxUpdateDateQuery;

	protected final String selectUpdateDateByPkQuery;

	protected final String countAllQuery;

	protected final TableMetadata delegate;

	protected final Map<String, SynchroColumnMetadata> columns;

	protected List<SynchroJoinMetadata> childJoins;

	protected List<SynchroJoinMetadata> parentJoins;

	protected List<SynchroJoinMetadata> joins;

	protected boolean hasJoins;

	protected boolean hasChildJoins;

	protected final List<String> columnNames;

	protected final List<String> selectColumnNames;

	protected final List<String> insertColumnNames;

	protected final int columnCount;

	protected final int selectColumnCount;

	protected final Set<String> pkNames;

	protected final int[] selectPkIndexs;

	protected String insertQuery;

	protected String updateQuery;

	protected final int updateDateIndexInSelectQuery;

	protected final boolean withUpdateDateColumn;

	protected final boolean withIdColumn;

	protected boolean isRoot;

	protected final String countQuery;

	protected final String countFromUpdateDateQuery;

	protected final String selectFromUpdateDateQuery;

	protected final String selectAllQuery;

	protected final String selectByPkQuery;

	protected final String deleteByPkQuery;

	protected final Map<String, List<String>> uniqueConstraints;

	protected final Map<String, DuplicateKeyStrategy> duplicateKeyStrategies;

	protected Map<String, String> selectPksByIndexQueries;

	protected Map<String, String> selectByFksWhereClauses;

	protected String sequenceName;

	protected String selectSequenceNextValString;

	protected String sequenceNextValString;

	protected List<SynchroInterceptor> interceptors;

	protected boolean hasUniqueConstraints;

	protected LockMode lockModeOnUpdate;

	protected boolean isBuild;

	// This field is reset when build() is called
	protected SynchroDatabaseMetadata dbMeta;

	// This field is reset when build() is called
	protected Dialect dialect;

	// This field is reset when build() is called
	protected EventBus eventBus;

	/**
	 * <p>Constructor for SynchroTableMetadata.</p>
	 *
	 * @param dbMeta a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 * @param delegate a {@link org.hibernate.tool.hbm2ddl.TableMetadata} object.
	 * @param dialect a {@link org.hibernate.dialect.Dialect} object.
	 * @param listeners a {@link java.util.List} object.
	 * @param tableName a {@link java.lang.String} object.
	 * @param columnFilter a {@link com.google.common.base.Predicate} object.
	 * @param enableQueries a boolean.
	 */
	protected SynchroTableMetadata(SynchroDatabaseMetadata dbMeta,
			TableMetadata delegate, Dialect dialect, List<Object> listeners,
			String tableName, Predicate<SynchroColumnMetadata> columnFilter,
			boolean enableQueries) {

		Preconditions.checkNotNull(delegate);
		Preconditions.checkNotNull(dbMeta);

		this.isBuild = false;
		this.dbMeta = dbMeta;
		this.dialect = dialect;
		this.delegate = delegate;
		this.eventBus = initEventBus(enableQueries, listeners,
				delegate.getName());
		this.interceptors = initInterceptors(listeners);

		SynchroDatabaseConfiguration config = dbMeta.getConfiguration();

		try {
			this.columns = initColumns(dialect, tableName, columnFilter);
			this.columnNames = initColumnNames(columns);
			this.columnCount = columnNames.size();
			this.pkNames = initPrimaryKeys(dbMeta);
			Preconditions.checkNotNull(pkNames);
			this.uniqueConstraints = Maps.newLinkedHashMap();
			this.duplicateKeyStrategies = Maps.newHashMap();
			initUniqueConstraints(dbMeta);
			this.hasUniqueConstraints = MapUtils
					.isNotEmpty(this.uniqueConstraints);
			this.withIdColumn = config.getColumnId() == null
					? false
					: columnNames.contains(config.getColumnId().toLowerCase());
			this.withUpdateDateColumn = config.getColumnUpdateDate() == null
					? false
					: columnNames.contains(config.getColumnUpdateDate()
							.toLowerCase());
			this.sequenceName = initSequenceName(dbMeta);
		} catch (Exception e) {
			throw new SynchroTechnicalException(t(
					"synchro.meta.table.instanciation.error",
					delegate.getName()), e);
		}

		if (enableQueries) {
			this.selectSequenceNextValString = createSelectSequenceNextValString(
					dialect, this.sequenceName);
			this.sequenceNextValString = createSequenceNextValString(dialect,
					this.sequenceName);
			this.countAllQuery = createAllCountQuery();
			this.countQuery = createCountQuery();
			this.countFromUpdateDateQuery = createCountFromUpdateDate(config);
			this.selectMaxUpdateDateQuery = createSelectMaxUpdateDateQuery(config);
			this.selectUpdateDateByPkQuery = createSelectUpdateDateByPkQuery(config);
			this.selectAllQuery = createSelectAllQuery();
			this.selectFromUpdateDateQuery = createSelectFromUpdateDateQuery(config);
			this.selectByPkQuery = createSelectByPkQuery();
			this.selectPksStrQuery = createSelectPksStrQuery(config);
			this.selectPksQuery = createSelectPksQuery(config);
			this.insertQuery = createInsertQuery();
			this.updateQuery = createUpdateQuery();
			this.selectPksByIndexQueries = createSelectPksByIndexQueries(config);
			this.selectByFksWhereClauses = createSelectByFksQueries(config);

			this.selectColumnNames = initColumnNamesFromQuery(this.selectAllQuery);
			this.insertColumnNames = initColumnNamesFromQuery(this.insertQuery);
			this.selectPkIndexs = createSelectPkIndex(this.selectColumnNames);
			Preconditions
					.checkArgument(
							this.selectPkIndexs.length == pkNames.size(),
							"Could not found all table PKs in the select query. This is require.");
			this.selectColumnCount = selectColumnNames.size();

			this.updateDateIndexInSelectQuery = selectColumnNames
					.indexOf(config.getColumnUpdateDate());
			this.deleteByPkQuery = createDeleteByPk();
		} else {
			this.selectSequenceNextValString = null;
			this.sequenceNextValString = null;
			this.countAllQuery = null;
			this.countQuery = null;
			this.countFromUpdateDateQuery = null;
			this.selectMaxUpdateDateQuery = null;
			this.selectUpdateDateByPkQuery = null;
			this.selectAllQuery = null;
			this.selectFromUpdateDateQuery = null;
			this.selectByPkQuery = null;
			this.selectPksStrQuery = null;
			this.selectPksQuery = null;
			this.insertQuery = null;
			this.updateQuery = null;
			this.selectPksByIndexQueries = null;
			this.selectColumnNames = Lists.newArrayList(); // empty list, to
															// avoid
															// NullPointerException
															// on
															// getSelectColumnIndex()
			this.insertColumnNames = Lists.newArrayList();
			this.selectColumnCount = 0;
			this.updateDateIndexInSelectQuery = -1;
			this.selectPkIndexs = null;
			this.deleteByPkQuery = null;
		}

		// Default value for joins (will be override in initJoins())
		this.joins = Lists.newArrayList();
		this.hasJoins = false;
		this.parentJoins = null;
		this.hasChildJoins = false;
		this.lockModeOnUpdate = LockMode.NONE;

		// Default values (could be override using interceptor)
		this.isRoot = false;
	}

	// for tests purposes
	SynchroTableMetadata() {
		isBuild = false;
		eventBus = null;
		delegate = null;
		columns = null;
		columnNames = null;
		selectSequenceNextValString = null;
		sequenceNextValString = null;
		joins = null;
		pkNames = null;
		selectPkIndexs = null;
		updateDateIndexInSelectQuery = -1;
		withUpdateDateColumn = false;
		withIdColumn = false;
		isRoot = false;
		insertQuery = null;
		updateQuery = null;
		countAllQuery = null;
		countQuery = null;
		countFromUpdateDateQuery = null;
		selectPksStrQuery = null;
		selectPksQuery = null;
		selectMaxUpdateDateQuery = null;
		selectUpdateDateByPkQuery = null;
		selectFromUpdateDateQuery = null;
		sequenceName = null;
		selectAllQuery = null;
		selectByPkQuery = null;
		columnCount = 0;
		selectPksByIndexQueries = null;
		uniqueConstraints = null;
		duplicateKeyStrategies = null;
		selectColumnNames = null;
		insertColumnNames = null;
		selectColumnCount = 0;
		lockModeOnUpdate = LockMode.NONE;
		deleteByPkQuery = null;
	}

	/**
	 * Initialize Join metadata. this need to have already loaded all tables.
	 *
	 * @param dbMeta
	 *            the Database metadata, with some preloaded tables inside
	 */
	public void initJoins(SynchroDatabaseMetadata dbMeta) {
		try {
			this.joins = initJoins(getCatalog(), getSchema(), getName(),
					dbMeta, this.columns);

			// Update fields on joins
			this.hasJoins = !joins.isEmpty();
			this.childJoins = initChildJoins(joins);
			this.parentJoins = initParentJoins(joins);
			this.hasChildJoins = !childJoins.isEmpty();

		} catch (Exception e) {
			throw new SynchroTechnicalException(t(
					"synchro.meta.table.instanciation.error",
					delegate.getName()), e);
		}
	}

	/**
	 * <p>Getter for the field <code>pkNames</code>.</p>
	 *
	 * @return a {@link java.util.Set} object.
	 */
	public Set<String> getPkNames() {
		return pkNames;
	}

	/**
	 * <p>isWithUpdateDateColumn.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isWithUpdateDateColumn() {
		return withUpdateDateColumn;
	}

	/**
	 * <p>isWithIdColumn.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isWithIdColumn() {
		return withIdColumn;
	}

	/**
	 * <p>isRoot.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isRoot() {
		return isRoot;
	}

	/**
	 * <p>setRoot.</p>
	 *
	 * @param isRoot a boolean.
	 */
	public void setRoot(boolean isRoot) {
		this.isRoot = isRoot;
	}

	/**
	 * <p>getColumnsCount.</p>
	 *
	 * @return a int.
	 */
	public int getColumnsCount() {
		return columnCount;
	}

	/**
	 * <p>Getter for the field <code>columnNames</code>.</p>
	 *
	 * @return a {@link java.util.Set} object.
	 */
	public Set<String> getColumnNames() {
		return ImmutableSet.copyOf(columnNames);
	}

	/**
	 * <p>getSelectColumnsCount.</p>
	 *
	 * @return a int.
	 */
	public int getSelectColumnsCount() {
		return selectColumnCount;
	}

	/**
	 * <p>Getter for the field <code>selectColumnNames</code>.</p>
	 *
	 * @return a {@link java.util.Set} object.
	 */
	public Set<String> getSelectColumnNames() {
		return ImmutableSet.copyOf(selectColumnNames);
	}

	/**
	 * <p>getColumnName.</p>
	 *
	 * @param columnIndex a int.
	 * @return a {@link java.lang.String} object.
	 */
	public String getColumnName(int columnIndex) {
		return columnNames.get(columnIndex);
	}

	/**
	 * <p>getColumnIndex.</p>
	 *
	 * @param name a {@link java.lang.String} object.
	 * @return a int.
	 */
	public int getColumnIndex(String name) {
		return columnNames.indexOf(name.toLowerCase());
	}

	/**
	 * <p>getSelectColumnIndex.</p>
	 *
	 * @param name a {@link java.lang.String} object.
	 * @return a int.
	 */
	public int getSelectColumnIndex(String name) {
		return selectColumnNames.indexOf(name.toLowerCase());
	}

	/**
	 * <p>getInsertColumnIndex.</p>
	 *
	 * @param name a {@link java.lang.String} object.
	 * @return a int.
	 */
	public int getInsertColumnIndex(String name) {
		return insertColumnNames.indexOf(name.toLowerCase());
	}

	/**
	 * <p>Getter for the field <code>delegate</code>.</p>
	 *
	 * @return a {@link org.hibernate.tool.hbm2ddl.TableMetadata} object.
	 */
	public TableMetadata getDelegate() {
		return delegate;
	}

	/**
	 * <p>getName.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getName() {
		return delegate.getName();
	}

	/**
	 * <p>getForeignKeyMetadata.</p>
	 *
	 * @param fk a {@link org.hibernate.mapping.ForeignKey} object.
	 * @return a {@link org.hibernate.tool.hbm2ddl.ForeignKeyMetadata} object.
	 */
	public ForeignKeyMetadata getForeignKeyMetadata(ForeignKey fk) {
		return delegate.getForeignKeyMetadata(fk);
	}

	/**
	 * <p>getColumnMetadata.</p>
	 *
	 * @param columnName a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.meta.SynchroColumnMetadata} object.
	 */
	public SynchroColumnMetadata getColumnMetadata(String columnName) {
		return columns.get(StringUtils.lowerCase(columnName));
	}

	/**
	 * <p>getSchema.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getSchema() {
		return delegate.getSchema();
	}

	/**
	 * <p>getCatalog.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getCatalog() {
		return delegate.getCatalog();
	}

	/**
	 * <p>getForeignKeyMetadata.</p>
	 *
	 * @param keyName a {@link java.lang.String} object.
	 * @return a {@link org.hibernate.tool.hbm2ddl.ForeignKeyMetadata} object.
	 */
	public ForeignKeyMetadata getForeignKeyMetadata(String keyName) {
		return delegate.getForeignKeyMetadata(keyName);
	}

	/**
	 * <p>getIndexMetadata.</p>
	 *
	 * @param indexName a {@link java.lang.String} object.
	 * @return a {@link org.hibernate.tool.hbm2ddl.IndexMetadata} object.
	 */
	public IndexMetadata getIndexMetadata(String indexName) {
		return delegate.getIndexMetadata(indexName);
	}

	/**
	 * <p>getColumn.</p>
	 *
	 * @param columnName a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.meta.SynchroColumnMetadata} object.
	 */
	public SynchroColumnMetadata getColumn(String columnName) {
		return columns.get(columnName);
	}

	/**
	 * <p>Getter for the field <code>joins</code>.</p>
	 *
	 * @return a {@link java.util.List} object.
	 */
	public List<SynchroJoinMetadata> getJoins() {
		return joins;
	}

	/**
	 * <p>Getter for the field <code>childJoins</code>.</p>
	 *
	 * @return a {@link java.util.List} object.
	 */
	public List<SynchroJoinMetadata> getChildJoins() {
		return childJoins;
	}

	/**
	 * <p>Getter for the field <code>parentJoins</code>.</p>
	 *
	 * @return a {@link java.util.List} object.
	 */
	public List<SynchroJoinMetadata> getParentJoins() {
		return parentJoins;
	}

	/**
	 * <p>hasJoins.</p>
	 *
	 * @return a boolean.
	 */
	public boolean hasJoins() {
		return hasJoins;
	}

	/**
	 * <p>hasChildJoins.</p>
	 *
	 * @return a boolean.
	 */
	public boolean hasChildJoins() {
		return hasChildJoins;
	}

	/**
	 * <p>Getter for the field <code>lockModeOnUpdate</code>.</p>
	 *
	 * @return a {@link org.hibernate.LockMode} object.
	 */
	public LockMode getLockModeOnUpdate() {
		return lockModeOnUpdate;
	}

	/**
	 * <p>setLockOnUpdate.</p>
	 *
	 * @param lockModeOnUpdate a {@link org.hibernate.LockMode} object.
	 */
	public void setLockOnUpdate(LockMode lockModeOnUpdate) {
		this.lockModeOnUpdate = lockModeOnUpdate;
	}

	/**
	 * <p>Setter for the field <code>sequenceName</code>.</p>
	 *
	 * @param sequenceName a {@link java.lang.String} object.
	 */
	public void setSequenceName(String sequenceName) {
		this.sequenceName = sequenceName;
	}

	/**
	 * <p>Getter for the field <code>interceptors</code>.</p>
	 *
	 * @return a {@link java.util.List} object.
	 */
	public List<SynchroInterceptor> getInterceptors() {
		return this.interceptors;
	}

	/**
	 * <p>addInterceptor.</p>
	 *
	 * @param interceptor a {@link fr.ifremer.common.synchro.intercept.SynchroInterceptor} object.
	 */
	public void addInterceptor(SynchroInterceptor interceptor) {

		this.interceptors.add(interceptor);
	}

	/**
	 * <p>containsInterceptor.</p>
	 *
	 * @param interceptor a {@link fr.ifremer.common.synchro.intercept.SynchroInterceptor} object.
	 * @return a boolean.
	 */
	public boolean containsInterceptor(SynchroInterceptor interceptor) {

		return this.interceptors.contains(interceptor);
	}

	/**
	 * <p>addSelectByFksWhereClause.</p>
	 *
	 * @param fkColumnNames a {@link java.util.Set} object.
	 * @param additionalWhereClause a {@link java.lang.String} object.
	 */
	public void addSelectByFksWhereClause(Set<String> fkColumnNames,
			String additionalWhereClause) {
		String key = toPkStr(fkColumnNames).toLowerCase();
		this.selectByFksWhereClauses.put(key, additionalWhereClause);
	}

	/**
	 * <p>addSelectByFksWhereClause.</p>
	 *
	 * @param fkColumnName a {@link java.lang.String} object.
	 * @param query a {@link java.lang.String} object.
	 */
	public void addSelectByFksWhereClause(String fkColumnName, String query) {
		this.selectByFksWhereClauses.put(fkColumnName.toLowerCase(), query);
	}

	/**
	 * <p>getSelectByFksWhereClause.</p>
	 *
	 * @param fkColumnNames a {@link java.util.Set} object.
	 * @return a {@link java.lang.String} object.
	 */
	public String getSelectByFksWhereClause(Set<String> fkColumnNames) {
		String key = toPkStr(fkColumnNames).toLowerCase();
		return this.selectByFksWhereClauses.get(key);
	}

	/**
	 * <p>getSelectByFksWhereClause.</p>
	 *
	 * @param fkColumnName a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	public String getSelectByFksWhereClause(String fkColumnName) {
		return this.selectByFksWhereClauses.get(fkColumnName.toLowerCase());
	}

	/**
	 * <p>getTableLogPrefix.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getTableLogPrefix() {
		return "[" + getName() + "]";
	}

	/** {@inheritDoc} */
	@Override
	public String toString() {
		return delegate.toString();
	}

	// ------------------------------------------------------------------------//
	// -- queries methods --//
	// ------------------------------------------------------------------------//

	/**
	 * <p>Getter for the field <code>sequenceName</code>.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getSequenceName() {
		return sequenceName;
	}

	/**
	 * <p>Getter for the field <code>insertQuery</code>.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getInsertQuery() {
		return insertQuery;
	}

	/**
	 * <p>Getter for the field <code>updateQuery</code>.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getUpdateQuery() {
		return updateQuery;
	}

	/**
	 * Obtains a SQL with one column output : a concatenation of all PK.
	 *
	 * @return a SQL select, with only one result column
	 */
	public String getSelectPksStrQuery() {
		return selectPksStrQuery;
	}

	/**
	 * <p>Getter for the field <code>selectPksQuery</code>.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getSelectPksQuery() {
		return selectPksQuery;
	}

	/**
	 * <p>Getter for the field <code>selectMaxUpdateDateQuery</code>.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getSelectMaxUpdateDateQuery() {
		return selectMaxUpdateDateQuery;
	}

	/**
	 * <p>Getter for the field <code>selectUpdateDateByPkQuery</code>.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getSelectUpdateDateByPkQuery() {
		return selectUpdateDateByPkQuery;
	}

	/**
	 * <p>getSelectDataQueryFromPk.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getSelectDataQueryFromPk() {
		return selectByPkQuery;
	}

	/**
	 * <p>Getter for the field <code>selectAllQuery</code>.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getSelectAllQuery() {
		return selectAllQuery;
	}

	/**
	 * <p>getSelectUpdatedDataQuery.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getSelectUpdatedDataQuery() {
		return selectFromUpdateDateQuery;
	}

	/**
	 * <p>Getter for the field <code>deleteByPkQuery</code>.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getDeleteByPkQuery() {
		return deleteByPkQuery;
	}

	/**
	 * <p>Getter for the field <code>sequenceNextValString</code>.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getSequenceNextValString() {
		return sequenceNextValString;
	}

	/**
	 * <p>Getter for the field <code>selectSequenceNextValString</code>.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getSelectSequenceNextValString() {
		return selectSequenceNextValString;
	}

	/**
	 * Count all rows (no where clause)
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getCountAllQuery() {
		return countAllQuery;
	}

	/**
	 * Count rows to synchronize (could have where clause).
	 * <br>
	 * Could be override using listener on SynchroQueryName.count
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getCountQuery() {
		return countQuery;
	}

	/**
	 * <p>getCountUpdatedDataQuery.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getCountUpdatedDataQuery() {
		if (countFromUpdateDateQuery == null) {
			return null;
		}
		return countFromUpdateDateQuery;
	}

	// ------------------------------------------------------------------------//
	// -- PK methods --//
	// ------------------------------------------------------------------------//

	/**
	 * <p>Getter for the field <code>selectPkIndexs</code>.</p>
	 *
	 * @return an array of int.
	 */
	public int[] getSelectPkIndexs() {
		return selectPkIndexs;
	}

	/**
	 * <p>isSimpleKey.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isSimpleKey() {
		return selectPkIndexs.length == 1;
	}

	/**
	 * <p>isSelectPrimaryKeysAsStringQueryEnable.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isSelectPrimaryKeysAsStringQueryEnable() {
		return selectPksStrQuery != null;
	}

	/**
	 * <p>getUpdateDate.</p>
	 *
	 * @param incomingData a {@link java.sql.ResultSet} object.
	 * @return a {@link java.sql.Timestamp} object.
	 * @throws java.sql.SQLException if any.
	 */
	public Timestamp getUpdateDate(ResultSet incomingData) throws SQLException {
		return incomingData.getTimestamp(updateDateIndexInSelectQuery + 1);
	}

	/**
	 * <p>getUpdateDate.</p>
	 *
	 * @param incomingData an array of {@link java.lang.Object} objects.
	 * @return a {@link java.sql.Timestamp} object.
	 * @throws java.sql.SQLException if any.
	 */
	public Timestamp getUpdateDate(Object[] incomingData) throws SQLException {
		return (Timestamp) incomingData[updateDateIndexInSelectQuery];
	}

	/**
	 * Add a unique constraint. <br>
	 * The given name must not exists in the constraint list. If you need to
	 * replace a existing constraint, use the method
	 * {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata#putUniqueConstraint(String, List, DuplicateKeyStrategy)}
	 * instead.
	 *
	 * @param indexName
	 *            the unique constraint name
	 * @param columnNames a {@link java.util.List} object.
	 * @param duplicateKeyStrategy a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata.DuplicateKeyStrategy} object.
	 */
	public void addUniqueConstraint(String indexName, List<String> columnNames,
			DuplicateKeyStrategy duplicateKeyStrategy) {
		Preconditions.checkNotNull(indexName);
		Preconditions.checkArgument(!uniqueConstraints.containsKey(indexName),
				"Duplicate unique constraints name");
		Preconditions.checkNotNull(CollectionUtils.isNotEmpty(columnNames));

		if (log.isDebugEnabled()) {
			log.debug(String.format("[%s] Add unique constraint: %s",
					getName(), indexName));
		}

		List<String> columnNamesLowerCase = Lists
				.newArrayListWithCapacity(columnNames.size());
		for (String columnName : columnNames) {
			columnNamesLowerCase.add(columnName.toLowerCase());
		}

		uniqueConstraints.put(indexName, columnNamesLowerCase);
		duplicateKeyStrategies.put(indexName, duplicateKeyStrategy);
		hasUniqueConstraints = true;
	}

	/**
	 * Add (or replace) a unique constraint
	 *
	 * @param indexName
	 *            the unique constraint name
	 * @param columnNames a {@link java.util.List} object.
	 * @param duplicateKeyStrategy a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata.DuplicateKeyStrategy} object.
	 */
	public void putUniqueConstraint(String indexName, List<String> columnNames,
			DuplicateKeyStrategy duplicateKeyStrategy) {
		Preconditions.checkNotNull(indexName);
		Preconditions.checkNotNull(CollectionUtils.isNotEmpty(columnNames));

		if (log.isDebugEnabled()) {
			log.debug(String.format("[%s] Add unique constraint: %s",
					getName(), indexName));
		}

		List<String> columnNamesLowerCase = Lists
				.newArrayListWithCapacity(columnNames.size());
		for (String columnName : columnNames) {
			columnNamesLowerCase.add(columnName.toLowerCase());
		}

		uniqueConstraints.put(indexName, columnNamesLowerCase);
		duplicateKeyStrategies.put(indexName, duplicateKeyStrategy);
		hasUniqueConstraints = true;
	}

	/**
	 * <p>Getter for the field <code>uniqueConstraints</code>.</p>
	 *
	 * @return a {@link java.util.Map} object.
	 */
	public Map<String, List<String>> getUniqueConstraints() {
		checkBuild();
		return this.uniqueConstraints;
	}

	/**
	 * <p>hasUniqueConstraint.</p>
	 *
	 * @param constraintName a {@link java.lang.String} object.
	 * @return a boolean.
	 */
	public boolean hasUniqueConstraint(String constraintName) {
		return this.uniqueConstraints.containsKey(constraintName);
	}

	/**
	 * <p>hasUniqueConstraints.</p>
	 *
	 * @return a boolean.
	 */
	public boolean hasUniqueConstraints() {
		return this.hasUniqueConstraints;
	}

	/**
	 * Allow to change a strategy for an unique constraint
	 *
	 * @param indexName a {@link java.lang.String} object.
	 * @param duplicateKeyStrategy a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata.DuplicateKeyStrategy} object.
	 */
	public void setUniqueConstraintStrategy(String indexName,
			DuplicateKeyStrategy duplicateKeyStrategy) {
		Preconditions.checkArgument(hasUniqueConstraint(indexName));

		duplicateKeyStrategies.put(indexName, duplicateKeyStrategy);
	}

	/**
	 * Return if the table has one or more index with the given
	 * DuplicateKeyStrategy
	 *
	 * @param filter a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata.DuplicateKeyStrategy} object.
	 * @return true if index exists with this strategy
	 */
	public boolean hasUniqueConstraints(DuplicateKeyStrategy filter) {
		Preconditions.checkNotNull(filter);
		for (DuplicateKeyStrategy indexStrategy : duplicateKeyStrategies
				.values()) {
			if (filter.equals(indexStrategy)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * <p>removeUniqueConstraints.</p>
	 *
	 * @param indexName a {@link java.lang.String} object.
	 */
	public void removeUniqueConstraints(String indexName) {
		checkBuild();
		this.uniqueConstraints.remove(indexName);
		this.duplicateKeyStrategies.remove(indexName);
	}

	/**
	 * <p>getDuplicatKeyStrategies.</p>
	 *
	 * @return a {@link java.util.Map} object.
	 */
	public Map<String, DuplicateKeyStrategy> getDuplicatKeyStrategies() {
		checkBuild();
		return this.duplicateKeyStrategies;
	}

	/**
	 * <p>getSelectPkByIndex.</p>
	 *
	 * @param indexName a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	public String getSelectPkByIndex(String indexName) {
		checkBuild();
		return MapUtils.getString(this.selectPksByIndexQueries, indexName);
	}

	/**
	 * Return queries, to count if a pK is used by other tables.<br>
	 * The SQL queries returned has params (<code>:pk1</code> to
	 * <code>:pkN</code>) that need to be bind.
	 *
	 * @param dbMeta
	 *            the metadata of the target database (This is required because
	 *            the SynchroTableMetadata could have be build on the soure
	 *            database).
	 * @throws java.sql.SQLException if any.
	 * @return a {@link java.util.Map} object.
	 */
	public Map<String, String> getCountPkReferenceQueries(
			DatabaseMetaData dbMeta) throws SQLException {
		checkBuild();
		Preconditions.checkNotNull(dbMeta);

		if (CollectionUtils.isEmpty(pkNames)) {
			return null;
		}

		Map<String, String> result;
		if (pkNames.size() == 1) {
			result = getCountPkReferenceQueriesUniquePk(dbMeta);
		} else {
			result = getCountPkReferenceQueriesCompositePk(dbMeta);
		}

		return result;
	}

	/**
	 * <p>getExportedKeys.</p>
	 *
	 * @param dbMeta a {@link java.sql.DatabaseMetaData} object.
	 * @return a {@link com.google.common.collect.Multimap} object.
	 * @throws java.sql.SQLException if any.
	 */
	public Multimap<String, String> getExportedKeys(DatabaseMetaData dbMeta)
			throws SQLException {
		Preconditions.checkArgument(isSimpleKey());

		String pkColumnName = pkNames.iterator().next();
		Multimap<String, String> result = ArrayListMultimap.create();

		// Exported keys (primary keys referenced by another table)
		ResultSet rs = null;
		try {
			rs = dbMeta.getExportedKeys(getCatalog(), getSchema(), getName());
			while (rs.next()) {
				String columnName = rs.getString("PKCOLUMN_NAME").toLowerCase();

				if (pkColumnName.equalsIgnoreCase(columnName)) {
					String fkTableName = rs.getString("FKTABLE_NAME")
							.toUpperCase();
					String fkColumnName = rs.getString("FKCOLUMN_NAME")
							.toLowerCase();
					result.put(fkTableName, fkColumnName);
				}
			}
		} finally {
			Daos.closeSilently(rs);
		}

		return result;
	}

	// ------------------------------------------------------------------------//
	// -- Protected methods --//
	// ------------------------------------------------------------------------//

	/**
	 * <p>initColumns.</p>
	 *
	 * @param dialect a {@link org.hibernate.dialect.Dialect} object.
	 * @param tableName a {@link java.lang.String} object.
	 * @param columnFilter a {@link com.google.common.base.Predicate} object.
	 * @return a {@link java.util.Map} object.
	 * @throws java.lang.NoSuchFieldException if any.
	 * @throws java.lang.SecurityException if any.
	 * @throws java.lang.IllegalArgumentException if any.
	 * @throws java.lang.IllegalAccessException if any.
	 */
	protected Map<String, SynchroColumnMetadata> initColumns(Dialect dialect,
			String tableName, Predicate<SynchroColumnMetadata> columnFilter)
			throws NoSuchFieldException, SecurityException,
			IllegalArgumentException, IllegalAccessException {

		Map<String, ColumnMetadata> delegateColumns = getColumns(delegate);
		// Use a TreeMap instead of a HashMap to sort column names in alpha order (mantis #30821)
		Map<String, SynchroColumnMetadata> columns = Maps.newTreeMap();
		// Dialect dialect = dbMeta.getDialect();
		for (String columnName : delegateColumns.keySet()) {
			ColumnMetadata delegateColumn = delegateColumns.get(columnName);
			SynchroColumnMetadata column = new SynchroColumnMetadata(tableName,
					delegateColumn, dialect);
			boolean skipColumn = columnFilter != null
					&& !columnFilter.apply(column);
			if (!skipColumn) {
				columns.put(columnName, column);
			}
		}

		return columns;
	}

	/**
	 * <p>initColumnNames.</p>
	 *
	 * @param columns a {@link java.util.Map} object.
	 * @return a {@link java.util.List} object.
	 */
	protected List<String> initColumnNames(
			Map<String, SynchroColumnMetadata> columns) {
		List<String> result = Lists
				.newArrayListWithExpectedSize(columns.size());
		for (String name : columns.keySet()) {
			result.add(name.toLowerCase());
		}
		return result;
	}

	/**
	 * <p>initColumnNamesFromQuery.</p>
	 *
	 * @param aQuery a {@link java.lang.String} object.
	 * @return a {@link java.util.List} object.
	 */
	protected List<String> initColumnNamesFromQuery(String aQuery) {
		return SynchroQueryBuilder.newBuilder(aQuery).getColumnNames();
	}

	/**
	 * <p>initPrimaryKeys.</p>
	 *
	 * @param dbMeta a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 * @return a {@link java.util.Set} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected Set<String> initPrimaryKeys(SynchroDatabaseMetadata dbMeta)
			throws SQLException {

		Set<String> result = Sets.newLinkedHashSet();
		ResultSet rs = dbMeta.getPrimaryKeys(getCatalog(), getSchema(),
				getName());
		try {

			while (rs.next()) {
				result.add(rs.getString("COLUMN_NAME").toLowerCase());
			}
			// Fire event to interceptors
			result = fireOnPkLoad(result);
			return ImmutableSet.copyOf(result);
		} finally {
			Daos.closeSilently(rs);
		}
	}

	/**
	 * <p>initUniqueConstraints.</p>
	 *
	 * @param dbMeta a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected void initUniqueConstraints(SynchroDatabaseMetadata dbMeta)
			throws SQLException {
		Preconditions.checkNotNull(this.uniqueConstraints);

		ResultSet rs = null;
		try {
			rs = dbMeta.getIndexInfo(getCatalog(), getSchema(), getName(),
					true /* unique */, true /*
											 * allow info to be approximated (do
											 * not re-compute statistics)
											 */
			);

			List<String> columnNames = Lists.newArrayList();
			String lastIndexName = null;
			while (rs.next()) {

				String indexName = rs.getString("INDEX_NAME");
				if (indexName != null) {
					indexName = indexName.toLowerCase();
					// If index changed, add the previous index to the result
					if (lastIndexName != null
							&& indexName.equals(lastIndexName) == false) {
						addUniqueConstraintsIfNotPK(indexName, columnNames,
						// Always reject, when constraints is define on database
						// level
								DuplicateKeyStrategy.REJECT);
						// Reset loop var
						lastIndexName = indexName;
						columnNames = Lists.newArrayList();
					}

					String columName = rs.getString("COLUMN_NAME")
							.toLowerCase();
					columnNames.add(columName);
				}
			}

			// Add the last index to the result
			if (lastIndexName != null) {
				addUniqueConstraintsIfNotPK(lastIndexName, columnNames,
				// Always reject, when constraints is define on database level
						DuplicateKeyStrategy.REJECT);
			}

		} finally {
			Daos.closeSilently(rs);
		}
	}

	/**
	 * <p>initInterceptors.</p>
	 *
	 * @param listeners a {@link java.util.List} object.
	 * @return a {@link java.util.List} object.
	 */
	protected List<SynchroInterceptor> initInterceptors(List<Object> listeners) {
		List<SynchroInterceptor> result = Lists.newArrayList();
		if (listeners != null) {
			for (Object listener : listeners) {
				if (listener instanceof SynchroInterceptor) {
					result.add((SynchroInterceptor) listener);
				}
			}
		}
		return result;
	}

	/**
	 * <p>createSelectPkIndex.</p>
	 *
	 * @param selectColumnNames a {@link java.util.List} object.
	 * @return an array of int.
	 */
	protected int[] createSelectPkIndex(List<String> selectColumnNames) {

		int[] result = new int[pkNames.size()];

		int pkI = 0;
		for (String pkName : pkNames) {
			String pkColumnName = pkName.toLowerCase();

			int i = 1;

			int index = -1;
			for (String columnName : selectColumnNames) {
				if (pkColumnName.equals(columnName)) {
					index = i;
				} else {
					i++;
				}
			}
			result[pkI++] = index;
		}
		return result;
	}

	/**
	 * <p>initJoins.</p>
	 *
	 * @param catalog a {@link java.lang.String} object.
	 * @param schema a {@link java.lang.String} object.
	 * @param tableName a {@link java.lang.String} object.
	 * @param dbMeta a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 * @param columns a {@link java.util.Map} object.
	 * @return a {@link java.util.List} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected List<SynchroJoinMetadata> initJoins(String catalog,
			String schema, String tableName, SynchroDatabaseMetadata dbMeta,
			Map<String, SynchroColumnMetadata> columns) throws SQLException {

		List<SynchroJoinMetadata> result = Lists.newArrayList();

		// Exported keys (primary keys referenced by another table)
		ResultSet rs = null;
		try {
			rs = dbMeta.getExportedKeys(catalog, schema, tableName);
			while (rs.next()) {
				String columnName = rs.getString("PKCOLUMN_NAME").toLowerCase();
				if (columns.containsKey(columnName)) {
					SynchroJoinMetadata join = new SynchroJoinMetadata(rs,
							this, dbMeta);

					// Fire event to interceptors
					fireOnJoinLoad(join);

					if (join.isValid()) {
						result.add(join);
					}
				}
			}
		} finally {
			Daos.closeSilently(rs);
		}

		// Imported keys (foreign keys that references another table)
		try {
			rs = dbMeta.getImportedKeys(catalog, schema, tableName);
			while (rs.next()) {
				String columnName = rs.getString("FKCOLUMN_NAME").toLowerCase();
				if (columns.containsKey(columnName)) {
					SynchroJoinMetadata join = new SynchroJoinMetadata(rs,
							this, dbMeta);

					// Fire event to interceptors
					fireOnJoinLoad(join);

					if (join.isValid()) {
						result.add(join);

						SynchroColumnMetadata column = columns.get(columnName);
						column.setParentJoin(join);
					}
				}
			}
		} finally {
			Daos.closeSilently(rs);
		}

		return result;
	}

	/**
	 * <p>initChildJoins.</p>
	 *
	 * @param joins a {@link java.util.List} object.
	 * @return a {@link java.util.List} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected List<SynchroJoinMetadata> initChildJoins(
			List<SynchroJoinMetadata> joins) throws SQLException {

		List<SynchroJoinMetadata> result = Lists
				.newArrayListWithExpectedSize(joins.size());

		for (SynchroJoinMetadata join : joins) {
			if (join.isChild()) {
				result.add(join);
			}
		}
		return result;
	}

	/**
	 * <p>initParentJoins.</p>
	 *
	 * @param joins a {@link java.util.List} object.
	 * @return a {@link java.util.List} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected List<SynchroJoinMetadata> initParentJoins(
			List<SynchroJoinMetadata> joins) throws SQLException {

		List<SynchroJoinMetadata> result = Lists
				.newArrayListWithExpectedSize(joins.size());

		for (SynchroJoinMetadata join : joins) {
			if (!join.isChild()) {
				result.add(join);
			}
		}
		return result;
	}

	/**
	 * <p>createInsertQuery.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	protected String createInsertQuery() {
		if (CollectionUtils.isEmpty(columnNames)) {
			return null;
		}

		StringBuilder queryParams = new StringBuilder();
		StringBuilder valueParams = new StringBuilder();

		for (String columnName : columnNames) {
			queryParams.append(", ").append(columnName);
			valueParams.append(", ?");
		}

		String sql = String.format("INSERT INTO %s (%s) VALUES (%s)",
				getName(), queryParams.substring(2), valueParams.substring(2));

		return fireOnCreateQuery(SynchroQueryName.insert, sql);
	}

	/**
	 * <p>createUpdateQuery.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	protected String createUpdateQuery() {
		// Could not update, because no PK found
		if (CollectionUtils.isEmpty(pkNames)) {
			return null;
		}
		if (CollectionUtils.isEmpty(columnNames)) {
			return null;
		}

		StringBuilder updateParams = new StringBuilder();

		for (String columnName : columnNames) {
			updateParams.append(", ").append(columnName);
			updateParams.append(" = ?");
		}

		String sql = String.format("UPDATE %s SET %s WHERE %s", getName(),
				updateParams.substring(2), createPkWhereClause());
		return fireOnCreateQuery(SynchroQueryName.update, sql);
	}

	/**
	 * <p>createDeleteByPk.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	protected String createDeleteByPk() {
		String sql = String.format("DELETE FROM %s WHERE %s", getName(),
				createPkWhereClause());
		return fireOnCreateQuery(SynchroQueryName.deleteByPk, sql);
	}

	/**
	 * <p>createSelectByPkQuery.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	protected String createSelectByPkQuery() {
		// Could not update, because no PK found
		if (CollectionUtils.isEmpty(pkNames)) {
			return null;
		}

		String sql = String.format("SELECT %s FROM %s t WHERE %s",
				createSelectParams("t"), getName(), createPkWhereClause("t"));

		return fireOnCreateQuery(SynchroQueryName.selectByPk, sql);
	}

	/**
	 * <p>createSelectPksStrQuery.</p>
	 *
	 * @param config a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String createSelectPksStrQuery(SynchroDatabaseConfiguration config) {
		// Could not update, because no PK found
		if (CollectionUtils.isEmpty(pkNames)) {
			return null;
		}

		String prefix = " || '" + PK_SEPARATOR + "' || ";
		StringBuilder pkParams = new StringBuilder();

		boolean allowUniqueOutputColumn = true;

		for (String columnName : pkNames) {

			// If some date conversion => not database independent
			SynchroColumnMetadata column = columns.get(columnName);

			// For date/timestamp : make sure the convertion to char give the
			// same result
			// -> add a cast to Timestamp
			if (column.getTypeCode() == Types.TIMESTAMP
					|| column.getTypeCode() == Types.DATE) {
				allowUniqueOutputColumn = false;
			}
			pkParams.append(prefix).append("t.").append(columnName);
		}

		if (!allowUniqueOutputColumn) {
			return null;
		}

		// Append updateDate is present
		if (withUpdateDateColumn) {
			pkParams.append(", ").append(config.getColumnUpdateDate());
		}

		String sql = String.format("SELECT %s FROM %s t",
				pkParams.substring(prefix.length()), getName());

		return fireOnCreateQuery(SynchroQueryName.selectPksStr, sql);
	}

	/**
	 * <p>createSelectPksQuery.</p>
	 *
	 * @param config a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String createSelectPksQuery(SynchroDatabaseConfiguration config) {
		// Could not update, because no PK found
		if (CollectionUtils.isEmpty(pkNames)) {
			return null;
		}

		StringBuilder pkParams = new StringBuilder();

		for (String columnName : pkNames) {
			pkParams.append(", t.").append(columnName);
		}

		// Append updateDate is present
		if (withUpdateDateColumn) {
			pkParams.append(", t.").append(config.getColumnUpdateDate());
		}

		String sql = String.format("SELECT %s FROM %s t",
				pkParams.substring(2), getName());

		return fireOnCreateQuery(SynchroQueryName.selectPks, sql);
	}

	/**
	 * <p>createSelectPksByIndexQueries.</p>
	 *
	 * @param config a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 * @return a {@link java.util.Map} object.
	 */
	protected Map<String, String> createSelectPksByIndexQueries(
			SynchroDatabaseConfiguration config) {
		// Could not update, because no PK found
		if (MapUtils.isEmpty(uniqueConstraints)) {
			return null;
		}
		// Could not update, because no PK found
		if (CollectionUtils.isEmpty(pkNames)) {
			return null;
		}
		Map<String, String> result = Maps.newHashMap();

		// Create the select clause
		StringBuilder pkParams = new StringBuilder();
		for (String columnName : pkNames) {
			pkParams.append(", t.").append(columnName);
		}

		// Append updateDate is present
		if (withUpdateDateColumn) {
			pkParams.append(", t.").append(config.getColumnUpdateDate());
		}

		// For each unique constraints
		for (Entry<String, List<String>> entry : uniqueConstraints.entrySet()) {

			// Create the where clause
			StringBuilder whereParams = new StringBuilder();
			List<String> columnNames = entry.getValue();
			for (String columnName : entry.getValue()) {
				SynchroColumnMetadata column = getColumn(columnName);
				if (column == null) {
					throw new SynchroTechnicalException(
							String.format(
									"%s Missing column [%s] declared in unique constraint with name [%s]",
									getTableLogPrefix(), columnName,
									entry.getKey()));
				}
				boolean isNullable = getColumn(columnName).isNullable();
				if (isNullable && columnNames.size() > 1/* see mantis 25893 */) {
					// If nullable column, use 'IS NULL' operator when binding
					// value = null (mantis #23769)
					whereParams.append(String.format(
							" AND (t.%s = ? OR (? is null and t.%s is null))",
							columnName, columnName));
				} else {
					whereParams.append(" AND t.").append(columnName)
							.append(" = ?");
				}
			}

			String sql = String.format("SELECT %s FROM %s t WHERE %s",
					pkParams.substring(2), getName(), whereParams.substring(5));

			sql = fireOnCreateQuery(SynchroQueryName.selectPksByIndex, sql);

			if (StringUtils.isNotBlank(sql)) {
				result.put(entry.getKey(), sql);
			}
		}

		return result;
	}

	/**
	 * <p>createSelectByFksQueries.</p>
	 *
	 * @param config a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 * @return a {@link java.util.Map} object.
	 */
	protected Map<String, String> createSelectByFksQueries(
			SynchroDatabaseConfiguration config) {
		return Maps.newHashMap();
	}

	/**
	 * <p>createSelectAllQuery.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	protected String createSelectAllQuery() {
		String sql = String.format("SELECT %s FROM %s t",
				createSelectParams("t"), getName());

		return fireOnCreateQuery(SynchroQueryName.select, sql);
	}

	/**
	 * <p>createSelectMaxUpdateDateQuery.</p>
	 *
	 * @param config a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String createSelectMaxUpdateDateQuery(
			SynchroDatabaseConfiguration config) {
		if (!withUpdateDateColumn) {
			return null;
		}
		String sql = String.format("SELECT max(t.%s) FROM %s t",
				config.getColumnUpdateDate(), getName());

		return fireOnCreateQuery(SynchroQueryName.selectMaxUpdateDate, sql);
	}

	/**
	 * <p>createSelectUpdateDateByPkQuery.</p>
	 *
	 * @param config a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String createSelectUpdateDateByPkQuery(
			SynchroDatabaseConfiguration config) {
		if (!withUpdateDateColumn) {
			return null;
		}
		String sql = String.format("SELECT t.%s FROM %s t WHERE %s",
				config.getColumnUpdateDate(), getName(),
				createPkWhereClause("t"));

		return fireOnCreateQuery(SynchroQueryName.selectUpdateDateByPk, sql);
	}

	/**
	 * <p>createSelectFromUpdateDateQuery.</p>
	 *
	 * @param config a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String createSelectFromUpdateDateQuery(
			final SynchroDatabaseConfiguration config) {
		if (!withUpdateDateColumn) {
			return null;
		}
		String sql = String.format("SELECT %s FROM %s t%s",
				createSelectParams("t"), getName(),
				createWithUpdateDateWhereClause(config, "t"));

		return fireOnCreateQuery(SynchroQueryName.selectFromUpdateDate, sql);
	}

	/**
	 * <p>createAllCountQuery.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	protected String createAllCountQuery() {
		String sql = String.format("SELECT count(*) FROM %s t", getName());

		// DO NOT send override query event (not where clause should be add)
		// return sqlfireOnCreateQuery(SynchroQueryName.countAll, sql);

		return sql;
	}

	/**
	 * <p>createCountQuery.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	protected String createCountQuery() {
		String sql = String.format("SELECT count(*) FROM %s t", getName());

		return fireOnCreateQuery(SynchroQueryName.count, sql);
	}

	/**
	 * <p>createCountFromUpdateDate.</p>
	 *
	 * @param config a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String createCountFromUpdateDate(
			final SynchroDatabaseConfiguration config) {
		String sql = String.format("SELECT count(*) FROM %s t%s", getName(),
				createWithUpdateDateWhereClause(config, "t"));

		return fireOnCreateQuery(SynchroQueryName.countFromUpdateDate, sql);
	}

	/**
	 * <p>createPkWhereClause.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String createPkWhereClause() {
		return createPkWhereClause(null);
	}

	/**
	 * <p>createPkWhereClause.</p>
	 *
	 * @param tableAlias a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	public String createPkWhereClause(String tableAlias) {
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(pkNames));
		StringBuilder pkParams = new StringBuilder();
		String prefix = tableAlias != null ? tableAlias + "." : "";

		for (String columnName : pkNames) {
			pkParams.append(" AND ").append(prefix).append(columnName)
					.append(" = ?");
		}

		return pkParams.substring(5);
	}

	/**
	 * <p>createWithUpdateDateWhereClause.</p>
	 *
	 * @param config a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String createWithUpdateDateWhereClause(
			final SynchroDatabaseConfiguration config) {
		return createWithUpdateDateWhereClause(config, null);
	}

	/**
	 * <p>createWithUpdateDateWhereClause.</p>
	 *
	 * @param config a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 * @param tableAlias a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String createWithUpdateDateWhereClause(
			final SynchroDatabaseConfiguration config, String tableAlias) {
		String whereClause;

		if (isWithUpdateDateColumn()) {
			String columnUpdateDate = config.getColumnUpdateDate();
			String prefix = tableAlias != null ? tableAlias + "." : "";
			// add a filter
			whereClause = String.format(" WHERE (%s%s IS NULL OR %s%s > :%s)",
					prefix, columnUpdateDate, prefix, columnUpdateDate,
					UPDATE_DATE_BINDPARAM);
		} else {
			whereClause = "";
		}
		return whereClause;
	}

	/**
	 * <p>createSelectParams.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	protected String createSelectParams() {
		return createSelectParams(null);
	}

	/**
	 * <p>createSelectParams.</p>
	 *
	 * @param tableAlias a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String createSelectParams(String tableAlias) {
		return createSelectParams(this.columnNames, tableAlias);
	}

	/**
	 * <p>createSelectParams.</p>
	 *
	 * @param columnNames a {@link java.util.List} object.
	 * @param tableAlias a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String createSelectParams(List<String> columnNames,
			String tableAlias) {
		Preconditions.checkArgument(
				CollectionUtils.isNotEmpty(columnNames),
				String.format("No column found for table: %s",
						delegate.getName()));

		StringBuilder queryParams = new StringBuilder();

		for (String columnName : columnNames) {
			queryParams.append(", ");
			if (tableAlias != null) {
				queryParams.append(tableAlias).append(".");
			}
			queryParams.append(columnName);
		}

		return queryParams.substring(2);
	}

	/**
	 * <p>initSequenceName.</p>
	 *
	 * @param dbMeta a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String initSequenceName(SynchroDatabaseMetadata dbMeta) {
		final Set<String> availableSequences = dbMeta.getSequences();
		final SynchroDatabaseConfiguration config = dbMeta.getConfiguration();
		final String sequenceSuffix = config.getSequenceSuffix();
		final int maxSqlNameLength = config.getMaxSqlNameLength();

		final String tableName = getName().toLowerCase();
		String sequenceName;

		// Compute the max size of
		final int maxLength = maxSqlNameLength - sequenceSuffix.length();
		if (maxLength > -0) {
			sequenceName = SynchroMetadataUtils.ensureMaximumNameLength(
					tableName, maxLength) + sequenceSuffix;
			if (availableSequences.contains(sequenceName.toLowerCase())) {
				return sequenceName;
			}
		}

		// If not found (with length limit), try without length limit
		sequenceName = tableName + sequenceSuffix;
		if (availableSequences.contains(sequenceName.toLowerCase())) {
			return sequenceName;
		}

		// sequence not found
		return null;
	}

	/**
	 * <p>createSelectSequenceNextValString.</p>
	 *
	 * @param dialect a {@link org.hibernate.dialect.Dialect} object.
	 * @param sequenceName a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String createSelectSequenceNextValString(Dialect dialect,
			String sequenceName) {
		if (StringUtils.isBlank(sequenceName)) {
			return null;
		}
		return dialect.getSelectSequenceNextValString(sequenceName);
	}

	/**
	 * <p>createSequenceNextValString.</p>
	 *
	 * @param dialect a {@link org.hibernate.dialect.Dialect} object.
	 * @param sequenceName a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String createSequenceNextValString(Dialect dialect,
			String sequenceName) {
		if (StringUtils.isBlank(sequenceName)) {
			return null;
		}
		return dialect.getSequenceNextValString(sequenceName);
	}

	/**
	 * <p>fireOnCreateQuery.</p>
	 *
	 * @param queryName a {@link fr.ifremer.common.synchro.query.SynchroQueryName} object.
	 * @param sql a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String fireOnCreateQuery(SynchroQueryName queryName, String sql) {
		if (eventBus == null) {
			return sql;
		}

		// Dispatch event to listeners
		CreateQueryEvent e = new CreateQueryEvent(this, queryName, sql);
		eventBus.post(e);

		// Return the updated sql query
		return e.sql;
	}

	/**
	 * <p>fireOnJoinLoad.</p>
	 *
	 * @param join a {@link fr.ifremer.common.synchro.meta.SynchroJoinMetadata} object.
	 * @return a {@link fr.ifremer.common.synchro.meta.SynchroJoinMetadata} object.
	 */
	protected SynchroJoinMetadata fireOnJoinLoad(SynchroJoinMetadata join) {
		if (eventBus == null || !join.isValid()) {
			return join;
		}

		// Dispatch event to listeners
		LoadJoinEvent e = new LoadJoinEvent(this, join);
		eventBus.post(e);

		// Return the updated join
		return e.join;
	}

	/**
	 * <p>fireOnPkLoad.</p>
	 *
	 * @param pk a {@link java.util.Set} object.
	 * @return a {@link java.util.Set} object.
	 */
	protected Set<String> fireOnPkLoad(Set<String> pk) {
		if (eventBus == null) {
			return pk;
		}

		// Dispatch event to listeners
		LoadPkEvent e = new LoadPkEvent(this, pk);
		eventBus.post(e);

		// Return the updated join
		return e.pk;
	}

	/**
	 * <p>build.</p>
	 *
	 * @return a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 */
	public SynchroTableMetadata build() {
		if (isBuild) {
			return this;
		}

		// Dispatch load event to listeners
		if (eventBus != null) {
			LoadTableEvent e = new LoadTableEvent(dbMeta, this);
			eventBus.post(e);
		}

		try {
			// Compute/Refresh some fields (after event dispatch)
			this.selectPksByIndexQueries = createSelectPksByIndexQueries(dbMeta
					.getConfiguration());
			this.selectSequenceNextValString = createSelectSequenceNextValString(
					dialect, this.sequenceName);
			this.sequenceNextValString = createSequenceNextValString(dialect,
					this.sequenceName);

		} catch (Exception e) {
			throw new SynchroTechnicalException(t(
					"synchro.meta.table.instanciation.error",
					delegate.getName()), e);
		}

		// Clean fields unused after build
		this.eventBus = null;
		this.dbMeta = null;
		this.dialect = null;

		if (log.isTraceEnabled()) {
			logQueriesMappings();
		}

		isBuild = true;
		return this;
	}

	/**
	 * <p>logQueriesMappings.</p>
	 */
	public void logQueriesMappings() {
		if (selectAllQuery == null || !log.isDebugEnabled()) {
			return;
		}
		List<String> selectColumnNames = SynchroQueryBuilder.newBuilder(
				this.selectAllQuery).getColumnNames();
		SynchroQueryBuilder insertQuery = SynchroQueryBuilder
				.newBuilder(this.insertQuery);
		SynchroQueryBuilder updateQuery = SynchroQueryBuilder
				.newBuilder(this.updateQuery);
		List<String> insertBindColumnNames = Lists.newArrayList();
		for (String columnName : insertQuery.getColumnNames()) {
			if ("?".equals(insertQuery.getColumnValue(columnName))) {
				insertBindColumnNames.add(columnName);
			}
		}
		List<String> updateBindColumnNames = Lists.newArrayList();
		for (String columnName : updateQuery.getColumnNames()) {
			if ("?".equals(updateQuery.getColumnValue(columnName))) {
				updateBindColumnNames.add(columnName);
			}
		}

		StringBuilder insertBuffer = new StringBuilder();
		insertBuffer.append(String.format("[%s] Insert mapping:", getName()));
		StringBuilder updateBuffer = new StringBuilder();
		updateBuffer.append(String.format("[%s] Update mapping:", getName()));
		int i = 0;
		int j = 0;
		for (String selectColumnName : selectColumnNames) {
			String insertColumnName = "N/A";
			if (i < insertBindColumnNames.size()) {
				insertColumnName = insertBindColumnNames.get(i++);
			}
			String updateColumnName = "N/A";
			if (j < updateBindColumnNames.size()) {
				updateColumnName = updateBindColumnNames.get(j++);
			}
			insertBuffer.append("\n\t").append(selectColumnName).append(" -> ")
					.append(insertColumnName);
			updateBuffer.append("\n\t").append(selectColumnName).append(" -> ")
					.append(updateColumnName);
		}
		if (insertBindColumnNames.size() > i) {
			for (int k = i; k < insertBindColumnNames.size(); k++) {
				insertBuffer.append("\n\tN/A  -> ").append(
						insertBindColumnNames.get(k));
			}
		}
		if (updateBindColumnNames.size() > j) {
			for (int k = j; k < updateBindColumnNames.size(); k++) {
				updateBuffer.append("\n\tN/A  -> ").append(
						updateBindColumnNames.get(k));
			}
		}

		log.trace(insertBuffer.toString());
		log.trace(updateBuffer.toString());
	}

	/**
	 * <p>initEventBus.</p>
	 *
	 * @param enableEvent a boolean.
	 * @param listeners a {@link java.util.List} object.
	 * @param tableName a {@link java.lang.String} object.
	 * @return a {@link com.google.common.eventbus.EventBus} object.
	 */
	protected EventBus initEventBus(boolean enableEvent,
			List<Object> listeners, String tableName) {
		if (CollectionUtils.isEmpty(listeners) || !enableEvent) {
			return null;
		}
		EventBus eventBus = new EventBus(new SubscriberExceptionHandler() {
			@Override
			public void handleException(Throwable exception,
					SubscriberExceptionContext context) {
				throw new SynchroTechnicalException(exception);
			}
		});

		if (log.isDebugEnabled()) {
			log.debug(String.format("[%s] Active listeners: ", tableName));
		}

		for (Object listener : listeners) {
			eventBus.register(listener);
			if (log.isDebugEnabled()) {
				log.debug(String.format("[%s]  - %s", tableName, listener
						.getClass().getName()));
			}
		}
		return eventBus;
	}

	/**
	 * <p>addUniqueConstraintsIfNotPK.</p>
	 *
	 * @param indexName a {@link java.lang.String} object.
	 * @param columnNames a {@link java.util.List} object.
	 * @param duplicateKeyStrategy a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata.DuplicateKeyStrategy} object.
	 */
	protected void addUniqueConstraintsIfNotPK(String indexName,
			List<String> columnNames, DuplicateKeyStrategy duplicateKeyStrategy) {
		Preconditions.checkArgument(!uniqueConstraints.containsKey(indexName),
				"Duplicate unique constraints name");
		Preconditions.checkNotNull(CollectionUtils.isNotEmpty(columnNames));

		// Do not add unique constraints if equals to PK constraints
		if (CollectionUtils.isEqualCollection(pkNames, columnNames)) {
			return;
		}
		if (log.isDebugEnabled()) {
			log.debug(String.format("[%s] Add unique constraint: %s",
					getName(), indexName));
		}

		uniqueConstraints.put(indexName, columnNames);
		duplicateKeyStrategies.put(indexName, duplicateKeyStrategy);
		hasUniqueConstraints = true;
	}

	/**
	 * <p>checkBuild.</p>
	 */
	protected void checkBuild() {
		Preconditions.checkState(isBuild,
				"table not build. Please call build() first.");
	}

	/**
	 * <p>getCountPkReferenceQueriesUniquePk.</p>
	 *
	 * @param dbMeta a {@link java.sql.DatabaseMetaData} object.
	 * @return a {@link java.util.Map} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected Map<String, String> getCountPkReferenceQueriesUniquePk(
			DatabaseMetaData dbMeta) throws SQLException {
		Preconditions.checkArgument(isSimpleKey());
		String pkColumnName = pkNames.iterator().next();

		// If a PK is exported (used by another table): store table and columns
		// in a multi map
		Multimap<String, String[]> columnMappingsByTable = ArrayListMultimap
				.create();
		ResultSet rs = null;
		try {
			rs = dbMeta.getExportedKeys(getCatalog(), getSchema(), getName());
			while (rs.next()) {
				String columnName = rs.getString("PKCOLUMN_NAME").toLowerCase();

				if (pkColumnName.equals(columnName)) {
					String fkTableName = rs.getString("FKTABLE_NAME")
							.toUpperCase();
					String fkColumnName = rs.getString("FKCOLUMN_NAME")
							.toLowerCase();
					columnMappingsByTable.put(fkTableName, new String[]{
							columnName, fkColumnName});
				}
			}
		} finally {
			Daos.closeSilently(rs);
		}

		// if PK exported: stop
		if (columnMappingsByTable.isEmpty()) {
			return null;
		}

		Map<String, String> result = Maps.newHashMap();
		for (String fkTableName : columnMappingsByTable.keySet()) {
			// Compute the where clause
			StringBuilder whereClause = new StringBuilder();
			Collection<String[]> columnMappings = columnMappingsByTable
					.get(fkTableName);

			// iterate on PK to keep PKs order in binding parameters
			for (String[] columnMapping : columnMappings) {
				String fkColumnName = columnMapping[1];
				// If more than one reference, need or OR (when 2 referencee one
				// a table - mantis #23134)
				whereClause.append(" OR ").append(fkColumnName).append("=:pk1");
			}

			String sql = String.format("select count(*) from %s where %s",
					fkTableName, whereClause.substring(4));
			result.put(fkTableName, sql);
		}

		return result;
	}

	/**
	 * <p>getCountPkReferenceQueriesCompositePk.</p>
	 *
	 * @param dbMeta a {@link java.sql.DatabaseMetaData} object.
	 * @return a {@link java.util.Map} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected Map<String, String> getCountPkReferenceQueriesCompositePk(
			DatabaseMetaData dbMeta) throws SQLException {
		Set<String> pkColumnNames = getPkNames();

		// If a PK is exported (used by another table): store table and columns
		// in a multi map
		Multimap<String, String[]> columnMappingsByTable = ArrayListMultimap
				.create();
		ResultSet rs = null;
		try {
			rs = dbMeta.getExportedKeys(getCatalog(), getSchema(), getName());
			while (rs.next()) {
				String columnName = rs.getString("PKCOLUMN_NAME").toLowerCase();

				if (pkColumnNames.contains(columnName)) {
					String fkTableName = rs.getString("FKTABLE_NAME")
							.toUpperCase();
					String fkColumnName = rs.getString("FKCOLUMN_NAME")
							.toLowerCase();
					columnMappingsByTable.put(fkTableName, new String[]{
							columnName, fkColumnName});
				}
			}
		} finally {
			Daos.closeSilently(rs);
		}

		// if no exported PK: stop
		if (columnMappingsByTable.isEmpty()) {
			return null;
		}

		Map<String, String> result = Maps.newHashMap();
		for (String fkTableName : columnMappingsByTable.keySet()) {
			// Compute the where clause
			StringBuilder whereClause = new StringBuilder();
			Collection<String[]> columnMappings = columnMappingsByTable
					.get(fkTableName);

			// iterate on PK to keep PKs order in binding parameters
			int pkIndex = 1;
			for (String pkColumnName : pkColumnNames) {
				String pkParamName = ":pk" + pkIndex;
				for (String[] columnMapping : columnMappings) {
					if (columnMapping[0].equals(pkColumnName)) {
						String fkColumnName = columnMapping[1];
						whereClause.append(" AND ").append(fkColumnName)
								.append("=").append(pkParamName);
					} else {
						// a fake condition, to be able to bind on each pk
						// column (see SynchroTableDaoImpl)
						whereClause.append(" AND ").append(pkParamName)
								.append(" is not null");
					}
				}
				pkIndex++;
			}

			String sql = String.format("select count(*) from %s where %s",
					fkTableName, whereClause.substring(5));
			result.put(fkTableName, sql);
		}

		return result;
	}

}
