package fr.ifremer.common.synchro.meta;

/*
 * #%L
 * Tutti :: Persistence API
 * $Id: TuttiEntities.java 1578 2014-02-07 15:31:18Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/trunk/tutti-persistence/src/main/java/fr/ifremer/tutti/persistence/entities/TuttiEntities.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Predicate;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import fr.ifremer.common.synchro.SynchroTechnicalException;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.sql.Types;
import java.util.List;
import java.util.Set;

/**
 * Useful method around DAO and entities.
 *
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 * @since 3.5.4
 */
public class SynchroMetadataUtils {
	/** Logger. */
	private static final Log log = LogFactory
			.getLog(SynchroMetadataUtils.class);

	/** Constant <code>ORACLE_EXCLUDE_TABLE_PATTERNS</code> */
	public static final List<String> ORACLE_EXCLUDE_TABLE_PATTERNS = Lists
			.newArrayList("BIN%", // Trash Oracle
					"MDR%" // Table metadata spatial
			);

	/**
	 * <p>Constructor for SynchroMetadataUtils.</p>
	 */
	protected SynchroMetadataUtils() {
		// helper class does not instantiate
	}

	/**
	 * <p>newAllTablesOraclePredicate.</p>
	 *
	 * @return a {@link com.google.common.base.Predicate} object.
	 */
	public static Predicate<String> newAllTablesOraclePredicate() {
		return newTablesOraclePredicate(null, null);
	}

	/**
	 * <p>newTablesOraclePredicate.</p>
	 *
	 * @param excludes a {@link java.util.Set} object.
	 * @param includes a {@link java.util.Set} object.
	 * @return a {@link com.google.common.base.Predicate} object.
	 */
	public static Predicate<String> newTablesOraclePredicate(
			Set<String> excludes, final Set<String> includes) {
		Set<String> excludesPatterns = Sets
				.newHashSet(ORACLE_EXCLUDE_TABLE_PATTERNS);

		if (CollectionUtils.isNotEmpty(excludes)) {
			excludesPatterns.addAll(excludes);
		}

		return newTablePredicate(excludesPatterns, includes);
	}

	/**
	 * <p>newTablePredicate.</p>
	 *
	 * @param excludes a {@link java.util.Set} object.
	 * @param includes a {@link java.util.Set} object.
	 * @return a {@link com.google.common.base.Predicate} object.
	 */
	public static Predicate<String> newTablePredicate(
			final Set<String> excludes, final Set<String> includes) {
		// If no filter
		if (CollectionUtils.isEmpty(excludes)
				&& CollectionUtils.isEmpty(includes)) {
			return null;
		}

		return new Predicate<String>() {
			public boolean apply(String tableName) {

				if (CollectionUtils.isNotEmpty(excludes)) {
					for (String excludePattern : excludes) {
						if (tableName.matches(excludePattern.replaceAll("%",
								".*"))) {
							return false;
						}
					}
				}
				if (CollectionUtils.isEmpty(includes)) {
					return true;
				}
				for (String includePattern : includes) {
					if (tableName.matches(includePattern.replaceAll("%", ".*"))) {
						return true;
					}
				}
				return false;
			}
		};
	}

	/**
	 * <p>newExcludeColumnPredicate.</p>
	 *
	 * @param columnExcludes a {@link java.util.Set} object.
	 * @return a {@link com.google.common.base.Predicate} object.
	 */
	public static Predicate<SynchroColumnMetadata> newExcludeColumnPredicate(
			final Set<String> columnExcludes) {
		if (columnExcludes == null || columnExcludes.isEmpty()) {
			return null;
		}

		return new Predicate<SynchroColumnMetadata>() {
			public boolean apply(SynchroColumnMetadata input) {
				return !columnExcludes.contains(input.getTableName()
						.toLowerCase() + "." + input.getName().toLowerCase())
						&& !columnExcludes.contains("%."
								+ input.getName().toLowerCase());
			}
		};
	}

	/**
	 * Check that the tow given shemas are compatible for a synchronize
	 * operation (same tables with same columns).
	 * <p>
	 * If <code>allowMissingOptionalColumn=true</code> then missing columns are
	 * allowed. Missing columns will be added to the given result.
	 * </p>
	 * If <code>allowAdditionalMandatoryColumnInSourceSchema=true</code> then
	 * additional mandatory columns in the source schema are allowed. It could
	 * be set to
	 * <code>false</code> for data synchronization, to avoid getting data from tables that could not be export later.
	 * <br>
	 * If schemas are incompatible, then a {@link fr.ifremer.common.synchro.SynchroTechnicalException} exception will be thrown.
	 *
	 * @param targetSchema
	 *            schema 1 to check
	 * @param sourceSchema
	 *            schema 2 to check
	 * @param allowMissingOptionalColumn
	 *            Is missing optional columns are allowed (in source or target
	 *            schema) ? If true, missing column will be ignore in
	 *            synchronization.
	 * @param allowAdditionalMandatoryColumnInSourceSchema
	 *            Is additional mandatory columns are allowed in source schema ?
	 *            If true, source schema could have more mandatory columns.
	 * @return a list of columns to exclude to avoid error
	 * @throws fr.ifremer.common.synchro.meta.SynchroSchemaValidationException
	 *             if schema are not compatible
	 * @throws fr.ifremer.common.synchro.SynchroTechnicalException if any.
	 */
	public static Set<String> checkSchemas(
			SynchroDatabaseMetadata sourceSchema,
			SynchroDatabaseMetadata targetSchema,
			boolean allowMissingOptionalColumn,
			boolean allowAdditionalMandatoryColumnInSourceSchema)
			throws SynchroSchemaValidationException, SynchroTechnicalException {

		if (!allowMissingOptionalColumn) {
			checkSchemasStrict(sourceSchema, targetSchema);
			return null;
		}

		return checkSchemasAllowMissingOptionalColumn(sourceSchema,
				targetSchema, allowAdditionalMandatoryColumnInSourceSchema);
	}

	/**
	 * Check that the tow given shemas are compatible for a synchronize
	 * operation (same tables with same columns). *
	 * <br>
	 * This method allow missing columns (if define as nullable in the target
	 * schema)
	 * <br>
	 * If schemas are incompatible, then a {@link fr.ifremer.common.synchro.SynchroTechnicalException}
	 * exception will be thrown.
	 *
	 * @param targetSchema
	 *            schema 1 to check
	 * @param sourceSchema
	 *            schema 2 to check
	 * @throws fr.ifremer.common.synchro.meta.SynchroSchemaValidationException
	 *             if schema are not compatible
	 * @param allowAdditionalMandatoryColumnInSourceSchema a boolean.
	 * @return a {@link java.util.Set} object.
	 */
	protected static Set<String> checkSchemasAllowMissingOptionalColumn(
			SynchroDatabaseMetadata sourceSchema,
			SynchroDatabaseMetadata targetSchema,
			boolean allowAdditionalMandatoryColumnInSourceSchema)
			throws SynchroSchemaValidationException {
		Set<String> missingColumns = Sets.newHashSet();

		Set<String> targetSchemaTableNames = targetSchema.getLoadedTableNames();
		Set<String> sourceSchemaTableNames = sourceSchema.getLoadedTableNames();

		// Check if table names are equals
		if (!targetSchemaTableNames.equals(sourceSchemaTableNames)) {
			Set<String> missingTargetTables = Sets.newHashSet();
			for (String targetTableName : targetSchemaTableNames) {
				if (!sourceSchemaTableNames.contains(targetTableName)) {
					missingTargetTables.add(targetTableName);
				}
			}
			Set<String> missingSourceTables = Sets.newHashSet();
			for (String sourceTableName : sourceSchemaTableNames) {
				if (!targetSchemaTableNames.contains(sourceTableName)) {
					missingSourceTables.add(sourceTableName);
				}
			}

			throw new SynchroSchemaValidationException(
					String.format(
							"Incompatible schemas.\nMissing tables in source database: %s\nMissing tables in target database: %s",
							missingTargetTables, missingSourceTables));
		}

		for (String tableName : targetSchemaTableNames) {
			SynchroTableMetadata targetTable = targetSchema
					.getLoadedTable(tableName);
			SynchroTableMetadata sourceTable = sourceSchema
					.getLoadedTable(tableName);
			Set<String> targetColumnNames = Sets.newHashSet(targetTable
					.getColumnNames());
			Set<String> sourceColumnNames = sourceTable.getColumnNames();

			// Check if columns names are equals
			if (!targetColumnNames.equals(sourceColumnNames)) {
				Set<String> missingMandatoryColumns = Sets.newTreeSet();
				Set<String> missingSourceColumnNames = Sets
						.newHashSet(sourceColumnNames);

				// Check if missing column (in source) are optional
				for (String targetColumnName : targetTable.getColumnNames()) {
					if (!sourceColumnNames.contains(targetColumnName)) {
						SynchroColumnMetadata targetColumn = targetTable
								.getColumnMetadata(targetColumnName);

						// Optional column: add it to the context (will be
						// ignore in during synchronization)
						if (targetColumn.isNullable()) {
							log.debug(String
									.format("Optional column not found in source database: %s.%s. Will be ignore.",
											tableName, targetColumnName));
							missingColumns.add(tableName + "."
									+ targetColumnName);
							targetColumnNames.remove(targetColumnName);
						}

						// Mandatory columns: add to list to check later
						else {
							log.warn(String
									.format("Column not found in source database: %s.%s",
											tableName, targetColumnName));
							missingMandatoryColumns.add(targetColumnName);
						}
					}

					missingSourceColumnNames.remove(targetColumnName);
				}

				// Check if missing column (in target) are optional
				for (String sourceColumnName : missingSourceColumnNames) {
					SynchroColumnMetadata sourceColumn = sourceTable
							.getColumnMetadata(sourceColumnName);
					if (allowAdditionalMandatoryColumnInSourceSchema
							|| sourceColumn.isNullable()) {
						log.debug(String
								.format("Optional column not found in target database: %s.%s. Will be ignore.",
										tableName, sourceColumnName));
						missingColumns.add(tableName + "." + sourceColumnName);
					} else {
						log.warn(String
								.format("Column not found in target database: %s.%s. Will be ignore.",
										tableName, sourceColumnName));
						missingMandatoryColumns.add(sourceColumnName);
					}
				}

				// Throw an exception if exists any missing column
				if (CollectionUtils.isNotEmpty(missingMandatoryColumns)) {
					throw new SynchroSchemaValidationException(
							String.format(
									"Incompatible schema of table: %s. Missing mandatory columns: %s",
									tableName, missingMandatoryColumns));
				}
			}

			// Check column types compatibility
			for (String columnName : targetColumnNames) {
				SynchroColumnMetadata targetColumn = targetTable
						.getColumnMetadata(columnName);
				SynchroColumnMetadata sourceColumn = sourceTable
						.getColumnMetadata(columnName);
				checkType(tableName, targetColumn, sourceColumn);
			}
		}

		return missingColumns;
	}

	/**
	 * Check that the tow given datasource shemas are compatible for a
	 * synchronize operation (same tables with same columns).
	 * <br>
	 * If schemas are incompatible, then a {@link fr.ifremer.common.synchro.SynchroTechnicalException}
	 * exception will be thrown.
	 *
	 * @throws fr.ifremer.common.synchro.meta.SynchroSchemaValidationException
	 *             if schema are not compatible
	 * @param sourceSchema a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 * @param targetSchema a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 */
	protected static void checkSchemasStrict(
			SynchroDatabaseMetadata sourceSchema,
			SynchroDatabaseMetadata targetSchema)
			throws SynchroSchemaValidationException {
		Set<String> sourceSchemaTableNames = sourceSchema.getLoadedTableNames();
		Set<String> targetSchemaTableNames = targetSchema.getLoadedTableNames();

		// Check if table names are equals
		if (!targetSchemaTableNames.equals(sourceSchemaTableNames)) {
			throw new SynchroSchemaValidationException(
					"Incompatible schemas: missing tables");
		}

		for (String tableName : sourceSchemaTableNames) {
			SynchroTableMetadata sourceTable = sourceSchema.getTable(tableName);
			SynchroTableMetadata targetTable = targetSchema.getTable(tableName);
			Set<String> sourceColumnNames = sourceTable.getColumnNames();
			Set<String> targetColumnNames = targetTable.getColumnNames();

			// Check if columns names are equals
			if (!targetColumnNames.equals(sourceColumnNames)) {
				throw new SynchroSchemaValidationException(
						"Incompatible schema of table: " + tableName);
			}

			// Check column types compatibility
			for (String columnName : targetColumnNames) {
				SynchroColumnMetadata sourceColumn = sourceTable
						.getColumnMetadata(columnName);
				SynchroColumnMetadata targetColumn = targetTable
						.getColumnMetadata(columnName);
				checkType(tableName, targetColumn, sourceColumn);
			}
		}
	}

	/**
	 * Check if types are compatible, and return a DataRetrievalFailureException
	 * if not compatible.
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @param internalColumn a {@link fr.ifremer.common.synchro.meta.SynchroColumnMetadata} object.
	 * @param externalColumn a {@link fr.ifremer.common.synchro.meta.SynchroColumnMetadata} object.
	 */
	public static void checkType(String tableName,
			SynchroColumnMetadata internalColumn,
			SynchroColumnMetadata externalColumn) {

		// If numeric
		if (isNumericType(internalColumn) && isNumericType(externalColumn)) {

			if (isIgnoreColumnSizeType(internalColumn)
					|| isIgnoreColumnSizeType(externalColumn)) {
				// OK
			} else {
				int internalColumnSize = internalColumn.getColumnSize();
				int externalColumnSize = externalColumn.getColumnSize();
				if (internalColumnSize > 0 && externalColumnSize > 0
						&& internalColumnSize < externalColumnSize) {
					throw new SynchroTechnicalException(
							String.format(
									"Incompatible type for column [%s.%s]: incompatible column size between source [%s] and target [%s]",
									tableName, internalColumn.getName(),
									externalColumnSize, internalColumnSize));
				}
				int internalDecimalDigits = internalColumn.getDecimalDigits();
				int externalDecimalDigits = externalColumn.getDecimalDigits();
				if (internalDecimalDigits > 0 && externalDecimalDigits > 0
						&& internalDecimalDigits < externalDecimalDigits) {
					throw new SynchroTechnicalException(
							String.format(
									"Incompatible type for column [%s.%s]: incompatible decimal digits between source [%s] and target [%s]",
									tableName, internalColumn.getName(),
									externalDecimalDigits,
									internalDecimalDigits));
				}
			}
		}

		// If Date
		else if (isDateType(internalColumn) && isDateType(externalColumn)) {
			// OK
		}

		// If Boolean
		else if (isBooleanType(internalColumn) && isBooleanType(externalColumn)) {
			// OK
		}

		// If Geometry
		else if (isGeometryType(internalColumn)
				&& isGeometryType(externalColumn)) {
			// OK
		}

		// Else : compare type code and name
		else {
			String internalColumnTypeName = internalColumn.getTypeName();
			String externalColumnTypeName = externalColumn.getTypeName();
			int internalColumnTypeCode = internalColumn.getTypeCode();
			int externalColumnTypeCode = externalColumn.getTypeCode();

			if (internalColumnTypeCode != externalColumnTypeCode
					&& internalColumnTypeName.equals(externalColumnTypeName) == false) {
				throw new SynchroTechnicalException(
						String.format(
								"Incompatible type for column [%s.%s] between source [%s] and target [%s]",
								tableName, internalColumn.getName(),
								externalColumnTypeName, internalColumnTypeName));
			}
		}
	}

	/**
	 * <p>isNumericType.</p>
	 *
	 * @param column a {@link fr.ifremer.common.synchro.meta.SynchroColumnMetadata} object.
	 * @return a boolean.
	 */
	public static boolean isNumericType(SynchroColumnMetadata column) {
		int typeCode = column.getTypeCode();
		if (typeCode == Types.BIGINT || typeCode == Types.INTEGER
				|| typeCode == Types.NUMERIC || typeCode == Types.DECIMAL
				|| typeCode == Types.FLOAT || typeCode == Types.REAL
				|| typeCode == Types.SMALLINT || typeCode == Types.TINYINT
				|| typeCode == Types.DOUBLE) {
			return true;
		}

		String columnTypeName = column.getTypeName();
		return columnTypeName.equals("NUMBER")
				|| columnTypeName.equals("INTEGER")
				|| columnTypeName.equals("SMALLINT");
	}

	/**
	 * <p>isDateType.</p>
	 *
	 * @param column a {@link fr.ifremer.common.synchro.meta.SynchroColumnMetadata} object.
	 * @return a boolean.
	 */
	public static boolean isDateType(SynchroColumnMetadata column) {
		String columnTypeName = column.getTypeName();
		return columnTypeName.equals("TIMESTAMP")
				|| columnTypeName.equals("DATE");
	}

	/**
	 * <p>isBooleanType.</p>
	 *
	 * @param column a {@link fr.ifremer.common.synchro.meta.SynchroColumnMetadata} object.
	 * @return a boolean.
	 */
	public static boolean isBooleanType(SynchroColumnMetadata column) {
		int typeCode = column.getTypeCode();
		int columnSize = column.getColumnSize();
		if ((typeCode == Types.CHAR || typeCode == Types.VARCHAR)
				&& columnSize == 1) {
			return true;
		}
		String columnTypeName = column.getTypeName();
		return columnTypeName.equals("BOOLEAN")
				|| (columnTypeName.equals("NUMBER") && columnSize == 1)
				|| (columnTypeName.equals("INTEGER") && columnSize == 1)
				|| (columnTypeName.equals("SMALLINT") && columnSize == 1)
				// BIT(1) -> Hsqldb 2.x default SQL type for boolean:
				|| (columnTypeName.equals("BIT") && columnSize == 1);
	}

	/**
	 * <p>isGeometryType.</p>
	 *
	 * @param column a {@link fr.ifremer.common.synchro.meta.SynchroColumnMetadata} object.
	 * @return a boolean.
	 */
	public static boolean isGeometryType(SynchroColumnMetadata column) {
		int typeCode = column.getTypeCode();
		int columnSize = column.getColumnSize();
		if (typeCode == Types.VARCHAR && columnSize > 1) {
			return true;
		}
		String columnTypeName = column.getTypeName();
		return columnTypeName.equals("SDO_GEOMETRY")
				|| columnTypeName.equals("GEOMETRY");
	}

	/**
	 * Must ignore column size comparison ?
	 * <br>
	 * (cf mantis #28694 - DO NOT compare decimal precision with Oracle type
	 * FLOAT)
	 * <br>
	 *
	 * @param column a {@link fr.ifremer.common.synchro.meta.SynchroColumnMetadata} object.
	 * @return a boolean.
	 */
	public static boolean isIgnoreColumnSizeType(SynchroColumnMetadata column) {
		int typeCode = column.getTypeCode();
		return (typeCode == Types.FLOAT || typeCode == Types.REAL);
	}

	/**
	 * <p>
	 * Trims the passed in value to the maximum name length.
	 * </p>
	 * If no maximum length has been set then this method does nothing.
	 *
	 * @param name
	 *            the name length to check and trim if necessary
	 * @param nameMaxLength
	 *            if this is not null, then the name returned will be trimmed to
	 *            this length (if it happens to be longer).
	 * @param nameMaxLength
	 *            if this is not null, then the name returned will be trimmed to
	 *            this length (if it happens to be longer).
	 * @return String the string to be used as SQL type
	 */
	public static String ensureMaximumNameLength(String name,
			Integer nameMaxLength) {
		if (StringUtils.isNotBlank(name) && nameMaxLength != null) {
			int max = nameMaxLength.intValue();
			if (name.length() > max) {
				name = name.substring(0, max);
			}
		}
		return name;
	}
}
