package fr.ifremer.common.synchro.meta;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import org.hibernate.internal.CoreMessageLogger;
import org.jboss.logging.Logger;

import java.io.Serializable;
import java.sql.ResultSet;
import java.sql.SQLException;

/**
 * Store metadata on a link between two tables.
 * <ul>
 * <li>Obtains owner table name {@link #getPkTable()};</li>
 * <li>Obtains linked column (for FK only) {@link #getFkColumn()} as a join
 * metadata: used to synchronize data in DataSynchroService (from adagio synchro module);</li>
 * </ul>
 *
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 * @since 3.5.2
 */
public class SynchroJoinMetadata implements Serializable {

	static final long serialVersionUID = -1L;

	private static final CoreMessageLogger LOG = Logger.getMessageLogger(
			CoreMessageLogger.class, SynchroJoinMetadata.class.getName());

	protected SynchroTableMetadata sourceTable;

	protected SynchroColumnMetadata sourceColumn;

	protected SynchroColumnMetadata targetColumn;

	protected SynchroTableMetadata targetTable;

	protected boolean isValid = false;

	protected boolean isChild = false;

	protected boolean isSourceFk = false;

	/**
	 * <p>Constructor for SynchroJoinMetadata.</p>
	 *
	 * @param rs a {@link java.sql.ResultSet} object.
	 * @param table a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param meta a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 * @throws java.sql.SQLException if any.
	 */
	public SynchroJoinMetadata(ResultSet rs, SynchroTableMetadata table,
			SynchroDatabaseMetadata meta) throws SQLException {

		// String pkCatalog = rs.getString("PKTABLE_CAT");
		// String pkSchema = rs.getString("PKTABLE_SCHEM");
		String pkTableName = rs.getString("PKTABLE_NAME").toLowerCase();
		String pkColumnName = rs.getString("PKCOLUMN_NAME").toLowerCase();
		String fkTableName = rs.getString("FKTABLE_NAME").toLowerCase();
		String fkColumnName = rs.getString("FKCOLUMN_NAME").toLowerCase();

		// Load PK metadata
		SynchroTableMetadata pkTable = meta.getLoadedTable(pkTableName);
		SynchroColumnMetadata pkColumn = null;
		if (pkTable == null) {
			if (LOG.isTraceEnabled()) {
				LOG.trace(String.format(
						"[%s] Ignore join to %s: table not found",
						table.getName(), pkTableName.toUpperCase()));
			}
			return;
		}
		pkColumn = pkTable.getColumn(pkColumnName);
		if (pkColumn == null) {
			if (LOG.isTraceEnabled()) {
				LOG.trace(String.format(
						"[%s] Ignore join to %s: column %s not found",
						table.getName(), pkTableName.toUpperCase(),
						pkColumnName.toUpperCase()));
			}
			return;
		}

		// Load FK metadata
		SynchroTableMetadata fkTable = meta.getLoadedTable(fkTableName);
		SynchroColumnMetadata fkColumn = null;
		if (fkTable != null) {
			fkColumn = fkTable.getColumn(fkColumnName);
		}
		if (fkTable == null) {
			// Ignoring the join, because referenced table not bound
			if (LOG.isTraceEnabled()) {
				LOG.trace(String.format(
						"[%s] Ignore join from %s: table not found",
						table.getName(), fkTableName.toUpperCase()));
			}
			return;
		}
		if (fkTable == null || fkColumn == null) {
			// Ignoring the join, because referenced column not bound
			if (LOG.isTraceEnabled()) {
				LOG.trace(String.format(
						"[%s] Ignore join from %s: column %s not found",
						table.getName(), fkTableName.toUpperCase(),
						fkColumnName.toUpperCase()));
			}
			return;
		}

		if (pkTable == table) {
			this.sourceTable = pkTable;
			this.sourceColumn = pkColumn;

			this.targetTable = fkTable;
			this.targetColumn = fkColumn;

			this.isChild = true;
			this.isSourceFk = false;
		} else {
			this.sourceTable = fkTable;
			this.sourceColumn = fkColumn;

			this.targetTable = pkTable;
			this.targetColumn = pkColumn;

			this.isChild = false;
			this.isSourceFk = true;
		}
		isValid = true;
	}

	/**
	 * <p>Constructor for SynchroJoinMetadata.</p>
	 *
	 * @param table a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param pkTable a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param pkColumn a {@link fr.ifremer.common.synchro.meta.SynchroColumnMetadata} object.
	 * @param fkTable a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param fkColumn a {@link fr.ifremer.common.synchro.meta.SynchroColumnMetadata} object.
	 */
	public SynchroJoinMetadata(SynchroTableMetadata table,
			SynchroTableMetadata pkTable, SynchroColumnMetadata pkColumn,
			SynchroTableMetadata fkTable, SynchroColumnMetadata fkColumn) {

		if (pkTable == table) {
			this.sourceTable = pkTable;
			this.sourceColumn = pkColumn;

			this.targetTable = fkTable;
			this.targetColumn = fkColumn;

			this.isChild = true;
			this.isSourceFk = false;
		} else {
			this.sourceTable = fkTable;
			this.sourceColumn = fkColumn;

			this.targetTable = pkTable;
			this.targetColumn = pkColumn;

			this.isChild = false;
			this.isSourceFk = true;
		}
		isValid = true;
	}

	/**
	 * <p>isChild.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isChild() {
		return isChild;
	}

	/**
	 * <p>hashCode.</p>
	 *
	 * @return a int.
	 */
	public int hashCode() {
		return sourceColumn.hashCode() + targetColumn.hashCode();
	}

	/** {@inheritDoc} */
	public boolean equals(Object obj) {
		if (obj instanceof SynchroJoinMetadata) {
			SynchroJoinMetadata otherJoin = (SynchroJoinMetadata) obj;
			return sourceColumn.equals(otherJoin.sourceColumn)
					&& targetColumn.equals(otherJoin.targetColumn);
		}
		return this.equals(obj);
	}

	/**
	 * <p>Getter for the field <code>targetTable</code>.</p>
	 *
	 * @return a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 */
	public SynchroTableMetadata getTargetTable() {
		return targetTable;
	}

	/**
	 * <p>Getter for the field <code>targetColumn</code>.</p>
	 *
	 * @return a {@link fr.ifremer.common.synchro.meta.SynchroColumnMetadata} object.
	 */
	public SynchroColumnMetadata getTargetColumn() {
		return targetColumn;
	}

	/**
	 * <p>Getter for the field <code>sourceTable</code>.</p>
	 *
	 * @return a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 */
	public SynchroTableMetadata getSourceTable() {
		return sourceTable;
	}

	/**
	 * <p>Getter for the field <code>sourceColumn</code>.</p>
	 *
	 * @return a {@link fr.ifremer.common.synchro.meta.SynchroColumnMetadata} object.
	 */
	public SynchroColumnMetadata getSourceColumn() {
		return sourceColumn;
	}

	/**
	 * <p>isValid.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isValid() {
		return isValid;
	}

	/**
	 * <p>Setter for the field <code>isValid</code>.</p>
	 *
	 * @param isValid a boolean.
	 */
	public void setIsValid(boolean isValid) {
		this.isValid = isValid;
	}

	/**
	 * <p>toString.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String toString() {
		return String.format("JoinMetadata(%s.%s = %s.%s)",
				sourceTable.getName(), sourceColumn.getName(),
				targetTable.getName(), targetColumn.getName());
	}

	/**
	 * <p>isSourceFk.</p>
	 *
	 * @return true if the source side = the FK side. false if source = PK
	 */
	public boolean isSourceFk() {
		return isSourceFk;
	}

	/**
	 * <p>isTargetPk.</p>
	 *
	 * @return true if the current table has the targeted PK. false if has the
	 *         FK
	 */
	public boolean isTargetPk() {
		return !isSourceFk;
	}

	/**
	 * <p>getPkTable.</p>
	 *
	 * @return a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 */
	public SynchroTableMetadata getPkTable() {
		return !isSourceFk ? sourceTable : targetTable;
	}

	/**
	 * <p>getPkColumn.</p>
	 *
	 * @return a {@link fr.ifremer.common.synchro.meta.SynchroColumnMetadata} object.
	 */
	public SynchroColumnMetadata getPkColumn() {
		return !isSourceFk ? sourceColumn : targetColumn;
	}

	/**
	 * <p>getFkTable.</p>
	 *
	 * @return a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 */
	public SynchroTableMetadata getFkTable() {
		return isSourceFk ? sourceTable : targetTable;
	}

	/**
	 * <p>getFkColumn.</p>
	 *
	 * @return a {@link fr.ifremer.common.synchro.meta.SynchroColumnMetadata} object.
	 */
	public SynchroColumnMetadata getFkColumn() {
		return isSourceFk ? sourceColumn : targetColumn;
	}
}
