package fr.ifremer.common.synchro.meta;

/*
 * #%L
 * Tutti :: Persistence
 * $Id: ReferentialSynchroDatabaseMetadata.java 1573 2014-02-04 16:41:40Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/trunk/tutti-persistence/src/main/java/fr/ifremer/adagio/core/service/technical/synchro/ReferentialSynchroDatabaseMetadata.java $
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import fr.ifremer.common.synchro.SynchroTechnicalException;
import fr.ifremer.common.synchro.dao.Daos;
import fr.ifremer.common.synchro.intercept.SynchroInterceptor;
import fr.ifremer.common.synchro.intercept.SynchroInterceptorUtils;
import fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.HibernateException;
import org.hibernate.cfg.Configuration;
import org.hibernate.dialect.Dialect;
import org.hibernate.exception.spi.SQLExceptionConverter;
import org.hibernate.internal.util.StringHelper;
import org.hibernate.mapping.Table;
import org.hibernate.tool.hbm2ddl.DatabaseMetadata;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import java.io.Closeable;
import java.io.Serializable;
import java.lang.reflect.Field;
import java.sql.*;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static org.nuiton.i18n.I18n.t;

/**
 * Created on 1/14/14.
 *
 * @author Tony Chemit (chemit@codelutin.com)
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 * @since 3.5
 */
public class SynchroDatabaseMetadata
		implements
			SynchroTableMetadataLoader,
			Closeable,
			Serializable {

	static final long serialVersionUID = -1L;

	/** Logger. */
	private static final Log log = LogFactory
			.getLog(SynchroDatabaseMetadata.class);

	private static final String TABLE_CATALOG_PATTERN = "TABLE_CAT";
	private static final String TABLE_TYPE_PATTERN = "TABLE_TYPE";
	private static final String TABLE_SCHEMA_PATTERN = "TABLE_SCHEM";
	private static final String REMARKS_PATTERN = "REMARKS";
	private static final String TABLE_NAME_PATTERN = "TABLE_NAME";

	/**
	 * Load the datasource schema for the given connection and dialect.
	 *
	 * @param connection
	 *            connection of the data source
	 * @param configuration
	 *            the configuration of database
	 * @return the database schema metadata
	 * @param tableNames a {@link java.util.Set} object.
	 */
	public static SynchroDatabaseMetadata loadDatabaseMetadata(
			Connection connection, SynchroDatabaseConfiguration configuration,
			Set<String> tableNames) {

		SynchroDatabaseMetadata meta = new SynchroDatabaseMetadata(connection,
				configuration);
		meta.prepare(tableNames);
		return meta;
	}

	/**
	 * Load the datasource schema for the given connection and dialect.
	 *
	 * @param connection
	 *            connection of the data source
	 * @param configuration
	 *            the configuration of database
	 * @return the database schema metadata
	 */
	public static SynchroDatabaseMetadata loadDatabaseMetadata(
			Connection connection, SynchroDatabaseConfiguration configuration) {

		SynchroDatabaseMetadata meta = new SynchroDatabaseMetadata(connection,
				configuration);
		return meta;
	}

	protected DatabaseMetadata hibernateMeta;

	protected DatabaseMetaData jdbcMeta;

	protected final Map<String, SynchroTableMetadata> tables;

	protected final SynchroDatabaseConfiguration config;

	protected final Set<String> sequences;

	protected final String[] types;

	protected SQLExceptionConverter sqlExceptionConverter;

	protected List<SynchroInterceptor> interceptors;

	protected List<Object> otherListeners;

	protected Predicate<SynchroColumnMetadata> columnFilter;

	protected boolean isClosed = false;

	/**
	 * <p>Constructor for SynchroDatabaseMetadata.</p>
	 *
	 * @param connection a {@link java.sql.Connection} object.
	 * @param config a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 */
	public SynchroDatabaseMetadata(Connection connection,
			SynchroDatabaseConfiguration config) {
		Preconditions.checkNotNull(connection);
		Preconditions.checkNotNull(config);

		this.config = config;
		this.interceptors = initInterceptors(config);
		this.otherListeners = Lists.newArrayList();

		this.sqlExceptionConverter = Daos.newSQLExceptionConverter(config
				.getDialect());

		// Build column filter
		this.columnFilter = SynchroMetadataUtils
				.newExcludeColumnPredicate(config.getColumnExcludesAsSet());
		this.tables = Maps.newLinkedHashMap();

		try {
			this.hibernateMeta = initHibernateDatabaseMetadata(connection,
					config, sqlExceptionConverter);

			this.sequences = initSequences(connection, config.getDialect());

			Field typesField = DatabaseMetadata.class.getDeclaredField("types");
			typesField.setAccessible(true);
			this.types = (String[]) typesField.get(this.hibernateMeta);

			this.jdbcMeta = connection.getMetaData();

		} catch (Exception e) {
			throw new SynchroTechnicalException(t(
					"synchro.meta.db.instanciation.error",
					config.getJdbcUrl()), e);
		}
	}

	/**
	 * <p>getTableCount.</p>
	 *
	 * @return a int.
	 */
	public int getTableCount() {
		return tables.size();
	}

	/**
	 * <p>getConfiguration.</p>
	 *
	 * @return a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 */
	public SynchroDatabaseConfiguration getConfiguration() {
		return this.config;
	}

	/**
	 * <p>getDialect.</p>
	 *
	 * @return a {@link org.hibernate.dialect.Dialect} object.
	 */
	public Dialect getDialect() {
		return this.config.getDialect();
	}

	/**
	 * <p>getInExpressionCountLimit.</p>
	 *
	 * @return a int.
	 */
	public int getInExpressionCountLimit() {
		return this.config.getDialect().getInExpressionCountLimit();
	}

	/**
	 * <p>isSequence.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @return a boolean.
	 */
	public boolean isSequence(String tableName) {
		String[] strings = StringHelper.split(".", tableName);
		return sequences.contains(StringUtils
				.lowerCase(strings[strings.length - 1]));
	}

	/**
	 * <p>getConnection.</p>
	 *
	 * @return a {@link java.sql.Connection} object.
	 * @throws java.sql.SQLException if any.
	 */
	public Connection getConnection() throws SQLException {
		checkNotClosed();
		return jdbcMeta.getConnection();
	}

	/** {@inheritDoc} */
	@Override
	public void close() {
		this.interceptors = null;
		this.otherListeners = null;
		this.sqlExceptionConverter = null;
		this.jdbcMeta = null;
		this.hibernateMeta = null;

		// Mark as closed
		this.isClosed = true;
	}

	/**
	 * <p>isClosed.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isClosed() {
		return this.isClosed;
	}

	/**
	 * <p>prepare.</p>
	 *
	 * @param tableNames a {@link java.util.Set} object.
	 */
	public void prepare(Set<String> tableNames) {
		checkNotClosed();
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(tableNames),
				"'tableNames' must be set and not empty");

		// Getting tables names to process
		boolean enableFilter = false;
		for (String tablePattern : tableNames) {
			enableFilter = tablePattern.contains("%");
			if (enableFilter)
				break;
		}

		Set<String> filteredTableNames;
		if (enableFilter) {
			filteredTableNames = getTableNames(tableNames, null);
		} else {
			filteredTableNames = tableNames;
		}

		for (String tableName : filteredTableNames) {

			if (log.isDebugEnabled()) {
				log.debug("Load metas of table: " + tableName);
			}
			getTableBeforeBuild(tableName);
		}

		for (SynchroTableMetadata table : tables.values()) {

			// Init joins metadata (must be call AFTER getTable())
			if (config.isFullMetadataEnable()) {

				if (log.isDebugEnabled()) {
					log.debug("Load joins of table: " + table.getName());
				}
				table.initJoins(this);
			}

			// Build the table
			table.build();
		}
	}

	/**
	 * Unload all loaded tables. Useful for Unit tests
	 */
	public void unloadTables() {
		tables.clear();
	}

	/** {@inheritDoc} */
	public SynchroTableMetadata getTable(String name) throws HibernateException {

		// Try to get from loaded table
		SynchroTableMetadata result = getLoadedTable(name);
		if (result != null) {
			return result;
		}

		// Get table, as not loaded
		result = getTableBeforeBuild(name).build();

		return result;
	}

	/**
	 * <p>getLoadedTable.</p>
	 *
	 * @param name a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @throws org.hibernate.HibernateException if any.
	 */
	public SynchroTableMetadata getLoadedTable(String name)
			throws HibernateException {
		return getLoadedTable(name, config.getJdbcSchema(),
				config.getJdbcCatalog());
	}

	/**
	 * <p>getLoadedTable.</p>
	 *
	 * @param name a {@link java.lang.String} object.
	 * @param schema a {@link java.lang.String} object.
	 * @param catalog a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @throws org.hibernate.HibernateException if any.
	 */
	public SynchroTableMetadata getLoadedTable(String name, String schema,
			String catalog) throws HibernateException {
		String key = Table.qualify(catalog, schema, name).toLowerCase();
		return tables.get(key);
	}

	/**
	 * Load tables names from database schema, using the given table filter.<br>
	 * This method call {@link #getTableNames(Set, Predicate)}
	 * with the table pattern "%".
	 *
	 * @param tableFilter
	 *            A filter predicate, to filter tables to retrieve. If null:
	 *            process all tables found.
	 * @return All tables names found in database, filtered using the given
	 *         tableFilter
	 * @see #getTableNames(Set,Predicate)
	 */
	public Set<String> getTableNames(Predicate<String> tableFilter) {
		return getTableNames(Sets.newHashSet("%"), tableFilter);
	}

	/**
	 * Load tables names from database schema, using the given table patterns
	 * list, and a optional filter. This use the JDBC metadata API.<br>
	 * This will include Tables and View objects. Synonyms ar includes only if
	 * enable in connection properties
	 *
	 * @param tablePatterns
	 *            A list of table pattern. Use the pattern '%' to get all
	 *            tables.
	 * @param tableFilter
	 *            Optional. A filter predicate, to filter tables to retrieve. If
	 *            null: process tables found from pattern will be return.
	 * @return All tables names found in database
	 * @see org.hibernate.cfg.AvailableSettings#ENABLE_SYNONYMS
	 * @see java.sql.DatabaseMetaData#getTables(String,String,String,String[])
	 */
	public Set<String> getTableNames(Set<String> tablePatterns,
			Predicate<String> tableFilter) {
		checkNotClosed();
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(tablePatterns));

		Set<String> tablenames = Sets.newLinkedHashSet();

		String jdbcSchema = config.getJdbcSchema();
		String jdbcCatalog = config.getJdbcCatalog();

		String[] types = null; // available types are: "TABLE", "VIEW",
								// "SYSTEM TABLE", "GLOBAL TEMPORARY",
								// "LOCAL TEMPORARY", "ALIAS", "SYNONYM"
		if (config != null && config.isSynonymsEnable()) {
			types = new String[]{"TABLE", "VIEW", "SYNONYM"};
		} else {
			types = new String[]{"TABLE", "VIEW"};
		}

		ResultSet rs = null;
		try {
			if (log.isDebugEnabled()) {
				log.debug("Getting table names, using filter");
			}

			for (String tablePattern : tablePatterns) {
				// first pass on the main schema
				rs = jdbcMeta.getTables(jdbcCatalog, jdbcSchema, tablePattern,
						types);
				while (rs.next()) {
					String tableName = rs.getString(TABLE_NAME_PATTERN); //$NON-NLS-1$
					if (!hibernateMeta.isSequence(tableName)
							&& (tableFilter == null || tableFilter
									.apply(tableName))) {
						if (log.isTraceEnabled()) {
							log.trace(" " + TABLE_CATALOG_PATTERN + "="
									+ rs.getString(TABLE_CATALOG_PATTERN) + " "
									+ TABLE_SCHEMA_PATTERN + "="
									+ rs.getString(TABLE_SCHEMA_PATTERN) + " "
									+ TABLE_NAME_PATTERN + "="
									+ rs.getString(TABLE_NAME_PATTERN) + " "
									+ TABLE_TYPE_PATTERN + "="
									+ rs.getString(TABLE_TYPE_PATTERN) + " "
									+ REMARKS_PATTERN + "="
									+ rs.getString(REMARKS_PATTERN));
						}
						tablenames.add(tableName);
					}
				}
			}
		} catch (SQLException e) {
			throw sqlExceptionConverter.convert(e,
					"Retrieving database table names", "n/a");
		} finally {
			Daos.closeSilently(rs);
		}

		return tablenames;
	}

	/**
	 * Return all root tables (top level tables).<br>
	 * Return only tables previously loaded using methods getTable() or
	 * loadDatabaseMetadata()<br> Will keep order given by method
	 * <code>context.getTableNames()</code>
	 *
	 * @return All loaded tables metadata
	 */
	public Set<String> getLoadedRootTableNames() {
		Set<String> tablenames = Sets.newLinkedHashSet();
		for (SynchroTableMetadata table : tables.values()) {
			if (table.isRoot()) {
				tablenames.add(table.getName());
			}
		}

		return tablenames;
	}

	/**
	 * Return all tables (already loaded).<br>
	 * Return only tables previously loaded using methods getTable() or
	 * loadDatabaseMetadata()
	 *
	 * @return All loaded tables names
	 */
	public Set<String> getLoadedTableNames() {
		Set<String> tablenames = Sets.newLinkedHashSet();
		for (SynchroTableMetadata table : tables.values()) {
			tablenames.add(table.getName());
		}

		return tablenames;
	}

	/**
	 * <p>getExportedKeys.</p>
	 *
	 * @see java.sql.DatabaseMetaData#getExportedKeys(String,String,String)
	 * @param catalog a {@link java.lang.String} object.
	 * @param schema a {@link java.lang.String} object.
	 * @param table a {@link java.lang.String} object.
	 * @return a {@link java.sql.ResultSet} object.
	 * @throws java.sql.SQLException if any.
	 */
	public ResultSet getExportedKeys(String catalog, String schema, String table)
			throws SQLException {
		checkNotClosed();
		return jdbcMeta.getExportedKeys(catalog, schema, table);
	}

	/**
	 * <p>getImportedKeys.</p>
	 *
	 * @see java.sql.DatabaseMetaData#getImportedKeys(String,String,String)
	 * @param catalog a {@link java.lang.String} object.
	 * @param schema a {@link java.lang.String} object.
	 * @param table a {@link java.lang.String} object.
	 * @return a {@link java.sql.ResultSet} object.
	 * @throws java.sql.SQLException if any.
	 */
	public ResultSet getImportedKeys(String catalog, String schema, String table)
			throws SQLException {
		checkNotClosed();
		return jdbcMeta.getImportedKeys(catalog, schema, table);
	}

	/**
	 * <p>getPrimaryKeys.</p>
	 *
	 * @see java.sql.DatabaseMetaData#getPrimaryKeys(String,String,String)
	 * @param catalog a {@link java.lang.String} object.
	 * @param schema a {@link java.lang.String} object.
	 * @param table a {@link java.lang.String} object.
	 * @return a {@link java.sql.ResultSet} object.
	 * @throws java.sql.SQLException if any.
	 */
	public ResultSet getPrimaryKeys(String catalog, String schema, String table)
			throws SQLException {
		checkNotClosed();
		return jdbcMeta.getPrimaryKeys(catalog, schema, table);
	}

	/**
	 * <p>getIndexInfo.</p>
	 *
	 * @see java.sql.DatabaseMetaData#getIndexInfo(String,String,String,boolean,boolean)
	 * @param catalog a {@link java.lang.String} object.
	 * @param schema a {@link java.lang.String} object.
	 * @param table a {@link java.lang.String} object.
	 * @param unique a boolean.
	 * @param approximate a boolean.
	 * @return a {@link java.sql.ResultSet} object.
	 * @throws java.sql.SQLException if any.
	 */
	public ResultSet getIndexInfo(String catalog, String schema, String table,
			boolean unique, boolean approximate) throws SQLException {
		checkNotClosed();
		return jdbcMeta.getIndexInfo(catalog, schema, table, unique,
				approximate);
	}

	/**
	 * <p>Getter for the field <code>sequences</code>.</p>
	 *
	 * @return a {@link java.util.Set} object.
	 */
	public Set<String> getSequences() {
		return sequences;
	}

	/**
	 * <p>register.</p>
	 *
	 * @param listener a {@link java.lang.Object} object.
	 */
	public void register(Object listener) {
		checkNotClosed();
		if (listener instanceof SynchroInterceptor) {
			// Add to interceptor list
			interceptors.add((SynchroInterceptor) listener);
		} else {
			otherListeners.add(listener);
		}
	}

	/**
	 * <p>unregister.</p>
	 *
	 * @param listener a {@link java.lang.Object} object.
	 */
	public void unregister(Object listener) {
		checkNotClosed();
		if (listener instanceof SynchroInterceptor) {
			// Remove to interceptor list
			interceptors.remove((SynchroInterceptor) listener);
		}

		else {
			otherListeners.remove(listener);
		}
	}

	/* -- Internal methods -- */

	/**
	 * <p>initInterceptors.</p>
	 *
	 * @param config a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 * @return a {@link java.util.List} object.
	 */
	protected List<SynchroInterceptor> initInterceptors(
			SynchroDatabaseConfiguration config) {
		// Load interceptors
		List<SynchroInterceptor> interceptors = SynchroInterceptorUtils
				.load(SynchroInterceptor.class);

		// Filter using config
		interceptors = SynchroInterceptorUtils.filter(interceptors, config);

		return interceptors;
	}

	/**
	 * <p>getTableBeforeBuild.</p>
	 *
	 * @param name a {@link java.lang.String} object.
	 * @return a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @throws org.hibernate.HibernateException if any.
	 */
	protected SynchroTableMetadata getTableBeforeBuild(String name)
			throws HibernateException {
		return getTableBeforeBuild(name, config.getJdbcSchema(),
				config.getJdbcCatalog(), false);
	}

	/**
	 * <p>getTableBeforeBuild.</p>
	 *
	 * @param name a {@link java.lang.String} object.
	 * @param schema a {@link java.lang.String} object.
	 * @param catalog a {@link java.lang.String} object.
	 * @param isQuoted a boolean.
	 * @return a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @throws org.hibernate.HibernateException if any.
	 */
	protected SynchroTableMetadata getTableBeforeBuild(String name,
			String schema, String catalog, boolean isQuoted)
			throws HibernateException {
		checkNotClosed();

		String key = Table.qualify(catalog, schema, name).toLowerCase();
		SynchroTableMetadata synchroTableMetadata = tables.get(key);
		if (synchroTableMetadata == null) {

			TableMetadata tableMetadata = hibernateMeta.getTableMetadata(
					name.toLowerCase(), schema, catalog, isQuoted);
			Preconditions.checkNotNull(tableMetadata, String.format(
					"Could not find db table '%s' (schema=%s, catalog=%s)",
					name, schema, catalog));

			boolean enableFullMetadata = getConfiguration()
					.isFullMetadataEnable();

			List<Object> listeners = enableFullMetadata
					? getListeners(tableMetadata)
					: null;

			synchroTableMetadata = new SynchroTableMetadata(this,
					tableMetadata, getDialect(), listeners, name, columnFilter,
					enableFullMetadata);

			Preconditions.checkNotNull(synchroTableMetadata,
					"Could not load metadata for table: " + name);

			tables.put(key, synchroTableMetadata);
		}
		return synchroTableMetadata;
	}

	/**
	 * <p>initHibernateDatabaseMetadata.</p>
	 *
	 * @param connection a {@link java.sql.Connection} object.
	 * @param configuration a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 * @param sqlExceptionConverter a {@link org.hibernate.exception.spi.SQLExceptionConverter} object.
	 * @return a {@link org.hibernate.tool.hbm2ddl.DatabaseMetadata} object.
	 * @throws java.sql.SQLException if any.
	 * @throws java.lang.NoSuchFieldException if any.
	 * @throws java.lang.SecurityException if any.
	 * @throws java.lang.IllegalArgumentException if any.
	 * @throws java.lang.IllegalAccessException if any.
	 */
	protected DatabaseMetadata initHibernateDatabaseMetadata(
			Connection connection, SynchroDatabaseConfiguration configuration,
			SQLExceptionConverter sqlExceptionConverter) throws SQLException,
			NoSuchFieldException, SecurityException, IllegalArgumentException,
			IllegalAccessException {

		Configuration hibernateConfiguration = configuration
				.createHibernateConfiguration();
		DatabaseMetadata hibernateMeta = new DatabaseMetadata(connection,
				configuration.getDialect(), hibernateConfiguration, true);

		// Apply the sql exception converter
		Field sqlExceptionConverterField = DatabaseMetadata.class
				.getDeclaredField("sqlExceptionConverter");
		sqlExceptionConverterField.setAccessible(true);
		sqlExceptionConverterField.set(hibernateMeta, sqlExceptionConverter);

		return hibernateMeta;
	}

	/**
	 * <p>initSequences.</p>
	 *
	 * @param connection a {@link java.sql.Connection} object.
	 * @param dialect a {@link org.hibernate.dialect.Dialect} object.
	 * @return a {@link java.util.Set} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected Set<String> initSequences(Connection connection, Dialect dialect)
			throws SQLException {
		Set<String> sequences = Sets.newHashSet();
		if (dialect.supportsSequences()) {
			String sql = dialect.getQuerySequencesString();
			if (sql != null) {

				Statement statement = null;
				ResultSet rs = null;
				try {
					statement = connection.createStatement();
					rs = statement.executeQuery(sql);

					while (rs.next()) {
						sequences.add(StringUtils.lowerCase(rs.getString(1))
								.trim());
					}
				} finally {
					rs.close();
					statement.close();
				}

			}
		}
		return sequences;
	}

	/**
	 * <p>getListeners.</p>
	 *
	 * @param table a {@link org.hibernate.tool.hbm2ddl.TableMetadata} object.
	 * @return a {@link java.util.List} object.
	 */
	protected List<Object> getListeners(final TableMetadata table) {
		Collection<SynchroInterceptor> filteredInterceptors = SynchroInterceptorUtils
				.filter(interceptors, this, table);

		List<Object> listeners = Lists.newArrayList();
		if (CollectionUtils.isNotEmpty(filteredInterceptors)) {
			listeners.addAll(filteredInterceptors);
		}
		if (CollectionUtils.isNotEmpty(otherListeners)) {
			listeners.addAll(otherListeners);
		}

		return listeners;
	}

	/**
	 * <p>checkNotClosed.</p>
	 */
	protected void checkNotClosed() {
		Preconditions.checkState(!isClosed, "This object is closed");
	}
}
