package fr.ifremer.common.synchro.meta;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Preconditions;
import org.hibernate.dialect.Dialect;
import org.hibernate.dialect.HSQLDialect;
import org.hibernate.tool.hbm2ddl.ColumnMetadata;

import java.io.Serializable;
import java.sql.Types;

/**
 * Overrides of the {@link org.hibernate.tool.hbm2ddl.ColumnMetadata} with some improvements:
 * <ul>
 * <li>Obtains owner table name {@link #getTableName()};</li>
 * <li>Obtains linked column (for FK only) {@link #getParentJoin()} as a join
 * metadata: used to synchronize data in DataSynchroService (from adagio synchro module);</li>
 * </ul>
 *
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 * @since 3.5.2
 */
public class SynchroColumnMetadata implements Serializable {

	static final long serialVersionUID = -1L;

	protected final ColumnMetadata delegate;

	protected final String tableName;

	protected final int columnSize;

	protected SynchroJoinMetadata parentJoin = null;

	/**
	 * <p>Constructor for SynchroColumnMetadata.</p>
	 *
	 * @param tableName a {@link java.lang.String} object.
	 * @param delegate a {@link org.hibernate.tool.hbm2ddl.ColumnMetadata} object.
	 * @param dialect a {@link org.hibernate.dialect.Dialect} object.
	 */
	public SynchroColumnMetadata(String tableName, ColumnMetadata delegate,
			Dialect dialect) {

		Preconditions.checkNotNull(delegate);

		this.delegate = delegate;
		this.tableName = tableName;
		this.columnSize = initColumnSize(dialect, delegate);
	}

	/**
	 * <p>Constructor for SynchroColumnMetadata.</p>
	 *
	 * @param beanToCopy a {@link fr.ifremer.common.synchro.meta.SynchroColumnMetadata} object.
	 */
	public SynchroColumnMetadata(SynchroColumnMetadata beanToCopy) {

		Preconditions.checkNotNull(beanToCopy);

		this.delegate = beanToCopy.delegate;
		this.tableName = beanToCopy.tableName;
		this.parentJoin = beanToCopy.parentJoin;
		this.columnSize = beanToCopy.columnSize;
	}

	/**
	 * Obtains owner table name
	 *
	 * @return The owner table name
	 */
	public String getTableName() {
		return tableName;
	}

	/**
	 * Obtains linked column (for FK only)<br>
	 * This is used to synchronize data in DataSynchroService (from adagio synchro module);
	 *
	 * @return the parent join metadata
	 * @see SynchroJoinMetadata
	 */
	public SynchroJoinMetadata getParentJoin() {
		return parentJoin;
	}

	/**
	 * <p>Setter for the field <code>parentJoin</code>.</p>
	 *
	 * @param parentJoin a {@link fr.ifremer.common.synchro.meta.SynchroJoinMetadata} object.
	 */
	public void setParentJoin(SynchroJoinMetadata parentJoin) {
		this.parentJoin = parentJoin;
	}

	/**
	 * <p>hashCode.</p>
	 *
	 * @see ColumnMetadata#hashCode()
	 * @return a int.
	 */
	public int hashCode() {
		return delegate.hashCode();
	}

	/**
	 * <p>getName.</p>
	 *
	 * @see ColumnMetadata#getName()
	 * @return a {@link java.lang.String} object.
	 */
	public String getName() {
		return delegate.getName();
	}

	/**
	 * <p>getTypeName.</p>
	 *
	 * @see ColumnMetadata#getTypeName()
	 * @return a {@link java.lang.String} object.
	 */
	public String getTypeName() {
		return delegate.getTypeName();
	}

	/**
	 * <p>Getter for the field <code>columnSize</code>.</p>
	 *
	 * @see ColumnMetadata#getColumnSize()
	 * @return a int.
	 */
	public int getColumnSize() {
		return columnSize;
	}

	/**
	 * <p>getDecimalDigits.</p>
	 *
	 * @see ColumnMetadata#getDecimalDigits()
	 * @return a int.
	 */
	public int getDecimalDigits() {
		return delegate.getDecimalDigits();
	}

	/**
	 * <p>getNullable.</p>
	 *
	 * @see ColumnMetadata#getTypeCode()
	 * @return a {@link java.lang.String} object.
	 */
	public String getNullable() {
		return delegate.getNullable();
	}

	/**
	 * <p>isNullable.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isNullable() {
		return "YES".equalsIgnoreCase(getNullable());
	}

	/**
	 * <p>toString.</p>
	 *
	 * @see ColumnMetadata#toString()
	 * @return a {@link java.lang.String} object.
	 */
	public String toString() {
		return delegate.toString();
	}

	/**
	 * <p>getTypeCode.</p>
	 *
	 * @see ColumnMetadata#getTypeCode()
	 * @return a int.
	 */
	public int getTypeCode() {
		return delegate.getTypeCode();
	}

	/** {@inheritDoc} */
	public boolean equals(Object obj) {
		if (obj instanceof ColumnMetadata) {
			return delegate.equals(obj);
		}
		return super.equals(obj);
	}

	/**
	 * <p>getSqlTypeName.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getSqlTypeName() {
		int size = getColumnSize();
		String typeName = getTypeName();

		if (size == 0) {
			return typeName;
		}

		StringBuilder sqlTypeName = new StringBuilder();
		sqlTypeName.append(typeName);
		sqlTypeName.append("(").append(size);

		int decimals = getDecimalDigits();
		if (decimals > 0) {
			sqlTypeName.append(",").append(decimals);
		}

		sqlTypeName.append(")");
		return sqlTypeName.toString();
	}

	/* -- Internal methods -- */

	/**
	 * Column size is computed once, because of some special case (for HsqlDB)
	 * 
	 * @return
	 */
	private int initColumnSize(Dialect dialect, ColumnMetadata delegate) {
		if (dialect instanceof HSQLDialect) {
			// Ignore some column size, on HSQLDB
			if (delegate.getTypeCode() == Types.INTEGER) {
				return 0;
			}
			if (delegate.getTypeCode() == Types.DOUBLE) {
				// Default value is 64
				return 0;
			}
			if (delegate.getTypeCode() == Types.BIGINT) {
				return 0;
			}
			if (delegate.getTypeCode() == Types.SMALLINT) {
				return 0;
			}
		}

		return delegate.getColumnSize();
	}
}
