package fr.ifremer.common.synchro.intercept;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.util.Collection;
import java.util.List;
import java.util.ServiceLoader;

import org.hibernate.tool.hbm2ddl.TableMetadata;

import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;

import fr.ifremer.common.synchro.SynchroTechnicalException;
import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration;

/**
 * <p>SynchroInterceptorUtils class.</p>
 *
 */
public class SynchroInterceptorUtils {

	private SynchroInterceptorUtils() {
		// helper class
	}

	/**
	 * <p>load.</p>
	 *
	 * @param clazz a {@link java.lang.Class} object.
	 * @param <T> a T object.
	 * @param <U> a U object.
	 * @return a {@link java.util.List} object.
	 */
	public static <T, U extends SynchroInterceptor> List<U> load(Class<U> clazz) {
		List<U> result = Lists.newArrayList();

		ServiceLoader<U> loader = ServiceLoader.load(clazz);
		for (U interceptor : loader) {
			result.add(interceptor);
		}
		return result;
	}

	/**
	 * <p>filter.</p>
	 *
	 * @param interceptors a {@link java.util.List} object.
	 * @param config a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 * @param <T> a T object.
	 * @param <U> a U object.
	 * @return a {@link java.util.List} object.
	 */
	public static <T, U extends SynchroInterceptor> List<U> filter(
			List<U> interceptors, final SynchroDatabaseConfiguration config) {
		// Retrieve overrides to apply on this table
		Collection<U> result = Collections2.filter(interceptors,
				new Predicate<SynchroInterceptor>() {
					public boolean apply(SynchroInterceptor interceptor) {
						return interceptor.apply(config);
					}
				});
		return ImmutableList.copyOf(result);
	}

	/**
	 * <p>filter.</p>
	 *
	 * @param interceptors a {@link java.util.List} object.
	 * @param meta a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 * @param table a {@link org.hibernate.tool.hbm2ddl.TableMetadata} object.
	 * @param <T> a T object.
	 * @param <U> a U object.
	 * @return a {@link java.util.List} object.
	 */
	public static <T, U extends SynchroInterceptor> List<U> filter(
			List<U> interceptors, final SynchroDatabaseMetadata meta,
			final TableMetadata table) {
		// Retrieve overrides to apply on this table
		Collection<U> result = Collections2.filter(interceptors,
				new Predicate<SynchroInterceptor>() {
					public boolean apply(SynchroInterceptor interceptor) {
						return interceptor.apply(meta, table);
					}
				});
		return ImmutableList.copyOf(result);
	}

	/**
	 * <p>chain.</p>
	 *
	 * @param interceptors a {@link java.util.List} object.
	 * @param meta a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 * @param table a {@link org.hibernate.tool.hbm2ddl.TableMetadata} object.
	 * @param chainClazz a {@link java.lang.Class} object.
	 * @param <T> a T object.
	 * @param <U> a U object.
	 * @return a U object.
	 */
	@SuppressWarnings("unchecked")
	public static <T, U extends SynchroInterceptor> U chain(
			List<U> interceptors, final SynchroDatabaseMetadata meta,
			final TableMetadata table,
			Class<? extends SynchroInterceptor> chainClazz) {
		Collection<U> filteredInterceptors = filter(interceptors, meta, table);

		try {
			return (U) createChain(filteredInterceptors, chainClazz);
		} catch (InstantiationException e) {
			throw new SynchroTechnicalException(
					"Could not instantiate interceptor : " + e.getMessage(), e);
		} catch (IllegalAccessException e) {
			throw new SynchroTechnicalException(
					"Could not instantiate interceptor : " + e.getMessage(), e);
		}
	}

	/**
	 * <p>chain.</p>
	 *
	 * @param interceptors a {@link java.util.Collection} object.
	 * @param chainClazz a {@link java.lang.Class} object.
	 * @param <T> a T object.
	 * @param <U> a U object.
	 * @param <V> a V object.
	 * @return a U object.
	 */
	@SuppressWarnings("unchecked")
	public static <T, U extends SynchroInterceptor, V extends SynchroInterceptor> U chain(
			Collection<U> interceptors, Class<V> chainClazz) {
		try {
			return (U) createChain(interceptors, chainClazz);
		} catch (InstantiationException e) {
			throw new SynchroTechnicalException(
					"Could not instantiate interceptor : " + e.getMessage(), e);
		} catch (IllegalAccessException e) {
			throw new SynchroTechnicalException(
					"Could not instantiate interceptor : " + e.getMessage(), e);
		}
	}

	/**
	 * <p>createChain.</p>
	 *
	 * @param interceptors a {@link java.util.Collection} object.
	 * @param chainClazz a {@link java.lang.Class} object.
	 * @param <T> a T object.
	 * @param <U> a U object.
	 * @return a {@link fr.ifremer.common.synchro.intercept.SynchroInterceptor} object.
	 * @throws java.lang.InstantiationException if any.
	 * @throws java.lang.IllegalAccessException if any.
	 */
	@SuppressWarnings("unchecked")
	protected static <T, U extends SynchroInterceptor> SynchroInterceptor createChain(
			Collection<U> interceptors,
			Class<? extends SynchroInterceptor> chainClazz)
			throws InstantiationException, IllegalAccessException {
		SynchroInterceptor result = null;
		SynchroInterceptor previous = null;
		for (U interceptor : interceptors) {
			SynchroInterceptor newChain = null;
			if (interceptor instanceof SynchroInterceptor) {
				newChain = (SynchroInterceptor) interceptor;
			} else {
				newChain = encapsulate(interceptor, chainClazz);
			}
			if (result == null) {
				result = newChain;
			} else if (previous != null) {
				previous.setNext((U) newChain);
			}
			previous = newChain;
		}
		return result;
	}

	/**
	 * <p>encapsulate.</p>
	 *
	 * @param delegate a U object.
	 * @param chainClass a {@link java.lang.Class} object.
	 * @param <T> a T object.
	 * @param <U> a U object.
	 * @return a {@link fr.ifremer.common.synchro.intercept.SynchroInterceptor} object.
	 * @throws java.lang.InstantiationException if any.
	 * @throws java.lang.IllegalAccessException if any.
	 */
	protected static <T, U extends SynchroInterceptor> SynchroInterceptor encapsulate(
			U delegate, Class<? extends SynchroInterceptor> chainClass)
			throws InstantiationException, IllegalAccessException {
		SynchroInterceptor chain = (SynchroInterceptor) chainClass
				.newInstance();
		chain.setNext(delegate);
		return chain;
	}
}
