package fr.ifremer.common.synchro.intercept;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import fr.ifremer.common.synchro.SynchroTechnicalException;
import fr.ifremer.common.synchro.dao.SynchroTableDao;
import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata;
import fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration;
import org.hibernate.tool.hbm2ddl.ColumnMetadata;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import java.io.IOException;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * <p>SynchroInterceptorBase class.</p>
 *
 */
public class SynchroInterceptorBase implements SynchroInterceptor {

	private SynchroInterceptor next;

	private boolean enableOnRead = false;

	private boolean enableOnWrite = false;

	private SynchroDatabaseConfiguration config;

	/**
	 * <p>Constructor for SynchroInterceptorBase.</p>
	 */
	public SynchroInterceptorBase() {
		this.next = null;
	}

	/** {@inheritDoc} */
	@Override
	public boolean apply(SynchroDatabaseConfiguration config) {
		this.config = config;
		return true;
	}

	/**
	 * <p>Constructor for SynchroInterceptorBase.</p>
	 *
	 * @param next a {@link fr.ifremer.common.synchro.intercept.SynchroInterceptor} object.
	 */
	public SynchroInterceptorBase(SynchroInterceptor next) {
		this.next = next;
	}

	/** {@inheritDoc} */
	@Override
	public SynchroInterceptorBase clone() {

		try {
			SynchroInterceptorBase newInstance = this.getClass().newInstance();
			newInstance.config = this.config;
			newInstance.enableOnRead = this.enableOnRead;
			newInstance.enableOnWrite = this.enableOnWrite;
			return newInstance;
		} catch (InstantiationException e) {
			throw new SynchroTechnicalException(
					String.format(
							"Could not clone the interceptor class %s. Make sure method clone() has been implemented.",
							this.getClass().getName()));
		} catch (IllegalAccessException e) {
			throw new SynchroTechnicalException(
					String.format(
							"Could not clone the interceptor class %s. Make sure method clone() has been implemented.",
							this.getClass().getName()));
		}
	}

	/** {@inheritDoc} */
	public boolean apply(SynchroDatabaseMetadata meta, TableMetadata table) {
		return false;
	}

	/** {@inheritDoc} */
	@Override
	public void setNext(SynchroInterceptor next) {
		this.next = next;
	}

	/**
	 * <p>Getter for the field <code>next</code>.</p>
	 *
	 * @return a {@link fr.ifremer.common.synchro.intercept.SynchroInterceptor} object.
	 */
	public SynchroInterceptor getNext() {
		return this.next;
	}

	/** {@inheritDoc} */
	@Override
	public final void close() throws IOException {
		doClose();

		if (next != null) {
			next.close();
		}
	}

	/** {@inheritDoc} */
	@Override
	public final void onRead(Object[] data, SynchroTableDao sourceDao,
			SynchroTableDao targetDao) throws SQLException {
		doOnRead(data, sourceDao, targetDao);

		if (next != null) {
			next.onRead(data, sourceDao, targetDao);
		}
	}

	/** {@inheritDoc} */
	@Override
	public final void onWrite(Object[] data, List<Object> pk,
			SynchroTableDao sourceDao, SynchroTableDao targetDao,
			SynchroOperationRepository buffer, boolean insert)
			throws SQLException {
		doOnWrite(data, pk, sourceDao, targetDao, buffer, insert);

		if (next != null) {
			next.onWrite(data, pk, sourceDao, targetDao, buffer, insert);
		}
	}

	/** {@inheritDoc} */
	@Override
	public final void onDelete(List<Object> pk, SynchroTableDao sourceDao,
			SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {
		doOnDelete(pk, sourceDao, targetDao, buffer);

		if (next != null) {
			next.onDelete(pk, sourceDao, targetDao, buffer);
		}
	}

	/** {@inheritDoc} */
	@Override
	public final void onDetach(List<Object> pk, SynchroTableDao sourceDao,
			SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {
		doOnDetach(pk, sourceDao, targetDao, buffer);

		if (next != null) {
			next.onDetach(pk, sourceDao, targetDao, buffer);
		}
	}

	/** {@inheritDoc} */
	@Override
	public boolean enableOnRead() {
		return enableOnRead;
	}

	/** {@inheritDoc} */
	@Override
	public boolean enableOnWrite() {
		return enableOnWrite;
	}

	/**
	 * <p>Setter for the field <code>enableOnRead</code>.</p>
	 *
	 * @param enableOnRead a boolean.
	 */
	public void setEnableOnRead(boolean enableOnRead) {
		this.enableOnRead = enableOnRead;
	}

	/**
	 * <p>Setter for the field <code>enableOnWrite</code>.</p>
	 *
	 * @param enableOnWrite a boolean.
	 */
	public void setEnableOnWrite(boolean enableOnWrite) {
		this.enableOnWrite = enableOnWrite;
	}

	/**
	 * Return the database configuration (without cast).<br>
	 * Subclasses could define another getter method, with cast
	 *
	 * @return a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 */
	protected SynchroDatabaseConfiguration getDefaultDatabaseConfiguration() {
		return config;
	}

	/* -- protected method -- */

	/**
	 * <p>doClose.</p>
	 *
	 * @throws java.io.IOException if any.
	 */
	protected void doClose() throws IOException {
	}

	/**
	 * <p>doOnRead.</p>
	 *
	 * @param data an array of {@link java.lang.Object} objects.
	 * @param sourceDao a {@link fr.ifremer.common.synchro.dao.SynchroTableDao} object.
	 * @param targetDao a {@link fr.ifremer.common.synchro.dao.SynchroTableDao} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected void doOnRead(Object[] data, SynchroTableDao sourceDao,
			SynchroTableDao targetDao) throws SQLException {
	}

	/**
	 * <p>doOnWrite.</p>
	 *
	 * @param data an array of {@link java.lang.Object} objects.
	 * @param pk a {@link java.util.List} object.
	 * @param sourceDao a {@link fr.ifremer.common.synchro.dao.SynchroTableDao} object.
	 * @param targetDao a {@link fr.ifremer.common.synchro.dao.SynchroTableDao} object.
	 * @param buffer a {@link fr.ifremer.common.synchro.intercept.SynchroOperationRepository} object.
	 * @param insert a boolean.
	 * @throws java.sql.SQLException if any.
	 */
	protected void doOnWrite(Object[] data, List<Object> pk,
			SynchroTableDao sourceDao, SynchroTableDao targetDao,
			SynchroOperationRepository buffer, boolean insert)
			throws SQLException {
	}

	/**
	 * <p>doOnDelete.</p>
	 *
	 * @param pk a {@link java.util.List} object.
	 * @param sourceDao a {@link fr.ifremer.common.synchro.dao.SynchroTableDao} object.
	 * @param targetDao a {@link fr.ifremer.common.synchro.dao.SynchroTableDao} object.
	 * @param buffer a {@link fr.ifremer.common.synchro.intercept.SynchroOperationRepository} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected void doOnDelete(List<Object> pk, SynchroTableDao sourceDao,
			SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {
	}

	/**
	 * <p>doOnDetach.</p>
	 *
	 * @param pk a {@link java.util.List} object.
	 * @param sourceDao a {@link fr.ifremer.common.synchro.dao.SynchroTableDao} object.
	 * @param targetDao a {@link fr.ifremer.common.synchro.dao.SynchroTableDao} object.
	 * @param buffer a {@link fr.ifremer.common.synchro.intercept.SynchroOperationRepository} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected void doOnDetach(List<Object> pk, SynchroTableDao sourceDao,
			SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {
	}

	/**
	 * <p>hasColumns.</p>
	 *
	 * @param table a {@link org.hibernate.tool.hbm2ddl.TableMetadata} object.
	 * @param columnNames a {@link java.lang.String} object.
	 * @return a boolean.
	 */
	protected boolean hasColumns(TableMetadata table, String... columnNames) {

		Map<String, ColumnMetadata> delegateColumns = SynchroTableMetadata
				.getColumns(table);
		for (String columnName : columnNames) {
			if (!delegateColumns.containsKey(columnName.toLowerCase())) {
				return false;
			}
		}
		return true;
	}

	/**
	 * <p>hasColumns.</p>
	 *
	 * @param table a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param columnNames a {@link java.lang.String} object.
	 * @return a boolean.
	 */
	protected boolean hasColumns(SynchroTableMetadata table,
			String... columnNames) {
		Set<String> delegateColumns = table.getColumnNames();
		for (String columnName : columnNames) {
			if (!delegateColumns.contains(columnName.toLowerCase())) {
				return false;
			}
		}
		return true;
	}

}
