package fr.ifremer.common.synchro.dao;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.collect.Multimap;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata;
import fr.ifremer.common.synchro.service.SynchroTableOperation;
import org.hibernate.dialect.Dialect;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * <p>SynchroTableDao interface.</p>
 *
 */
public interface SynchroTableDao extends SynchroBaseDao {

	/**
	 * <p>getTable.</p>
	 *
	 * @return a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 */
	SynchroTableMetadata getTable();

	/**
	 * <p>setCurrentOperation.</p>
	 *
	 * @param pendingChangesBuffer a {@link fr.ifremer.common.synchro.service.SynchroTableOperation} object.
	 */
	void setCurrentOperation(SynchroTableOperation pendingChangesBuffer);

	/**
	 * <p>getCurrentOperation.</p>
	 *
	 * @return a {@link fr.ifremer.common.synchro.service.SynchroTableOperation} object.
	 */
	SynchroTableOperation getCurrentOperation();

	/**
	 * Obtains the max update date found in table's rows
	 *
	 * @throws java.sql.SQLException if any.
	 * @return a {@link java.sql.Timestamp} object.
	 */
	Timestamp getLastUpdateDate() throws SQLException;

	/**
	 * Count the number of rows in the table.<br>
	 * Zero value is always exact, even if approximate=true
	 *
	 * @param approximate
	 *            is approximation allow ? (e.g. for big table, approximate is
	 *            must faster)
	 * @throws java.sql.SQLException if any.
	 * @return a long.
	 */
	long countAll(boolean approximate) throws SQLException;

	/**
	 * Return if a PK already exists on table
	 *
	 * @param pk a {@link java.util.List} object.
	 * @throws java.sql.SQLException if any.
	 * @return a boolean.
	 */
	boolean exists(List<Object> pk) throws SQLException;

	/**
	 * Count the number of rows updated since the given date
	 *
	 * @param fromDate a {@link java.util.Date} object.
	 * @throws java.sql.SQLException if any.
	 * @deprecated use count(Map&lt;String, Object&gt;) instead
	 * @return a long.
	 */
	@Deprecated
	long countDataToUpdate(Date fromDate) throws SQLException;

	/**
	 * Count the number of rows
	 *
	 * @param bindings
	 *            binding parameters
	 * @throws java.sql.SQLException if any.
	 * @return a long.
	 */
	long countData(Map<String, Object> bindings) throws SQLException;

	/**
	 * Count the number of rows, by a set of FK value
	 *
	 * @param bindings
	 *            binding parameters
	 * @throws java.sql.SQLException if any.
	 * @param fkColumnNames a {@link java.util.Set} object.
	 * @param fkColumnsValues a {@link java.util.List} object.
	 * @return a long.
	 */
	long countDataByFks(Set<String> fkColumnNames,
			List<List<Object>> fkColumnsValues, Map<String, Object> bindings)
			throws SQLException;

	/**
	 * Obtains all primary keys of a table. If more than one key in tha table,
	 * use a serialization {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata#toPkStr}
	 *
	 * @throws java.sql.SQLException if any.
	 * @return a {@link java.util.Set} object.
	 */
	Set<String> getPksStr() throws SQLException;

	/**
	 * Obtains existing PKs with update dates (by local id)
	 *
	 * @throws java.sql.SQLException if any.
	 * @return a {@link java.util.Map} object.
	 */
	Map<String, Timestamp> getPksStrWithUpdateDate() throws SQLException;

	/**
	 * <p>getPksStrWithUpdateDateByFks.</p>
	 *
	 * @param fkColumnNames a {@link java.util.Set} object.
	 * @param fkColumnsValues a {@link java.util.List} object.
	 * @param bindings a {@link java.util.Map} object.
	 * @return a {@link java.util.Map} object.
	 * @throws java.sql.SQLException if any.
	 */
	Map<String, Timestamp> getPksStrWithUpdateDateByFks(
			Set<String> fkColumnNames, List<List<Object>> fkColumnsValues,
			Map<String, Object> bindings) throws SQLException;

	/**
	 * Getting all row depending of a set of column values
	 * <br>
	 * i.e. [[COLUMN_FK1=VAL1_1, COLUMN_FK2=VAL1_2], [COLUMN_FK1=VAL2_1,
	 * COLUMN_FK2=VAL2_2]]
	 *
	 * @param fkColumnNames a {@link java.util.Set} object.
	 * @param fkColumnsValues a {@link java.util.List} object.
	 * @param bindings a {@link java.util.Map} object.
	 * @throws java.sql.SQLException if any.
	 * @return a {@link java.sql.ResultSet} object.
	 */
	ResultSet getDataByFks(Set<String> fkColumnNames,
			List<List<Object>> fkColumnsValues, Map<String, Object> bindings)
			throws SQLException;

	/**
	 * Getting all PK depending of a set of column values
	 * <br>
	 * i.e. [[COLUMN_FK1=VAL1_1, COLUMN_FK2=VAL1_2], [COLUMN_FK1=VAL2_1,
	 * COLUMN_FK2=VAL2_2]]
	 *
	 * @param fkColumnNames a {@link java.util.Set} object.
	 * @param fkColumnsValues a {@link java.util.List} object.
	 * @param bindings a {@link java.util.Map} object.
	 * @return list of pk found
	 * @throws java.sql.SQLException if any.
	 */
	List<List<Object>> getPksByFks(Set<String> fkColumnNames,
			List<List<Object>> fkColumnsValues, Map<String, Object> bindings)
			throws SQLException;

	/**
	 * Return all PK from rows that NOT match given values
	 *
	 * @param fkColumnNames a {@link java.util.Set} object.
	 * @param fkColumnsValues a {@link java.util.List} object.
	 * @param bindings a {@link java.util.Map} object.
	 * @return list of pk found, that NOT match the given FK values
	 * @throws java.sql.SQLException if any.
	 */
	List<List<Object>> getPksByNotFoundFks(Set<String> fkColumnNames,
			List<List<Object>> fkColumnsValues, Map<String, Object> bindings)
			throws SQLException;

	/**
	 * Same, but return a list of pkStr
	 *
	 * @param fkColumnNames a {@link java.util.Set} object.
	 * @param fkColumnsValues a {@link java.util.List} object.
	 * @param bindings a {@link java.util.Map} object.
	 * @throws java.sql.SQLException if any.
	 * @return a {@link java.util.Set} object.
	 */
	Set<String> getPksStrByFks(Set<String> fkColumnNames,
			List<List<Object>> fkColumnsValues, Map<String, Object> bindings)
			throws SQLException;

	/**
	 * Return that NOT match given values
	 *
	 * @param fkColumnNames a {@link java.util.Set} object.
	 * @param fkColumnsValues a {@link java.util.List} object.
	 * @param bindings a {@link java.util.Map} object.
	 * @throws java.sql.SQLException if any.
	 * @return a {@link java.util.Set} object.
	 */
	Set<String> getPksStrByNotFoundFks(Set<String> fkColumnNames,
			List<List<Object>> fkColumnsValues, Map<String, Object> bindings)
			throws SQLException;

	/**
	 * Obtains all data updated since the given date (or all data if the given
	 * date is null
	 *
	 * @param fromDate a {@link java.util.Date} object.
	 * @return A Resulset with expected data.
	 * @throws java.sql.SQLException if any.
	 * @deprecated use getData(map) instead
	 */
	@Deprecated
	ResultSet getDataToUpdate(Date fromDate) throws SQLException;

	/**
	 * Obtains all data updated since the given date (or all data if the given
	 * date is null
	 *
	 * @return A Resulset with expected data.
	 * @throws java.sql.SQLException if any.
	 * @param bindings a {@link java.util.Map} object.
	 */
	ResultSet getData(Map<String, Object> bindings) throws SQLException;

	/**
	 * Read resultset, and return the current row as array
	 *
	 * @param incomingData a {@link java.sql.ResultSet} object.
	 * @throws java.sql.SQLException if any.
	 * @return an array of {@link java.lang.Object} objects.
	 */
	Object[] getDataAsArray(ResultSet incomingData) throws SQLException;

	/**
	 * Obtains a row, using values of each PK
	 *
	 * @param pk
	 *            values of PK column
	 * @return the existing row in the table
	 * @throws java.sql.SQLException if any.
	 */
	Object[] findByPk(List<Object> pk) throws SQLException;

	/**
	 * Obtains a PK from the current ResultSet's row
	 *
	 * @param incomingData a {@link java.sql.ResultSet} object.
	 * @throws java.sql.SQLException if any.
	 * @return a {@link java.util.List} object.
	 */
	List<Object> getPk(ResultSet incomingData) throws SQLException;

	/**
	 * Obtains a PK from a row, given as an array
	 *
	 * @param incomingData an array of {@link java.lang.Object} objects.
	 * @throws java.sql.SQLException if any.
	 * @return a {@link java.util.List} object.
	 */
	List<Object> getPk(Object[] incomingData) throws SQLException;

	/**
	 * Obtains a PK from the current ResultSet's row, after interceptors
	 * transformation
	 *
	 * @param incomingData a {@link java.sql.ResultSet} object.
	 * @param transform
	 *            apply transformation (interceptor.onRead) ?
	 * @throws java.sql.SQLException if any.
	 * @return a {@link java.util.List} object.
	 */
	List<Object> getPk(ResultSet incomingData, boolean transform)
			throws SQLException;

	/**
	 * Delete all rows of a table
	 *
	 * @throws java.sql.SQLException if any.
	 */
	void deleteAll() throws SQLException;

	/**
	 * Insert into a table the current row of the given {@link java.sql.ResultSet}
	 *
	 * @param incomingData
	 *            a {@link java.sql.ResultSet} with a row ready to be read (
	 *            {@link java.sql.ResultSet#next()} should have been call before)
	 * @throws java.sql.SQLException if any.
	 */
	void executeInsert(ResultSet incomingData) throws SQLException;

	/**
	 * Insert into a table the current row of the given incoming data
	 * <br>
	 * Same as {@link #executeInsert(ResultSet)}, but using a array instead
	 * of a ResultSet.
	 *
	 * @throws java.sql.SQLException if any.
	 * @param row an array of {@link java.lang.Object} objects.
	 */
	void executeInsert(Object[] row) throws SQLException;

	/**
	 * Update one table's row, using the current row of the given
	 * {@link java.sql.ResultSet}
	 *
	 * @param pk
	 *            the PK of the row to update
	 * @param incomingData
	 *            a {@link java.sql.ResultSet} with a row ready to be read (
	 *            {@link java.sql.ResultSet#next()} should have been call before)
	 * @throws java.sql.SQLException if any.
	 */
	void executeUpdate(List<Object> pk, ResultSet incomingData)
			throws SQLException;

	/**
	 * Update one table's row, using the current row of the given array of
	 * values
	 * <br>
	 * Same as {@link #executeUpdate(List,ResultSet)}, but using a array instead
	 * of a ResultSet.
	 *
	 * @param pk
	 *            the PK of the row to update
	 * @param row
	 *            a array of object
	 * @throws java.sql.SQLException if any.
	 */
	void executeUpdate(List<Object> pk, Object[] row) throws SQLException;

	/**
	 * Update one column of one table row.
	 *
	 * @param columnName a {@link java.lang.String} object.
	 * @param pk a {@link java.util.List} object.
	 * @param value a {@link java.lang.Object} object.
	 * @throws java.sql.SQLException if any.
	 */
	void executeUpdateColumn(String columnName, List<Object> pk, Object value)
			throws SQLException;

	/**
	 * Flush all pending updates (i.e. pending batch statement)
	 *
	 * @throws java.sql.SQLException if any.
	 */
	void flush() throws SQLException;

	/**
	 * Obtains the number of updated rows
	 *
	 * @return the number of updated rows
	 */
	int getUpdateCount();

	/**
	 * Obtains the number of updated rows (update on the given column)
	 *
	 * @return the number of updated rows
	 * @param columnName a {@link java.lang.String} object.
	 */
	int getUpdateColumnCount(String columnName);

	/**
	 * Obtains the number of inserted rows
	 *
	 * @return the number of inserted rows
	 */
	int getInsertCount();

	/**
	 * Obtains the number of deleted rows
	 *
	 * @return the number of deleted rows
	 */
	int getDeleteCount();

	/**
	 * Getting the database dialect
	 *
	 * @return a {@link org.hibernate.dialect.Dialect} object.
	 */
	Dialect getDialect();

	/**
	 * Getting all PKs corresponding to unique constraints
	 *
	 * @param inconmingData a {@link java.sql.ResultSet} object.
	 * @return a {@link java.util.Map} object.
	 * @throws java.sql.SQLException if any.
	 */
	Map<String, List<Object>> getPkFromUniqueConstraints(ResultSet inconmingData)
			throws SQLException;

	/**
	 * Retrieve update_date from the table, by PK
	 *
	 * @param pk a {@link java.util.List} object.
	 * @return a {@link java.sql.Timestamp} object.
	 * @throws java.sql.SQLException if any.
	 */
	Timestamp getUpdateDateByPk(List<Object> pk) throws SQLException;

	/**
	 * <p>getUpdateDate.</p>
	 *
	 * @param incomingData a {@link java.sql.ResultSet} object.
	 * @param transform a boolean.
	 * @return a {@link java.sql.Timestamp} object.
	 * @throws java.sql.SQLException if any.
	 */
	Timestamp getUpdateDate(ResultSet incomingData, boolean transform)
			throws SQLException;

	/**
	 * Try to lock the row, by pk. If could not lock, return false
	 *
	 * @param pk a {@link java.util.List} object.
	 * @return true if the lock is acquire, false is could not lock
	 * @throws java.sql.SQLException if any.
	 */
	boolean lock(List<Object> pk) throws SQLException;

	/**
	 * <p>setSourceDao.</p>
	 *
	 * @param sourceDao a {@link fr.ifremer.common.synchro.dao.SynchroTableDao} object.
	 */
	void setSourceDao(SynchroTableDao sourceDao);

	/**
	 * Delete a row, by PK.
	 *
	 * @param pk
	 *            PK of the row to delete
	 * @param checkPkNotUsed
	 *            check if the PK is not used anymore. If yes, throw an
	 *            exception
	 * @throws java.sql.SQLException if any.
	 */
	void executeDelete(List<Object> pk, boolean checkPkNotUsed)
			throws SQLException;

	/**
	 * Detach an row. By default, to nothing but call interceptor.onDetach()
	 *
	 * @param pk
	 *            PK of the row to detach
	 * @throws java.sql.SQLException if any.
	 */
	void executeDetach(List<Object> pk) throws SQLException;

	/**
	 * Prepare to run dao (reset count...)
	 */
	void prepare();

	/**
	 * <p>executeDeleteByFk.</p>
	 *
	 * @param fkColumnName a {@link java.lang.String} object.
	 * @param fkColumnValue a {@link java.lang.String} object.
	 * @param additionalWhereClause a {@link java.lang.String} object.
	 * @param bindings a {@link java.util.Map} object.
	 * @return a boolean.
	 * @throws java.sql.SQLException if any.
	 */
	boolean executeDeleteByFk(String fkColumnName, String fkColumnValue,
			String additionalWhereClause, Map<String, Object> bindings)
			throws SQLException;

	/**
	 * <p>getLastGeneratedPk.</p>
	 *
	 * @return a {@link java.util.List} object.
	 */
	List<Object> getLastGeneratedPk();

	/**
	 * <p>transformOnRead.</p>
	 *
	 * @param fkColumnNames a {@link java.util.Set} object.
	 * @param fkSourceColumnValues a {@link java.util.List} object.
	 * @return a {@link java.util.List} object.
	 * @throws java.sql.SQLException if any.
	 */
	List<List<Object>> transformOnRead(Set<String> fkColumnNames,
			List<List<Object>> fkSourceColumnValues) throws SQLException;

	/**
	 * <p>transformColumnNames.</p>
	 *
	 * @param fkColumnNames a {@link java.util.Set} object.
	 * @return a {@link java.util.Set} object.
	 * @throws java.sql.SQLException if any.
	 */
	Set<String> transformColumnNames(Set<String> fkColumnNames)
			throws SQLException;

	/**
	 * <p>getObject.</p>
	 *
	 * @param incomingData a {@link java.sql.ResultSet} object.
	 * @param index a int.
	 * @return a {@link java.lang.Object} object.
	 * @throws java.sql.SQLException if any.
	 */
	Object getObject(ResultSet incomingData, int index) throws SQLException;

	/**
	 * <p>getExportedKeys.</p>
	 *
	 * @return a {@link com.google.common.collect.Multimap} object.
	 * @throws java.sql.SQLException if any.
	 */
	Multimap<String, String> getExportedKeys() throws SQLException;
}
