package fr.ifremer.common.synchro.dao;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2015 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.dialect.Dialect;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;

import fr.ifremer.common.synchro.SynchroTechnicalException;
import fr.ifremer.common.synchro.config.SynchroConfiguration;
import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata;
import fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration;

/**
 * <p>SynchroBaseDaoImpl class.</p>
 *
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 * @since 3.8.0
 */
public class SynchroBaseDaoImpl implements SynchroBaseDao {

	/** Logger. */
	private static final Log log = LogFactory.getLog(SynchroBaseDaoImpl.class);

	private Connection connection;

	private DaoStats stats;

	private final LoadingCache<String, PreparedStatement> statementCache;

	private final LoadingCache<String, Optional<Object>> valueCache;

	private DaoFactory daoFactory;

	private int batchSize;

	private final String deleteTempQueryParameterQuery;

	private final String deleteTempQueryParameterQueryWithLike;

	private final String insertIntoTempQueryParameterQuery;

	private boolean debug;

	/**
	 * Constructor for unit test
	 *
	 * @param dbMeta a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 * @throws java.sql.SQLException if any.
	 */
	public SynchroBaseDaoImpl(SynchroDatabaseMetadata dbMeta)
			throws SQLException {
		this(dbMeta.getConnection(), dbMeta.getConfiguration(), dbMeta,
				new DaoFactoryImpl(dbMeta), new DaoStats(),
				DaoFactory.DEFAULT_STATEMENT_CACHE_SIZE,
				DaoFactory.DEFAULT_STATEMENT_CACHE_DURATION_SECONDS,
				DaoFactory.DEFAULT_VALUE_CACHE_SIZE,
				DaoFactory.DEFAULT_VALUE_CACHE_DURATION_SECONDS);
	}

	/**
	 * Base constructor for production
	 *
	 * @param connection a {@link java.sql.Connection} object.
	 * @param stats a {@link fr.ifremer.common.synchro.dao.DaoStats} object.
	 * @param statementCacheSize a int.
	 * @param statementCacheDurationInSecond a int.
	 * @param valueCacheSize a int.
	 * @param valueCacheDurationInSecond a int.
	 * @param configuration a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 * @param dbMeta a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 * @param daoFactory a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 */
	public SynchroBaseDaoImpl(Connection connection,
                              SynchroDatabaseConfiguration configuration,
                              SynchroDatabaseMetadata dbMeta,
                              DaoFactory daoFactory,
                              DaoStats stats,
                              int statementCacheSize,
                              int statementCacheDurationInSecond,
                              int valueCacheSize,
                              int valueCacheDurationInSecond) {
		this.connection = connection;
		this.daoFactory = daoFactory;
		this.stats = stats;
		debug = log.isTraceEnabled();
		this.batchSize = SynchroConfiguration.getInstance()
				.getImportJdbcBatchSize();
		Preconditions.checkArgument(this.batchSize > 0);

		// Init cache loader
		int validStatementCacheSize = statementCacheSize > 0
				? statementCacheSize
				: DaoFactory.DEFAULT_STATEMENT_CACHE_SIZE;
		int validValueCacheSize = valueCacheSize > 0
				? valueCacheSize
				: DaoFactory.DEFAULT_VALUE_CACHE_SIZE;
		this.statementCache = initStatementCache(validStatementCacheSize,
				statementCacheDurationInSecond);
		this.valueCache = initValueCache(validValueCacheSize,
				valueCacheDurationInSecond);

		// Init query on table 'TEMP_QUERY_PARAMETER'
		Preconditions
				.checkNotNull(
						dbMeta.getTable(TEMP_QUERY_PARAMETER_TABLE),
						String.format(
								"%s could not be started without '%s' table. Please add this technical table to schema",
								getClass().getSimpleName(),
								TEMP_QUERY_PARAMETER_TABLE));
		this.insertIntoTempQueryParameterQuery = initInsertIntoTempQueryParameterQuery(configuration
				.getDialect());
		this.deleteTempQueryParameterQuery = initDeleteTempQueryParameterQuery(false);
		this.deleteTempQueryParameterQueryWithLike = initDeleteTempQueryParameterQuery(true);
	}

	/** {@inheritDoc} */
	@Override
	public Connection getConnection() {
		return connection;
	}

	/** {@inheritDoc} */
	@Override
	public DaoFactory getDaoFactory() {
		return daoFactory;
	}

	/** {@inheritDoc} */
	@Override
	public void close() throws IOException {

		// Clean the statements cache (should call onRemoval() and close all
		// statements)
		statementCache.invalidateAll();

		// Clean the values cache
		valueCache.invalidateAll();

		daoFactory = null;
		stats = null;
		connection = null;
	}

	/** {@inheritDoc} */
	@Override
	public PreparedStatement getPreparedStatement(String sql)
			throws SQLException {
		Preconditions.checkNotNull(sql);

		if (log.isDebugEnabled()) {
			log.debug(String.format("Prepare statement: %s",
					sql));
		}

		try {
			// Encapsulate to a not closeable statement, to avoid the call of
			// close() method
			return new NotCloseablePreparedStatement(statementCache.get(sql));
		} catch (ExecutionException e) {
			throw new SynchroTechnicalException(String.format(
					"[%s] Error during statement loading (using cache): %s",
					sql, e.getMessage()), e);
		}
	}

	/** {@inheritDoc} */
	@SuppressWarnings("unchecked")
	@Override
	public <T> T getUniqueTyped(String sql, Object[] bindingValues)
			throws SQLException {
		String cacheKey = new StringBuilder().append(sql).append("#")
				.append(SynchroTableMetadata.toPkStr(bindingValues)).toString();

		stats.incrementValue();

		// Read from cache
		Optional<Object> optionalResult = valueCache.getIfPresent(cacheKey);

		// If not null value, then return this value
		if (optionalResult != null && optionalResult.isPresent()) {
			return (T) optionalResult.get();
		}

		// If nothing in the cache, or a null value: try again (null could have
		// be changed !)
		Object result = newSqlExecuteUnique(cacheKey);

		// Update the cache, if need
		if (result != null) {
			valueCache.put(cacheKey, Optional.of(result));
		}

		return (T) result;
	}

	/** {@inheritDoc} */
	@Override
	public String getInsertIntoTempQueryParameterQuery() {
		return insertIntoTempQueryParameterQuery;
	}

	/** {@inheritDoc} */
	@Override
	public void cleanTempQueryParameter() throws SQLException {
		if (debug) {
			log.debug(String.format("Deleting all synchro rows from %s",
					TEMP_QUERY_PARAMETER_TABLE.toUpperCase()));
		}

		String sql = initCleanTempQueryParameterQuery();

		PreparedStatement statement = null;
		try {
			// Get statement from pool
			statement = getPreparedStatement(sql);

			// execute the delete statement
			int deletedRows = statement.executeUpdate();
			if (debug) {
				log.trace(String
						.format("%s rows delete from TEMP_QUERY_PARAMETER",
								deletedRows));
			}
		} catch (Exception e) {
			throw new SynchroTechnicalException(String.format(
					"Could not delete all synchro rows from %s",
					TEMP_QUERY_PARAMETER_TABLE), e);
		} finally {
			Daos.closeSilently(statement);
		}
	}

	/** {@inheritDoc} */
	public void executeDeleteTempQueryParameter(String queryParameterName,
			boolean likeOperatorForParameterName, int queryPersonId)
			throws SQLException {
		if (debug) {
			log.debug(String.format("Deleting rows from %s",
					TEMP_QUERY_PARAMETER_TABLE.toUpperCase()));
		}

		String sql = likeOperatorForParameterName
				? deleteTempQueryParameterQueryWithLike
				: deleteTempQueryParameterQuery;

		PreparedStatement statement = null;
		try {
			// Get statement from pool
			statement = getPreparedStatement(sql);

			// execute the delete statement
			statement.setString(1, queryParameterName);
			statement.setInt(2, queryPersonId);
			int deletedRows = statement.executeUpdate();
			if (debug) {
				log.trace(String
						.format("%s rows delete from TEMP_QUERY_PARAMETER",
								deletedRows));
			}
		} catch (Exception e) {
			throw new SynchroTechnicalException(String.format(
					"Could not delete from table %s",
					TEMP_QUERY_PARAMETER_TABLE), e);
		} finally {
			Daos.closeSilently(statement);
		}
	}

	/** {@inheritDoc} */
	@Override
	public void executeInsertIntoTempQueryParameter(
			List<Object> alphaNumericalValues, String queryParameterName,
			int queryPersonId) throws SQLException {

		// Delete existing rows in TEMP_QUERY_PARAMETER
		executeDeleteTempQueryParameter(queryParameterName, false/* = */,
				queryPersonId);

		if (debug) {
			log.debug(String.format("Setting query parameters into %s",
					TEMP_QUERY_PARAMETER_TABLE.toUpperCase()));
		}

		// Query temp parameter (for query parameter with many values)
		PreparedStatement statement = null;
		try {
			try {
				statement = getPreparedStatement(this.insertIntoTempQueryParameterQuery);
			} catch (SQLException e) {
				throw new SynchroTechnicalException(
						String.format(
								"Error while creating statement on table %s. Make sure this table exist.",
								TEMP_QUERY_PARAMETER_TABLE), e);
			}

			int rowCount = 0;
			for (Object alphaNumericalValue : alphaNumericalValues) {
				rowCount++;
				statement.setString(1, queryParameterName); // PARAMETER_NAME
				statement.setNull(2, Types.INTEGER); // GROUPING_KEY
				statement.setNull(3, Types.INTEGER); // NUMERICAL_VALUE
				statement.setObject(4, alphaNumericalValue); // ALPHANUMERICAL_VALUE
				statement.setInt(5, queryPersonId);
				statement.addBatch();
				if (rowCount % batchSize == 0) {
					statement.executeBatch();
					statement.clearBatch();
				}
			}

			if (rowCount % batchSize != 0) {
				statement.executeBatch();
				statement.clearBatch();
			}

		} catch (Exception e) {
			throw new SynchroTechnicalException(String.format(
					"Could not insert into table %s",
					TEMP_QUERY_PARAMETER_TABLE), e);
		} finally {
			Daos.closeSilently(statement);
		}
	}

	/* -- Internal methods -- */

	/**
	 * <p>initStatementCache.</p>
	 *
	 * @param maximumSize a int.
	 * @param durationInSeconds a int.
	 * @return a {@link com.google.common.cache.LoadingCache} object.
	 */
	protected LoadingCache<String, PreparedStatement> initStatementCache(
			final int maximumSize, int durationInSeconds) {
		return CacheBuilder
				.newBuilder()
				.maximumSize(maximumSize)
				.expireAfterAccess(durationInSeconds, TimeUnit.SECONDS)
				.removalListener(
						new RemovalListener<String, PreparedStatement>() {
							@Override
							public void onRemoval(
									RemovalNotification<String, PreparedStatement> notification) {
								// When removal, close the statement
								closeStatement(notification.getValue());
							}
						}).build(new CacheLoader<String, PreparedStatement>() {
					public PreparedStatement load(String sql)
							throws SQLException {
						return newPreparedStatement(sql);
					}
				});
	}

	/**
	 * <p>newPreparedStatement.</p>
	 *
	 * @param sql a {@link java.lang.String} object.
	 * @return a {@link java.sql.PreparedStatement} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected PreparedStatement newPreparedStatement(String sql)
			throws SQLException {
		stats.incrementStatement();
		PreparedStatement ps = this.connection.prepareStatement(sql);
		if (log.isTraceEnabled()) {
			log.trace(String.format("Create new prepared statement [%s]",
					ps.toString()));
		}
		return ps;
	}

	/**
	 * <p>closeStatement.</p>
	 *
	 * @param ps a {@link java.sql.PreparedStatement} object.
	 */
	protected void closeStatement(PreparedStatement ps) {
		if (log.isTraceEnabled()) {
			log.trace(String.format("Closing prepared statement [%s]",
					ps.toString()));
		}
		Daos.closeSilently(ps);
	}

	/**
	 * <p>initValueCache.</p>
	 *
	 * @param maximumSize a int.
	 * @param durationInSeconds a int.
	 * @return a {@link com.google.common.cache.LoadingCache} object.
	 */
	protected LoadingCache<String, Optional<Object>> initValueCache(
			final int maximumSize, int durationInSeconds) {
		return CacheBuilder.newBuilder().maximumSize(maximumSize)
				.expireAfterWrite(durationInSeconds, TimeUnit.SECONDS)
				.build(new CacheLoader<String, Optional<Object>>() {
					public Optional<Object> load(String sqlAndBindings)
							throws SQLException {
						return Optional
								.fromNullable(newSqlExecuteUnique(sqlAndBindings));
					}
				});
	}

	/**
	 * <p>newSqlExecuteUnique.</p>
	 *
	 * @param sqlAndBindings a {@link java.lang.String} object.
	 * @return a {@link java.lang.Object} object.
	 * @throws java.sql.SQLException if any.
	 */
	protected Object newSqlExecuteUnique(String sqlAndBindings)
			throws SQLException {

		stats.incrementExecuteValue();

		String[] sqlParts = sqlAndBindings.split("#");
		String sql = sqlParts[0];
		List<Object> bindingValues = SynchroTableMetadata
				.fromPkStr(sqlParts[1]);

		PreparedStatement statement = null;
		ResultSet rs = null;
		try {
			statement = getPreparedStatement(sql);

			int i = 1;
			for (Object value : bindingValues) {
				statement.setObject(i++, value);
			}

			rs = statement.executeQuery();
			if (!rs.next()) {
				return null;
			}
			Object result = rs.getObject(1);
			if (rs.next()) {
				throw new SynchroTechnicalException(
						"Executed query has more than one row: " + sql);
			}

			return result;
		} finally {
			Daos.closeSilently(rs);
			Daos.closeSilently(statement);
		}
	}

	/**
	 * <p>initCleanTempQueryParameterQuery.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	protected String initCleanTempQueryParameterQuery() {
		String sql = String.format(
				"DELETE FROM %s WHERE parameter_name like '%s'",
				TEMP_QUERY_PARAMETER_TABLE, TEMP_QUERY_PARAMETER_PARAM_PREFIX
						+ "%");

		return sql;
	}

	/**
	 * <p>initInsertIntoTempQueryParameterQuery.</p>
	 *
	 * @param dialect a {@link org.hibernate.dialect.Dialect} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String initInsertIntoTempQueryParameterQuery(Dialect dialect) {
		String sequenceNextValString = dialect
				.getSelectSequenceNextValString(TEMP_QUERY_PARAMETER_SEQUENCE);

		String sql = String
				.format("INSERT INTO %s (ID, PARAMETER_NAME, GROUPING_KEY, NUMERICAL_VALUE, ALPHANUMERICAL_VALUE, %s)"
						+ " VALUES (%s, ?, ?, ?, ?, ?)",
						TEMP_QUERY_PARAMETER_TABLE, SynchroConfiguration
								.getInstance()
								.getTempQueryParemeterUserIdColumn(),
						sequenceNextValString);

		return sql;
	}

	/**
	 * <p>initDeleteTempQueryParameterQuery.</p>
	 *
	 * @param likeOperatorForParameterName a boolean.
	 * @return a {@link java.lang.String} object.
	 */
	protected String initDeleteTempQueryParameterQuery(
			boolean likeOperatorForParameterName) {
		String sql = String.format(
				"DELETE FROM %s WHERE parameter_name %s ? AND %s=?",
				TEMP_QUERY_PARAMETER_TABLE, likeOperatorForParameterName
						? "LIKE"
						: "=", SynchroConfiguration.getInstance()
						.getTempQueryParemeterUserIdColumn());

		return sql;
	}
}
