package fr.ifremer.common.synchro.dao;

/*
 * #%L
 * Tutti :: Persistence API
 * $Id: TuttiEntities.java 1578 2014-02-07 15:31:18Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/trunk/tutti-persistence/src/main/java/fr/ifremer/tutti/persistence/entities/TuttiEntities.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.io.File;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.Properties;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.JDBCException;
import org.hibernate.Session;
import org.hibernate.cfg.AvailableSettings;
import org.hibernate.cfg.Configuration;
import org.hibernate.cfg.Environment;
import org.hibernate.dialect.Dialect;
import org.hibernate.exception.GenericJDBCException;
import org.hibernate.exception.spi.SQLExceptionConversionDelegate;
import org.hibernate.exception.spi.SQLExceptionConverter;
import org.nuiton.version.Version;
import org.nuiton.version.Versions;

import com.google.common.base.Preconditions;

import fr.ifremer.common.synchro.SynchroTechnicalException;

/**
 * Usefull method around DAO and entities.
 *
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 * @since 3.5.4
 */
public class Daos {
	/** Logger. */
	private static final Log log = LogFactory.getLog(Daos.class);

	private final static String JDBC_URL_PREFIX_HSQLDB = "jdbc:hsqldb:";
	private final static String JDBC_URL_PREFIX_HSQLDB_FILE = JDBC_URL_PREFIX_HSQLDB
			+ "file:";
	private final static String JDBC_URL_PREFIX_ORACLE = "jdbc:oracle:";

	/**
	 * <p>Constructor for Daos.</p>
	 */
	protected Daos() {
		// helper class does not instantiate
	}

	/**
	 * Create a new hibernate configuration, with all hbm.xml files for the
	 * schema need for app
	 *
	 * @return the hibernate Configuration
	 * @param jdbcUrl a {@link java.lang.String} object.
	 * @param username a {@link java.lang.String} object.
	 * @param password a {@link java.lang.String} object.
	 * @param catalog a {@link java.lang.String} object.
	 * @param schema a {@link java.lang.String} object.
	 * @param dialect a {@link java.lang.String} object.
	 * @param driver a {@link java.lang.String} object.
	 */
	public static Properties getConnectionProperties(String jdbcUrl,
			String username, String password, String catalog, String schema,
			String dialect, String driver) {

		// Building a new configuration
		Properties p = new Properties();

		// Set driver
		p.setProperty(Environment.DRIVER, driver);

		// Set hibernate dialect
		p.setProperty(Environment.DIALECT, dialect);

		// To be able to retrieve connection
		p.setProperty(Environment.URL, jdbcUrl);
		p.setProperty(Environment.USER, username);
		p.setProperty(Environment.PASS, password);

		if (StringUtils.isNotBlank(schema)) {
			p.setProperty(Environment.DEFAULT_SCHEMA, schema);
		}

		if (StringUtils.isNotBlank(catalog)) {
			p.setProperty(Environment.DEFAULT_CATALOG, catalog);
		}

		// Try with synonyms enable
		p.setProperty(AvailableSettings.ENABLE_SYNONYMS, "true");

		// Pour tester avec le metadata generic (normalement plus long pour
		// Oracle)
		// cfg.setProperty("hibernatetool.metadatadialect",
		// "org.hibernate.cfg.rveng.dialect.JDBCMetaDataDialect");
		if (jdbcUrl.startsWith("jdbc:oracle")) {
			p.setProperty("hibernatetool.metadatadialect",
					"org.hibernate.cfg.rveng.dialect.OracleMetaDataDialect");
		}

		return p;
	}

	/**
	 * Override default Hibernate 'org.hibernate.tool.hbm2ddl.DatabaseMetadata',
	 * because of th use of deprecated method buildSqlExceptionConverter() (see
	 * https://hibernate.atlassian.net/browse/HHH-9131)
	 *
	 * @param dialect a {@link org.hibernate.dialect.Dialect} object.
	 * @return a {@link org.hibernate.exception.spi.SQLExceptionConverter} object.
	 */
	public static SQLExceptionConverter newSQLExceptionConverter(
			final Dialect dialect) {
		// Build a valid sql converter
		return new SQLExceptionConverter() {
			private static final long serialVersionUID = 5458961195167573495L;

			SQLExceptionConversionDelegate delegate = dialect
					.buildSQLExceptionConversionDelegate();;

			@Override
			public JDBCException convert(SQLException sqlException,
					String message, String sql) {
				JDBCException exception = delegate.convert(sqlException,
						message, sql);
				if (exception != null) {
					return exception;
				}
				return new GenericJDBCException(message, sqlException, sql);
			}
		};
	}

	/**
	 * <p>shutdownDatabase.</p>
	 *
	 * @param connectionProperties a {@link java.util.Properties} object.
	 * @throws java.sql.SQLException if any.
	 */
	public static void shutdownDatabase(Properties connectionProperties)
			throws SQLException {
		Connection conn = Daos.createConnection(connectionProperties);
		try {
			shutdownDatabase(conn);
		} finally {
			closeSilently(conn);
		}
	}

	/**
	 * <p>closeSilently.</p>
	 *
	 * @param statement a {@link java.sql.Statement} object.
	 */
	public static void closeSilently(Statement statement) {
		try {
			if (statement != null && !statement.isClosed()) {
				statement.close();
			}
		} catch (AbstractMethodError e) {
			try {
				statement.close();
			} catch (SQLException e1) {
			}
			if (log.isDebugEnabled()) {
				log.debug("Fix this linkage error, damned hsqlsb 1.8.0.7:(");
			}
		} catch (IllegalAccessError e) {
			if (log.isDebugEnabled()) {
				log.debug("Fix this IllegalAccessError error, damned hsqlsb 1.8.0.7:(");
			}
		} catch (Exception e) {
			if (log.isErrorEnabled()) {
				log.error("Could not close statement, but do not care", e);
			}
		}
	}

	/**
	 * <p>closeSilently.</p>
	 *
	 * @param connection a {@link java.sql.Connection} object.
	 */
	public static void closeSilently(Connection connection) {
		try {
			if (connection != null && !connection.isClosed()) {
				connection.close();
			}
		} catch (Exception e) {
			try {
				connection.close();
			} catch (SQLException e1) {
			}
			if (log.isErrorEnabled()) {
				log.error("Could not close connection, but do not care", e);
			}
		}
	}

	/**
	 * <p>closeSilently.</p>
	 *
	 * @param resultSet a {@link java.sql.ResultSet} object.
	 */
	public static void closeSilently(ResultSet resultSet) {
		try {
			if (resultSet != null && !resultSet.isClosed()) {

				resultSet.close();
			}
		} catch (AbstractMethodError e) {
			try {
				resultSet.close();
			} catch (SQLException e1) {
			}
			if (log.isDebugEnabled()) {
				log.debug("Fix this linkage error, damned hsqlsb 1.8.0.7:(");
			}
		} catch (IllegalAccessError e) {
			if (log.isDebugEnabled()) {
				log.debug("Fix this IllegalAccessError error, damned hsqlsb 1.8.0.7:(");
			}
		} catch (Exception e) {
			if (log.isErrorEnabled()) {
				log.error("Could not close statement, but do not care", e);
			}
		}
	}

	/**
	 * <p>closeSilently.</p>
	 *
	 * @param session a {@link org.hibernate.Session} object.
	 */
	public static void closeSilently(Session session) {
		try {
			if (session != null && session.isOpen()) {

				session.close();
			}
		} catch (Exception e) {
			if (log.isErrorEnabled()) {
				log.error("Could not close session, but do not care", e);
			}
		}
	}

	/**
	 * <p>createConnection.</p>
	 *
	 * @param connectionProperties a {@link java.util.Properties} object.
	 * @return a {@link java.sql.Connection} object.
	 * @throws java.sql.SQLException if any.
	 */
	public static Connection createConnection(Properties connectionProperties)
			throws SQLException {
		return createConnection(
				connectionProperties.getProperty(Environment.URL),
				connectionProperties.getProperty(Environment.USER),
				connectionProperties.getProperty(Environment.PASS));
	}

	/**
	 * <p>createConnection.</p>
	 *
	 * @param jdbcUrl a {@link java.lang.String} object.
	 * @param user a {@link java.lang.String} object.
	 * @param password a {@link java.lang.String} object.
	 * @return a {@link java.sql.Connection} object.
	 * @throws java.sql.SQLException if any.
	 */
	public static Connection createConnection(String jdbcUrl, String user,
			String password) throws SQLException {
		Connection connection = DriverManager.getConnection(jdbcUrl, user,
				password);
		connection.setAutoCommit(false);
		return connection;
	}

	/**
	 * <p>fillConnectionProperties.</p>
	 *
	 * @param p a {@link java.util.Properties} object.
	 * @param url a {@link java.lang.String} object.
	 * @param username a {@link java.lang.String} object.
	 * @param password a {@link java.lang.String} object.
	 */
	public static void fillConnectionProperties(Properties p, String url,
			String username, String password) {
		p.put(Environment.URL, url);
		p.put(Environment.USER, username);
		p.put(Environment.PASS, password);
	}

	/**
	 * <p>getJdbcUrl.</p>
	 *
	 * @param directory a {@link java.io.File} object.
	 * @param dbName a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	public static String getJdbcUrl(File directory, String dbName) {
		String jdbcUrl = JDBC_URL_PREFIX_HSQLDB_FILE
				+ directory.getAbsolutePath() + "/" + dbName;
		jdbcUrl = jdbcUrl.replaceAll("\\\\", "/");
		return jdbcUrl;
	}

	/**
	 * <p>isFileDatabase.</p>
	 *
	 * @param jdbcUrl a {@link java.lang.String} object.
	 * @return a boolean.
	 */
	public static boolean isFileDatabase(String jdbcUrl) {
		Preconditions.checkNotNull(jdbcUrl);
		return jdbcUrl.startsWith(JDBC_URL_PREFIX_HSQLDB_FILE);
	}

	/**
	 * <p>isOracleDatabase.</p>
	 *
	 * @param jdbcUrl a {@link java.lang.String} object.
	 * @return a boolean.
	 */
	public static boolean isOracleDatabase(String jdbcUrl) {
		Preconditions.checkNotNull(jdbcUrl);
		return jdbcUrl.startsWith(JDBC_URL_PREFIX_ORACLE);
	}

	/**
	 * Check if connection properties are valid. Try to open a SQL connection,
	 * then close it. If no error occur, the connection is valid.
	 *
	 * @param connectionProperties a {@link java.util.Properties} object.
	 * @return a boolean.
	 */
	public static boolean isValidConnectionProperties(
			Properties connectionProperties) {
		return isValidConnectionProperties(
				connectionProperties.getProperty(Environment.DRIVER),
				connectionProperties.getProperty(Environment.URL),
				connectionProperties.getProperty(Environment.USER),
				connectionProperties.getProperty(Environment.PASS));
	}

	/**
	 * Check if connection properties are valid. Try to open a SQL connection,
	 * then close it. If no error occur, the connection is valid.
	 *
	 * @param jdbcDriver a {@link java.lang.String} object.
	 * @param jdbcUrl a {@link java.lang.String} object.
	 * @param user a {@link java.lang.String} object.
	 * @param password a {@link java.lang.String} object.
	 * @return a boolean.
	 */
	public static boolean isValidConnectionProperties(String jdbcDriver,
			String jdbcUrl, String user, String password) {
		String driverClassName = jdbcDriver;
		try {
			Class<?> driverClass = Class.forName(driverClassName);
			DriverManager.registerDriver((java.sql.Driver) driverClass
					.newInstance());
		} catch (Exception e) {
			log.error("Could not load JDBC Driver: " + e.getMessage(), e);
			return false;
		}

		Connection connection = null;
		try {
			connection = createConnection(jdbcUrl, user, password);
			return true;
		} catch (SQLException e) {
			log.error("Could not connect to database: " + e.getMessage().trim());
		} finally {
			Daos.closeSilently(connection);
		}
		return false;
	}

	/**
	 * <p>getUrl.</p>
	 *
	 * @param connectionProperties a {@link java.util.Properties} object.
	 * @return a {@link java.lang.String} object.
	 */
	public static String getUrl(Properties connectionProperties) {
		return connectionProperties.getProperty(Environment.URL);
	}

	/**
	 * <p>getUrl.</p>
	 *
	 * @param connection a {@link java.sql.Connection} object.
	 * @return a {@link java.lang.String} object.
	 */
	public static String getUrl(Connection connection) {
		try {
			return connection.getMetaData().getURL();
		} catch (SQLException e) {
			throw new SynchroTechnicalException(
					"Could not get JDBC URL from connection", e);
		}
	}

	/**
	 * <p>getDialect.</p>
	 *
	 * @param connectionProperties a {@link java.util.Properties} object.
	 * @return a {@link org.hibernate.dialect.Dialect} object.
	 */
	public static Dialect getDialect(Properties connectionProperties) {
		return Dialect.getDialect(connectionProperties);
	}

	/**
	 * <p>getConfiguration.</p>
	 *
	 * @param connectionProperties a {@link java.util.Properties} object.
	 * @return a {@link org.hibernate.cfg.Configuration} object.
	 */
	public static Configuration getConfiguration(Properties connectionProperties) {
		return new Configuration().setProperties(connectionProperties);
	}

	/**
	 * <p>shutdownDatabase.</p>
	 *
	 * @param connection a {@link java.sql.Connection} object.
	 */
	public static void shutdownDatabase(Connection connection) {
		try {
			String jdbcUrl = connection.getMetaData().getURL();
			if (isFileDatabase(jdbcUrl)) {
				sqlUpdate(connection, "SHUTDOWN");
			}
		} catch (SQLException e) {
			e.printStackTrace();
		}

	}

	/**
	 * <p>sqlUpdate.</p>
	 *
	 * @param connection a {@link java.sql.Connection} object.
	 * @param sql a {@link java.lang.String} object.
	 * @return a int.
	 */
	public static int sqlUpdate(Connection connection, String sql) {
		Statement stmt = null;
		try {
			stmt = connection.createStatement();
		} catch (SQLException ex) {
			closeSilently(stmt);
			throw new SynchroTechnicalException(
					"Could not open database connection", ex);
		}

		// Log using a special logger
		if (log.isDebugEnabled()) {
			log.debug(sql);
		}

		try {
			return stmt.executeUpdate(sql);
		} catch (SQLException ex) {
			throw new SynchroTechnicalException("Could not execute query: "
					+ sql, ex);
		} finally {
			closeSilently(stmt);
		}
	}

	/**
	 * <p>isHsqldbDatabase.</p>
	 *
	 * @param jdbcUrl a {@link java.lang.String} object.
	 * @return a boolean.
	 */
	public static boolean isHsqldbDatabase(String jdbcUrl) {
		Preconditions.checkNotNull(jdbcUrl);
		return jdbcUrl.startsWith(JDBC_URL_PREFIX_HSQLDB);
	}

	/**
	 * <p>setIntegrityConstraints.</p>
	 *
	 * @param connectionProperties a {@link java.util.Properties} object.
	 * @param enableIntegrityConstraints a boolean.
	 * @throws java.sql.SQLException if any.
	 */
	public static void setIntegrityConstraints(Properties connectionProperties,
			boolean enableIntegrityConstraints) throws SQLException {
		// Execute the SQL order
		Connection connection = null;
		try {
			connection = createConnection(connectionProperties);
			setIntegrityConstraints(connection, enableIntegrityConstraints);
		} finally {
			closeSilently(connection);
		}

	}

	/**
	 * <p>setIntegrityConstraints.</p>
	 *
	 * @param connection a {@link java.sql.Connection} object.
	 * @param enableIntegrityConstraints a boolean.
	 * @throws java.sql.SQLException if any.
	 */
	public static void setIntegrityConstraints(Connection connection,
			boolean enableIntegrityConstraints) throws SQLException {
		String jdbcUrl = connection.getMetaData().getURL();

		String sql = null;
		// if HSQLDB
		if (isHsqldbDatabase(jdbcUrl)) {
			Version hsqldbVersion = getDatabaseVersion(connection);

			// 1.8 :
			if ("1.8".equals(hsqldbVersion.toString())) {
				sql = "SET REFERENTIAL_INTEGRITY %s";
			}

			// 2.x :
			else {
				sql = "SET DATABASE REFERENTIAL INTEGRITY %s";
			}
			sql = String.format(sql, enableIntegrityConstraints
					? "TRUE"
					: "FALSE");
		}

		// Oracle, etc : not supported operation
		else {
			throw new SynchroTechnicalException(
					String.format(
							"Could not disable integrity constraints on database: %s. Not implemented for this DBMS.",
							jdbcUrl));
		}

		sqlUpdate(connection, sql);
	}

	/**
	 * <p>getCurrentTimestamp.</p>
	 *
	 * @param connectionProperties a {@link java.util.Properties} object.
	 * @return a {@link java.sql.Timestamp} object.
	 */
	public static Timestamp getCurrentTimestamp(Properties connectionProperties) {

		Dialect dialect = Dialect.getDialect(connectionProperties);
		Preconditions
				.checkNotNull(dialect,
						"Could not found hibernate dialect for this connection properties.");

		Connection connection = null;
		try {
			connection = createConnection(connectionProperties);

			return getCurrentTimestamp(connection, dialect);
		} catch (SQLException e) {
			throw new SynchroTechnicalException(
					"Could not connect to given database", e);
		} finally {
			Daos.closeSilently(connection);
		}

	}

	/**
	 * <p>getCurrentTimestamp.</p>
	 *
	 * @param connection a {@link java.sql.Connection} object.
	 * @param dialect a {@link org.hibernate.dialect.Dialect} object.
	 * @return a {@link java.sql.Timestamp} object.
	 */
	public static Timestamp getCurrentTimestamp(Connection connection,
			Dialect dialect) {
		String sql = dialect.getCurrentTimestampSelectString();

		Statement statement = null;
		ResultSet rs = null;
		try {

			statement = connection.createStatement();
			rs = statement.executeQuery(sql);
			rs.next();
			Timestamp result = rs.getTimestamp(1);
			return result;
		} catch (SQLException e) {
			throw new SynchroTechnicalException(
					"Error while executing function SYSTIMESTAMP query.", e);
		} finally {
			Daos.closeSilently(rs);
			Daos.closeSilently(statement);
		}

	}

	/**
	 * the value 0 if the two Timestamp objects are equal; a value less than 0
	 * if t1 object is before t2; and a value greater than 0 if t1 object is
	 * after t2.
	 *
	 * @param t1 a {@link java.sql.Timestamp} object.
	 * @param t2 a {@link java.sql.Timestamp} object.
	 * @return a int.
	 */
	public static int compareUpdateDates(Timestamp t1, Timestamp t2) {
		if (t1 == null && t2 == null) {
			return 0;
		}
		// t1 after t2
		if (t2 == null) {
			return 1;
		}
		// t1 before t2
		if (t1 == null) {
			return -1;
		}
		return t1.compareTo(t2);
	}

	/**
	 * <p>isUpdateDateBefore.</p>
	 *
	 * @param t1 a {@link java.sql.Timestamp} object.
	 * @param t2 a {@link java.sql.Timestamp} object.
	 * @return a boolean.
	 */
	public static boolean isUpdateDateBefore(Timestamp t1, Timestamp t2) {
		return compareUpdateDates(t1, t2) < 0;
	}

	/**
	 * <p>getDatabaseVersion.</p>
	 *
	 * @param connection a {@link java.sql.Connection} object.
	 * @return a {@link org.nuiton.version.Version} object.
	 * @throws java.sql.SQLException if any.
	 */
	public static Version getDatabaseVersion(Connection connection)
			throws SQLException {
		int majorVersion = connection.getMetaData().getDatabaseMajorVersion();
		int minorVersion = connection.getMetaData().getDatabaseMinorVersion();
		return Versions.valueOf(String.format("%d.%d", majorVersion,
				minorVersion));
	}

	/* -- protected -- */

}
