package fr.ifremer.common.synchro.config;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import static org.nuiton.i18n.I18n.t;

import java.io.File;
import java.util.Locale;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.config.ApplicationConfig;
import org.nuiton.config.ApplicationConfigHelper;
import org.nuiton.config.ApplicationConfigProvider;
import org.nuiton.config.ArgumentsParserException;

import com.google.common.base.Charsets;

import fr.ifremer.common.synchro.SynchroTechnicalException;
import fr.ifremer.common.synchro.dao.Daos;

/**
 * <p>SynchroConfiguration class.</p>
 *
 */
public class SynchroConfiguration {
	/** Logger. */
	private static final Log log = LogFactory
			.getLog(SynchroConfiguration.class);

	/**
	 * Delegate application config.
	 */
	protected final ApplicationConfig applicationConfig;

	private static SynchroConfiguration instance;

	/**
	 * <p>Getter for the field <code>instance</code>.</p>
	 *
	 * @return a {@link fr.ifremer.common.synchro.config.SynchroConfiguration} object.
	 */
	public static SynchroConfiguration getInstance() {
		return instance;
	}

	/**
	 * <p>Setter for the field <code>instance</code>.</p>
	 *
	 * @param instance a {@link fr.ifremer.common.synchro.config.SynchroConfiguration} object.
	 */
	public static void setInstance(SynchroConfiguration instance) {
		SynchroConfiguration.instance = instance;
	}

	protected final String[] optionKeyToNotSave;

	protected File configFile;

	/**
	 * <p>Constructor for SynchroConfiguration.</p>
	 *
	 * @param applicationConfig a {@link org.nuiton.config.ApplicationConfig} object.
	 */
	public SynchroConfiguration(ApplicationConfig applicationConfig) {
		super();
		this.applicationConfig = applicationConfig;
		this.optionKeyToNotSave = null;
	}

	/**
	 * <p>Constructor for SynchroConfiguration.</p>
	 *
	 * @param file a {@link java.lang.String} object.
	 * @param args a {@link java.lang.String} object.
	 */
	public SynchroConfiguration(String file, String... args) {
		super();
		this.applicationConfig = new ApplicationConfig();
		this.applicationConfig.setEncoding(Charsets.UTF_8.name());
		this.applicationConfig.setConfigFileName(file);

		// get all config providers
		Set<ApplicationConfigProvider> providers = ApplicationConfigHelper
				.getProviders(null, null, null, true);

		// load all default options
		ApplicationConfigHelper.loadAllDefaultOption(applicationConfig,
				providers);

		// Load actions
		for (ApplicationConfigProvider provider : providers) {
			applicationConfig.loadActions(provider.getActions());
		}

		// get all transient and final option keys
		Set<String> optionToSkip = ApplicationConfigHelper
				.getTransientOptionKeys(providers);

		if (log.isDebugEnabled()) {
			log.debug("Option that won't be saved: " + optionToSkip);
		}
		optionKeyToNotSave = optionToSkip.toArray(new String[optionToSkip
				.size()]);

		try {
			applicationConfig.parse(args);

		} catch (ArgumentsParserException e) {
			throw new SynchroTechnicalException(t("synchro.config.parse.error"),
					e);
		}

		File allegroBasedir = applicationConfig
				.getOptionAsFile(SynchroConfigurationOption.BASEDIR.getKey());

		if (allegroBasedir == null) {
			allegroBasedir = new File("");
		}
		if (!allegroBasedir.isAbsolute()) {
			allegroBasedir = new File(allegroBasedir.getAbsolutePath());
		}
		if (allegroBasedir.getName().equals("..")) {
			allegroBasedir = allegroBasedir.getParentFile().getParentFile();
		}
		if (allegroBasedir.getName().equals(".")) {
			allegroBasedir = allegroBasedir.getParentFile();
		}
		if (log.isInfoEnabled()) {
			log.info("Application basedir: " + allegroBasedir);
		}
		applicationConfig.setOption(
				SynchroConfigurationOption.BASEDIR.getKey(),
				allegroBasedir.getAbsolutePath());
	}

	/**
	 * <p>Getter for the field <code>configFile</code>.</p>
	 *
	 * @return a {@link java.io.File} object.
	 */
	public File getConfigFile() {
		if (configFile == null) {
			File dir = getBasedir();
			if (dir == null || !dir.exists()) {
				dir = new File(applicationConfig.getUserConfigDirectory());
			}
			configFile = new File(dir, applicationConfig.getConfigFileName());
		}
		return configFile;
	}

	/**
	 * <p>getBasedir.</p>
	 *
	 * @return a {@link java.io.File} object.
	 */
	public File getBasedir() {
		File result = applicationConfig
				.getOptionAsFile(SynchroConfigurationOption.BASEDIR.getKey());
		return result;
	}

	/**
	 * <p>getDataDirectory.</p>
	 *
	 * @return a {@link java.io.File} object.
	 */
	public File getDataDirectory() {
		File result = applicationConfig
				.getOptionAsFile(SynchroConfigurationOption.DATA_DIRECTORY
						.getKey());
		return result;
	}

	/**
	 * <p>Getter for the field <code>applicationConfig</code>.</p>
	 *
	 * @return a {@link org.nuiton.config.ApplicationConfig} object.
	 */
	public ApplicationConfig getApplicationConfig() {
		return applicationConfig;
	}

	/**
	 * <p>getDbDirectory.</p>
	 *
	 * @return a {@link java.io.File} object.
	 */
	public File getDbDirectory() {
		return applicationConfig
				.getOptionAsFile(SynchroConfigurationOption.DB_DIRECTORY
						.getKey());
	}

	/**
	 * <p>setDbDirectory.</p>
	 *
	 * @param dbDirectory a {@link java.io.File} object.
	 */
	public void setDbDirectory(File dbDirectory) {
		applicationConfig.setOption(
				SynchroConfigurationOption.DB_DIRECTORY.getKey(),
				dbDirectory.getPath());
	}

	/**
	 * <p>getDbAttachmentDirectory.</p>
	 *
	 * @return a {@link java.io.File} object.
	 */
	public File getDbAttachmentDirectory() {
		return applicationConfig
				.getOptionAsFile(SynchroConfigurationOption.DB_ATTACHMENT_DIRECTORY
						.getKey());
	}

	/**
	 * <p>getCacheDirectory.</p>
	 *
	 * @return a {@link java.io.File} object.
	 */
	public File getCacheDirectory() {
		return applicationConfig
				.getOptionAsFile(SynchroConfigurationOption.DB_CACHE_DIRECTORY
						.getKey());
	}

	/**
	 * <p>getDbBackupDirectory.</p>
	 *
	 * @return a {@link java.io.File} object.
	 */
	public File getDbBackupDirectory() {
		return applicationConfig
				.getOptionAsFile(SynchroConfigurationOption.DB_BACKUP_DIRECTORY
						.getKey());
	}

	/**
	 * <p>getHibernateDialect.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getHibernateDialect() {
		return applicationConfig
				.getOption(SynchroConfigurationOption.HIBERNATE_DIALECT
						.getKey());
	}

	/**
	 * <p>getJdbcDriver.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getJdbcDriver() {
		return applicationConfig
				.getOption(SynchroConfigurationOption.JDBC_DRIVER.getKey());
	}

	/**
	 * <p>getJdbcURL.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getJdbcURL() {
		return applicationConfig.getOption(SynchroConfigurationOption.JDBC_URL
				.getKey());
	}

	/**
	 * <p>getDbName.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getDbName() {
		return applicationConfig.getOption(SynchroConfigurationOption.DB_NAME
				.getKey());
	}

	/**
	 * <p>getDbValidationQuery.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getDbValidationQuery() {
		return applicationConfig
				.getOption(SynchroConfigurationOption.DB_VALIDATION_QUERY
						.getKey());
	}

	/**
	 * <p>getJdbcUsername.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getJdbcUsername() {
		return applicationConfig
				.getOption(SynchroConfigurationOption.JDBC_USERNAME.getKey());
	}

	/**
	 * <p>getJdbcPassword.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getJdbcPassword() {
		return applicationConfig
				.getOption(SynchroConfigurationOption.JDBC_PASSWORD.getKey());
	}

	/**
	 * <p>getJdbcSchema.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getJdbcSchema() {
		return applicationConfig
				.getOption(SynchroConfigurationOption.JDBC_SCHEMA.getKey());
	}

	/**
	 * <p>getJdbcCatalog.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getJdbcCatalog() {
		return applicationConfig
				.getOption(SynchroConfigurationOption.JDBC_CATALOG.getKey());
	}

	/**
	 * <p>getI18nDirectory.</p>
	 *
	 * @return a {@link java.io.File} object.
	 */
	public File getI18nDirectory() {
		return applicationConfig
				.getOptionAsFile(SynchroConfigurationOption.I18N_DIRECTORY
						.getKey());
	}

	/**
	 * <p>getI18nLocale.</p>
	 *
	 * @return a {@link java.util.Locale} object.
	 */
	public Locale getI18nLocale() {
		return applicationConfig
				.getOptionAsLocale(SynchroConfigurationOption.I18N_LOCALE
						.getKey());
	}

	/**
	 * <p>setI18nLocale.</p>
	 *
	 * @param locale a {@link java.util.Locale} object.
	 */
	public void setI18nLocale(Locale locale) {
		applicationConfig.setOption(
				SynchroConfigurationOption.I18N_LOCALE.getKey(),
				locale.toString());
	}

	/**
	 * <p>getImportJdbcURL.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getImportJdbcURL() {
		return applicationConfig
				.getOption(SynchroConfigurationOption.IMPORT_JDBC_URL.getKey());
	}

	/**
	 * <p>getImportJdbcUsername.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getImportJdbcUsername() {
		return applicationConfig
				.getOption(SynchroConfigurationOption.IMPORT_JDBC_USERNAME
						.getKey());
	}

	/**
	 * <p>getImportJdbcPassword.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getImportJdbcPassword() {
		return applicationConfig
				.getOption(SynchroConfigurationOption.IMPORT_JDBC_PASSWORD
						.getKey());
	}

	/**
	 * <p>getImportJdbcSchema.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getImportJdbcSchema() {
		return applicationConfig
				.getOption(SynchroConfigurationOption.IMPORT_JDBC_SCHEMA
						.getKey());
	}

	/**
	 * <p>getImportJdbcCatalog.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getImportJdbcCatalog() {
		return applicationConfig
				.getOption(SynchroConfigurationOption.IMPORT_JDBC_CATALOG
						.getKey());
	}

	/**
	 * <p>getImportJdbcDriver.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getImportJdbcDriver() {
		return applicationConfig
				.getOption(SynchroConfigurationOption.IMPORT_JDBC_DRIVER
						.getKey());
	}

	/**
	 * <p>getImportJdbcBatchSize.</p>
	 *
	 * @return a int.
	 */
	public int getImportJdbcBatchSize() {
		// If batch disable, force to batch-size=1
		if (!isImportJdbcBatchEnable()) {
			return 1;
		}

		int importJdbcBatchSize = applicationConfig
				.getOptionAsInt(SynchroConfigurationOption.IMPORT_JDBC_BATCH_SIZE
						.getKey());

		// If negative value, use the default value
		if (importJdbcBatchSize <= 0) {
			importJdbcBatchSize = Integer
					.parseInt(SynchroConfigurationOption.IMPORT_JDBC_BATCH_SIZE
							.getDefaultValue());
			log.warn(String
					.format("Invalid batch size [%s] on configuration property [%s]. Will use default value [%s]",
							importJdbcBatchSize,
							SynchroConfigurationOption.IMPORT_JDBC_BATCH_SIZE
									.getKey(), importJdbcBatchSize));
		}
		return importJdbcBatchSize;
	}

	/**
	 * <p>getImportJdbcFetchSize.</p>
	 *
	 * @return a int.
	 */
	public int getImportJdbcFetchSize() {
		return applicationConfig
				.getOptionAsInt(SynchroConfigurationOption.IMPORT_JDBC_FETCH_SIZE
						.getKey());
	}

	/**
	 * <p>isImportJdbcBatchEnable.</p>
	 *
	 * @return a boolean.
	 */
	public boolean isImportJdbcBatchEnable() {
		return applicationConfig
				.getOptionAsBoolean(SynchroConfigurationOption.IMPORT_JDBC_BATCH_ENABLE
						.getKey());
	}

	/**
	 * <p>getImportHibernateDialect.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getImportHibernateDialect() {
		return applicationConfig
				.getOption(SynchroConfigurationOption.IMPORT_HIBERNATE_DIALECT
						.getKey());
	}

	/**
	 * <p>getTempQueryParemeterUserIdColumn.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getTempQueryParemeterUserIdColumn() {
		return applicationConfig
				.getOption(SynchroConfigurationOption.TEMP_QUERY_PARAMETER_USER_ID
						.getKey());
	}

	/**
	 * <p>getConnectionProperties.</p>
	 *
	 * @return a {@link java.util.Properties} object.
	 */
	public Properties getConnectionProperties() {
		return Daos.getConnectionProperties(getJdbcURL(), getJdbcUsername(),
				getJdbcPassword(), getJdbcCatalog(), getJdbcSchema(),
				getHibernateDialect(), getJdbcDriver());
	}

	/**
	 * <p>getImportConnectionProperties.</p>
	 *
	 * @return a {@link java.util.Properties} object.
	 */
	public Properties getImportConnectionProperties() {
		return Daos.getConnectionProperties(getImportJdbcURL(),
				getImportJdbcUsername(), getImportJdbcPassword(),
				getImportJdbcCatalog(), getImportJdbcSchema(),
				getImportHibernateDialect(), getImportJdbcDriver());
	}

	/**
	 * <p>getColumnId.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getColumnId() {
		return applicationConfig
				.getOption(SynchroConfigurationOption.DB_COLUMN_ID.getKey());
	}

	/**
	 * <p>getColumnUpdateDate.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getColumnUpdateDate() {
		return applicationConfig
				.getOption(SynchroConfigurationOption.DB_COLUMN_UPDATE_DATE
						.getKey());
	}

	/**
	 * <p>getSequenceSuffix.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String getSequenceSuffix() {
		return applicationConfig
				.getOption(SynchroConfigurationOption.DB_SEQUENCE_SUFFIX
						.getKey());
	}

	/**
	 * <p>getMaxSqlNameLength.</p>
	 *
	 * @return a int.
	 */
	public int getMaxSqlNameLength() {
		return applicationConfig
				.getOptionAsInt(SynchroConfigurationOption.DB_MAX_SQL_NAME_LENGTH
						.getKey());
	}
}
