package fr.ifremer.adagio.synchro.service;

/*
 * #%L
 * Tutti :: Persistence API
 * $Id: ReferentialSynchroResult.java 1486 2014-01-15 08:43:26Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/trunk/tutti-persistence/src/main/java/fr/ifremer/adagio/core/service/technical/synchro/ReferentialSynchroResult.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.sql.Timestamp;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Joiner;
import com.google.common.base.Objects;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;

import fr.ifremer.adagio.synchro.SynchroTechnicalException;
import fr.ifremer.adagio.synchro.dao.Daos;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata;
import fr.ifremer.adagio.synchro.type.ProgressionModel;

/**
 * Result of a referential synchronize operation.
 * 
 * @author tchemit <chemit@codelutin.com>
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since 3.5.2
 */
public class SynchroResult {

	protected Exception error;

	/**
	 * Number of rows detected to update (per table).
	 * 
	 * @since 1.0
	 */
	protected final Map<String, Integer> rowHits = Maps.newTreeMap();

	/**
	 * Number of insert done (per table).
	 * 
	 * @since 1.0
	 */
	protected final Map<String, Integer> insertHits = Maps.newTreeMap();

	/**
	 * Number of update done (per table).
	 * 
	 * @since 1.0
	 */
	protected final Map<String, Integer> updateHits = Maps.newTreeMap();

	/**
	 * Number of delete done (per table).
	 * 
	 * @since 3.7.0
	 */
	protected final Map<String, Integer> deletesHits = Maps.newTreeMap();

	/**
	 * timestamp of last update date (per table).
	 * 
	 * @since 1.0
	 */
	protected final Map<String, Timestamp> updateDateHits = Maps.newTreeMap();

	/**
	 * rejected rows, as a CSV string.
	 * <p/>
	 * i.e. : [PK_AS_STR];[REJECT_STATUS];[UPDATE_DATE_NANOS_TIME] 25364~~~1;BAD_UPDATE_DATE;12437789901254
	 * 
	 * @since 3.7
	 */
	protected final Map<String, String> rejectedRows = Maps.newTreeMap();

	/**
	 * missing updates on source DB, by tableName, and source pkStr
	 * 
	 * @since 3.7
	 */
	protected final Map<String, Map<String, Map<String, Object>>> sourceMissingUpdates = Maps.newIdentityHashMap();

	/**
	 * PK str that need to be reverted (new importation)
	 * 
	 * @since 3.7.2
	 */
	protected final Multimap<String, String> sourceMissingReverts = ArrayListMultimap.create();

	/**
	 * PK str that need to be deleted
	 * 
	 * @since 3.7.2
	 */
	protected final Multimap<String, String> sourceMissingDeletes = ArrayListMultimap.create();

	/**
	 * All table treated.
	 * 
	 * @since 1.0
	 */
	protected final Set<String> tableNames = Sets.newHashSet();

	protected String targetUrl;

	protected String sourceUrl;

	/**
	 * Transient, because not need to serialize it
	 */
	protected transient final ProgressionModel progressionModel = new ProgressionModel(this);

	public SynchroResult() {
	}

	public SynchroResult(String targetUrl, String sourceUrl) {
		this.targetUrl = targetUrl;
		this.sourceUrl = sourceUrl;
	}

	public void setLocalUrl(String targetUrl) {
		this.targetUrl = targetUrl;
	}

	public void setRemoteUrl(String sourceUrl) {
		this.sourceUrl = sourceUrl;
	}

	public boolean isSuccess() {
		return error == null;
	}

	public Exception getError() {
		return error;
	}

	public void setError(Exception error) {
		this.error = error;
	}

	public ProgressionModel getProgressionModel() {
		return progressionModel;
	}

	public Set<String> getTableNames() {
		return ImmutableSet.copyOf(tableNames);
	}

	public int getTotalRows() {
		int result = 0;
		for (Integer nb : rowHits.values()) {
			result += nb;
		}
		return result;
	}

	public int getTotalInserts() {
		int result = 0;
		for (Integer nb : insertHits.values()) {
			result += nb;
		}
		return result;
	}

	public int getTotalUpdates() {
		int result = 0;
		for (Integer nb : updateHits.values()) {
			result += nb;
		}
		return result;
	}

	public int getTotalDeletes() {
		int result = 0;
		for (Integer nb : deletesHits.values()) {
			result += nb;
		}
		return result;
	}

	public int getTotalRejects() {
		int result = 0;
		for (String rows : rejectedRows.values()) {
			result += StringUtils.countMatches(rows, "\n");
		}
		return result;
	}

	public int getTotalTreated() {
		return getTotalInserts() + getTotalUpdates() + getTotalDeletes() + getTotalRejects();
	}

	public int getNbRows(String tableName) {
		Integer result = rowHits.get(tableName);
		if (result == null) {
			result = 0;
		}
		return result;
	}

	public int getNbInserts(String tableName) {
		Integer result = insertHits.get(tableName);
		if (result == null) {
			result = 0;
		}
		return result;
	}

	public int getNbUpdates(String tableName) {
		Integer result = updateHits.get(tableName);
		if (result == null) {
			result = 0;
		}
		return result;
	}

	public int getNbDeletes(String tableName) {
		Integer result = deletesHits.get(tableName);
		if (result == null) {
			result = 0;
		}
		return result;
	}

	public String getRejectedRows(String tableName) {
		String result = rejectedRows.get(tableName);
		if (result == null) {
			return "";
		}
		return result;
	}

	public void addRows(String tableName, int nb) {
		if (nb > 0) {
			rowHits.put(tableName, getNbRows(tableName) + nb);
		}
	}

	public void addUpdates(String tableName, int nb) {
		if (nb > 0) {
			updateHits.put(tableName, getNbUpdates(tableName) + nb);
		}
	}

	public void addInserts(String tableName, int nb) {
		if (nb > 0) {
			insertHits.put(tableName, getNbInserts(tableName) + nb);
		}
	}

	public void addDeletes(String tableName, int nb) {
		if (nb > 0) {
			deletesHits.put(tableName, getNbDeletes(tableName) + nb);
		}
	}

	public void addReject(String tableName, String... rowInfo) {
		StringBuilder rows = new StringBuilder(getRejectedRows(tableName));
		rejectedRows.put(tableName,
				Joiner.on(';').appendTo(rows, rowInfo).append('\n').toString());
	}

	public void addSourceMissingColumnUpdate(String tableName, String columnName, List<Object> pk, Object columnValue) {
		addSourceMissingColumnUpdate(
				tableName,
				columnName,
				SynchroTableMetadata.toPkStr(pk),
				columnValue);
	}

	public void addSourceMissingColumnUpdate(String tableName, String columnName, String pkStr, Object columnValue) {
		Map<String, Map<String, Object>> tableMissingSourceUpdates = sourceMissingUpdates.get(tableName);
		if (tableMissingSourceUpdates == null) {
			tableMissingSourceUpdates = Maps.newHashMap();
			sourceMissingUpdates.put(tableName, tableMissingSourceUpdates);
		}
		Map<String, Object> destRows = tableMissingSourceUpdates.get(columnName);
		if (destRows == null) {
			destRows = Maps.newHashMap();
			tableMissingSourceUpdates.put(columnName, destRows);
		}
		if (destRows.containsKey(pkStr)) {
			Object oldValue = destRows.get(pkStr);
			if (!Objects.equal(oldValue, columnValue)) {
				throw new SynchroTechnicalException(
						String.format(
								"Could not update a row column twice, with two differents values: table [%s] pk [%s] - existing %s: [%s] new: [%s]",
								tableName,
								pkStr,
								columnName,
								oldValue,
								columnValue
								));
			}
		}
		destRows.put(pkStr, columnValue);
	}

	public Timestamp getUpdateDate(String tableName) {
		return updateDateHits.get(tableName);
	}

	public void setUpdateDate(String tableName, Timestamp t) {
		updateDateHits.put(tableName, t);
	}

	public Map<String, Timestamp> getUpdateDateHits() {
		return updateDateHits;
	}

	public Map<String, String> getRejectedRows() {
		return rejectedRows;
	}

	public Map<String, Map<String, Map<String, Object>>> getSourceMissingUpdates() {
		return sourceMissingUpdates;
	}

	public void addTableName(String tableName) {
		tableNames.add(tableName);
	}

	public String getLocalUrl() {
		return targetUrl;
	}

	public String getRemoteUrl() {
		return sourceUrl;
	}

	public void addSourceMissingRevert(String tableName, String pkStr) {
		sourceMissingReverts.put(tableName, pkStr);
	}

	public Multimap<String, String> getSourceMissingReverts() {
		return sourceMissingReverts;
	}

	public void addSourceMissingDelete(String tableName, String pkStr) {
		sourceMissingDeletes.put(tableName, pkStr);
	}

	public Multimap<String, String> getSourceMissingDeletes() {
		return sourceMissingDeletes;
	}

	/**
	 * Clear every fields
	 */
	public void clear() {
		rowHits.clear();
		insertHits.clear();
		updateHits.clear();
		deletesHits.clear();
		updateDateHits.clear();
		rejectedRows.clear();
		sourceMissingUpdates.clear();
		sourceMissingReverts.clear();
		sourceMissingDeletes.clear();
		error = null;
	}

	public void addAll(SynchroResult anotherResult) {

		// row hits
		for (Entry<String, Integer> entry : anotherResult.rowHits.entrySet()) {
			addRows(entry.getKey(), entry.getValue());
		}

		for (Entry<String, Integer> entry : anotherResult.insertHits.entrySet()) {
			addInserts(entry.getKey(), entry.getValue());
		}
		for (Entry<String, Integer> entry : anotherResult.updateHits.entrySet()) {
			addUpdates(entry.getKey(), entry.getValue());
		}
		for (Entry<String, Integer> entry : anotherResult.deletesHits.entrySet()) {
			addDeletes(entry.getKey(), entry.getValue());
		}
		for (String tableName : anotherResult.updateDateHits.keySet()) {
			Timestamp newUpdateDate = anotherResult.updateDateHits.get(tableName);
			Timestamp previousUpdateDate = getUpdateDate(tableName);
			if (Daos.isUpdateDateBefore(previousUpdateDate, newUpdateDate)) {
				setUpdateDate(tableName, newUpdateDate);
			}
		}
		for (Entry<String, String> entry : anotherResult.rejectedRows.entrySet()) {
			addReject(entry.getKey(), entry.getValue());
		}
		for (String tableName : anotherResult.sourceMissingUpdates.keySet()) {
			Map<String, Map<String, Object>> tableMissingSourceUpdates = anotherResult.sourceMissingUpdates.get(tableName);
			for (String columnName : tableMissingSourceUpdates.keySet()) {
				Map<String, Object> destRows = tableMissingSourceUpdates.get(columnName);
				for (String pkStr : destRows.keySet()) {
					Object columnValue = destRows.get(pkStr);
					addSourceMissingColumnUpdate(tableName, columnName, pkStr, columnValue);
				}
			}
		}
		sourceMissingReverts.putAll(anotherResult.sourceMissingReverts);
		sourceMissingDeletes.putAll(anotherResult.sourceMissingDeletes);
	}

}
