package fr.ifremer.adagio.synchro.service;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.sql.Timestamp;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;

import fr.ifremer.adagio.synchro.SynchroTechnicalException;
import fr.ifremer.adagio.synchro.config.SynchroConfiguration;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;

/**
 * A context class, need for referential and data synchronization
 * Created on 5/05/14.
 * 
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since 3.5.3
 */
public class SynchroContext {

	protected SynchroDatabaseConfiguration sourceDatabaseConfiguration;

	protected SynchroDatabaseMetadata sourceMeta;

	protected SynchroDatabaseConfiguration targetDatabaseConfiguration;

	protected SynchroDatabaseMetadata targetMeta;

	protected SynchroResult result;

	protected Set<String> tableNames;

	protected Map<String, SynchroTableOperation> tableContextMap;

	protected Timestamp lastSynchronizationDate;

	/**
	 * Create a new synchro context
	 * 
	 * @param tableNames
	 *            Table set to includes. If null, all tables will be processed by table filter
	 * @param sourceConnectionProperties
	 *            Connection properties for source database
	 * @param targetConnectionProperties
	 *            Connection properties for target database
	 * @param result
	 *            Store the synchronization result
	 * @return a new synchro context
	 */
	public static SynchroContext newContext(
			Set<String> tableNames,
			Properties sourceConnectionProperties,
			Properties targetConnectionProperties,
			SynchroResult result) {
		Preconditions.checkNotNull(sourceConnectionProperties);
		Preconditions.checkNotNull(targetConnectionProperties);
		SynchroConfiguration config = SynchroConfiguration.getInstance();

		return newContext(tableNames,
				new SynchroDatabaseConfiguration(config, sourceConnectionProperties, false),
				new SynchroDatabaseConfiguration(config, targetConnectionProperties, true),
				result);
	}

	/**
	 * Create a new synchro context
	 * 
	 * @param tableNames
	 *            Table set to includes. If null, all tables will be processed by table filter
	 * @param source
	 *            Connection configuration for source database
	 * @param target
	 *            Connection configuration for target database
	 * @param result
	 *            Store the synchronization result
	 * @return a new synchro context
	 */
	public static SynchroContext newContext(
			Set<String> tableNames,
			SynchroDatabaseConfiguration source,
			SynchroDatabaseConfiguration target,
			SynchroResult result) {
		Preconditions.checkNotNull(source);
		Preconditions.checkNotNull(target);

		SynchroContext context = new SynchroContext();
		context.setTableNames(tableNames);
		context.setSource(source);
		context.setTarget(target);
		context.setResult(result);
		return context;
	}

	/* -- non static methods-- */

	public SynchroContext() {
		this.tableContextMap = Maps.newHashMap();
	}

	public SynchroContext(SynchroContext otherBean) {
		copy(otherBean);
	}

	public SynchroResult getResult() {
		return result;
	}

	public void setResult(SynchroResult result) {
		this.result = result;
	}

	public void setSource(SynchroDatabaseConfiguration sourceSettings) {
		this.sourceDatabaseConfiguration = sourceSettings;
	}

	public void setTarget(SynchroDatabaseConfiguration targetSettings) {
		this.targetDatabaseConfiguration = targetSettings;
	}

	public void setTableNames(Set<String> tableNames) {
		this.tableNames = tableNames;
	}

	public SynchroDatabaseConfiguration getSource() {
		return sourceDatabaseConfiguration;
	}

	public SynchroDatabaseConfiguration getTarget() {
		return targetDatabaseConfiguration;
	}

	public Set<String> getTableNames() {
		return this.tableNames;
	}

	public Timestamp getLastSynchronizationDate() {
		return lastSynchronizationDate;
	}

	public void setLastSynchronizationDate(Timestamp lastSynchronizationDate) {
		this.lastSynchronizationDate = lastSynchronizationDate;
	}

	public SynchroDatabaseMetadata getSourceMeta() {
		return sourceMeta;
	}

	public void setSourceMeta(SynchroDatabaseMetadata sourceMeta) {
		this.sourceMeta = sourceMeta;
	}

	public SynchroDatabaseMetadata getTargetMeta() {
		return targetMeta;
	}

	public void setTargetMeta(SynchroDatabaseMetadata targetMeta) {
		this.targetMeta = targetMeta;
	}

	public SynchroDatabaseMetadata getMeta(SynchroDatabaseConfiguration configuration) {
		Preconditions.checkNotNull(configuration);

		if (configuration == targetDatabaseConfiguration) {
			return targetMeta;
		}
		return sourceMeta;
	}

	/**
	 * Return the metadata to use for synchronization. By default, the target metadata.
	 * This could be the source is targetConfiguration.isQueryMetadataEnable() == false
	 * 
	 * @return
	 */
	public SynchroDatabaseMetadata getMetaWithQueryEnable() {
		Preconditions.checkNotNull(targetDatabaseConfiguration);
		Preconditions.checkNotNull(sourceDatabaseConfiguration);
		Preconditions.checkNotNull(targetMeta);
		Preconditions.checkNotNull(sourceMeta);
		Preconditions.checkArgument(targetDatabaseConfiguration.isFullMetadataEnable() || sourceDatabaseConfiguration.isFullMetadataEnable());

		if (targetDatabaseConfiguration.isFullMetadataEnable()) {
			return targetMeta;
		}
		return sourceMeta;
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder("synchronisation context:");
		if (getSource() != null) {
			sb.append("\n  ").append(getSource().toString());
		}
		if (getTarget() != null) {
			sb.append("\n  ").append(getTarget().toString());
		}
		sb.append("\n  table includes: ");
		if (getTableNames() != null) {
			sb.append(Joiner.on(',').join(getTableNames()));
		}
		sb.append("\n  last synchronization date: ").append(getLastSynchronizationDate());
		return sb.toString();
	}

	public void copy(SynchroContext otherBean) {
		this.tableContextMap = otherBean.tableContextMap;
		this.result = otherBean.result;
		this.sourceDatabaseConfiguration = otherBean.sourceDatabaseConfiguration;
		this.sourceMeta = otherBean.sourceMeta;
		this.targetDatabaseConfiguration = otherBean.targetDatabaseConfiguration;
		this.targetMeta = otherBean.targetMeta;
		this.tableNames = otherBean.tableNames;
		this.tableContextMap = otherBean.tableContextMap;
		this.lastSynchronizationDate = otherBean.lastSynchronizationDate;
	}
}
