package fr.ifremer.adagio.synchro.query;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.util.Arrays;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;

import fr.ifremer.adagio.synchro.query.internal.SynchroAbstractQuery;
import fr.ifremer.adagio.synchro.query.internal.SynchroDeleteQuery;
import fr.ifremer.adagio.synchro.query.internal.SynchroInsertQuery;
import fr.ifremer.adagio.synchro.query.internal.SynchroSelectQuery;
import fr.ifremer.adagio.synchro.query.internal.SynchroUpdateQuery;

public abstract class SynchroQueryBuilder {

	public static final String COLUMN_VALUE_TABLE_SEQUENCE = "~~sequence~~";

	protected static final String ALIAS_VAR = "~~alias~~";
	protected static final String NO_ALIAS_STR = "";

	private static Pattern SELECT_FROM_PATTERN = Pattern.compile("^[\\s]*SELECT(.*)FROM(.+)");
	private static Pattern WHERE_PATTERN = Pattern.compile("^(.*)WHERE(.+)");
	private static Pattern COLUMN_WITH_ALIAS_PATTERN = Pattern.compile("([a-zA-Z0-9_]+)[.]([a-zA-Z0-9_]+)");

	public static SynchroQueryBuilder newBuilder(String sql) {
		return newBuilder(null, sql);
	}

	public static SynchroQueryBuilder newBuilder(SynchroQueryName queryName, String sql) {
		Preconditions.checkNotNull(sql);

		String upperCaseQuery = sql.toUpperCase();

		SynchroAbstractQuery query = null;
		if (upperCaseQuery.startsWith("SELECT")) {
			query = parseSelectQuery(queryName, sql);
		}
		if (upperCaseQuery.startsWith("INSERT")) {
			query = parseInsertQuery(queryName, sql);
		}
		if (upperCaseQuery.startsWith("UPDATE")) {
			query = parseUpdateQuery(queryName, sql);
		}
		if (upperCaseQuery.startsWith("DELETE")) {
			query = parseDeleteQuery(queryName, sql);
		}
		if (query == null) {
			throw new IllegalArgumentException(String.format("Unable to parse sql: %s", sql));
		}

		return query;
	}

	protected static SynchroAbstractQuery parseSelectQuery(SynchroQueryName queryName, String sql) {

		Matcher matcher = SELECT_FROM_PATTERN.matcher(sql);
		if (!matcher.matches()) {
			throw new IllegalArgumentException(
					String.format("Bad sql query: [%s]. Expected: SELECT <...> FROM <...> WHERE <...>", sql));
		}

		String selectClause = matcher.group(1).trim();
		String fromClause = matcher.group(2).trim();

		matcher = WHERE_PATTERN.matcher(fromClause);
		String whereClause = null;
		if (matcher.matches()) {
			fromClause = matcher.group(1).trim();
			whereClause = matcher.group(2).trim();
		}

		// Select part
		List<String> columnNames = Lists.newArrayList();
		String tableAlias = null;
		boolean hasColumnDistinct = false;
		{
			if (selectClause.startsWith("DISTINCT")) {
				selectClause = selectClause.substring("DISTINCT".length()).trim();
				hasColumnDistinct = true;
			}
			Iterable<String> parts = Splitter.on(',')
					.trimResults()
					.omitEmptyStrings().split(selectClause);
			for (String columnName : parts) {
				Matcher columnAliasMatcher = COLUMN_WITH_ALIAS_PATTERN.matcher(columnName);
				if (columnAliasMatcher.matches()) {
					// Set table alias only once
					if (tableAlias == null) {
						tableAlias = columnAliasMatcher.group(1);
					}
					columnName = ALIAS_VAR + columnAliasMatcher.group(2);
				}
				columnNames.add(columnName);
			}
		}

		// From part
		String tableName;
		{
			String[] parts = fromClause.split("[ \t]+");
			if (parts.length == 2) {
				tableName = parts[0];
				tableAlias = parts[1];
				fromClause = null;
			}
			else {
				tableName = null;
			}
		}

		SynchroAbstractQuery selectQuery = new SynchroSelectQuery(queryName, tableName, columnNames, fromClause, whereClause);
		selectQuery.setTableAlias(tableAlias);
		selectQuery.setColumnDistinct(hasColumnDistinct);
		return selectQuery;
	}

	protected static SynchroAbstractQuery parseInsertQuery(SynchroQueryName queryName, String sql) {
		String[] parts = sql.split("(INTO)|(VALUES[ \t\\(]+)");
		if (parts.length != 2 && parts.length != 3) {
			throw new IllegalArgumentException(
					String.format("Bad sql query: [%s]. Expected: INSERT INTO <...> <...> VALUES <...>", sql));
		}
		String[] part1 = parts[1].trim().split("[ \t,\\(\\)]+");
		List<String> columnNames = Lists.newArrayList(Arrays.asList(part1));
		String tableName = columnNames.remove(0);

		// remove last parenthesis
		if (parts[2].lastIndexOf(")") == parts[2].length() - 1) {
			parts[2] = parts[2].substring(0, parts[2].length() - 1);
		}
		String[] part2 = parts[2].trim().split("[\t,\\(\\)]*,[ \t\\(\\)]*");
		List<String> columnvalues = Lists.newArrayList(Arrays.asList(part2));
		if (columnvalues.size() != columnNames.size()) {
			throw new IllegalArgumentException(
					String.format("Bad sql query: [%s]. Not same number of columns and values", sql));
		}
		SynchroAbstractQuery query = new SynchroInsertQuery(queryName, tableName, columnNames, columnvalues);
		return query;
	}

	protected static SynchroAbstractQuery parseUpdateQuery(SynchroQueryName queryName, String sql) {

		String[] parts = sql.split("(UPDATE)|(SET)|(WHERE)");
		if (parts.length != 3 && parts.length != 4) {
			throw new IllegalArgumentException(
					String.format("Bad sql query: [%s]. Expected: UPDATE <...> SET <...> WHERE <...>", sql));
		}
		String tableName = parts[1].trim();

		String[] colsAndVal = parts[2].trim().split(",");
		List<String> columnNames = Lists.newArrayListWithExpectedSize(colsAndVal.length);
		List<String> columnValues = Lists.newArrayListWithExpectedSize(colsAndVal.length);
		for (String colAndVal : colsAndVal) {
			String[] colAndValParts = colAndVal.trim().split("[ \t=]+");
			if (colAndValParts.length != 2) {
				throw new IllegalArgumentException(
						String.format("Bad sql query: [%s]. Expected: UPDATE <...> SET <...> WHERE <...>", sql));
			}
			columnNames.add(colAndValParts[0]);
			columnValues.add(colAndValParts[1]);
		}
		String whereClause = null;
		if (parts.length == 4) {
			whereClause = parts[3].trim();
		}
		SynchroUpdateQuery query = new SynchroUpdateQuery(queryName, tableName, columnNames, columnValues);
		query.setWhereClause(whereClause);
		return query;
	}

	protected static SynchroAbstractQuery parseDeleteQuery(SynchroQueryName queryName, String sql) {

		SynchroAbstractQuery selectQuery = new SynchroDeleteQuery(queryName, null);
		return selectQuery;
	}

	public String build() {
		return toSql();
	}

	/* -- public abstract method -- */

	// getters :
	public abstract String toSql();

	public abstract boolean constainsColumn(String columnName);

	public abstract int getColumnCount();

	public abstract String getColumnValue(String columnName);

	public abstract List<String> getColumnNames();

	public abstract List<String> getColumnNamesWithAlias();

	// operations :

	public abstract SynchroQueryBuilder replaceColumn(String oldColumn, String newColumn);

	public abstract SynchroQueryBuilder addColumn(String columnName, String value);

	public abstract SynchroQueryBuilder deleteColumn(String columnName);

	public abstract SynchroQueryBuilder deleteColumnIfExists(String columnName);

	public abstract SynchroQueryBuilder setColumnValue(String columnName, String value);

	public abstract SynchroQueryBuilder setWhereClause(String whereClause);

	public abstract SynchroQueryBuilder addWhere(SynchroQueryOperator operator, String whereClause);

	public abstract SynchroQueryBuilder addWhere(SynchroQueryOperator operator, String columnName, String columnValue);

	public abstract SynchroQueryBuilder addWhere(SynchroQueryOperator operator, String columnName, String columnOperator, String columnValue);

	public abstract SynchroQueryBuilder setColumnDistinct(boolean columnDistinct);

	/**
	 * Add a join (inner or outer) to the main table (in the 'FROM' clause)
	 * 
	 * @param joinClause
	 */
	public abstract SynchroQueryBuilder addJoin(String joinClause);
}
