package fr.ifremer.adagio.synchro.meta;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.sql.Types;

import org.hibernate.dialect.Dialect;
import org.hibernate.dialect.HSQLDialect;
import org.hibernate.tool.hbm2ddl.ColumnMetadata;

import com.google.common.base.Preconditions;

/**
 * Overrides of the {@link ColumnMetadata} with some improvements:
 * <ul>
 * <li>Obtains owner table name {@link #getTableName()};</li>
 * <li>Obtains linked column (for FK only) {@link #getParentJoin()} as a join metadata: used to synchronize data in
 * {@link DataSynchroService};</li>
 * </ul>
 * 
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since 3.5.2
 * 
 */
public class SynchroColumnMetadata {

	protected final ColumnMetadata delegate;

	protected final String tableName;

	protected final int columnSize;

	protected SynchroJoinMetadata parentJoin = null;

	public SynchroColumnMetadata(
			String tableName,
			ColumnMetadata delegate,
			Dialect dialect) {

		Preconditions.checkNotNull(delegate);

		this.delegate = delegate;
		this.tableName = tableName;
		this.columnSize = initColumnSize(dialect, delegate);
	}

	public SynchroColumnMetadata(
			SynchroColumnMetadata beanToCopy) {

		Preconditions.checkNotNull(beanToCopy);

		this.delegate = beanToCopy.delegate;
		this.tableName = beanToCopy.tableName;
		this.parentJoin = beanToCopy.parentJoin;
		this.columnSize = beanToCopy.columnSize;
	}

	/**
	 * Obtains owner table name
	 * 
	 * @return The owner table name
	 */
	public String getTableName() {
		return tableName;
	}

	/**
	 * Obtains linked column (for FK only)<br/>
	 * This is used to synchronize data in {@link DataSynchroService};
	 * 
	 * @return the parent join metadata
	 * @see SynchroJoinMetadata
	 */
	public SynchroJoinMetadata getParentJoin() {
		return parentJoin;
	}

	public void setParentJoin(SynchroJoinMetadata parentJoin) {
		this.parentJoin = parentJoin;
	}

	/**
	 * @see ColumnMetadata#hashCode()
	 */
	public int hashCode() {
		return delegate.hashCode();
	}

	/**
	 * @see ColumnMetadata#getName()
	 */
	public String getName() {
		return delegate.getName();
	}

	/**
	 * @see ColumnMetadata#getTypeName()
	 */
	public String getTypeName() {
		return delegate.getTypeName();
	}

	/**
	 * @see ColumnMetadata#getColumnSize()
	 */
	public int getColumnSize() {
		return columnSize;
	}

	/**
	 * @see ColumnMetadata#getDecimalDigits()
	 */
	public int getDecimalDigits() {
		return delegate.getDecimalDigits();
	}

	/**
	 * @see ColumnMetadata#getTypeCode()
	 */
	public String getNullable() {
		return delegate.getNullable();
	}

	public boolean isNullable() {
		return "YES".equalsIgnoreCase(getNullable());
	}

	/**
	 * @see ColumnMetadata#toString()
	 */
	public String toString() {
		return delegate.toString();
	}

	/**
	 * @see ColumnMetadata#getTypeCode()
	 */
	public int getTypeCode() {
		return delegate.getTypeCode();
	}

	/**
	 * @see ColumnMetadata#delegate(Object)
	 */
	public boolean equals(Object obj) {
		if (obj instanceof ColumnMetadata) {
			return delegate.equals(obj);
		}
		return super.equals(obj);
	}

	public String getSqlTypeName() {
		int size = getColumnSize();
		String typeName = getTypeName();

		if (size == 0) {
			return typeName;
		}

		StringBuilder sqlTypeName = new StringBuilder();
		sqlTypeName.append(typeName);
		sqlTypeName.append("(").append(size);

		int decimals = getDecimalDigits();
		if (decimals > 0) {
			sqlTypeName.append(",").append(decimals);
		}

		sqlTypeName.append(")");
		return sqlTypeName.toString();
	}

	/* -- Internal methods -- */

	/**
	 * Column size is computed once, because of some special case (for HsqlDB)
	 * 
	 * @return
	 */
	private int initColumnSize(Dialect dialect, ColumnMetadata delegate) {
		if (dialect instanceof HSQLDialect) {
			// Ignore some column size, on HSQLDB
			if (delegate.getTypeCode() == Types.INTEGER) {
				// TODO : we could also return 10 ??
				return 0;
			}
			if (delegate.getTypeCode() == Types.DOUBLE) {
				// Default value is 64
				return 0;
			}
			if (delegate.getTypeCode() == Types.BIGINT) {
				// TODO : we could also return 0 ??
				return 0;
			}
			if (delegate.getTypeCode() == Types.SMALLINT) {
				// TODO : we could also return 0 ??
				return 0;
			}
		}

		return delegate.getColumnSize();
	}
}
