package fr.ifremer.adagio.synchro.dao;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import static org.nuiton.i18n.I18n.t;

import java.io.IOException;
import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import oracle.sql.TIMESTAMP;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.LockMode;
import org.hibernate.dialect.Dialect;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import fr.ifremer.adagio.synchro.SynchroBusinessException;
import fr.ifremer.adagio.synchro.SynchroTechnicalException;
import fr.ifremer.adagio.synchro.config.SynchroConfiguration;
import fr.ifremer.adagio.synchro.intercept.SynchroInterceptor;
import fr.ifremer.adagio.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.adagio.synchro.intercept.SynchroInterceptorUtils;
import fr.ifremer.adagio.synchro.intercept.SynchroOperationRepository;
import fr.ifremer.adagio.synchro.meta.SynchroColumnMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroJoinMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata.DuplicateKeyStrategy;
import fr.ifremer.adagio.synchro.query.SynchroQueryBuilder;
import fr.ifremer.adagio.synchro.query.SynchroQueryOperator;
import fr.ifremer.adagio.synchro.query.internal.SynchroInsertQuery;
import fr.ifremer.adagio.synchro.query.internal.SynchroSelectQuery;
import fr.ifremer.adagio.synchro.service.SynchroDatabaseConfiguration;
import fr.ifremer.adagio.synchro.service.SynchroTableOperation;

public class SynchroTableDaoImpl implements SynchroTableDao {

	/** Logger. */
	private static final Log log =
			LogFactory.getLog(SynchroTableDaoImpl.class);

	/**
	 * A name to use for column parameter_name, when inserting into table TEMP_QUERY_PARAMETER
	 */
	protected static final String TEMP_QUERY_PARAMETER_PARAM_NAME = SynchroBaseDao.TEMP_QUERY_PARAMETER_PARAM_PREFIX + "dao";

	protected Connection connection;

	protected SynchroTableMetadata table;

	protected Dialect dialect;

	protected final PreparedStatement insertStatement;

	protected final PreparedStatement updateStatement;

	protected final PreparedStatement deleteStatement;

	protected final List<PreparedStatement> selectStatements;

	protected final Map<String, PreparedStatement> updateColumnStatements;

	protected final PreparedStatement lockStatement;

	protected final Map<PreparedStatement, int[]> selectPkByIndexStatements;

	protected final int[] incominDataIndexedNeedForIndex;

	protected final int incomingDataColumnCount;

	protected final int batchSize;

	protected final int fetchSize;

	protected final String tableName;

	protected final boolean debug;

	protected boolean isJdbcBatchEnable;

	protected int insertCount = 0;

	protected int updateCount = 0;

	protected Map<String, Integer> updateColumnCounts = Maps.newHashMap();

	protected int deleteCount = 0;

	protected SynchroInterceptor readInterceptor;

	protected SynchroInterceptor writeInterceptor;

	protected SynchroTableOperation pendingOperationBuffer;

	protected final String sequenceNextValString;

	protected final boolean isMirrorDatabase;

	protected final boolean keepWhereClauseOnQueriesByFks;

	protected final boolean hasSequence;

	protected final boolean enableInsertWithPkBind;

	protected final int[] insertPkIndexes;

	protected int[] selectPkIndexs;

	protected final boolean enableWrite;

	protected List<Object> lastGeneratedPk;

	protected SynchroTableDao sourceDao;

	protected final String countAllApproxQuery;

	/**
	 * Delegated dao, with base capabilities.
	 * Not defined has super class, to avoid to have many common statements
	 */
	protected final SynchroBaseDao delegate;

	public SynchroTableDaoImpl(
			SynchroDatabaseMetadata dbMeta,
			SynchroTableMetadata table) throws SQLException {
		this(dbMeta.getConnection(), dbMeta.getConfiguration(), table, new SynchroBaseDaoImpl(dbMeta));
	}

	public SynchroTableDaoImpl(
			Connection connection,
			SynchroDatabaseConfiguration configuration,
			SynchroTableMetadata table,
			SynchroBaseDao delegate) throws SQLException {
		this.dialect = configuration.getDialect();
		this.connection = connection;
		this.table = table;
		this.delegate = delegate;

		this.selectStatements = Lists.newArrayList();
		this.updateColumnStatements = Maps.newHashMap();
		this.tableName = table.getName();
		this.enableWrite = !configuration.isReadOnly() && configuration.isTarget();
		this.isMirrorDatabase = configuration.isMirrorDatabase();
		this.keepWhereClauseOnQueriesByFks = configuration.isKeepWhereClauseOnQueriesByFks();
		this.selectPkIndexs = table.getSelectPkIndexs();
		this.countAllApproxQuery = initCountAllApproxQuery(dialect, configuration, table);

		// Batch size
		this.batchSize = SynchroConfiguration.getInstance().getImportJdbcBatchSize();
		Preconditions.checkArgument(this.batchSize > 0);
		this.fetchSize = SynchroConfiguration.getInstance().getImportJdbcFetchSize();
		Preconditions.checkArgument(this.fetchSize > 0);

		this.isJdbcBatchEnable = this.batchSize > 1;
		this.debug = log.isDebugEnabled();

		this.readInterceptor = createReadInterceptor(table);

		// Prepare for Write
		if (enableWrite) {
			this.sequenceNextValString = createSequenceNextValString(this.dialect, table);
			this.hasSequence = StringUtils.isNotBlank(sequenceNextValString);

			this.incomingDataColumnCount = createIncomingDataColumnCount(table);

			// Init write interceptors
			this.writeInterceptor = createWriteInterceptor(table);

			// Prepare insert statement
			this.enableInsertWithPkBind = !isMirrorDatabase && hasSequence && (writeInterceptor != null || !isJdbcBatchEnable);
			if (enableInsertWithPkBind) {
				String insertWithPkBindQuery = createInsertWithPkBindQuery(table);
				this.insertStatement = connection.prepareStatement(insertWithPkBindQuery);
				this.insertPkIndexes = initInsertPkIndexes(insertWithPkBindQuery);
			}
			else {
				this.insertStatement = createInsertStatement(connection, this.dialect, table);
				this.insertPkIndexes = null;
			}

			// Init select PK from unique constraints
			this.selectPkByIndexStatements = createSelectPkByIndexStatement(connection, table);
			this.incominDataIndexedNeedForIndex = createIncomingDataIndexedNeedForIndex(selectPkByIndexStatements, incomingDataColumnCount);

			// Prepare update statement
			this.updateStatement = createUpdateStatement(connection, table);

			this.lockStatement = createLockStatement(connection, table);

			this.deleteStatement = createDeleteStatement(connection, table);

			// Could not use JDBC batch is unique must be checked - mantis #23599
			this.isJdbcBatchEnable = this.batchSize > 1
					&& (!table.hasUniqueConstraints(DuplicateKeyStrategy.REJECT)
					|| !configuration.isCheckUniqueConstraintBetweenInputRows());
		}
		else {
			this.insertStatement = null;
			this.updateStatement = null;
			this.lockStatement = null;
			this.writeInterceptor = null;
			this.sequenceNextValString = null;
			this.hasSequence = false;
			this.incomingDataColumnCount = -1;
			this.selectPkByIndexStatements = null;
			this.incominDataIndexedNeedForIndex = null;
			this.enableInsertWithPkBind = false;
			this.insertPkIndexes = null;
			this.deleteStatement = null;
			this.isJdbcBatchEnable = this.batchSize > 1;
		}

		this.lastGeneratedPk = null;
	}

	/* -- public methods -- */

	@Override
	public void cleanTempQueryParameter() throws SQLException {
		delegate.cleanTempQueryParameter();
	}

	@Override
	public void executeDeleteTempQueryParameter(String queryParameterName, boolean likeOperatorForParameterName, int queryPersonId)
			throws SQLException {
		delegate.executeDeleteTempQueryParameter(queryParameterName, likeOperatorForParameterName, queryPersonId);
	}

	@Override
	public void executeInsertIntoTempQueryParameter(List<Object> parameterValues, String queryParameterName, int queryPersonId) throws SQLException {
		delegate.executeInsertIntoTempQueryParameter(parameterValues, queryParameterName, queryPersonId);
	}

	@Override
	public Connection getConnection() {
		return delegate.getConnection();
	}

	@Override
	public String getInsertIntoTempQueryParameterQuery() {
		return delegate.getInsertIntoTempQueryParameterQuery();
	}

	@Override
	public <T> T getUniqueTyped(String sql, Object[] bindingValues) throws SQLException {
		return delegate.getUniqueTyped(sql, bindingValues);
	}

	@Override
	public DaoFactory getDaoFactory() {
		return delegate.getDaoFactory();
	}

	@Override
	public void setSourceDao(SynchroTableDao sourceDao) {
		this.sourceDao = sourceDao;
	}

	@Override
	public Dialect getDialect() {
		return dialect;
	}

	@Override
	public SynchroTableMetadata getTable() {
		return table;
	}

	public void setCurrentOperation(SynchroTableOperation pendingChangesBuffer) {
		this.pendingOperationBuffer = pendingChangesBuffer;
	}

	public SynchroTableOperation getCurrentOperation() {
		return this.pendingOperationBuffer;
	}

	public int getInsertCount() {
		return insertCount;
	}

	public int getUpdateCount() {
		return updateCount;
	}

	public int getUpdateColumnCount(String columnName) {
		return MapUtils.getIntValue(updateColumnCounts, columnName);
	}

	public int getTotalUpdateColumnCount() {
		int result = 0;
		for (String columnName : updateColumnStatements.keySet()) {
			result += MapUtils.getIntValue(updateColumnCounts, columnName);
		}
		return result;
	}

	public int getDeleteCount() {
		return deleteCount;
	}

	public void flush() throws SQLException {
		if (!isJdbcBatchEnable) {
			return;
		}

		checkWriteEnable();
		if (insertCount > 0 && insertCount % batchSize != 0) {
			insertStatement.executeBatch();
			insertStatement.clearBatch();
		}
		if (updateCount > 0 && updateCount % batchSize != 0) {
			updateStatement.executeBatch();
			updateStatement.clearBatch();
		}
		if (deleteCount > 0 && deleteCount % batchSize != 0) {
			deleteStatement.executeBatch();
			deleteStatement.clearBatch();
		}
		for (String columnName : updateColumnStatements.keySet()) {
			int updateColumnCount = getUpdateColumnCount(columnName);
			PreparedStatement updateColumnStatement = updateColumnStatements.get(columnName);
			if (updateColumnCount > 0 && updateColumnCount % batchSize != 0) {
				updateColumnStatement.executeBatch();
				updateColumnStatement.clearBatch();
			}
		}
	}

	@Override
	public void close() throws IOException {
		Daos.closeSilently(insertStatement);
		Daos.closeSilently(updateStatement);
		Daos.closeSilently(deleteStatement);
		Daos.closeSilently(lockStatement);
		closeSelectStatements();
		closeUpdateColumnStatements();
		closeSelectPkByIndexStatements();

		IOUtils.closeQuietly(writeInterceptor);
		writeInterceptor = null;

		IOUtils.closeQuietly(readInterceptor);
		readInterceptor = null;

		pendingOperationBuffer = null;
		connection = null;
		dialect = null;
		table = null;
	}

	/**
	 * Gets the last updateDate for the given {@code table} using
	 * the given datasource.
	 * 
	 * @return the last update date of the given table, or {@code null} if table does not use a updateDate columns or if
	 *         there
	 *         is no data in table.
	 */
	public Timestamp getLastUpdateDate() throws SQLException {
		return SynchroTableDaoUtils.getLastUpdateDate(this.table, getConnection());
	}

	@Override
	public long countAll(boolean approximate) throws SQLException {

		String sql = (approximate && this.countAllApproxQuery != null)
				? this.countAllApproxQuery
				: table.getCountAllQuery();

		PreparedStatement statement = getConnection().prepareStatement(sql);
		ResultSet resultSet = null;
		long result = 0;
		boolean errorOnCount = false;
		try {
			resultSet = statement.executeQuery();
			if (!resultSet.next()) {
				// could not count: error occured
				errorOnCount = true;
			} else {
				result = resultSet.getLong(1);
			}
		} finally {
			Daos.closeSilently(resultSet);
			Daos.closeSilently(statement);
		}

		// If approximate query return 0 row: get the exact rows count
		if (approximate && this.countAllApproxQuery != null && (result == 0 || errorOnCount)) {
			// Do it again without approximation (zero must always be exact)
			return countAll(false);
		}

		// if rows still could not be counted, there is definitely a problem
		if (errorOnCount) {
			throw new SynchroTechnicalException(String.format("[%s] Count query returned no row : %s", tableName, sql));
		}

		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see fr.ifremer.adagio.synchro.dao.SynchroTableDao#countDataToUpdate(java.util.Date)
	 */
	public long countDataToUpdate(Date fromDate) throws SQLException {

		Map<String, Object> bindings = Maps.newHashMap();
		if (fromDate != null) {
			bindings.put(SynchroTableMetadata.UPDATE_DATE_BINDPARAM, fromDate);
		}

		return countData(bindings);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see fr.ifremer.adagio.synchro.dao.SynchroTableDao#count(java.util.Map)
	 */
	public long countData(Map<String, Object> bindings) throws SQLException {

		// Prepare bindings
		Map<String, Object> validBindings = prepareBindings(bindings);

		String sql = table.getCountQuery();
		if (validBindings.containsKey(SynchroTableMetadata.UPDATE_DATE_BINDPARAM)) {
			sql = table.getCountUpdatedDataQuery();
		}

		PreparedStatement statement = null;
		ResultSet resultSet = null;
		try {

			statement = prepareAndBindStatement(sql, bindings, "count");
			resultSet = statement.executeQuery();
			resultSet.next();

			long result = resultSet.getLong(1);
			return result;
		} finally {
			Daos.closeSilently(resultSet);
			Daos.closeSilently(statement);
		}
	}

	@Override
	public long countDataByFks(Set<String> fkColumnNames, List<List<Object>> fkColumnsValues, Map<String, Object> bindings) throws SQLException {

		Preconditions.checkArgument(CollectionUtils.isNotEmpty(fkColumnNames));
		String queryParameterName = "values";

		// Prepare bindings
		Map<String, Object> validBindings = prepareBindings(bindings);

		String baseSql = table.getCountQuery();
		if (validBindings.containsKey(SynchroTableMetadata.UPDATE_DATE_BINDPARAM)) {
			baseSql = table.getCountUpdatedDataQuery();
		}

		String sql = createSelectByManyFksUsingTempParameterTable(baseSql, fkColumnNames, queryParameterName);

		if (log.isDebugEnabled()) {
			log.debug(String.format("[%s] Prepare select query: %s", table.getName(), sql));
		}

		// Insert values into table TEMP_QUERY_PARAMETER
		Integer userId = (Integer) bindings.get("userId");
		if (userId == null) {
			userId = new Integer(-1);
		}
		executeInsertIntoTempQueryParameter(fkColumnNames, fkColumnsValues, queryParameterName, userId, false);

		// Then execute the query
		PreparedStatement statement = null;
		ResultSet resultSet = null;
		try {
			statement = prepareAndBindStatement(sql, bindings, "countByFks");
			registerSelectStatementAndClosePrevious(statement);
			resultSet = statement.executeQuery();
			resultSet.next();

			return resultSet.getLong(1);
		} finally {
			Daos.closeSilently(resultSet);
			Daos.closeSilently(statement);
		}
	}

	/**
	 * @param fromDate
	 * @param bindings
	 * @return
	 * @throws SQLException
	 * @Deprecated use getData() instead
	 */
	public ResultSet getDataToUpdate(Date fromDate) throws SQLException {

		Map<String, Object> bindings = Maps.newHashMap();
		if (fromDate != null) {
			bindings.put(SynchroTableMetadata.UPDATE_DATE_BINDPARAM, fromDate);
		}

		return getData(bindings);
	}

	@Override
	public ResultSet getData(Map<String, Object> bindings) throws SQLException {

		// Copy bindings
		Map<String, Object> validBindings = prepareBindings(bindings);

		String sql = table.getSelectAllQuery();
		if (table.isWithUpdateDateColumn() && MapUtils.getObject(validBindings, SynchroTableMetadata.UPDATE_DATE_BINDPARAM) != null) {
			sql = table.getSelectUpdatedDataQuery();
		}
		if (log.isDebugEnabled()) {
			log.debug(String.format("[%s] Prepare select query: %s", table.getName(), sql));
		}

		PreparedStatement statement = prepareAndBindSelectStatement(sql, bindings, "select");
		registerSelectStatementAndClosePrevious(statement);

		statement.setFetchSize(batchSize);

		ResultSet result = statement.executeQuery();
		return result;
	}

	@Override
	public Timestamp getUpdateDateByPk(List<Object> pk) throws SQLException {
		if (!table.isWithUpdateDateColumn()) {
			return null;
		}
		String sql = table.getSelectUpdateDateByPkQuery();

		if (sql == null) {
			return null;
		}

		if (debug) {
			log.debug(String.format("[%s] Execute getUpdateDateByPk query: %s", table.getName(), sql));
		}

		PreparedStatement statement = getConnection().prepareStatement(sql);
		ResultSet resultSet = null;
		try {
			int pkIndex = 1;
			for (Object value : pk) {
				statement.setObject(pkIndex, value);
				pkIndex++;
			}

			resultSet = statement.executeQuery();
			if (!resultSet.next()) {
				return null;
			}
			Timestamp result = resultSet.getTimestamp(1);
			return result;
		} finally {
			Daos.closeSilently(resultSet);
			Daos.closeSilently(statement);
		}
	}

	public void deleteAll() throws SQLException {
		PreparedStatement deleteStatement =
				getConnection().prepareStatement(
						"DELETE FROM " + table.getName());
		try {
			deleteStatement.execute();
		} finally {
			Daos.closeSilently(deleteStatement);
		}
	}

	@Override
	public boolean executeDeleteByFk(String fkColumnName, String fkColumnValue, String additionalWhereClause, Map<String, Object> bindings)
			throws SQLException {
		String sql = String.format("DELETE FROM %s WHERE %s=?%s",
				table.getName(),
				fkColumnName,
				StringUtils.isNotBlank(additionalWhereClause) ? " AND " + additionalWhereClause : ""
				);

		PreparedStatement deleteStatement = prepareAndBindStatement(sql, bindings, "delete");

		deleteStatement.setObject(1, fkColumnValue);

		try {
			int rowCount = deleteStatement.executeUpdate();
			deleteCount += rowCount;

			return deleteCount > 0;
		} finally {
			Daos.closeSilently(deleteStatement);
		}
	}

	@Override
	public void executeDelete(List<Object> pk, boolean checkPkNotUsed)
			throws SQLException {

		// Check is the PK not used anymore
		if (checkPkNotUsed) {
			checkPkNotUsedBeforeDelete(pk);
		}
		if (writeInterceptor != null) {
			writeInterceptor.onDelete(pk, sourceDao, this, pendingOperationBuffer);
		}

		int i = 1;
		for (Object value : pk) {
			deleteStatement.setObject(i++, value);
		}

		deleteCount++;

		if (!isJdbcBatchEnable) {
			checkWriteEnable();
			if (debug) {
				log.debug(String.format("[%s] Execute delete query (pk:%s)", tableName, SynchroTableMetadata.toPkStr(pk)));
			}
			int nbRowDeleted = deleteStatement.executeUpdate();
			if (debug) {
				// If =0 (row already deleted) do not throw exception
				if (nbRowDeleted == 0) {
					log.debug(String.format("[%s] Row not deleted (pk:%s) - maybe already deleted ?", tableName,
							SynchroTableMetadata.toPkStr(pk)));
				}

				// If > 0 (bad PK definition ?) throw an exception
				else if (nbRowDeleted > 1) {
					throw new SynchroTechnicalException(String.format("[%s] Could not delete a row (pk:%s): more than one row affected.", tableName,
							SynchroTableMetadata.toPkStr(pk)));
				}
			}
		}
		else {
			deleteStatement.addBatch();
			if (deleteCount > 0 && deleteCount % batchSize == 0) {
				checkWriteEnable();
				deleteStatement.executeBatch();
				deleteStatement.clearBatch();
			}
		}
	}

	@Override
	public void executeDetach(List<Object> pk) throws SQLException {
		if (writeInterceptor != null) {
			writeInterceptor.onDetach(pk, sourceDao, this, pendingOperationBuffer);
		}
	}

	public Object[] findByPk(List<Object> pk) throws SQLException {
		String sql = table.getSelectDataQueryFromPk();

		PreparedStatement selectStatement = getPreparedStatement(sql);
		int columnCountIndex = 1;

		for (Object pkColumn : pk) {
			selectStatement.setObject(columnCountIndex++, pkColumn);
		}

		int columnsCount = table.getSelectColumnsCount();

		ResultSet resultSet = null;
		try {
			resultSet = selectStatement.executeQuery();
			resultSet.next();

			Object[] result = new Object[columnsCount];

			for (int i = 1; i <= columnsCount; i++) {

				result[i - 1] = resultSet.getObject(i);
			}
			return result;
		} finally {
			Daos.closeSilently(resultSet);
			Daos.closeSilently(selectStatement);
		}
	}

	public boolean exists(List<Object> pk) throws SQLException {
		String sql = table.getSelectDataQueryFromPk();

		PreparedStatement selectStatement = getPreparedStatement(sql);
		ResultSet resultSet = null;
		try {

			int columnCountIndex = 1;
			for (Object pkColumn : pk) {
				selectStatement.setObject(columnCountIndex++, pkColumn);
			}

			resultSet = selectStatement.executeQuery();
			return resultSet.next();
		} finally {
			Daos.closeSilently(resultSet);
			Daos.closeSilently(selectStatement);
		}
	}

	public Set<String> getPksStr() throws SQLException {
		Set<String> result = Sets.newHashSet();

		// If simple SQL (select with only one output String column)
		if (table.isSelectPrimaryKeysAsStringQueryEnable()) {
			String sql = table.getSelectPksStrQuery();
			PreparedStatement statement = getConnection().prepareStatement(sql);
			ResultSet resultSet = null;
			try {
				resultSet = statement.executeQuery();

				while (resultSet.next()) {
					String pk = resultSet.getString(1);
					result.add(pk);
				}
			} finally {
				Daos.closeSilently(resultSet);
				Daos.closeSilently(statement);
			}
		}

		// If more than one one column as String, in the select query
		else {
			String sql = table.getSelectPksQuery();
			PreparedStatement statement = getConnection().prepareStatement(sql);
			ResultSet resultSet = null;
			try {
				resultSet = statement.executeQuery();

				int columnCount = table.getPkNames().size();
				List<Object> pks = Lists.newArrayListWithCapacity(columnCount);
				while (resultSet.next()) {
					for (int i = 1; i <= columnCount; i++) {
						Object pk = resultSet.getObject(i);
						pks.add(pk);
					}
					result.add(SynchroTableMetadata.toPkStr(pks));
					pks.clear();
				}
			} finally {
				Daos.closeSilently(resultSet);
				Daos.closeSilently(statement);
			}
		}

		return result;
	}

	public List<Object> generateNewPk() throws SQLException {
		Preconditions.checkArgument(hasSequence);

		PreparedStatement statement = getPreparedStatement(sequenceNextValString);
		try {
			ResultSet resultSet = statement.executeQuery();
			if (!resultSet.next()) {
				return null;
			}
			Integer sequenceValue = resultSet.getInt(1);
			return Lists.newArrayList((Object) sequenceValue);
		} catch (SQLException sqle) {
			// Log with the sql query (mantis #23225)
			log.error(t("adagio.synchro.dao.generateNewPk.error.log", table.getName(), sqle.getMessage(), sequenceNextValString));
			throw new SynchroTechnicalException(
					t("adagio.synchro.dao.generateNewPk.error", sqle.getMessage(), sequenceNextValString));
		} finally {
			Daos.closeSilently(statement);
		}
	}

	public void executeInsert(ResultSet incomingData) throws SQLException {
		List<Object> params = null;
		if (debug) {
			params = Lists.newArrayList();
		}

		List<Object> pk = null;
		if (enableInsertWithPkBind) {
			pk = generateNewPk();
		}

		if (writeInterceptor != null) {
			transformAndSetData(insertStatement, incomingData, pk, pendingOperationBuffer, writeInterceptor, params);
		}
		else {
			setData(insertStatement, incomingData, params);
		}

		// bind the pk
		if (enableInsertWithPkBind) {
			int i = 0;
			for (Object value : pk) {
				insertStatement.setObject(insertPkIndexes[i++], value);
			}
		}

		insertCount++;

		if (!isJdbcBatchEnable) {
			checkWriteEnable();
			if (pk == null) {
				pk = getPk(incomingData);
			}
			if (debug) {
				log.debug(String.format("[%s] Execute insert query (pk:%s), params: %s", tableName, SynchroTableMetadata.toPkStr(pk), params));
			}
			int nbRowInsert = insertStatement.executeUpdate();
			if (debug) {
				Preconditions.checkArgument(nbRowInsert == 1,
						String.format("[%s] Could not insert a row into the table (pk:%s), params: %s", tableName,
								SynchroTableMetadata.toPkStr(pk), params));
			}
		}
		else {
			insertStatement.addBatch();
			if (insertCount > 0 && insertCount % batchSize == 0) {
				checkWriteEnable();
				insertStatement.executeBatch();
				insertStatement.clearBatch();
			}
		}
	}

	public void executeInsert(Object[] incomingData) throws SQLException {

		List<Object> params = null;
		if (debug) {
			params = Lists.newArrayList();
		}

		List<Object> pk = null;
		if (enableInsertWithPkBind) {
			pk = generateNewPk();

			// Remember the PK
			this.lastGeneratedPk = pk;
		}

		if (writeInterceptor != null) {
			transformAndSetData(insertStatement, incomingData, pk, pendingOperationBuffer, writeInterceptor, params);
		}
		else {
			setData(insertStatement, incomingData, params);
		}
		insertCount++;

		if (!isJdbcBatchEnable) {
			checkWriteEnable();
			if (pk == null) {
				pk = getPk(incomingData);
			}
			if (debug) {
				log.debug(String.format("%s Execute insert query (pk:%s), params: %s", tableName, SynchroTableMetadata.toPkStr(pk), params));
			}
			insertStatement.executeUpdate();
		}
		else {
			insertStatement.addBatch();
			if (insertCount > 0 && insertCount % batchSize == 0) {
				checkWriteEnable();
				insertStatement.executeBatch();
				insertStatement.clearBatch();
			}
		}
	}

	@Override
	public void executeUpdate(List<Object> pk, ResultSet incomingData) throws SQLException {

		List<Object> params = null;
		if (debug) {
			params = Lists.newArrayList();
		}

		if (writeInterceptor != null) {
			transformAndSetData(updateStatement, incomingData, pk, pendingOperationBuffer, writeInterceptor, params);
		}
		else {
			setData(updateStatement, incomingData, params);
		}

		int columnCountIndex = incomingDataColumnCount + 1;
		for (Object pkColumn : pk) {
			updateStatement.setObject(columnCountIndex++, pkColumn);
		}

		updateCount++;

		if (!isJdbcBatchEnable) {
			checkWriteEnable();
			if (debug) {
				log.debug(String.format("%s Execute update query (pk:%s), params: %s", tableName, pk, params));
			}
			int nbRowUpdated = updateStatement.executeUpdate();
			if (debug) {
				Preconditions.checkArgument(nbRowUpdated == 1, String.format("%s rows has been updated, but expected 1 row.", nbRowUpdated));
			}
		}
		else {
			updateStatement.addBatch();
			if (updateCount > 0 && updateCount % batchSize == 0) {
				checkWriteEnable();
				updateStatement.executeBatch();
				updateStatement.clearBatch();
			}
		}
	}

	@Override
	public void executeUpdate(List<Object> pk, Object[] row) throws SQLException {
		List<Object> params = null;
		if (debug) {
			params = Lists.newArrayList();
		}

		if (writeInterceptor != null) {
			transformAndSetData(updateStatement, row, pk, pendingOperationBuffer, writeInterceptor, params);
		}
		else {
			setData(updateStatement, row, params);
		}

		int columnCountIndex = incomingDataColumnCount + 1;

		for (Object pkColumn : pk) {
			updateStatement.setObject(columnCountIndex++, pkColumn);
		}

		updateCount++;

		if (!isJdbcBatchEnable) {
			if (debug) {
				log.debug(String.format("%s Execute update query (pk:%s), params: %s", tableName, pk, params));
			}
			int nbRowUpdated = updateStatement.executeUpdate();
			if (debug) {
				Preconditions.checkArgument(nbRowUpdated == 1, String.format("%s rows has been updated, but expected 1 row.", nbRowUpdated));
			}
		}
		else {
			updateStatement.addBatch();
			if (updateCount > 0 && updateCount % batchSize == 0) {
				updateStatement.executeBatch();
				updateStatement.clearBatch();
			}
		}
	}

	@Override
	public void executeUpdateColumn(String columnName, List<Object> pk, Object columnValue) throws SQLException {

		PreparedStatement updateColumnStatement = updateColumnStatements.get(columnName);
		if (updateColumnStatement == null) {
			updateColumnStatement = createUpdateColumnStatement(columnName);
			updateColumnStatements.put(columnName, updateColumnStatement);
		}

		// Execute update on the pk's row
		executeUpdateColumn(updateColumnStatement, pk, columnName, columnValue);
	}

	@Override
	public ResultSet getDataByFks(Set<String> fkColumnNames, List<List<Object>> fkColumnsValues, Map<String, Object> bindings) throws SQLException {

		int nbValues = fkColumnsValues.size();
		boolean insertUsingTempQueryParameter = (fkColumnNames.size() > 1
				|| (dialect.getInExpressionCountLimit() > 0 && nbValues > dialect.getInExpressionCountLimit())
				|| nbValues > 5000 /* Always use TempQueryParameter if too many rows */
				);

		if (insertUsingTempQueryParameter) {
			return getDataByFksUsingTempParameterTable(fkColumnNames, fkColumnsValues, bindings);
		}

		// Is could not use tempQueryParameterEnable, make sure there is only one column
		if (fkColumnNames.size() > 1) {
			throw new SynchroTechnicalException(
					"getDataByFks() without using TempQueryParameter not implemented yet. Please enable tempQueryParameter use in the database configuration");
		}

		return getDataByFkUsingIn(fkColumnNames.iterator().next(), fkColumnsValues, bindings);
	}

	@Override
	public List<List<Object>> getPksByFks(Set<String> fkColumnNames, List<List<Object>> fkColumnsValues, Map<String, Object> bindings)
			throws SQLException {
		return getPksByFks(fkColumnNames, fkColumnsValues, bindings, true);
	}

	@Override
	public List<List<Object>> getPksByNotFoundFks(Set<String> fkColumnNames, List<List<Object>> fkColumnsValues, Map<String, Object> bindings)
			throws SQLException {
		return getPksByFks(fkColumnNames, fkColumnsValues, bindings, false);
	}

	@Override
	public Set<String> getPksStrByFks(Set<String> fkColumnNames, List<List<Object>> fkColumnsValues, Map<String, Object> bindings)
			throws SQLException {
		Map<String, Timestamp> result = getPksStrWithUpdateDateByFks(fkColumnNames, fkColumnsValues, bindings);
		if (MapUtils.isEmpty(result)) {
			return null;
		}
		return result.keySet();
	}

	@Override
	public Map<String, Timestamp> getPksStrWithUpdateDateByFks(Set<String> fkColumnNames, List<List<Object>> fkColumnsValues,
			Map<String, Object> bindings) throws SQLException {
		return getPksStrWithUpdateDateByFks(fkColumnNames, fkColumnsValues, bindings, true);
	}

	@Override
	public Set<String> getPksStrByNotFoundFks(Set<String> fkColumnNames, List<List<Object>> fkColumnsValues, Map<String, Object> bindings)
			throws SQLException {
		Map<String, Timestamp> result = getPksStrWithUpdateDateByFks(fkColumnNames, fkColumnsValues, bindings, false);
		if (MapUtils.isEmpty(result)) {
			return null;
		}
		return result.keySet();
	}

	@Override
	public Map<String, Timestamp> getPksStrWithUpdateDate() throws SQLException {
		Preconditions.checkArgument(table.isWithUpdateDateColumn());

		Map<String, Timestamp> result = Maps.newHashMap();

		// If simple SQL (select with only one output String column)
		if (table.isSelectPrimaryKeysAsStringQueryEnable()) {
			String sql = table.getSelectPksStrQuery();
			PreparedStatement statement = getConnection().prepareStatement(sql);
			ResultSet resultSet = null;
			try {
				resultSet = statement.executeQuery();

				while (resultSet.next()) {
					String pk = resultSet.getString(1);
					Timestamp updateDate = resultSet.getTimestamp(2);
					result.put(pk, updateDate);
				}
			} finally {
				Daos.closeSilently(resultSet);
				Daos.closeSilently(statement);
			}
		}

		// If more than one one column as String, in the select query
		else {
			String sql = table.getSelectPksQuery();
			PreparedStatement statement = getConnection().prepareStatement(sql);
			ResultSet resultSet = null;

			try {
				resultSet = statement.executeQuery();

				int pkCount = table.getPkNames().size();
				List<Object> pks = Lists.newArrayListWithCapacity(pkCount);
				while (resultSet.next()) {
					for (int i = 1; i <= pkCount; i++) {
						Object pk = resultSet.getObject(i);
						pks.add(pk);
					}
					Timestamp updateDate = resultSet.getTimestamp(pkCount + 1);
					result.put(SynchroTableMetadata.toPkStr(pks), updateDate);
					pks.clear();
				}
			} finally {
				Daos.closeSilently(resultSet);
				Daos.closeSilently(statement);
			}
		}

		return result;
	}

	@Override
	public Set<String> transformColumnNames(Set<String> fkColumnNames) throws SQLException {
		Set<String> result = Sets.newLinkedHashSet();
		List<String> insertColumnNames = ((SynchroInsertQuery) SynchroQueryBuilder.newBuilder(table.getInsertQuery())).getBindingColumnNames();

		for (String columnName : fkColumnNames) {
			int fkSelectIndex = table.getSelectColumnIndex(columnName);
			String insertColumnName = null;
			if (fkSelectIndex == -1) {
				throw new SynchroTechnicalException(String.format("Could not found column %s in the select query. Unable to remap column names",
						columnName));
			}
			insertColumnName = insertColumnNames.get(fkSelectIndex);
			if (StringUtils.isBlank(insertColumnName)) {
				throw new SynchroTechnicalException(String.format("Could not found column %s in the insert query. Unable to remap column names",
						columnName));
			}
			result.add(insertColumnName);
		}
		return result;
	}

	@Override
	public List<List<Object>> transformOnRead(Set<String> fkColumnNames, List<List<Object>> fkSourceColumnValues) throws SQLException {
		if (this.readInterceptor == null) {
			return fkSourceColumnValues;
		}

		List<List<Object>> result = Lists.newArrayListWithCapacity(fkSourceColumnValues.size());

		int[] fkSelectIndexes = new int[fkColumnNames.size()];
		int i = 0;
		for (String columnName : fkColumnNames) {
			int fkSelectIndex = table.getSelectColumnIndex(columnName);
			fkSelectIndexes[i++] = fkSelectIndex;
		}

		Object[] fakeIncominData = new Object[incomingDataColumnCount];
		for (List<Object> values : fkSourceColumnValues) {
			i = 0;
			for (Object value : values) {
				int fkSelectIndex = fkSelectIndexes[i++];
				fakeIncominData[fkSelectIndex] = value;
			}

			fakeIncominData = transformDataOnRead(fakeIncominData, this.readInterceptor);

			List<Object> transformedValues = Lists.newArrayListWithCapacity(fkSelectIndexes.length);
			for (int fkSelectIndex : fkSelectIndexes) {
				transformedValues.add(fakeIncominData[fkSelectIndex]);
			}
			result.add(transformedValues);
		}

		return result;
	}

	/**
	 * Prepare the DAO for another execution.
	 * <p/>
	 * This method could be used by Factory.
	 */
	@Override
	public void prepare() {
		insertCount = 0;
		updateCount = 0;
		deleteCount = 0;
		updateColumnCounts.clear();
		lastGeneratedPk = null;
		closeUpdateColumnStatements();
	}

	@Override
	public PreparedStatement getPreparedStatement(String sql) throws SQLException {

		try {
			// If a factory is define: use it
			if (delegate != null) {
				return delegate.getPreparedStatement(sql);
			}

			// else, use a normal statement creation
			return connection.prepareStatement(sql);

		} catch (SQLException e) {
			throw new SynchroTechnicalException(
					String.format(
							"Error while getting pooled statement for query [%s]. Make sure this table exists in database.",
							sql),
					e);
		}
	}

	/* -- Abstract method -- */

	/* -- Protected methods -- */

	protected void executeInsertIntoTempQueryParameter(Set<String> columnNames,
			List<List<Object>> values,
			String queryParameterName,
			int queryPersonId,
			boolean transform)
			throws SQLException {

		// Delete existing rows
		delegate.executeDeleteTempQueryParameter(queryParameterName + "_%", true/* like */, queryPersonId);

		if (debug) {
			log.debug(String.format("Setting query parameters into %s", SynchroBaseDao.TEMP_QUERY_PARAMETER_TABLE));
		}

		Object[] fakeIncominData = null;
		int[] columnIndexesInSelect = null;
		if (transform && readInterceptor != null) {
			fakeIncominData = new Object[incomingDataColumnCount];
			columnIndexesInSelect = new int[columnNames.size()];
			int i = 0;
			for (String columnName : columnNames) {
				columnIndexesInSelect[i++] = table.getSelectColumnIndex(columnName);
			}
		}

		PreparedStatement statement = null;
		try {
			String sql = delegate.getInsertIntoTempQueryParameterQuery();
			statement = getPreparedStatement(sql);

			int rowCount = 1;
			int insertCount = 1;
			for (List<Object> rowValues : values) {
				int columnIndex = 0;
				for (Object columnValue : rowValues) {

					// Transformed the given value, using read interceptor
					if (fakeIncominData != null) {
						int fakeColumnIndex = columnIndexesInSelect[columnIndex];
						fakeIncominData[fakeColumnIndex] = columnValue;
						columnValue = transformDataOnRead(fakeIncominData, readInterceptor)[fakeColumnIndex];
					}

					statement.setString(1, queryParameterName + "_" + columnIndex);
					statement.setInt(2, rowCount);
					statement.setObject(3, columnValue);
					statement.setInt(4, queryPersonId);
					statement.addBatch();
					columnIndex++;
					insertCount++;
					if (insertCount % batchSize == 0) {
						statement.executeBatch();
						statement.clearBatch();
					}
				}
				rowCount++;
			}

			if (insertCount % batchSize != 0) {
				statement.executeBatch();
				statement.clearBatch();
			}

		} catch (Exception e) {
			throw new SynchroTechnicalException(String.format("Could not insert into table %s", TEMP_QUERY_PARAMETER_TABLE), e);
		} finally {
			Daos.closeSilently(statement);
		}
	}

	protected void executeUpdateColumn(PreparedStatement updateColumnStatement, List<Object> pk, String columnName, Object columnValue)
			throws SQLException {
		List<Object> params = null;
		if (debug) {
			params = Lists.newArrayList();
		}

		// Transform the column value
		// (skip if no interceptor OR no source Dao - could append in SynchroService.finish())
		if (readInterceptor != null && sourceDao != null) {
			columnValue = transformDataOnRead(columnName, columnValue, readInterceptor);
		}

		int columnCountIndex = 1;
		updateColumnStatement.setObject(columnCountIndex++, columnValue);
		if (debug) {
			params.add(columnValue);
		}
		for (Object pkColumn : pk) {
			updateColumnStatement.setObject(columnCountIndex++, pkColumn);
			if (debug) {
				params.add(pkColumn);
			}

		}

		int updateColumnCount = MapUtils.getIntValue(updateColumnCounts, columnName) + 1;
		updateColumnCounts.put(columnName, updateColumnCount);

		if (!isJdbcBatchEnable) {
			checkWriteEnable();
			if (debug) {
				log.debug(String.format("[%s] Execute update query (pk:%s), params: %s", tableName, pk, params));
			}
			int nbRowUpdated = updateColumnStatement.executeUpdate();
			if (debug) {
				Preconditions.checkArgument(nbRowUpdated == 1,
						String.format("[%s]   %s rows has been updated, but expected 1 row.", tableName, nbRowUpdated));
			}
		}
		else {
			updateColumnStatement.addBatch();
			if (updateColumnCount % batchSize == 0) {
				checkWriteEnable();
				updateColumnStatement.executeBatch();
				updateColumnStatement.clearBatch();
			}
		}
	}

	protected void closeSelectStatements() {
		if (CollectionUtils.isNotEmpty(selectStatements)) {
			for (PreparedStatement statement : selectStatements) {
				Daos.closeSilently(statement);
			}
			selectStatements.clear();
		}
	}

	protected void closeUpdateColumnStatements() {
		if (MapUtils.isNotEmpty(updateColumnStatements)) {
			for (PreparedStatement statement : updateColumnStatements.values()) {
				Daos.closeSilently(statement);
			}
			updateColumnStatements.clear();
		}
	}

	protected void closeSelectPkByIndexStatements() {
		if (MapUtils.isNotEmpty(selectPkByIndexStatements)) {
			for (PreparedStatement statement : selectPkByIndexStatements.keySet()) {
				Daos.closeSilently(statement);
			}
			selectPkByIndexStatements.clear();
		}
	}

	protected void registerSelectStatementAndClosePrevious(PreparedStatement newStatement) {
		closeSelectStatements();
		selectStatements.add(newStatement);
	}

	protected void transformAndSetData(PreparedStatement statement, ResultSet incomingData, List<Object> pk,
			SynchroOperationRepository transformBuffer, SynchroInterceptor interceptor, List<Object> debugParams)
			throws SQLException {
		Preconditions.checkNotNull(transformBuffer);

		// Transform data
		Object[] row = transformDataOnWrite(incomingData, pk, interceptor, transformBuffer);

		// Set the data into the statement
		setData(statement, row, debugParams);
	}

	protected void transformAndSetData(PreparedStatement statement, Object[] incomingData, List<Object> pk,
			SynchroOperationRepository transformBuffer, SynchroInterceptor interceptor, List<Object> debugParams)
			throws SQLException {
		Preconditions.checkNotNull(transformBuffer);

		// Transform data
		Object[] row = transformDataOnWrite(incomingData, pk, interceptor, transformBuffer);

		// Set the data into the statement
		setData(statement, row, debugParams);
	}

	protected Object[] transformDataOnWrite(ResultSet incomingData, List<Object> pk, SynchroInterceptor interceptor, SynchroOperationRepository buffer)
			throws SQLException {
		Preconditions.checkNotNull(interceptor);

		// Get data
		Object[] result = getData(incomingData);

		// Transform values
		try {
			interceptor.onWrite(result, pk, sourceDao, this, buffer);
		} catch (SynchroBusinessException e) {
			throw e;
		} catch (Exception e) {
			throw new SynchroTechnicalException(t("adagio.synchro.dao.write.tranformData.error", table.getName(), e.getMessage()), e);
		}

		return result;
	}

	protected Object[] transformDataOnWrite(Object[] incomingData, List<Object> pk, SynchroInterceptor interceptor, SynchroOperationRepository buffer)
			throws SQLException {
		Preconditions.checkNotNull(interceptor);

		// Transform values
		try {
			interceptor.onWrite(incomingData, pk, sourceDao, this, buffer);
		} catch (SynchroBusinessException e) {
			throw e;
		} catch (Exception e) {
			throw new SynchroTechnicalException(t("adagio.synchro.dao.write.tranformData.error", table.getName(), e.getMessage()), e);
		}

		return incomingData;
	}

	protected Object[] transformDataOnRead(ResultSet incomingData, SynchroInterceptor interceptor)
			throws SQLException {
		// Get data
		Object[] result = getData(incomingData);

		// Transform values
		try {
			interceptor.onRead(result, sourceDao, this);
		} catch (Exception e) {
			throw new SynchroTechnicalException(t("adagio.synchro.dao.read.tranformData.error", table.getName(), e.getMessage()), e);
		}

		return result;
	}

	protected Object[] transformDataOnRead(Object[] incomingData, SynchroInterceptor interceptor)
			throws SQLException {
		// Transform values
		try {
			interceptor.onRead(incomingData, sourceDao, this);
		} catch (SynchroBusinessException e) {
			throw e;
		} catch (Exception e) {
			throw new SynchroTechnicalException(t("adagio.synchro.dao.read.tranformData.error", table.getName(), e.getMessage()), e);
		}

		return incomingData;
	}

	protected Object transformDataOnRead(String columnName, Object columnValue, SynchroInterceptor interceptor)
			throws SQLException {
		int columnIndexInInsertQuery = table.getInsertColumnIndex(columnName);

		// Create a fake row, to be able to transform it
		Object[] fakeIncomingData = new Object[incomingDataColumnCount];
		fakeIncomingData[columnIndexInInsertQuery] = columnValue;

		// Apply transformation
		fakeIncomingData = transformDataOnRead(fakeIncomingData, readInterceptor);

		// Return the transformed value
		return fakeIncomingData[columnIndexInInsertQuery];
	}

	protected void setData(PreparedStatement statement, Object[] values, List<Object> debugParams) throws SQLException {
		for (int c = 1; c <= incomingDataColumnCount; c++) {
			Object object = values[c - 1];
			statement.setObject(c, object);
			if (debug) {
				debugParams.add(object);
			}
		}
	}

	protected void setData(PreparedStatement statement, ResultSet incomingData, List<Object> debugParams) throws SQLException {
		for (int c = 1; c <= incomingDataColumnCount; c++) {
			Object object = getObject(incomingData, c);
			statement.setObject(c, object);
			if (debug) {
				debugParams.add(object);
			}
		}
	}

	protected Object[] getData(ResultSet incomingData) throws SQLException {
		Object[] result = new Object[incomingDataColumnCount];
		for (int c = 1; c <= incomingDataColumnCount; c++) {
			Object object = getObject(incomingData, c);
			result[c - 1] = object;
		}

		return result;
	}

	public Object getObject(ResultSet incomingData, int index) throws SQLException {
		Object object = incomingData.getObject(index);

		// Convert Oracle TIMESTAMP into standard java.sql.Timestamp
		// This is required to have comparable pkStr between DBMS, when PK as some dates columns
		if (object instanceof TIMESTAMP) {
			object = ((TIMESTAMP) object).timestampValue();
		}

		// Convert into integer if not a decimal number
		// This is required to have comparable pkStr between DBMS, when ID are in BigDecimal
		else if (object instanceof BigDecimal && ((BigDecimal) object).scale() == 0) {
			object = ((BigDecimal) object).intValue();
		}
		return object;
	}

	@Override
	public List<Object> getPk(ResultSet incomingData) throws SQLException {
		List<Object> result = Lists.newArrayListWithCapacity(selectPkIndexs.length);

		for (int pkIndex : selectPkIndexs) {
			Object pk = getObject(incomingData, pkIndex);
			result.add(pk);
		}
		return result;
	}

	@Override
	public List<Object> getPk(ResultSet incomingData, boolean transform) throws SQLException {
		if (transform && readInterceptor != null) {
			// remove not Pk column (to avoid error when REMOTE_ID not found (on table LANDING))
			Object[] fakeIncominData = new Object[incomingDataColumnCount];
			for (int pkIndex : selectPkIndexs) {
				fakeIncominData[pkIndex - 1] = incomingData.getObject(pkIndex);
			}
			fakeIncominData = transformDataOnRead(fakeIncominData, readInterceptor);
			return getPk(fakeIncominData);
		}

		return getPk(incomingData);
	}

	@Override
	public Timestamp getUpdateDate(ResultSet incomingData, boolean transform) throws SQLException {
		if (transform && readInterceptor != null) {
			Object[] data = transformDataOnRead(incomingData, readInterceptor);
			return table.getUpdateDate(data);
		}

		return table.getUpdateDate(incomingData);
	}

	public List<Object> getPk(Object[] incomingData) throws SQLException {
		List<Object> result = Lists.newArrayListWithCapacity(selectPkIndexs.length);
		for (int pkIndex : selectPkIndexs) {
			Object pk = incomingData[pkIndex - 1];
			result.add(pk);
		}
		return result;
	}

	@Override
	public Map<String, List<Object>> getPkFromUniqueConstraints(ResultSet incomingData) throws SQLException {

		Object[] data = null;
		if (readInterceptor != null) {
			// Copy all fields need to execute 'getPkFromUniqueConstraints' query
			// (do not try to convert all row. e.g. when process LANDING, interceptor on LANDING.FISHING_TRIP_FK coul
			// throw exception)
			Object[] fakeIncomingData = new Object[incomingDataColumnCount];
			for (int index : incominDataIndexedNeedForIndex) {
				fakeIncomingData[index] = getObject(incomingData, index + 1);
			}

			data = transformDataOnRead(fakeIncomingData, readInterceptor);
		}
		else {
			data = getData(incomingData);
		}
		return getPkFromUniqueConstraints(data);
	}

	@Override
	public boolean lock(List<Object> pk) throws SQLException {
		if (lockStatement == null) {
			return true;
		}

		int i = 1;
		for (Object value : pk) {
			lockStatement.setObject(i++, value);
		}

		try {

			lockStatement.execute();
		} catch (SQLException e) {
			if (debug) {
				log.debug(String.format("[%s] Could not acquire lock, for pk: %s. %s", table.getName(), pk, e.getMessage()));
			}
			return false;
		}

		return true;
	}

	public List<Object> getLastGeneratedPk() {
		return this.lastGeneratedPk;
	}

	/* -- internal methods -- */

	protected List<List<Object>> getPksByFks(Set<String> fkColumnNames, List<List<Object>> fkColumnsValues, Map<String, Object> bindings,
			boolean returnIfMatchValues)
			throws SQLException {
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(fkColumnNames));
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(fkColumnsValues));

		// Insert values into table TEMP_QUERY_PARAMETER
		Integer userId = (Integer) bindings.get("userId");
		if (userId == null) {
			userId = new Integer(-1);
		}
		executeInsertIntoTempQueryParameter(fkColumnNames, fkColumnsValues, TEMP_QUERY_PARAMETER_PARAM_NAME, userId, false);

		// Execute query and fetch into result list
		List<List<Object>> result = Lists.newArrayList();

		PreparedStatement statement = null;
		try {
			// Prepare then execute the sql query
			String sql = returnIfMatchValues
					? createSelectByManyFksUsingTempParameterTable(table.getSelectPksQuery(), fkColumnNames, TEMP_QUERY_PARAMETER_PARAM_NAME)
					: createSelectNotMatchManyFksUsingTempParameterTable(table.getSelectPksQuery(), fkColumnNames, TEMP_QUERY_PARAMETER_PARAM_NAME);
			statement = prepareAndBindStatement(sql, bindings, "selectPksByFks");
			statement.setFetchSize(batchSize);
			ResultSet rows = statement.executeQuery();

			int pkCount = table.getPkNames().size();
			while (rows.next()) {
				List<Object> pk = Lists.newArrayListWithCapacity(pkCount);

				for (int i = 1; i <= pkCount; i++) {
					Object value = getObject(rows, i);
					pk.add(value);
				}
				result.add(pk);
			}

			return result;
		} finally {
			Daos.closeSilently(statement);
		}
	}

	/**
	 * @param fkColumnNames
	 * @param fkColumnsValues
	 * @param bindings
	 * @param returnIfMatchValues
	 *            true to return rows that match FKvalues, false to return row that not match
	 * @return
	 * @throws SQLException
	 */
	protected Map<String, Timestamp> getPksStrWithUpdateDateByFks(Set<String> fkColumnNames, List<List<Object>> fkColumnsValues,
			Map<String, Object> bindings,
			boolean returnIfMatchValues) throws SQLException {
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(fkColumnNames));
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(fkColumnsValues));

		boolean enableUpdateDate = table.isWithUpdateDateColumn();

		// Insert values into table TEMP_QUERY_PARAMETER
		Integer userId = (Integer) bindings.get("userId");
		if (userId == null) {
			userId = new Integer(-1);
		}
		executeInsertIntoTempQueryParameter(fkColumnNames, fkColumnsValues, TEMP_QUERY_PARAMETER_PARAM_NAME, userId, false);

		// Execute query and fetch into result list
		Map<String, Timestamp> result = Maps.newHashMap();
		Timestamp fakeTimestamp = new Timestamp(0);

		PreparedStatement statement = null;
		try {

			// Fetch directly as pkStr, if enable
			if (table.isSelectPrimaryKeysAsStringQueryEnable()) {

				// Prepare then execute the sql query
				String sql = returnIfMatchValues
						? createSelectByManyFksUsingTempParameterTable(table.getSelectPksStrQuery(), fkColumnNames, TEMP_QUERY_PARAMETER_PARAM_NAME)
						: createSelectNotMatchManyFksUsingTempParameterTable(table.getSelectPksStrQuery(), fkColumnNames,
								TEMP_QUERY_PARAMETER_PARAM_NAME);

				statement = prepareAndBindStatement(sql, bindings, "selectPksStrByFks");
				statement.setFetchSize(batchSize);
				ResultSet rows = statement.executeQuery();

				while (rows.next()) {
					String pkStr = rows.getString(1);
					if (enableUpdateDate) {
						Timestamp updateDate = rows.getTimestamp(2);
						result.put(pkStr, updateDate);
					}
					else {
						result.put(pkStr, fakeTimestamp);
					}
				}

			}

			// Fetch from a pk list to pkStr
			else {
				// Prepare then execute the sql query
				String sql = returnIfMatchValues
						? createSelectByManyFksUsingTempParameterTable(table.getSelectPksQuery(), fkColumnNames, TEMP_QUERY_PARAMETER_PARAM_NAME)
						: createSelectNotMatchManyFksUsingTempParameterTable(table.getSelectPksQuery(), fkColumnNames,
								TEMP_QUERY_PARAMETER_PARAM_NAME);

				statement = prepareAndBindStatement(sql, bindings, "selectPksByFks");
				statement.setFetchSize(batchSize);
				ResultSet rows = statement.executeQuery();

				int pkCount = table.getPkNames().size();
				List<Object> pk = Lists.newArrayListWithCapacity(pkCount);
				while (rows.next()) {
					for (int i = 1; i <= pkCount; i++) {
						Object value = getObject(rows, i);
						pk.add(value);
					}

					String pkStr = SynchroTableMetadata.toPkStr(pk);
					pk.clear();

					if (enableUpdateDate) {
						Timestamp updateDate = rows.getTimestamp(pkCount + 1);
						result.put(pkStr, updateDate);
					}
					else {
						result.put(pkStr, fakeTimestamp);
					}
				}
			}

			return result;
		} finally {
			Daos.closeSilently(statement);
		}
	}

	protected Map<String, List<Object>> getPkFromUniqueConstraints(Object[] data) throws SQLException {
		Map<String, List<Object>> result = Maps.newHashMap();

		Iterator<String> constraintNames = Lists.newArrayList(table.getUniqueConstraints().keySet()).iterator();

		// For each index (with statement already prepared)
		for (Entry<PreparedStatement, int[]> entry : selectPkByIndexStatements.entrySet()) {
			PreparedStatement statement = entry.getKey();
			int[] columnIndexes = entry.getValue();
			String constraintName = constraintNames.next();

			// Bind the select query with index columns, then execute the query
			int i = 1;
			for (int columnIndex : columnIndexes) {
				Object columnValue = data[columnIndex - 1];

				statement.setObject(i++, columnValue);
			}

			ResultSet pkResultset = statement.executeQuery();

			// Fetch result and store each PKs found
			List<Object> pk = null;
			int pkColumnCount = table.getPkNames().size();
			while (pkResultset.next()) {
				if (pk != null) {
					log.warn(String.format("More than one row when check unique constraints %s, when expected only one row.", constraintName));
					break;
				}
				pk = Lists.newArrayListWithCapacity(pkColumnCount);
				for (int j = 1; j <= pkColumnCount; j++) {
					Object value = getObject(pkResultset, j);
					pk.add(value);
				}
			}
			if (pk != null) {
				result.put(constraintName, pk);
			}
			Daos.closeSilently(pkResultset);
		}

		return result;
	}

	protected ResultSet getDataByFkUsingIn(String fkColumnName, List<List<Object>> values, Map<String, Object> bindings) throws SQLException {

		String sql = createSelectByFkWithInQuery(table.getSelectAllQuery(), fkColumnName, values.size());

		Preconditions.checkNotNull(sql, String.format("Columns %s is not referenced for table %s", fkColumnName, table.getName()));

		if (log.isDebugEnabled()) {
			log.debug(String.format("[%s] Prepare select query: %s", table.getName(), sql));
		}

		PreparedStatement statement = prepareAndBindSelectStatement(sql, bindings, "select");
		registerSelectStatementAndClosePrevious(statement);

		int paramIndex = 1;
		for (List<Object> value : values) {
			statement.setObject(paramIndex, value.iterator().next());
			paramIndex++;
		}

		statement.setFetchSize(batchSize);

		ResultSet result = statement.executeQuery();
		return result;
	}

	protected ResultSet getDataByFkUsingTempParameterTable(String fkColumnName, List<Object> values, Map<String, Object> bindings)
			throws SQLException {
		String queryParameterName = fkColumnName;

		String sql = createSelectByOneFkUsingTempParameterTable(table.getSelectAllQuery(), fkColumnName, queryParameterName);

		if (log.isDebugEnabled()) {
			log.debug(String.format("[%s] Prepare select query: %s", table.getName(), sql));
		}

		// Insert values into table TEMP_QUERY_PARAMETER
		Integer userId = (Integer) bindings.get("userId");
		if (userId == null) {
			userId = new Integer(-1);
		}
		delegate.executeInsertIntoTempQueryParameter(values, queryParameterName, userId);

		// Then execute the query
		PreparedStatement statement = prepareAndBindSelectStatement(sql, bindings, "selectByFk");
		registerSelectStatementAndClosePrevious(statement);
		statement.setFetchSize(batchSize);
		ResultSet result = statement.executeQuery();

		return result;
	}

	protected ResultSet getDataByFksUsingTempParameterTable(Set<String> columnNames, List<List<Object>> columnValues, Map<String, Object> bindings)
			throws SQLException {
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(columnNames));

		String baseSql = table.getSelectAllQuery();

		String sql = createSelectByManyFksUsingTempParameterTable(baseSql, columnNames, TEMP_QUERY_PARAMETER_PARAM_NAME);

		if (log.isDebugEnabled()) {
			log.debug(String.format("[%s] Prepare select query: %s", table.getName(), sql));
		}

		// Insert values into table TEMP_QUERY_PARAMETER
		Integer userId = (Integer) bindings.get("userId");
		if (userId == null) {
			userId = new Integer(-1);
		}
		executeInsertIntoTempQueryParameter(columnNames, columnValues, TEMP_QUERY_PARAMETER_PARAM_NAME, userId, false);

		// Then execute the query
		PreparedStatement statement = prepareAndBindSelectStatement(sql, bindings, "select");
		registerSelectStatementAndClosePrevious(statement);
		statement.setFetchSize(batchSize);
		ResultSet result = statement.executeQuery();
		return result;
	}

	private SynchroInterceptor createReadInterceptor(SynchroTableMetadata table) {
		List<SynchroInterceptor> interceptors = table.getInterceptors();
		if (CollectionUtils.isEmpty(interceptors)) {
			return null;
		}
		List<SynchroInterceptor> readInterceptors = Lists.newArrayList();

		try {
			for (SynchroInterceptor interceptor : interceptors) {
				if (interceptor.enableOnRead()) {
					readInterceptors.add(interceptor);
				}
			}
		} catch (Exception e) {
			throw new SynchroTechnicalException("Could not initialize DAO read interceptors.", e);
		}

		if (CollectionUtils.isEmpty(readInterceptors)) {
			return null;
		}

		return SynchroInterceptorUtils.chain(readInterceptors, SynchroInterceptorBase.class);
	}

	private SynchroInterceptor createWriteInterceptor(SynchroTableMetadata table) {
		List<SynchroInterceptor> interceptors = table.getInterceptors();
		if (CollectionUtils.isEmpty(interceptors)) {
			return null;
		}
		List<SynchroInterceptor> writeInterceptors = Lists.newArrayList();

		try {
			for (SynchroInterceptor interceptor : interceptors) {
				if (interceptor.enableOnWrite()) {
					SynchroInterceptor newInterceptor = interceptor.clone();
					writeInterceptors.add(newInterceptor);
				}
			}
		} catch (Exception e) {
			throw new SynchroTechnicalException("Could not initialize DAO read interceptors.", e);
		}

		if (CollectionUtils.isEmpty(writeInterceptors)) {
			return null;
		}

		return SynchroInterceptorUtils.chain(writeInterceptors, SynchroInterceptorBase.class);
	}

	protected String createSelectSequenceNextValString(Dialect dialect, SynchroTableMetadata table) {
		String sequenceName = table.getSequenceName();
		if (StringUtils.isBlank(sequenceName)) {
			return null;
		}
		return dialect.getSelectSequenceNextValString(sequenceName);
	}

	protected String createSequenceNextValString(Dialect dialect, SynchroTableMetadata table) {
		String sequenceName = table.getSequenceName();
		if (StringUtils.isBlank(sequenceName)) {
			return null;
		}
		// Check sequence name validity, for Oracle
		if (dialect.getClass().getSimpleName().startsWith("Oracle")) {
			// TODO : get the maxSequenceName it from dialect ?
			int maxSequenceName = 30;
			if (sequenceName.length() > maxSequenceName) {
				throw new SynchroTechnicalException(String.format("Sequence name '%s': exceed the database max length of %s caracters", sequenceName,
						maxSequenceName));
			}
		}
		return dialect.getSequenceNextValString(sequenceName);
	}

	protected int createIncomingDataColumnCount(SynchroTableMetadata table) {
		String selectSql = table.getSelectAllQuery();

		// Make sure the PK is present as the last bind param
		int selectColumnCount = SynchroQueryBuilder.newBuilder(selectSql)
				.getColumnCount();

		return selectColumnCount;
	}

	protected PreparedStatement createInsertStatement(Connection connection, Dialect dialect, SynchroTableMetadata table)
			throws SQLException {
		String sql = table.getInsertQuery();

		String dialectSelectSequenceNextValString = createSelectSequenceNextValString(dialect, table);
		String oldSequenceNextValString = table.getSelectSequenceNextValString();

		// If the insert query call a sequence
		if (StringUtils.isNotBlank(dialectSelectSequenceNextValString)
				&& StringUtils.isNotBlank(oldSequenceNextValString)
				&& sql.contains(oldSequenceNextValString)) {

			// Adapt the sequence call for the current dialect
			sql = sql.replaceAll(oldSequenceNextValString, dialectSelectSequenceNextValString);
		}

		if (log.isDebugEnabled()) {
			log.debug(String.format("[%s] Prepare insert query: %s", table.getName(), sql));
		}

		PreparedStatement statement = connection.prepareStatement(sql);
		return statement;
	}

	protected String createInsertWithPkBindQuery(SynchroTableMetadata table)
			throws SQLException {
		Preconditions.checkArgument(CollectionUtils.size(table.getPkNames()) > 0, "Table should have a PK to create insert query with PK Bind");

		SynchroQueryBuilder query = SynchroQueryBuilder.newBuilder(table.getInsertQuery());
		for (String pkName : table.getPkNames()) {
			// Make sure the PK is binding
			query.setColumnValue(pkName, "?");
		}

		String sql = query.build();

		if (log.isDebugEnabled()) {
			log.debug(String.format("[%s] Prepare insert query (force PK binding): %s", table.getName(), sql));
		}

		return sql;
	}

	private int[] initInsertPkIndexes(String insertWithPkBindQuery) {
		Set<String> pkNames = table.getPkNames();

		SynchroInsertQuery insertQuery = (SynchroInsertQuery) SynchroQueryBuilder.newBuilder(insertWithPkBindQuery);

		int[] insertPkIndexes = new int[pkNames.size()];
		int i = 0;
		for (String pkName : pkNames) {
			int pkIndex = insertQuery.getBindingColumnNames().indexOf(pkName);
			if (pkIndex != -1) {
				insertPkIndexes[i++] = pkIndex + 1;
			}
			else {
				throw new SynchroTechnicalException(String.format("Could not retrieve PK column %s in the sinsert query", pkName));
			}
		}

		return insertPkIndexes;
	}

	private String initCountAllApproxQuery(Dialect dialect,
			SynchroDatabaseConfiguration configuration,
			SynchroTableMetadata table) {
		String result = null;

		// Special case for Oracle: use
		if (Daos.isOracleDatabase(configuration.getUrl())) {
			String jdbcSchema = configuration.getJdbcSchema();
			String jdbcUser = configuration.getJdbcUser();

			// If schema is empty, or = user
			if (StringUtils.isBlank(jdbcSchema)
					|| jdbcSchema.equalsIgnoreCase(jdbcUser)) {
				result = String.format("SELECT num_rows FROM user_tables WHERE table_name='%s'",
						table.getName());
			}

			// If schema != user
			else {
				result = String.format("SELECT num_rows FROM all_tables WHERE owner='%s' AND table_name='%s'",
						jdbcSchema,
						table.getName());
			}
		}

		return result;

	}

	private PreparedStatement createUpdateStatement(Connection connection, SynchroTableMetadata table) throws SQLException {
		String sql = table.getUpdateQuery();

		// Check error (due to mistake in Interceptors...)
		Preconditions.checkArgument(sql.toUpperCase().startsWith("UPDATE"),
				String.format("[%s] Update SQL query should be like 'UPDATE ...' but was: %s", table.getName(), sql));

		if (log.isDebugEnabled()) {
			log.debug(String.format("[%s] Prepare update query: %s", table.getName(), sql));
		}

		PreparedStatement statement = connection.prepareStatement(sql);
		return statement;
	}

	private PreparedStatement createDeleteStatement(Connection connection, SynchroTableMetadata table) throws SQLException {
		String sql = table.getDeleteByPkQuery();

		// Check error (due to mistake in Interceptors...)
		Preconditions.checkArgument(sql.toUpperCase().startsWith("DELETE"),
				String.format("[%s] Update SQL query should be like 'DELETE ...' but was: %s", table.getName(), sql));

		if (log.isDebugEnabled()) {
			log.debug(String.format("[%s] Prepare delete query: %s", table.getName(), sql));
		}

		PreparedStatement statement = connection.prepareStatement(sql);
		return statement;
	}

	private PreparedStatement createLockStatement(Connection connection, SynchroTableMetadata table) throws SQLException {

		LockMode lockMode = table.getLockModeOnUpdate();

		String selectPkSql = StringUtils.trimToNull(table.getSelectPksQuery());
		String forUpdateString = StringUtils.trimToNull(dialect.getForUpdateString(lockMode));

		if (lockMode == LockMode.NONE || selectPkSql == null || forUpdateString == null) {
			return null;
		}

		String sql = String.format("%s WHERE %s %s",
				selectPkSql,
				table.createPkWhereClause("t"),
				forUpdateString);

		PreparedStatement statement = connection.prepareStatement(sql);
		return statement;
	}

	private Map<PreparedStatement, int[]> createSelectPkByIndexStatement(Connection connection, SynchroTableMetadata table) throws SQLException {
		Map<String, List<String>> uniqueConstraints = table.getUniqueConstraints();
		if (MapUtils.isEmpty(uniqueConstraints)) {
			return null;
		}

		String pkName = null;
		if (enableInsertWithPkBind) {
			pkName = table.getPkNames().iterator().next();
		}
		// Retrieve the column names as define in the insert order
		SynchroQueryBuilder insertQuery = SynchroQueryBuilder.newBuilder(table.getInsertQuery());
		Map<String, Integer> columnNameIndexes = Maps.newHashMap();
		int i = 1;
		for (String columnName : insertQuery.getColumnNames()) {
			String insertColumnValue = insertQuery.getColumnValue(columnName);
			if ("?".equals(insertColumnValue) || insertColumnValue.startsWith(":")) {
				columnNameIndexes.put(columnName, i);
			}
			else if (enableInsertWithPkBind && columnName.equals(pkName)) {
				columnNameIndexes.put(pkName, i);
			}
			i++;
		}

		Map<PreparedStatement, int[]> results = Maps.newLinkedHashMap();

		for (Entry<String, List<String>> entry : uniqueConstraints.entrySet()) {
			String indexName = entry.getKey();
			List<String> columnNames = entry.getValue();

			String sql = table.getSelectPkByIndex(indexName);
			PreparedStatement statement = connection.prepareStatement(sql);
			selectStatements.add(statement);

			int[] bindingColumnIndexes = new int[columnNames.size()];
			int j = 0;
			for (String columnName : columnNames) {
				Integer bindingColumnIndex = columnNameIndexes.get(columnName);

				if (bindingColumnIndex == null) {
					log.warn(String.format("[%s] Ignore unique constraints %s, because column %s not bind in insert query",
							table.getName(),
							indexName,
							columnName));
					results.remove(statement);
					break;
				}
				bindingColumnIndexes[j++] = bindingColumnIndex;

				// WARNING: If nullable column, should be bind twice
				if (table.getColumn(columnName).isNullable()) {
					bindingColumnIndexes = ArrayUtils.add(bindingColumnIndexes, j++, bindingColumnIndex);
				}
			}
			results.put(statement, bindingColumnIndexes);
		}

		return results;
	}

	protected int[] createIncomingDataIndexedNeedForIndex(Map<PreparedStatement, int[]> selectPkByIndexStatements, int incomingDataColumnCount) {
		if (MapUtils.isEmpty(selectPkByIndexStatements)) {
			return null;
		}

		List<Integer> result = Lists.newArrayList();
		for (int[] indexForUniqueConstraint : selectPkByIndexStatements.values()) {
			for (int colIndex : indexForUniqueConstraint) {
				if (!result.contains(result)) {
					result.add(colIndex - 1);
				}
			}
		}

		return toArray(result);
	}

	protected String createSelectByFkWithInQuery(String selectQuery, String columnName, int nbValues) {
		// Params
		StringBuilder params = new StringBuilder();
		for (int i = 0; i < nbValues; i++) {
			params.append(",?");
		}

		SynchroSelectQuery query = null;
		if (keepWhereClauseOnQueriesByFks) {
			query = (SynchroSelectQuery) SynchroQueryBuilder.newBuilder(selectQuery);
		}
		else {
			// Create a new select, with a simple from clause 'FROM TABLE t'
			List<String> columnNames = SynchroQueryBuilder.newBuilder(selectQuery).getColumnNamesWithAlias();
			query = new SynchroSelectQuery(null, table.getName(), columnNames, null);
			query.setWhereClause(null); // make sure there is no where clause
		}

		// Create a new select, with a simple from clause 'FROM TABLE t'
		query.setTableAlias("t");

		// Add inner join join to TempQueryParameter table
		query.addWhere(SynchroQueryOperator.AND,
				String.format("t.%s IN (%s)",
						columnName,
						params.substring(1)));

		return query.build();
	}

	protected String createSelectByOneFkUsingTempParameterTable(String selectQuery, String fkColumnName, String queryParameterName) {

		SynchroSelectQuery query = null;
		if (keepWhereClauseOnQueriesByFks) {
			query = (SynchroSelectQuery) SynchroQueryBuilder.newBuilder(selectQuery);
		}
		else {
			// Create a new select, with a simple from clause 'FROM TABLE t'
			List<String> columnNames = SynchroQueryBuilder.newBuilder(selectQuery).getColumnNamesWithAlias();
			query = new SynchroSelectQuery(null, table.getName(), columnNames, null);
			query.setWhereClause(null); // make sure there is no where clause
		}

		query.setTableAlias("t");

		// Add inner join join to TempQueryParameter table
		query.addJoin(String.format(
				"INNER JOIN TEMP_QUERY_PARAMETER p on t.%s = p.ALPHANUMERICAL_VALUE AND p.PARAMETER_NAME='%s' AND p.PERSON_FK=:userId",
				fkColumnName,
				queryParameterName));

		return query.build();
	}

	protected String createSelectByManyFksUsingTempParameterTable(String selectQuery, Set<String> fkColumnNames, String queryParameterName) {

		SynchroSelectQuery query = null;
		if (keepWhereClauseOnQueriesByFks) {
			query = (SynchroSelectQuery) SynchroQueryBuilder.newBuilder(selectQuery);
		}
		else {
			// Create a new select, with a simple from clause 'FROM TABLE t'
			List<String> columnNames = SynchroQueryBuilder.newBuilder(selectQuery).getColumnNamesWithAlias();
			query = new SynchroSelectQuery(null, table.getName(), columnNames, null);
			query.setWhereClause(null); // make sure there is no where clause
		}

		query.setTableAlias("t");

		// Add a join to TempQueryParameter table
		query.addJoin(createJoinUsingTempParameterTable(fkColumnNames, queryParameterName));

		return query.build();
	}

	protected String createJoinUsingTempParameterTable(Set<String> fkColumnNames, String queryParameterName) {

		// Compute the inner join clause
		int index = 0;
		StringBuilder joinBuffer = new StringBuilder();
		for (String columnName : fkColumnNames) {
			String alias = "tqp_" + index;
			joinBuffer.append(String.format(" INNER JOIN TEMP_QUERY_PARAMETER %s",
					alias
					));
			joinBuffer.append(String.format(" ON %s.alphanumerical_value=t.%s",
					alias,
					columnName
					));
			joinBuffer.append(String.format(" AND %s.parameter_name='%s_%s'",
					alias,
					queryParameterName,
					index
					));
			if (index > 0) {
				joinBuffer.append(String.format(" AND %s.numerical_value=tqp_0.numerical_value",
						alias
						));
			}
			index++;
		}
		return joinBuffer.toString();
	}

	protected String createSelectNotMatchManyFksUsingTempParameterTable(String selectQuery, Set<String> fkColumnNames, String queryParameterName) {

		SynchroSelectQuery query = null;
		if (keepWhereClauseOnQueriesByFks) {
			query = (SynchroSelectQuery) SynchroQueryBuilder.newBuilder(selectQuery);
		}
		else {
			// Create a new select, with a simple from clause 'FROM TABLE t'
			List<String> columnNames = SynchroQueryBuilder.newBuilder(selectQuery).getColumnNamesWithAlias();
			query = new SynchroSelectQuery(null, table.getName(), columnNames, null);
			query.setWhereClause(null); // make sure there is no where clause
		}

		query.setTableAlias("t");

		// Add a join to TempQueryParameter table
		addNotInUsingTempParameterTable(query, fkColumnNames, queryParameterName);

		return query.build();
	}

	protected SynchroSelectQuery addNotInUsingTempParameterTable(SynchroSelectQuery query, Set<String> fkColumnNames, String queryParameterName) {

		// Compute the inner join clause
		int index = 0;
		StringBuilder joinBuffer = new StringBuilder();
		StringBuilder whereBuffer = new StringBuilder();
		for (String columnName : fkColumnNames) {
			String alias = "tqp_" + index;
			joinBuffer.append(String.format(" LEFT OUTER JOIN TEMP_QUERY_PARAMETER %s",
					alias
					));
			joinBuffer.append(String.format(" ON %s.alphanumerical_value=t.%s",
					alias,
					columnName
					));
			joinBuffer.append(String.format(" AND %s.parameter_name='%s_%s'",
					alias,
					queryParameterName,
					index
					));
			if (index > 0) {
				joinBuffer.append(String.format(" AND %s.numerical_value=tqp_0.numerical_value",
						alias
						));
			}
			if (index > 0) {
				whereBuffer.append(" AND ");
			}
			whereBuffer.append(String.format("%s.id is null",
					alias
					));
			index++;
		}

		query.addJoin(joinBuffer.toString());

		// Add where clause
		query.addWhere(SynchroQueryOperator.AND, whereBuffer.toString());

		return query;
	}

	protected String createUpdateColumnQuery(String columnName, String referenceTableName) {
		// Could not update, because no PK found
		if (CollectionUtils.isEmpty(table.getPkNames())) {
			return null;
		}

		String result = String.format("UPDATE %s SET %s = ? WHERE %s",
				table.getName(),
				columnName,
				table.createPkWhereClause()
				);
		return result;

	}

	protected PreparedStatement prepareAndBindSelectStatement(String sql, Map<String, Object> bindingMap, String queryTypeName) throws SQLException {
		PreparedStatement statement = prepareAndBindStatement(sql, bindingMap, queryTypeName);
		statement.setFetchSize(batchSize);
		statement.setFetchDirection(ResultSet.FETCH_FORWARD);
		return statement;
	}

	protected PreparedStatement prepareAndBindStatement(String sql, Map<String, Object> bindingMap, String queryTypeName) throws SQLException {
		StringBuilder sb = new StringBuilder();

		StringBuilder debugParams = null;
		if (debug) {
			debugParams = new StringBuilder();
		}

		List<Object> orderedBindingValues = Lists.newArrayList();
		Matcher paramMatcher = Pattern.compile(":[a-zA-Z_0-9]+").matcher(sql);
		int offset = 0;
		while (paramMatcher.find()) {
			String bindingName = sql.substring(paramMatcher.start() + 1, paramMatcher.end());
			Object bindingValue = bindingMap.get(bindingName);
			if (bindingValue == null && !bindingMap.containsKey(bindingName)) {
				log.error(t("adagio.synchro.bindingQuery.error.log",
						table.getName(),
						bindingName,
						sql));
				throw new SynchroTechnicalException(t("adagio.synchro.bindingQuery.error",
						table.getName()));
			}
			orderedBindingValues.add(bindingValue);
			sb.append(sql.substring(offset, paramMatcher.start()))
					.append("?");
			offset = paramMatcher.end();

			if (debug) {
				debugParams.append(", ").append(bindingValue);
			}
		}
		if (offset > 0) {
			if (offset < sql.length()) {
				sb.append(sql.substring(offset));
			}
			sql = sb.toString();
		}

		if (debug) {
			log.debug(String.format("[%s] Execute %s query: %s", table.getName(), queryTypeName, sql));
			log.debug(String.format("[%s]          with params: [%s]", table.getName(), debugParams.length() > 2 ? debugParams.substring(2)
					: "no binding"));
		}

		PreparedStatement statement = connection.prepareStatement(sql);

		int index = 1;
		for (Object value : orderedBindingValues) {
			statement.setObject(index, value);
			index++;
		}

		return statement;
	}

	/**
	 * Make sure all types are valid for bindings.
	 * <p/>
	 * For example, convert : Date -> Timestamp
	 * 
	 * @param bindings
	 * @return
	 */
	protected Map<String, Object> prepareBindings(Map<String, Object> bindings) {
		if (MapUtils.isEmpty(bindings)) {
			return Maps.newHashMap();
		}
		// Copy bindings
		Map<String, Object> result = Maps.newHashMap(bindings);

		for (Entry<String, Object> entry : bindings.entrySet()) {
			String bindingName = entry.getKey();
			Object bindingValue = entry.getValue();

			// Tranform Date -> Timestamp
			if (bindingValue instanceof Date) {
				bindingValue = new Timestamp(((Date) bindingValue).getTime());
				result.put(bindingName, bindingValue);
			}
		}

		return result;
	}

	protected PreparedStatement createUpdateColumnStatement(String columnName) throws SQLException {

		SynchroColumnMetadata column = table.getColumn(columnName);
		if (column == null) {
			throw new SynchroTechnicalException(String.format("No column %s found in table %s", columnName.toUpperCase(), table.getName()));
		}
		SynchroJoinMetadata join = column.getParentJoin();

		String referenceTableName = null;
		if (join != null) {
			referenceTableName = join.getTargetTable().getName();
		}

		String sql = createUpdateColumnQuery(columnName, referenceTableName);

		PreparedStatement updateColumnStatement = connection.prepareStatement(sql);

		return updateColumnStatement;
	}

	protected void checkPkNotUsedBeforeDelete(List<Object> pk) throws SQLException, DataIntegrityViolationOnDeleteException {

		// Retrieve queries to count rows that used a table
		Map<String, String> countPkReferenceQueries = table.getCountPkReferenceQueries(getConnection().getMetaData());

		// No queries to execute
		if (MapUtils.isEmpty(countPkReferenceQueries)) {
			return;
		}

		Set<String> existingReferenceTables = Sets.newHashSet();
		for (String fkTableName : countPkReferenceQueries.keySet()) {
			String countQuery = countPkReferenceQueries.get(fkTableName);
			long count = executeCountQueryByPk(countQuery, pk);
			if (count > 0) {
				existingReferenceTables.add(fkTableName);
			}
		}

		// If ths PK is still referenced: throw a error
		if (CollectionUtils.isNotEmpty(existingReferenceTables)) {
			// throw (mantis #23383)
			throw new DataIntegrityViolationOnDeleteException(
					t("adagio.synchro.synchronize.checkPkNotUsed.error",
							table.getName(),
							SynchroTableMetadata.toPkStr(pk),
							Joiner.on(',').join(existingReferenceTables)
					),
					table.getName(),
					SynchroTableMetadata.toPkStr(pk),
					existingReferenceTables);
		}
	}

	protected long executeCountQueryByPk(String countQuery, List<Object> pk) throws SQLException {

		// Prepare the binding params, from PK
		Map<String, Object> bindings = Maps.newHashMap();
		int i = 1;
		for (Object value : pk) {
			bindings.put("pk" + i, value);
			i++;
		}

		PreparedStatement statement = null;
		ResultSet resultSet = null;
		try {

			statement = prepareAndBindStatement(countQuery, bindings, "countQueryByPk");
			resultSet = statement.executeQuery();

			// Check the query result
			if (!resultSet.next()) {
				throw new SynchroTechnicalException(String.format("Invalid count query, because no row are returned [%s]", countQuery));
			}
			Object value = resultSet.getObject(1);
			if (value == null || !(value instanceof Number)) {
				throw new SynchroTechnicalException(String.format("Invalid count query, because not returning a Number value [%s]", countQuery));
			}

			// Final cast and return
			Number count = (Number) value;
			return count.longValue();
		} finally {
			Daos.closeSilently(resultSet);
			Daos.closeSilently(statement);
		}
	}

	private void checkWriteEnable() {
		Preconditions.checkArgument(enableWrite, "Unable to write data, because the target database is read only.");
	}

	/**
	 * Convert into int[] a not null list of not null integers
	 * 
	 * @param list
	 *            a not null list
	 * @return an int array
	 */
	private int[] toArray(List<Integer> list) {
		int[] result = new int[list.size()];
		int i = 0;
		for (Integer v : list) {
			result[i++] = v;
		}
		return result;
	}
}
