package fr.ifremer.adagio.synchro.dao;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.hibernate.dialect.Dialect;

import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata;
import fr.ifremer.adagio.synchro.service.SynchroTableOperation;

public interface SynchroTableDao extends SynchroBaseDao {

	SynchroTableMetadata getTable();

	void setCurrentOperation(SynchroTableOperation pendingChangesBuffer);

	SynchroTableOperation getCurrentOperation();

	/**
	 * Obtains the max update date found in table's rows
	 * 
	 * @return
	 * @throws SQLException
	 */
	Timestamp getLastUpdateDate() throws SQLException;

	/**
	 * Count the number of rows in the table.<br/>
	 * Zero value is always exact, even if approximate=true
	 * 
	 * @param approximate
	 *            is approximation allow ? (e.g. for big table, approximate is must faster)
	 * @return
	 * @throws SQLException
	 */
	long countAll(boolean approximate) throws SQLException;

	/**
	 * Return if a PK already exists on table
	 * 
	 * @param pk
	 * @return
	 * @throws SQLException
	 */
	boolean exists(List<Object> pk) throws SQLException;

	/**
	 * Count the number of rows updated since the given date
	 * 
	 * @param fromDate
	 * @return
	 * @throws SQLException
	 * @deprecated use count(Map<String, Object>) instead
	 */
	@Deprecated
	long countDataToUpdate(Date fromDate) throws SQLException;

	/**
	 * Count the number of rows
	 * 
	 * @param bindings
	 *            binding parameters
	 * @return
	 * @throws SQLException
	 */
	long countData(Map<String, Object> bindings) throws SQLException;

	/**
	 * Count the number of rows, by a set of FK value
	 * 
	 * @param bindings
	 *            binding parameters
	 * @return
	 * @throws SQLException
	 */
	long countDataByFks(Set<String> fkColumnNames, List<List<Object>> fkColumnsValues, Map<String, Object> bindings) throws SQLException;

	/**
	 * Obtains all primary keys of a table.
	 * If more than one key in tha table, use a serialization {@link SynchroTableMetadata#toPkStr}
	 * 
	 * @return
	 * @throws SQLException
	 */
	Set<String> getPksStr() throws SQLException;

	/**
	 * Obtains existing PKs with update dates (by local id)
	 * 
	 * @return
	 * @throws SQLException
	 */
	Map<String, Timestamp> getPksStrWithUpdateDate() throws SQLException;

	Map<String, Timestamp> getPksStrWithUpdateDateByFks(Set<String> fkColumnNames, List<List<Object>> fkColumnsValues, Map<String, Object> bindings)
			throws SQLException;

	/**
	 * Getting all row depending of a set of column values
	 * <p/>
	 * i.e. [[COLUMN_FK1=VAL1_1, COLUMN_FK2=VAL1_2], [COLUMN_FK1=VAL2_1, COLUMN_FK2=VAL2_2]]
	 * 
	 * @param fkColumnNames
	 * @param fkColumnsValues
	 * @param bindings
	 * @return
	 * @throws SQLException
	 */
	ResultSet getDataByFks(Set<String> fkColumnNames, List<List<Object>> fkColumnsValues, Map<String, Object> bindings) throws SQLException;

	/**
	 * Getting all PK depending of a set of column values
	 * <p/>
	 * i.e. [[COLUMN_FK1=VAL1_1, COLUMN_FK2=VAL1_2], [COLUMN_FK1=VAL2_1, COLUMN_FK2=VAL2_2]]
	 * 
	 * @param fkColumnNames
	 * @param fkColumnsValues
	 * @param bindings
	 * @return list of pk found
	 * @throws SQLException
	 */
	List<List<Object>> getPksByFks(Set<String> fkColumnNames, List<List<Object>> fkColumnsValues, Map<String, Object> bindings) throws SQLException;

	/**
	 * Return all PK from rows that NOT match given values
	 * 
	 * @param fkColumnNames
	 * @param fkColumnsValues
	 * @param bindings
	 * @return list of pk found, that NOT match the given FK values
	 * @throws SQLException
	 */
	List<List<Object>> getPksByNotFoundFks(Set<String> fkColumnNames, List<List<Object>> fkColumnsValues, Map<String, Object> bindings)
			throws SQLException;

	/**
	 * Same, but return a list of pkStr
	 * 
	 * @param fkColumnNames
	 * @param fkColumnsValues
	 * @param bindings
	 * @return
	 * @throws SQLException
	 */
	Set<String> getPksStrByFks(Set<String> fkColumnNames, List<List<Object>> fkColumnsValues, Map<String, Object> bindings) throws SQLException;

	/**
	 * Return that NOT match given values
	 * 
	 * @param fkColumnNames
	 * @param fkColumnsValues
	 * @param bindings
	 * @return
	 * @throws SQLException
	 */
	Set<String> getPksStrByNotFoundFks(Set<String> fkColumnNames, List<List<Object>> fkColumnsValues, Map<String, Object> bindings)
			throws SQLException;

	/**
	 * Obtains all data updated since the given date (or all data if the given date is null
	 * 
	 * @param fromDate
	 * @return A Resulset with expected data.
	 * @throws SQLException
	 * @deprecated use getData(map) instead
	 */
	@Deprecated
	ResultSet getDataToUpdate(Date fromDate) throws SQLException;

	/**
	 * Obtains all data updated since the given date (or all data if the given date is null
	 * 
	 * @param fromDate
	 * @return A Resulset with expected data.
	 * @throws SQLException
	 */
	ResultSet getData(Map<String, Object> bindings) throws SQLException;

	/**
	 * Obtains a row, using values of each PK
	 * 
	 * @param pk
	 *            values of PK column
	 * @return the existing row in the table
	 * @throws SQLException
	 */
	Object[] findByPk(List<Object> pk) throws SQLException;

	/**
	 * Obtains a PK from the current ResultSet's row
	 * 
	 * @param incomingData
	 * @param tableBuffer
	 * @return
	 * @throws SQLException
	 */
	List<Object> getPk(ResultSet incomingData) throws SQLException;

	/**
	 * Obtains a PK from the current ResultSet's row, after interceptors transformation
	 * 
	 * @param incomingData
	 * @param transform
	 *            apply transformation (interceptor.onRead) ?
	 * @return
	 * @throws SQLException
	 */
	List<Object> getPk(ResultSet incomingData, boolean transform) throws SQLException;

	/**
	 * Delete all rows of a table
	 * 
	 * @throws SQLException
	 */
	void deleteAll() throws SQLException;

	/**
	 * Insert into a table the current row of the given {@link ResultSet}
	 * 
	 * @param pk
	 *            the PK of the current row (only need for logging)
	 * @param incomingData
	 *            a {@link ResultSet} with a row ready to be read ({@link ResultSet#next()} should have been call
	 *            before)
	 * @throws SQLException
	 */
	void executeInsert(ResultSet incomingData) throws SQLException;

	/**
	 * Insert into a table the current row of the given incoming data
	 * <p/>
	 * Same as {@link #executeInsert(List,ResultSet)}, but using a array instead of a ResultSet.
	 * 
	 * @param incomingData
	 *            a array of object
	 * @throws SQLException
	 */
	void executeInsert(Object[] row) throws SQLException;

	/**
	 * Update one table's row, using the current row of the given {@link ResultSet}
	 * 
	 * @param pk
	 *            the PK of the row to update
	 * @param incomingData
	 *            a {@link ResultSet} with a row ready to be read ({@link ResultSet#next()} should have been call
	 *            before)
	 * @throws SQLException
	 */
	void executeUpdate(List<Object> pk, ResultSet incomingData) throws SQLException;

	/**
	 * Update one table's row, using the current row of the given array of values
	 * <p/>
	 * Same as {@link #executeUpdate(List,ResultSet)}, but using a array instead of a ResultSet.
	 * 
	 * @param pk
	 *            the PK of the row to update
	 * @param row
	 *            a array of object
	 * @throws SQLException
	 */
	void executeUpdate(List<Object> pk, Object[] row) throws SQLException;

	/**
	 * Update one column of one table row.
	 * 
	 * @param columnName
	 * @param pk
	 * @param value
	 * @throws SQLException
	 */
	void executeUpdateColumn(String columnName, List<Object> pk, Object value) throws SQLException;

	/**
	 * Flush all pending updates (i.e. pending batch statement)
	 * 
	 * @throws SQLException
	 */
	void flush() throws SQLException;

	/**
	 * Obtains the number of updated rows
	 * 
	 * @return the number of updated rows
	 */
	int getUpdateCount();

	/**
	 * Obtains the number of updated rows (update on the given column)
	 * 
	 * @return the number of updated rows
	 */
	int getUpdateColumnCount(String columnName);

	/**
	 * Obtains the number of inserted rows
	 * 
	 * @return the number of inserted rows
	 */
	int getInsertCount();

	/**
	 * Obtains the number of deleted rows
	 * 
	 * @return the number of deleted rows
	 */
	int getDeleteCount();

	/**
	 * Getting the database dialect
	 * 
	 * @return
	 */
	Dialect getDialect();

	/**
	 * Getting all PKs corresponding to unique constraints
	 * 
	 * @param inconmingData
	 * @return
	 */
	Map<String, List<Object>> getPkFromUniqueConstraints(ResultSet inconmingData) throws SQLException;

	/**
	 * Retrieve update_date from the table, by PK
	 * 
	 * @param pk
	 * @return
	 */
	Timestamp getUpdateDateByPk(List<Object> pk) throws SQLException;

	Timestamp getUpdateDate(ResultSet incomingData, boolean transform) throws SQLException;

	/**
	 * Try to lock the row, by pk. If could not lock, return false
	 * 
	 * @param pk
	 * @return true if the lock is acquire, false is could not lock
	 * @throws SQLException
	 */
	boolean lock(List<Object> pk) throws SQLException;

	void setSourceDao(SynchroTableDao sourceDao);

	/**
	 * Delete a row, by PK.
	 * 
	 * @param pk
	 *            PK of the row to delete
	 * @param checkPkNotUsed
	 *            check if the PK is not used anymore. If yes, throw an exception
	 */
	void executeDelete(List<Object> pk, boolean checkPkNotUsed) throws SQLException;

	/**
	 * Detach an row. By default, to nothing but call interceptor.onDetach()
	 * 
	 * @param pk
	 *            PK of the row to detach
	 */
	void executeDetach(List<Object> pk) throws SQLException;

	/**
	 * Prepare to run dao (reset count...)
	 */
	void prepare();

	boolean executeDeleteByFk(String fkColumnName, String fkColumnValue, String additionalWhereClause, Map<String, Object> bindings)
			throws SQLException;

	List<Object> getLastGeneratedPk();

	List<List<Object>> transformOnRead(Set<String> fkColumnNames, List<List<Object>> fkSourceColumnValues) throws SQLException;

	Set<String> transformColumnNames(Set<String> fkColumnNames) throws SQLException;

}