package fr.ifremer.adagio.synchro.service;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import fr.ifremer.adagio.synchro.intercept.SynchroOperationRepository;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata;

public class SynchroTableOperation implements SynchroOperationRepository {

	private Map<String, Map<String, Object>> missingUpdates = Maps.newHashMap();

	/**
	 * Entities (list of pks) to delete
	 */
	private List<List<Object>> missingDeletes = Lists.newArrayList();

	/**
	 * Entities (list of pks) to detach from server (e.g. reset the remote_id...)
	 */
	private List<List<Object>> missingDetachs = Lists.newArrayList();

	private Map<String, Map<Set<String>, List<List<Object>>>> childrenToUpdate = Maps.newHashMap();

	private Map<String, Map<Set<String>, List<List<Object>>>> childrenToDelete = Maps.newHashMap();

	private Map<String, Map<Set<String>, List<List<Object>>>> childrenToDetach = Maps.newHashMap();

	private final String tableName;

	private final SynchroContext context;

	/**
	 * Say if progressionModel should be increment, when processing the operation.<br/>
	 * Usually, enableProgress is set to <code>true</true> for "root" tables.
	 */
	private boolean enableProgress = false;

	/**
	 * This is used to disable add into missing deletes, when a deletes has already be tried.<br/>
	 * {@see SynchroServiceImpl#deleteRows()}
	 */
	private boolean allowMissingDeletes = true;

	public SynchroTableOperation(String tableName, SynchroContext context) {
		this.tableName = tableName;
		this.context = context;
	}

	public SynchroTableOperation(String parentTableName, String tableName, String filterColumName, List<List<Object>> filterColumValues,
			SynchroContext context) {
		this.tableName = parentTableName;
		this.context = context;
		addChildrenToUpdateFromManyColumns(tableName, Sets.newHashSet(filterColumName), filterColumValues);
	}

	public SynchroTableOperation(String parentTableName, String tableName, Set<String> filterColumNames, List<List<Object>> filterColumValues,
			SynchroContext context) {
		this.tableName = parentTableName;
		this.context = context;
		addChildrenToUpdateFromManyColumns(tableName, filterColumNames, filterColumValues);
	}

	public String getTableName() {
		return tableName;
	}

	public boolean isAllowMissingDeletes() {
		return allowMissingDeletes;
	}

	public void setAllowMissingDeletes(boolean allowMissingDeletes) {
		this.allowMissingDeletes = allowMissingDeletes;
	}

	@Override
	public SynchroContext getSynchroContext() {
		return context;
	}

	public boolean isEnableProgress() {
		return enableProgress;
	}

	public void setEnableProgress(boolean enableProgress) {
		this.enableProgress = enableProgress;
	}

	@Override
	public void addMissingColumnUpdate(String columnName, List<Object> pk, Object columnValue) {
		Preconditions.checkNotNull(pk);
		addMissingColumnUpdate(columnName,
				SynchroTableMetadata.toPkStr(pk),
				columnValue);
	}

	@Override
	public void addMissingColumnUpdate(String columnName, String pkStr, Object columnValue) {
		Preconditions.checkNotNull(pkStr);
		Map<String, Object> destRows = missingUpdates.get(columnName);
		if (destRows == null) {
			destRows = Maps.newIdentityHashMap();
			missingUpdates.put(columnName, destRows);
		}
		destRows.put(pkStr, columnValue);
	}

	@Override
	public void addAllMissingDelete(List<List<Object>> pks) {
		Preconditions.checkNotNull(pks);
		missingDeletes.addAll(pks);
	}

	@Override
	public void addMissingDelete(List<Object> pk) {
		Preconditions.checkNotNull(pk);
		missingDeletes.add(pk);
	}

	@Override
	public void addMissingDelete(String pkStr) {
		Preconditions.checkNotNull(pkStr);
		List<Object> pk = SynchroTableMetadata.fromPkStr(pkStr);
		missingDeletes.add(pk);
	}

	@Override
	public void addAllMissingDetach(List<List<Object>> pks) {
		Preconditions.checkNotNull(pks);
		missingDetachs.addAll(pks);
	}

	@Override
	public void addMissingDetach(List<Object> pk) {
		Preconditions.checkNotNull(pk);
		missingDetachs.add(pk);
	}

	@Override
	public void addMissingDetach(String pkStr) {
		Preconditions.checkNotNull(pkStr);
		List<Object> pk = SynchroTableMetadata.fromPkStr(pkStr);
		missingDetachs.add(pk);
	}

	@Override
	public void addAllMissingColumnUpdates(Map<String, Map<String, Object>> missingUpdates) {
		this.missingUpdates.putAll(missingUpdates);
	}

	@Override
	public void addChildToUpdateFromOneColumn(String childTablename, String columnName, Object columnValue) {
		addChildToUpdateFromManyColumns(childTablename, ImmutableSet.of(columnName), ImmutableList.of(columnValue));
	}

	@Override
	public void addChildrenToUpdateFromOneColumn(String childTablename, String columnName, List<Object> columnValues) {
		List<List<Object>> values = Lists.newArrayListWithCapacity(columnValues.size());
		for (Object singleValue : columnValues) {
			values.add(ImmutableList.of(singleValue));
		}
		addChildrenToUpdateFromManyColumns(childTablename, ImmutableSet.of(columnName), values);
	}

	@Override
	public void addChildToUpdateFromManyColumns(String childTablename, Set<String> columnNames, List<Object> columnValues) {
		Map<Set<String>, List<List<Object>>> childTable = childrenToUpdate.get(childTablename);
		if (childTable == null) {
			childTable = Maps.newHashMap();
			childrenToUpdate.put(childTablename, childTable);
		}
		List<List<Object>> childColumnValues = childTable.get(columnNames);
		if (childColumnValues == null) {
			childColumnValues = Lists.newArrayList();
			childTable.put(columnNames, childColumnValues);
		}
		childColumnValues.add(columnValues);
	}

	@Override
	public void addChildrenToUpdateFromManyColumns(String childTablename, Set<String> columnNames, List<List<Object>> columnValues) {
		Map<Set<String>, List<List<Object>>> childTable = childrenToUpdate.get(childTablename);
		if (childTable == null) {
			childTable = Maps.newHashMap();
			childrenToUpdate.put(childTablename, childTable);
		}
		List<List<Object>> childColumnValues = childTable.get(columnNames);
		if (childColumnValues == null) {
			childColumnValues = Lists.newArrayList();
			childTable.put(columnNames, childColumnValues);
		}
		childColumnValues.addAll(columnValues);
	}

	@Override
	public void addChildToDeleteFromOneColumn(String childTablename, String columnName, Object columnValue) {
		addChildToDeleteFromManyColumns(childTablename, ImmutableSet.of(columnName), ImmutableList.of(columnValue));
	}

	@Override
	public void addChildrenToDeleteFromOneColumn(String childTablename, String columnName, List<Object> columnValues) {
		List<List<Object>> values = Lists.newArrayListWithCapacity(columnValues.size());
		for (Object singleValue : columnValues) {
			values.add(ImmutableList.of(singleValue));
		}
		addChildrenToDeleteFromManyColumns(childTablename, ImmutableSet.of(columnName), values);
	}

	@Override
	public void addChildToDeleteFromManyColumns(String childTablename, Set<String> columnNames, List<Object> columnValue) {
		Map<Set<String>, List<List<Object>>> childTable = childrenToDelete.get(childTablename);
		if (childTable == null) {
			childTable = Maps.newHashMap();
			childrenToDelete.put(childTablename, childTable);
		}
		List<List<Object>> childColumnValues = childTable.get(columnNames);
		if (childColumnValues == null) {
			childColumnValues = Lists.newArrayList();
			childTable.put(columnNames, childColumnValues);
		}
		childColumnValues.add(columnValue);
	}

	@Override
	public void addChildrenToDeleteFromManyColumns(String childTablename, Set<String> columnNames, List<List<Object>> columnValues) {
		Map<Set<String>, List<List<Object>>> childTable = childrenToDelete.get(childTablename);
		if (childTable == null) {
			childTable = Maps.newHashMap();
			childrenToDelete.put(childTablename, childTable);
		}
		List<List<Object>> childColumnValues = childTable.get(columnNames);
		if (childColumnValues == null) {
			childColumnValues = Lists.newArrayList();
			childTable.put(columnNames, childColumnValues);
		}
		childColumnValues.addAll(columnValues);
	}

	@Override
	public void addChildToDetachFromOneColumn(String childTablename, String columnName, Object columnValue) {
		addChildToDetachFromManyColumn(childTablename, ImmutableSet.of(columnName), ImmutableList.of(columnValue));
	}

	@Override
	public void addChildToDetachFromManyColumn(String childTablename, Set<String> columnNames, List<Object> columnValue) {
		Map<Set<String>, List<List<Object>>> childTable = childrenToDetach.get(childTablename);
		if (childTable == null) {
			childTable = Maps.newHashMap();
			childrenToDetach.put(childTablename, childTable);
		}
		List<List<Object>> childColumnValues = childTable.get(columnNames);
		if (childColumnValues == null) {
			childColumnValues = Lists.newArrayList();
			childTable.put(columnNames, childColumnValues);
		}
		childColumnValues.add(columnValue);
	}

	@Override
	public void addChildrenToDetachFromManyColumns(String childTablename, Set<String> columnNames, List<List<Object>> columnValues) {
		Map<Set<String>, List<List<Object>>> childTable = childrenToDetach.get(childTablename);
		if (childTable == null) {
			childTable = Maps.newHashMap();
			childrenToDetach.put(childTablename, childTable);
		}
		List<List<Object>> childColumnValues = childTable.get(columnNames);
		if (childColumnValues == null) {
			childColumnValues = Lists.newArrayList();
			childTable.put(columnNames, childColumnValues);
		}
		childColumnValues.addAll(columnValues);
	}

	public Map<String, Map<Set<String>, List<List<Object>>>> getChildrenToUpdate() {
		return ImmutableMap.copyOf(childrenToUpdate);
	}

	public boolean hasChildrenToUpdate() {
		return MapUtils.isNotEmpty(childrenToUpdate);
	}

	public Map<String, Map<Set<String>, List<List<Object>>>> getChildrenToDelete() {
		return ImmutableMap.copyOf(childrenToDelete);
	}

	public Map<String, Map<Set<String>, List<List<Object>>>> getChildrenToDetach() {
		return ImmutableMap.copyOf(childrenToDetach);
	}

	public boolean hasChildrenToDelete() {
		return MapUtils.isNotEmpty(childrenToDelete);
	}

	public boolean hasChildrenToDetach() {
		return MapUtils.isNotEmpty(childrenToDetach);
	}

	public Map<String, Map<String, Object>> getMissingUpdates() {
		return ImmutableMap.copyOf(missingUpdates);
	}

	public List<List<Object>> getMissingDeletes() {
		return ImmutableList.copyOf(missingDeletes);
	}

	public List<List<Object>> getMissingDetachs() {
		return ImmutableList.copyOf(missingDetachs);
	}

	public boolean isEmpty() {
		return MapUtils.isEmpty(missingUpdates)
				&& CollectionUtils.isEmpty(missingDeletes)
				&& CollectionUtils.isEmpty(missingDetachs)
				&& MapUtils.isEmpty(childrenToUpdate)
				&& MapUtils.isEmpty(childrenToDelete)
				&& MapUtils.isEmpty(childrenToDetach);
	}

	public void clearMissingUpdates() {
		missingUpdates.clear();
	}

	public void clearMissingDeletes() {
		missingDeletes.clear();
	}

	public void clearMissingDetachs() {
		missingDetachs.clear();
	}

	public void clearChildrenToUpdate() {
		childrenToUpdate.clear();
	}

	public void clearChildrenToDelete() {
		childrenToDelete.clear();
	}

	public void clearChildrenToDetach() {
		childrenToDetach.clear();
	}
}
