package fr.ifremer.adagio.synchro.service;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import static org.nuiton.i18n.I18n.t;

import java.io.File;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Collection;
import java.util.Date;
import java.util.Deque;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import javax.sql.DataSource;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.cfg.Environment;
import org.nuiton.util.TimeLog;
import org.springframework.dao.DataRetrievalFailureException;
import org.springframework.jdbc.datasource.DataSourceUtils;
import org.springframework.transaction.interceptor.TransactionInterceptor;

import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Queues;
import com.google.common.collect.Sets;

import fr.ifremer.adagio.synchro.SynchroTechnicalException;
import fr.ifremer.adagio.synchro.config.SynchroConfiguration;
import fr.ifremer.adagio.synchro.dao.DaoFactoryImpl;
import fr.ifremer.adagio.synchro.dao.Daos;
import fr.ifremer.adagio.synchro.dao.DataIntegrityViolationOnDeleteException;
import fr.ifremer.adagio.synchro.dao.SynchroTableDao;
import fr.ifremer.adagio.synchro.dao.SynchroTableDaoUtils;
import fr.ifremer.adagio.synchro.intercept.SynchroDeletedRowException;
import fr.ifremer.adagio.synchro.intercept.SynchroDuplicateRowException;
import fr.ifremer.adagio.synchro.intercept.SynchroRejectRowException;
import fr.ifremer.adagio.synchro.meta.SynchroColumnMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroJoinMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroMetadataUtils;
import fr.ifremer.adagio.synchro.meta.SynchroSchemaValidationException;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata.DuplicateKeyStrategy;
import fr.ifremer.adagio.synchro.type.ProgressionModel;

public class SynchroServiceImpl implements SynchroService {

	/** Logger. */
	private static final Log log =
			LogFactory.getLog(SynchroServiceImpl.class);

	protected static final TimeLog TIME =
			new TimeLog(SynchroServiceImpl.class);

	/* A timer, with longer time (FK check are always long !) */
	protected static final TimeLog TIME_DELETION =
			new TimeLog(SynchroServiceImpl.class,
					3000l * 1000000l /* =3s */,
					5000l * 1000000l /* =5s */);

	protected final SynchroConfiguration config;

	protected final int MAX_ROW_COUNT_FOR_PK_PRELOADING = 50000;

	protected final int batchSize;

	protected final boolean debug;

	protected final DataSource dataSource;

	private final boolean disableIntegrityConstraints;

	private final boolean allowMissingOptionalColumn;

	private final boolean allowAdditionalMandatoryColumnInSourceSchema;

	private final boolean keepWhereClauseOnQueriesByFks;

	private int daoCacheSize = -1;

	private int statementCacheSize = -1;

	private final LoadingCache<SynchroContext, Map<String, Object>> defaultBindingCache;

	/**
	 * @param dataSource
	 * @param config
	 * @param tableToIncludes
	 *            tables (names) to synchronize
	 * @param disableIntegrityConstraints
	 * @param allowMissingOptionalColumn
	 * @param allowAdditionalMandatoryColumnInSourceSchema
	 * @param trace
	 */
	public SynchroServiceImpl(DataSource dataSource,
			SynchroConfiguration config,
			boolean disableIntegrityConstraints,
			boolean allowMissingOptionalColumn,
			boolean allowAdditionalMandatoryColumnInSourceSchema,
			boolean keepWhereClauseOnQueriesByFks) {
		Preconditions.checkNotNull(config);

		this.dataSource = dataSource;
		this.config = config;
		this.batchSize = config.getImportJdbcBatchSize();
		this.defaultBindingCache = initBindingCache(5);
		this.disableIntegrityConstraints = disableIntegrityConstraints;
		this.allowMissingOptionalColumn = allowMissingOptionalColumn;
		this.allowAdditionalMandatoryColumnInSourceSchema = allowAdditionalMandatoryColumnInSourceSchema;
		this.keepWhereClauseOnQueriesByFks = keepWhereClauseOnQueriesByFks;
		this.debug = log.isTraceEnabled();
	}

	/**
	 * @param tableToIncludes
	 *            tables (names) to synchronize
	 * @param couldDisableIntegrityConstraints
	 * @param allowMissingOptionalColumn
	 * @param allowAdditionalMandatoryColumnInSourceSchema
	 * @param trace
	 */
	public SynchroServiceImpl(
			boolean couldDisableIntegrityConstraints,
			boolean allowMissingOptionalColumn,
			boolean allowAdditionalMandatoryColumnInSourceSchema,
			boolean keepWhereClauseOnQueriesByFks) {
		this(null /* no dataSource */,
				SynchroConfiguration.getInstance(),
				couldDisableIntegrityConstraints,
				allowMissingOptionalColumn,
				allowAdditionalMandatoryColumnInSourceSchema,
				keepWhereClauseOnQueriesByFks);
	}

	public SynchroContext createSynchroContext(File sourceDbDirectory, Set<String> tableToIncludes) {

		String dbName = config.getDbName();
		Properties targetConnectionProperties = config.getConnectionProperties();

		Properties sourceConnectionProperties = new Properties(targetConnectionProperties);
		sourceConnectionProperties.setProperty(Environment.URL,
				Daos.getJdbcUrl(sourceDbDirectory, dbName));

		return this.createSynchroContext(sourceConnectionProperties, tableToIncludes);
	}

	public SynchroContext createSynchroContext(Properties sourceConnectionProperties, Set<String> tableToIncludes) {
		Preconditions.checkNotNull(sourceConnectionProperties);

		Properties targetConnectionProperties = config.getConnectionProperties();

		SynchroContext context = SynchroContext.newContext(
				tableToIncludes,
				sourceConnectionProperties,
				targetConnectionProperties,
				new SynchroResult());

		// Apply some constructor options
		context.getTarget().setKeepWhereClauseOnQueriesByFks(keepWhereClauseOnQueriesByFks);
		context.getSource().setKeepWhereClauseOnQueriesByFks(keepWhereClauseOnQueriesByFks);

		return context;
	}

	@Override
	public void prepare(SynchroContext context) {
		Preconditions.checkNotNull(context);
		if (log.isDebugEnabled()) {
			log.debug("Preparing for synchronization - " + context.toString());
		}

		SynchroDatabaseConfiguration source = context.getSource();
		Preconditions.checkNotNull(source);
		source.setReadOnly(true);

		SynchroDatabaseConfiguration target = context.getTarget();
		Preconditions.checkNotNull(target);
		target.setReadOnly(true);

		// Make sure one DB has metadata enable
		boolean useTargetMetadata = target.isFullMetadataEnable();
		source.setFullMetadataEnable(!useTargetMetadata);

		Set<String> tableToIncludes = context.getTableNames();
		if (CollectionUtils.isEmpty(tableToIncludes)) {
			log.info(t("adagio.synchro.prepare.noTableFilter"));
		}

		SynchroResult result = context.getResult();
		Preconditions.checkNotNull(result);

		result.setLocalUrl(target.getUrl());
		result.setRemoteUrl(source.getUrl());

		Connection targetConnection = null;
		Connection sourceConnection = null;
		DaoFactoryImpl sourceDaoFactory = null;
		DaoFactoryImpl targetDaoFactory = null;

		try {

			ProgressionModel progressionModel = result.getProgressionModel();
			progressionModel.setMessage(t("adagio.synchro.prepare.step1"));

			// create target connection
			targetConnection = createConnection(target);

			progressionModel.setMessage(t("adagio.synchro.prepare.step2"));

			// create source Connection
			sourceConnection = createConnection(source);

			// load metas and check it
			SynchroDatabaseMetadata dbMeta = null;
			{
				// load metas
				SynchroDatabaseMetadata targetMeta =
						loadDatabaseMetadata(
								targetConnection,
								target,
								tableToIncludes);
				// targetMeta.close(); // close to free memory

				SynchroDatabaseMetadata sourceMeta =
						loadDatabaseMetadata(
								sourceConnection,
								source,
								tableToIncludes);
				// sourceMeta.close(); // close to free memory

				progressionModel.setMessage(t("adagio.synchro.prepare.step3"));

				// check schema
				checkSchemasAndExcludeMissingColumns(source,
						target,
						sourceMeta,
						targetMeta,
						allowMissingOptionalColumn,
						allowAdditionalMandatoryColumnInSourceSchema,
						result);
				if (!result.isSuccess()) {
					return;
				}

				// Retrieve the meta to use
				dbMeta = useTargetMetadata ? targetMeta : sourceMeta;
			}

			// Create DAO factories
			sourceDaoFactory = newDaoFactory(sourceConnection, source, dbMeta);
			targetDaoFactory = newDaoFactory(targetConnection, target, dbMeta);

			// Clean temp query parameter table
			sourceDaoFactory.getDao().cleanTempQueryParameter();
			targetDaoFactory.getDao().cleanTempQueryParameter();

			// prepare model (compute update date, count rows to update,...)
			Set<String> rootTableNames = dbMeta.getLoadedRootTableNames();
			if (CollectionUtils.isEmpty(rootTableNames)) {
				log.warn(t("adagio.synchro.prepare.noRootTable"));
			}

			for (String tableName : rootTableNames) {

				long t0 = TimeLog.getTime();

				progressionModel.setMessage(t("adagio.synchro.prepare.step4", tableName));

				SynchroTableMetadata table = dbMeta.getLoadedTable(tableName);

				if (log.isDebugEnabled()) {
					log.debug("Prepare table: " + tableName);
				}
				prepareRootTable(
						sourceDaoFactory,
						targetDaoFactory,
						table,
						context,
						result);

				TIME.log(t0, "prepare table " + tableName);
			}

			long totalRows = result.getTotalRows();
			if (log.isInfoEnabled()) {
				log.info("Total root rows to update: " + totalRows);
			}

			// Always rollback when prepare
			rollbackSilently(targetConnection);

		} catch (SQLException e) {
			rollbackSilently(targetConnection);
			result.setError(e);

		} finally {
			IOUtils.closeQuietly(sourceDaoFactory);
			IOUtils.closeQuietly(targetDaoFactory);
			closeSilently(sourceConnection);
			closeSilently(targetConnection);
			releaseContext(context);
		}
	}

	@Override
	public void synchronize(SynchroContext context) {
		Preconditions.checkNotNull(context);
		if (log.isDebugEnabled()) {
			log.debug("Starting synchronization - " + context.toString());
		}

		SynchroDatabaseConfiguration source = context.getSource();
		Preconditions.checkNotNull(source);
		source.setReadOnly(true); // no writes in the source db

		SynchroDatabaseConfiguration target = context.getTarget();
		Preconditions.checkNotNull(target);
		target.setReadOnly(false); // enable writes on target

		// Make sure one DB has metadata enable
		boolean useTargetMetadata = target.isFullMetadataEnable();
		source.setFullMetadataEnable(!useTargetMetadata);

		Set<String> tableNames = context.getTableNames();

		SynchroResult result = context.getResult();
		Preconditions.checkNotNull(result);

		Connection targetConnection = null;
		Connection sourceConnection = null;
		DaoFactoryImpl sourceDaoFactory = null;
		DaoFactoryImpl targetDaoFactory = null;

		try {

			// create source Connection
			sourceConnection = createConnection(source);

			// create target connection
			targetConnection = createConnection(target);

			// load metas
			SynchroDatabaseMetadata dbMeta = null;
			if (useTargetMetadata) {
				log.debug("Loading target database metadata...");
				dbMeta = loadDatabaseMetadata(
						targetConnection,
						target,
						tableNames);
			}
			else {
				log.debug("Loading source database metadata...");
				dbMeta = loadDatabaseMetadata(
						sourceConnection,
						source,
						tableNames);
			}

			// Create DAO factories
			sourceDaoFactory = newDaoFactory(sourceConnection, source, dbMeta);
			targetDaoFactory = newDaoFactory(targetConnection, target, dbMeta);

			// set total in progression model
			ProgressionModel progressionModel = result.getProgressionModel();
			progressionModel.setTotal(result.getTotalRows() + 1);
			progressionModel.setCurrent(0);
			progressionModel.setMessage(t("adagio.synchro.synchronize.step0"));

			// prepare target (e.g. desactivate constraints if need)
			prepareSynch(targetConnection, context);

			try {
				// Clean temp query parameter table
				sourceDaoFactory.getDao().cleanTempQueryParameter();
				targetDaoFactory.getDao().cleanTempQueryParameter();

				// Create a operation stack with all root tables
				Deque<SynchroTableOperation> pendingOperations = Queues.newArrayDeque(
						getRootOperations(
								sourceDaoFactory,
								targetDaoFactory,
								dbMeta,
								context));
				progressionModel.increments(1);

				// Process table operation, while some exists
				while (!pendingOperations.isEmpty()) {
					// process the first operation (on top of the stack)
					SynchroTableOperation operation = pendingOperations.pop();

					// Execute synchronization operation
					synchronizeOperation(
							operation,
							dbMeta,
							sourceDaoFactory,
							targetDaoFactory,
							context,
							result,
							pendingOperations);
				}

			} finally {
				releaseSynch(targetConnection, context);
			}

			if (log.isInfoEnabled()) {
				long totalInserts = result.getTotalInserts();
				long totalUpdates = result.getTotalUpdates();
				long totalDeletes = result.getTotalDeletes();
				long totalRejects = result.getTotalRejects();
				log.info("Total root rows to treat: " + result.getTotalRows());
				log.info("Total rows inserted: " + totalInserts);
				log.info("Total rows  updated: " + totalUpdates);
				log.info("Total rows  deleted: " + totalDeletes);
				log.info("Total rows rejected: " + totalRejects);
				log.info("Total rows  treated: " + (totalInserts + totalUpdates + totalDeletes + totalRejects));
				if (totalRejects > 0) {
					log.warn(String.format("Some rows has been rejected (%s rows)", totalRejects));
				}
			}

			progressionModel.setMessage(t("adagio.synchro.synchronize.step2"));
			progressionModel.setCurrent(progressionModel.getTotal());

			if (target.isReadOnly()) {
				log.error("Could not commit because database is set as readOnly in the configuration.");
				targetConnection.rollback();
				return;
			}

			// Final commit
			commit(targetConnection);

		} catch (Exception e) {
			rollbackSilently(targetConnection);
			updateResultOnSynchronizeError(e, result);
		} finally {
			IOUtils.closeQuietly(sourceDaoFactory);
			IOUtils.closeQuietly(targetDaoFactory);
			closeSilently(sourceConnection);
			closeSilently(targetConnection);
		}
	}

	@Override
	public Timestamp getSourceLastUpdateDate(SynchroContext context) {
		if (log.isInfoEnabled()) {
			log.info("Read max(update_date) on referential tables...");
		}

		SynchroDatabaseConfiguration source = context.getSource();
		source.setFullMetadataEnable(true);
		source.setReadOnly(true);

		Set<String> tableNames = context.getTableNames();
		if (CollectionUtils.isEmpty(tableNames)) {
			log.info(t("adagio.synchro.prepare.noTableFilter"));
		}

		// set total in progression model
		Preconditions.checkNotNull(context.getResult());
		ProgressionModel progressionModel = context.getResult().getProgressionModel();
		progressionModel.setTotal(tableNames.size());

		Connection connection = null;
		try {

			progressionModel.setMessage(t("adagio.synchro.referential.lastUpdateDate.step1"));
			if (log.isDebugEnabled()) {
				log.debug(t("adagio.synchro.referential.lastUpdateDate.step1"));
			}

			long t0 = TimeLog.getTime();

			// create source Connection
			connection = createConnection(source);

			// load metas
			SynchroDatabaseMetadata dbMeta =
					loadDatabaseMetadata(
							connection,
							source,
							tableNames);

			// Update tableNames with table found, and set progression model total
			Set<String> rootTableNames = dbMeta.getLoadedRootTableNames();
			progressionModel.setTotal(rootTableNames.size());

			Timestamp result = null;

			// For each table
			for (String tableName : rootTableNames) {

				progressionModel.setMessage(t("adagio.synchro.referential.lastUpdateDate.step2", tableName));
				if (log.isDebugEnabled()) {
					log.debug(t("adagio.synchro.referential.lastUpdateDate.step2", tableName));
				}

				SynchroTableMetadata table = dbMeta.getTable(tableName);

				// get last updateDate used by db
				Timestamp updateDate = SynchroTableDaoUtils.getLastUpdateDate(table, connection);

				if (Daos.compareUpdateDates(result, updateDate) < 0) {
					result = updateDate;
				}

				progressionModel.increments(1);
			}

			if (log.isInfoEnabled()) {
				log.info(String.format("Read last update_date on referential tables [%s] ", result));
				TIME.log(t0, "Read last update_date on referential tables");
			}

			return result;

		} catch (Exception e) {
			log.error("Error while reading last update_date on referential tables", e);
			throw new DataRetrievalFailureException("Error while reading last update_date on referential tables", e);
		} finally {
			closeSilently(connection);
		}
	}

	@Override
	public void finish(SynchroContext synchroContext, SynchroResult serverExportResult, Map<RejectedRowStatus, RejectedRowStrategy> rejectStrategies) {
		Preconditions.checkNotNull(synchroContext);
		Preconditions.checkArgument(MapUtils.isNotEmpty(rejectStrategies));

		if (log.isDebugEnabled()) {
			log.debug(String.format("Finish (update source DB and resolve rejects with strateies %s): %s", rejectStrategies,
					synchroContext.toString()));
		}

		SynchroResult result = synchroContext.getResult();
		ProgressionModel progressionModel = result.getProgressionModel();

		// WARNING : target connection is now the source DB
		SynchroDatabaseConfiguration target = synchroContext.getSource();
		target.setReadOnly(false);
		target.setFullMetadataEnable(true);
		target.setIsTarget(true);

		Set<String> tableNames = synchroContext.getTableNames();
		if (CollectionUtils.isEmpty(tableNames)) {
			log.info(t("adagio.synchro.prepare.noTableFilter"));
		}

		Connection targetConnection = null;
		DaoFactoryImpl targetDaoFactory = null;

		try {
			// create source Connection
			targetConnection = createConnection(target);

			// load metas
			SynchroDatabaseMetadata dbMeta =
					loadDatabaseMetadata(
							targetConnection,
							target,
							tableNames);

			// Create dao factory
			targetDaoFactory = newDaoFactory(targetConnection, target, dbMeta);

			// Create a operation stack with all operations
			Deque<SynchroTableOperation> pendingOperations = Queues.newArrayDeque(
					getSourceMissingOperations(serverExportResult, synchroContext));

			// Resolve rejects (will add operations to queue if necessary)
			if (serverExportResult.getRejectedRows() != null
					&& !serverExportResult.getRejectedRows().isEmpty()) {
				resolveRejects(targetConnection,
						dbMeta,
						targetDaoFactory,
						synchroContext,
						serverExportResult.getRejectedRows(),
						rejectStrategies,
						pendingOperations);
			}

			// Compute the progression total
			progressionModel.setCurrent(0);
			progressionModel.setTotal(pendingOperations.size());

			// Process table operation, while some exists
			while (!pendingOperations.isEmpty()) {
				// process the first operation (on top of the stack)
				SynchroTableOperation operation = pendingOperations.pop();

				// Execute synchronization operation
				synchronizeOperation(
						operation,
						dbMeta,
						null,
						targetDaoFactory,
						synchroContext,
						result,
						pendingOperations);

			}

			// Final commit
			commit(targetConnection);

		} catch (Exception e) {
			rollbackSilently(targetConnection);
			result.setError(e);
		} finally {
			IOUtils.closeQuietly(targetDaoFactory);
			closeSilently(targetConnection);
		}

	}

	protected void setDaoCacheSize(int daoFactoryCacheSize) {
		this.daoCacheSize = daoFactoryCacheSize;
	}

	protected void setStatementCacheSize(int daoFactoryStatementCacheSize) {
		this.statementCacheSize = daoFactoryStatementCacheSize;
	}

	/* -- Internal methods -- */

	protected void checkSchemasAndExcludeMissingColumns(
			SynchroDatabaseConfiguration source,
			SynchroDatabaseConfiguration target,
			SynchroDatabaseMetadata sourceMeta,
			SynchroDatabaseMetadata targetMeta,
			boolean allowMissingOptionalColumn,
			boolean allowAdditionalMandatoryColumnInSourceSchema,
			SynchroResult result) {
		try {
			Set<String> columnExcludes = SynchroMetadataUtils.checkSchemas(
					sourceMeta,
					targetMeta,
					allowMissingOptionalColumn,
					allowAdditionalMandatoryColumnInSourceSchema);

			// Add column to exclude to configuration
			if (CollectionUtils.isNotEmpty(columnExcludes)) {
				if (log.isDebugEnabled()) {
					log.debug(String.format("Some missing optional columns will be skipped: %s", columnExcludes));
				}
				source.addColumnExcludes(columnExcludes);
				target.addColumnExcludes(columnExcludes);
			}

		} catch (SynchroTechnicalException e) {
			result.setError(e);
		} catch (SynchroSchemaValidationException e) {
			log.error(e.getMessage());
			result.setError(e);
		}
	}

	protected void reportProgress(SynchroResult result, SynchroTableDao dao, int countR, String tablePrefix) {
		if (dao.getCurrentOperation().isEnableProgress()) {
			if (countR % batchSize == 0) {
				result.getProgressionModel().increments(batchSize);
			}
		}

		if (countR % (batchSize * 10) == 0) {
			if (log.isInfoEnabled()) {
				log.info(String.format("%s Done: %s (inserts: %s, updates: %s deletes: %s)",
						tablePrefix,
						countR,
						dao.getInsertCount(),
						dao.getUpdateCount(),
						dao.getDeleteCount()));
			}
		}
	}

	protected Connection createConnection(SynchroDatabaseConfiguration databaseConfiguration) throws SQLException {
		return createConnection(
				databaseConfiguration.getUrl(),
				databaseConfiguration.getUser(),
				databaseConfiguration.getPassword());
	}

	protected Connection createConnection(String jdbcUrl,
			String user,
			String password) throws SQLException {
		Preconditions.checkArgument(StringUtils.isNotBlank(jdbcUrl));

		// If same URL as datasource, use the dataSource
		Connection connection = null;
		if (isManagedByDataSource(jdbcUrl) && this.dataSource != null) {
			connection = DataSourceUtils.getConnection(this.dataSource);
		}
		else {
			connection = DriverManager.getConnection(jdbcUrl,
					user,
					password);
		}
		connection.setAutoCommit(false);
		return connection;
	}

	protected void closeSilently(Connection connection) {
		if (connection == null) {
			return;
		}
		try {
			// If same URL as datasource, use the dataSource
			if (isManagedByDataSource(connection)
					&& this.dataSource != null) {
				DataSourceUtils.releaseConnection(connection, this.dataSource);
			}
			else {
				Daos.closeSilently(connection);
			}
		} catch (SQLException e) {
			// silent !
		}

	}

	protected boolean isManagedByDataSource(Connection connection) throws SQLException {
		String jdbcUrl = connection.getMetaData().getURL();
		// If same URL as datasource, use the dataSource
		return this.dataSource != null && isManagedByDataSource(jdbcUrl);
	}

	/**
	 * Apply a commit on transaction, only if there is no Spring transaction managment enable.
	 * 
	 * @param connection
	 * @throws SQLException
	 */
	protected void commit(Connection connection) throws SQLException {
		if (connection == null) {
			return;
		}

		if (dataSource == null
				|| !isManagedByDataSource(connection)
				|| !DataSourceUtils.isConnectionTransactional(connection, dataSource)) {
			connection.commit();
		}
	}

	/**
	 * Apply a rollback on transaction, only if there is no Spring transaction managment enable.
	 * 
	 * @param connection
	 * @throws SQLException
	 */
	protected void rollbackSilently(Connection connection) {
		if (connection == null) {
			return;
		}
		try {
			if (dataSource == null
					|| !isManagedByDataSource(connection)
					|| !DataSourceUtils.isConnectionTransactional(connection, dataSource)) {
				connection.rollback();
			}
			else {
				TransactionInterceptor.currentTransactionStatus().setRollbackOnly();
			}
		} catch (SQLException e) {
			// silent !
		}
	}

	/**
	 * Getting the default binding to use for select queries
	 * 
	 * @param context
	 * @return
	 */
	protected final Map<String, Object> getSelectBindings(SynchroContext context) {
		try {
			return defaultBindingCache.get(context);
		} catch (ExecutionException e) {
			return ImmutableMap.copyOf(createDefaultSelectBindings(context));
		}
	}

	/**
	 * Could be override to put some values need by overidden select queries
	 * 
	 * @param context
	 * @param tableName
	 * @return
	 */
	protected Map<String, Object> createDefaultSelectBindings(SynchroContext context) {

		Map<String, Object> defaultBinding = Maps.newHashMap();
		if (context.getLastSynchronizationDate() != null) {
			defaultBinding.put(SynchroTableMetadata.UPDATE_DATE_BINDPARAM, context.getLastSynchronizationDate());
		}

		return defaultBinding;
	}

	/**
	 * Could be override to put wome values need for select queries
	 * 
	 * @param context
	 * @param tableName
	 * @return
	 */
	protected Map<String, Object> createSelectBindingsForTable(SynchroContext context, String tableName) {
		// Get defaults binding from context
		Map<String, Object> bindings = Maps.newHashMap(getSelectBindings(context));

		Date tableUpdateDate = context.getResult().getUpdateDate(tableName);

		// if update date exists for the table (=some rows exists on table)
		if (tableUpdateDate != null || context.getTarget().isMirrorDatabase()) {

			Timestamp lastSynchronizationDate = context.getLastSynchronizationDate();

			// If context has a date, use it instead
			if (lastSynchronizationDate != null) {
				bindings.put(SynchroTableMetadata.UPDATE_DATE_BINDPARAM, lastSynchronizationDate);
			}
			else if (tableUpdateDate != null) {
				bindings.put(SynchroTableMetadata.UPDATE_DATE_BINDPARAM, tableUpdateDate);
			}
		}

		return bindings;
	}

	protected void disableIntegrityConstraints(Properties connectionProperties, SynchroResult result) {
		result.getProgressionModel().setMessage(t("adagio.synchro.synchronize.disableIntegrityConstraints"));
		if (log.isDebugEnabled()) {
			log.debug(t("adagio.synchro.synchronize.disableIntegrityConstraints"));
		}
		try {
			Daos.setIntegrityConstraints(connectionProperties, false);
		} catch (SynchroTechnicalException e) {
			result.setError(e);
		} catch (SQLException e) {
			result.setError(e);
		}
	}

	/**
	 * Utility method to known if the connection could be manage by the dataSource bean.
	 * <p/>
	 * Could be subclasses if dataSource configuration change (i.e. in core-allegro-ui-wicket)
	 * 
	 * @param jdbcUrl
	 * @return true
	 */
	protected boolean isManagedByDataSource(String jdbcUrl) {
		return Objects.equals(config.getJdbcURL(), jdbcUrl);
	}

	/**
	 * Load (or create) a database metadata, for the given configuration and connection.
	 * By default, a new object is return (no cache).
	 * </p>
	 * Could be override by subclasses (i.e. to use cache, ...)
	 * 
	 * @param connection
	 * @param dbConfiguration
	 * @param tableNames
	 * @return
	 */
	protected SynchroDatabaseMetadata loadDatabaseMetadata(Connection connection,
			SynchroDatabaseConfiguration dbConfiguration,
			Set<String> tableNames) {
		if (log.isDebugEnabled()) {
			log.debug(String.format("Loading database metadata... [%s]", dbConfiguration.getUrl()));
		}

		SynchroDatabaseMetadata result =
				SynchroDatabaseMetadata.loadDatabaseMetadata(
						connection,
						dbConfiguration,
						tableNames);

		return result;
	}

	protected DaoFactoryImpl newDaoFactory(Connection connection,
			SynchroDatabaseConfiguration dbConfig,
			SynchroDatabaseMetadata dbMeta) {
		return new DaoFactoryImpl(connection, dbConfig, dbMeta, daoCacheSize, statementCacheSize);
	}

	protected LoadingCache<SynchroContext, Map<String, Object>> initBindingCache(final int maximumSize) {
		return CacheBuilder.newBuilder()
				.maximumSize(maximumSize)
				.expireAfterAccess(2, TimeUnit.MINUTES)
				.build(
						new CacheLoader<SynchroContext, Map<String, Object>>() {
							public Map<String, Object> load(SynchroContext context) throws SQLException {
								return ImmutableMap.copyOf(createDefaultSelectBindings(context));
							}
						});
	}

	protected final void synchronizeOperation(
			SynchroTableOperation operation,
			SynchroDatabaseMetadata targetMeta,
			DaoFactoryImpl sourceDaoFactory,
			DaoFactoryImpl targetDaoFactory,
			SynchroContext context,
			SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations
			) throws SQLException {

		boolean hasMissingUpdates = MapUtils.isNotEmpty(operation.getMissingUpdates());
		boolean hasMissingDeletes = CollectionUtils.isNotEmpty(operation.getMissingDeletes());
		boolean hasMissingDetachs = CollectionUtils.isNotEmpty(operation.getMissingDetachs());
		boolean hasChildrenToUpdate = operation.hasChildrenToUpdate();
		boolean hasChildrenToDelete = operation.hasChildrenToDelete();
		boolean hasChildrenToDetach = operation.hasChildrenToDetach();

		boolean rootTableOperation = !hasChildrenToUpdate
				&& !hasChildrenToDelete
				&& !hasMissingUpdates
				&& !hasMissingDeletes
				&& !hasMissingDetachs
				&& !hasChildrenToDetach;

		// Root table
		if (rootTableOperation) {
			String tableName = operation.getTableName();
			SynchroTableMetadata table = targetMeta.getTable(tableName);
			long t0 = TimeLog.getTime();

			if (log.isDebugEnabled()) {
				log.debug("Synchronize root table: " + tableName);
			}
			long countToUpdate = result.getNbRows(tableName);

			if (countToUpdate > 0) {

				synchronizeRootTable(
						table,
						sourceDaoFactory,
						targetDaoFactory,
						context,
						result,
						operation,
						pendingOperations);

				TIME.log(t0, "synchronize root table " + tableName);

			}

			// Add to pending operations (if not empty)
			addToPendingOperationsIfNotEmpty(operation, pendingOperations, context);
		}

		// missing delete
		else if (hasMissingDeletes) {
			String tableName = operation.getTableName();
			SynchroTableMetadata table = targetMeta.getTable(tableName);

			// Get child to update and clean operation
			List<List<Object>> pksToDeletes = operation.getMissingDeletes();
			operation.clearMissingDeletes();

			long t0 = TimeLog.getTime();

			if (log.isDebugEnabled()) {
				log.debug("Execute missing deletes on table: " + tableName);
			}

			synchronizeDeletes(operation,
					table,
					pksToDeletes,
					sourceDaoFactory,
					targetDaoFactory,
					context,
					result,
					pendingOperations);

			TIME_DELETION.log(t0, "Execute update deletes reference on table " + tableName);

			// Add to pending operations (if not empty)
			addToPendingOperationsIfNotEmpty(operation, pendingOperations, context);
		}

		// missing update :
		else if (hasMissingUpdates) {

			String tableName = operation.getTableName();
			SynchroTableMetadata table = targetMeta.getTable(tableName);

			// Get child to update and clean operation
			Map<String, Map<String, Object>> missingUpdates = operation.getMissingUpdates();
			operation.clearMissingUpdates();

			long t0 = TimeLog.getTime();

			if (log.isDebugEnabled()) {
				log.debug("Update missing references on table: " + tableName);
			}

			synchronizeColumnUpdates(operation,
					table,
					missingUpdates,
					sourceDaoFactory,
					targetDaoFactory,
					context,
					result,
					pendingOperations);

			TIME.log(t0, "Update missing reference on table " + tableName);

			// Add to pending operations (if not empty)
			addToPendingOperationsIfNotEmpty(operation, pendingOperations, context);
		}

		// missing detachment
		else if (hasMissingDetachs) {
			String tableName = operation.getTableName();
			SynchroTableMetadata table = targetMeta.getTable(tableName);

			// Get detachment infos and clean operation
			List<List<Object>> missingDetachs = operation.getMissingDetachs();
			operation.clearMissingDetachs();

			long t0 = TimeLog.getTime();

			if (log.isDebugEnabled()) {
				log.debug("Detachs rows on table: " + tableName);
			}

			synchronizeDetachs(operation,
					table,
					missingDetachs,
					sourceDaoFactory,
					targetDaoFactory,
					context,
					result,
					pendingOperations);

			TIME.log(t0, "Detachs rows on table " + tableName);

			// Add to pending operations (if not empty)
			addToPendingOperationsIfNotEmpty(operation, pendingOperations, context);
		}

		// Update child tables (selected by one column)
		else if (hasChildrenToUpdate) {

			// Get child to update and clean operation
			Map<String, Map<Set<String>, List<List<Object>>>> childrenToUpdate = operation.getChildrenToUpdate();
			operation.clearChildrenToUpdate();

			for (Entry<String, Map<Set<String>, List<List<Object>>>> entry : childrenToUpdate.entrySet()) {
				String tableName = entry.getKey();
				Map<Set<String>, List<List<Object>>> childToUpdate = entry.getValue();

				SynchroTableOperation childOperation = new SynchroTableOperation(tableName, context);
				SynchroTableMetadata table = targetMeta.getLoadedTable(tableName);

				long t0 = TimeLog.getTime();

				if (table != null) {
					if (log.isDebugEnabled()) {
						log.debug(String.format("Synchronize child table: %s (child of %s)", tableName, operation.getTableName()));
					}

					for (Set<String> columnNames : childToUpdate.keySet()) {
						List<List<Object>> columnsValues = childToUpdate.get(columnNames);

						synchronizeChildrenByFks(
								childOperation,
								table,
								columnNames,
								columnsValues,
								sourceDaoFactory,
								targetDaoFactory,
								context,
								result,
								pendingOperations);

					}
					TIME.log(t0, "synchronize child table " + tableName);

					// Add to pending operations (if not empty)
					addToPendingOperationsIfNotEmpty(childOperation, pendingOperations, context);
				}
			}
		}

		// Delete child tables (selected by one column)
		else if (hasChildrenToDelete) {

			// Get child to update and clean operation
			Map<String, Map<Set<String>, List<List<Object>>>> childrenTablesToDelete = operation.getChildrenToDelete();
			operation.clearChildrenToDelete();

			for (Entry<String, Map<Set<String>, List<List<Object>>>> entry : childrenTablesToDelete.entrySet()) {
				String tableName = entry.getKey();
				Map<Set<String>, List<List<Object>>> childrenToDelete = entry.getValue();

				SynchroTableOperation childOperation = new SynchroTableOperation(tableName, context);
				SynchroTableMetadata table = targetMeta.getLoadedTable(tableName);

				long t0 = TimeLog.getTime();

				if (table != null) {
					if (log.isDebugEnabled()) {
						log.debug(String.format("Delete from child table: %s (child of %s)", tableName, operation.getTableName()));
					}

					for (Set<String> columnNames : childrenToDelete.keySet()) {
						List<List<Object>> columnsValues = childrenToDelete.get(columnNames);

						synchronizeChildrenToDeletes(
								childOperation,
								table,
								columnNames,
								columnsValues,
								sourceDaoFactory,
								targetDaoFactory,
								context,
								result,
								pendingOperations);

					}
					TIME_DELETION.log(t0, "delete child table " + tableName);

					// Add to pending operations (if not empty)
					// DO NOT add addToPending here, because this is done in method synchronizeChildrenToDeletes()
					// (special queue management for deletion)
					// addToPendingOperationsIfNotEmpty(childOperation, pendingOperations, context);

				}
			}
		}

		// Detach child tables (selected by one column)
		else if (hasChildrenToDetach) {

			// Get child to update and clean operation
			Map<String, Map<Set<String>, List<List<Object>>>> childrenTablesToDetach = operation.getChildrenToDetach();
			operation.clearChildrenToDetach();

			for (Entry<String, Map<Set<String>, List<List<Object>>>> entry : childrenTablesToDetach.entrySet()) {
				String tableName = entry.getKey();
				Map<Set<String>, List<List<Object>>> childrenToDetach = entry.getValue();

				SynchroTableOperation childOperation = new SynchroTableOperation(tableName, context);
				SynchroTableMetadata table = targetMeta.getLoadedTable(tableName);

				long t0 = TimeLog.getTime();

				if (table != null) {
					if (log.isDebugEnabled()) {
						log.debug(String.format("Detach from child table: %s (child of %s)", tableName, operation.getTableName()));
					}

					for (Set<String> columnNames : childrenToDetach.keySet()) {
						List<List<Object>> columnsValues = childrenToDetach.get(columnNames);

						synchronizeChildrenToDetach(
								childOperation,
								table,
								columnNames,
								columnsValues,
								sourceDaoFactory,
								targetDaoFactory,
								context,
								result,
								pendingOperations);

					}
					TIME.log(t0, "detach child table " + tableName);

					addToPendingOperationsIfNotEmpty(childOperation, pendingOperations, context);
				}
			}
		}
	}

	protected final void addToPendingOperationsIfNotEmpty(SynchroTableOperation operation,
			Deque<SynchroTableOperation> pendingOperations,
			SynchroContext context) {
		boolean hasChildToUpdate = MapUtils.isNotEmpty(operation.getChildrenToUpdate());
		boolean hasChildToDelete = MapUtils.isNotEmpty(operation.getChildrenToDelete());
		boolean hasChildToDetach = MapUtils.isNotEmpty(operation.getChildrenToDetach());
		boolean hasMissingUpdates = MapUtils.isNotEmpty(operation.getMissingUpdates());
		boolean hasMissingDeletes = CollectionUtils.isNotEmpty(operation.getMissingDeletes());
		boolean hasMissingDetach = CollectionUtils.isNotEmpty(operation.getMissingDetachs());
		boolean isEmpty = !hasChildToUpdate && !hasChildToDelete && !hasMissingUpdates && !hasMissingDeletes && !hasMissingDetach & !hasChildToDetach;

		if (isEmpty) {
			return;
		}

		// Disable progression if reused
		operation.setEnableProgress(false);

		if (hasChildToUpdate || hasChildToDelete) {
			// if only child to update : put in the begin
			if (!hasMissingUpdates && !hasMissingDeletes) {
				pendingOperations.addFirst(operation);
			}

			// If has child to update AND (missing updates or deletes) : split in many operations
			else {
				if (hasMissingUpdates) {
					// Put missing updates at the end
					SynchroTableOperation newOperation = new SynchroTableOperation(operation.getTableName(), context);
					newOperation.addAllMissingColumnUpdates(operation.getMissingUpdates());
					pendingOperations.add(newOperation);
					operation.clearMissingUpdates();
				}
				if (hasMissingDeletes) {
					// Put missing deletes at the end
					SynchroTableOperation newOperation = new SynchroTableOperation(operation.getTableName(), context);
					newOperation.addAllMissingDelete(operation.getMissingDeletes());
					pendingOperations.add(newOperation);
					operation.clearMissingDeletes();
				}

				// Put child to update/delete on the top
				pendingOperations.addFirst(operation);
			}
		}

		// If has only missing updates or deletes
		else {
			// Put at the end
			pendingOperations.add(operation);
		}
	}

	protected final void synchronizeRootTable(
			SynchroTableMetadata table,
			DaoFactoryImpl sourceDaoFactory,
			DaoFactoryImpl targetDaoFactory,
			SynchroContext context,
			SynchroResult result,
			SynchroTableOperation operation,
			Deque<SynchroTableOperation> pendingOperations) throws SQLException {

		String tableName = table.getName();
		boolean enableUpdateDate = table.isWithUpdateDateColumn();

		result.getProgressionModel().setMessage(t("adagio.synchro.synchronize.step1", tableName));

		SynchroTableDao sourceDao = sourceDaoFactory.getSourceDao(table);
		SynchroTableDao targetDao = targetDaoFactory.getTargetDao(table, sourceDao, operation);

		ResultSet dataToUpdate = null;
		try {
			// Get existing Pks and updateDates from the target db
			long existingRowCount = targetDao.countAll(true);
			boolean doGetExistingPks = existingRowCount > 0
					&& existingRowCount <= MAX_ROW_COUNT_FOR_PK_PRELOADING;

			Set<String> existingPks = null;
			Map<String, Timestamp> existingUpdateDates = null;

			// Getting existing Pks only for not big table
			if (doGetExistingPks) {
				if (enableUpdateDate) {
					existingUpdateDates = targetDao.getPksStrWithUpdateDate();
					existingPks = existingUpdateDates.keySet();
				}
				else {
					existingPks = targetDao.getPksStr();
				}
			}

			else if (existingRowCount == 0) {
				// Avoid a new call of countAll() in updateTablexxx methods
				existingPks = Sets.newHashSet();
			}

			// Create a binding for the table, then get data to update from source db
			Map<String, Object> bindings = createSelectBindingsForTable(context, table.getName());
			dataToUpdate = sourceDao.getData(bindings);

			if (table.hasUniqueConstraints()) {
				updateTableWithUniqueConstraints(
						sourceDao,
						targetDao,
						dataToUpdate,
						existingPks,
						false,
						existingUpdateDates,
						context,
						result,
						pendingOperations);
			}
			else {
				updateTable(
						targetDao,
						dataToUpdate,
						existingPks,
						false,
						existingUpdateDates,
						context,
						result,
						pendingOperations);
			}
		} finally {
			Daos.closeSilently(dataToUpdate);
		}
	}

	/**
	 * Récupération des données d'une table à partir de valeurs de FKs
	 * <p/>
	 * Sert aussi bien quand les tables parentes ont une clef simple ou un clé composite
	 */
	protected void synchronizeChildrenByFks(
			SynchroTableOperation operation,
			SynchroTableMetadata table,
			Set<String> fkColumnNames,
			List<List<Object>> fkSourceColumnValues,
			DaoFactoryImpl sourceDaoFactory,
			DaoFactoryImpl targetDaoFactory,
			SynchroContext context,
			SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations) throws SQLException {

		String tableName = table.getName();
		boolean enableUpdateDate = table.isWithUpdateDateColumn() && table.isRoot();

		result.getProgressionModel().setMessage(t("adagio.synchro.synchronize.step1", tableName));

		SynchroTableDao sourceDao = sourceDaoFactory.getSourceDao(table);
		SynchroTableDao targetDao = targetDaoFactory.getTargetDao(table, sourceDao, operation);

		Map<String, Object> bindings = createSelectBindingsForTable(context, tableName);

		ResultSet dataToUpdate = null;
		try {
			// Get existing Pks and updateDates from the target db
			long existingRowCount = targetDao.countAll(true);
			Set<String> existingPks = null;
			Map<String, Timestamp> existingUpdateDates = null;

			if (existingRowCount > 0) {
				Set<String> fkTargetColumnNames = targetDao.transformColumnNames(fkColumnNames);
				List<List<Object>> fkTargetColumnValues = targetDao.transformOnRead(fkColumnNames, fkSourceColumnValues);
				if (enableUpdateDate) {
					existingUpdateDates = targetDao.getPksStrWithUpdateDateByFks(fkTargetColumnNames, fkTargetColumnValues, bindings);
					existingPks = existingUpdateDates.keySet();
				}
				else {
					existingPks = targetDao.getPksStrByFks(fkTargetColumnNames, fkTargetColumnValues, bindings);
				}
			}
			// Make sure the list is never null
			// and avoid a new call of countAll() in updateTablexxx methods
			if (existingPks == null) {
				existingPks = Sets.newHashSet();
			}

			// Create a binding for the table, then get data to update from source db
			dataToUpdate = sourceDao.getDataByFks(fkColumnNames, fkSourceColumnValues, bindings);

			// Table with a REMOTE_ID (and ID), or other unique constraint
			if (table.hasUniqueConstraints()) {
				updateTableWithUniqueConstraints(
						sourceDao,
						targetDao,
						dataToUpdate,
						existingPks,
						true, // manage deletion
						existingUpdateDates,
						context,
						result,
						pendingOperations);
			}

			// Association tables, ...
			else {
				updateTable(
						targetDao,
						dataToUpdate,
						existingPks,
						true, // manage deletion
						existingUpdateDates,
						context,
						result,
						pendingOperations);
			}
		} finally {
			Daos.closeSilently(dataToUpdate);
		}
	}

	/**
	 * Delete all child rows, by FK on parent table
	 */
	protected final void synchronizeChildrenToDeletes(
			SynchroTableOperation operation,
			SynchroTableMetadata table,
			Set<String> fkColumnNames,
			List<List<Object>> fkColumnValues,
			DaoFactoryImpl sourceDaoFactory,
			DaoFactoryImpl targetDaoFactory,
			SynchroContext context,
			SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations) throws SQLException {

		String tableName = table.getName();
		boolean hasChildTables = table.hasChildJoins();
		boolean checkPkNotUsed = isCheckPkNotUsedBeforeDelete(context);

		result.getProgressionModel().setMessage(t("adagio.synchro.synchronize.step5", tableName));

		SynchroTableDao sourceDao = sourceDaoFactory.getSourceDao(table);
		SynchroTableDao targetDao = targetDaoFactory.getTargetDao(table, sourceDao, operation);

		Map<String, Object> bindings = createSelectBindingsForTable(context, tableName);
		List<List<Object>> pksToDelete = targetDao.getPksByFks(fkColumnNames, fkColumnValues, bindings);

		if (CollectionUtils.isNotEmpty(pksToDelete)) {

			// If has children, add deletion to pending operations
			if (hasChildTables) {
				operation.addAllMissingDelete(pksToDelete);

				// Put again on the top of the queue
				pendingOperations.addFirst(operation);

				// Then add children on top
				addDeleteChildrenToDeque(table, pksToDelete, pendingOperations, context);
			}

			// If no children: do deletion
			else {
				deleteRows(targetDao,
						pksToDelete,
						checkPkNotUsed,
						context,
						result,
						pendingOperations);

				addToPendingOperationsIfNotEmpty(operation, pendingOperations, context);
			}
		}
	}

	/**
	 * Delete all child rows, by FK on parent table
	 */
	protected final void synchronizeChildrenToDetach(
			SynchroTableOperation operation,
			SynchroTableMetadata table,
			Set<String> fkColumnNames,
			List<List<Object>> fkColumnValues,
			DaoFactoryImpl sourceDaoFactory,
			DaoFactoryImpl targetDaoFactory,
			SynchroContext context,
			SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations) throws SQLException {

		String tableName = table.getName();
		boolean hasChildTables = table.hasChildJoins();

		result.getProgressionModel().setMessage(t("adagio.synchro.synchronize.step6", tableName));

		SynchroTableDao sourceDao = null;
		if (sourceDaoFactory != null) {
			sourceDao = sourceDaoFactory.getSourceDao(table);
		}
		SynchroTableDao targetDao = targetDaoFactory.getTargetDao(table, sourceDao, operation);

		Map<String, Object> bindings = createSelectBindingsForTable(context, tableName);
		List<List<Object>> pksToDetach = targetDao.getPksByFks(fkColumnNames, fkColumnValues, bindings);

		if (CollectionUtils.isNotEmpty(pksToDetach)) {

			detachRows(targetDao,
					pksToDetach,
					context,
					result,
					pendingOperations);

			// If has children, add detachment to pending operations
			if (hasChildTables) {
				// Then add children on top
				addDetachChildrenToDeque(table, pksToDetach, pendingOperations, context);
			}
		}
	}

	protected void synchronizeColumnUpdates(
			SynchroTableOperation operation,
			SynchroTableMetadata table,
			Map<String, Map<String, Object>> columnUpdatesByPkStr,
			DaoFactoryImpl sourceDaoFactory,
			DaoFactoryImpl targetDaoFactory,
			SynchroContext context,
			SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations
			) throws SQLException {

		String tableName = table.getName();

		result.getProgressionModel().setMessage(t("adagio.synchro.synchronize.step4", tableName));

		SynchroTableDao sourceDao = null;
		if (sourceDaoFactory != null) {
			sourceDao = sourceDaoFactory.getSourceDao(table);
		}
		SynchroTableDao targetDao = targetDaoFactory.getTargetDao(table, sourceDao, operation);

		// For each column to update
		for (String columnName : columnUpdatesByPkStr.keySet()) {
			Map<String, Object> valuesByPkStr = columnUpdatesByPkStr.get(columnName);

			// Update the column
			updateColumn(targetDao,
					columnName,
					valuesByPkStr,
					result,
					pendingOperations);
		}
	}

	protected void synchronizeDetachs(
			SynchroTableOperation operation,
			SynchroTableMetadata table,
			List<List<Object>> pksToDetach,
			DaoFactoryImpl sourceDaoFactory,
			DaoFactoryImpl targetDaoFactory,
			SynchroContext context,
			SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations
			) throws SQLException {

		String tableName = table.getName();
		boolean hasChildJoin = table.hasChildJoins();

		result.getProgressionModel().setMessage(t("adagio.synchro.synchronize.step6", tableName));

		SynchroTableDao sourceDao = null;
		if (sourceDaoFactory != null) {
			sourceDao = sourceDaoFactory.getSourceDao(table);
		}
		SynchroTableDao targetDao = targetDaoFactory.getTargetDao(table, sourceDao, operation);

		detachRows(targetDao,
				pksToDetach,
				context,
				result,
				pendingOperations);

		if (hasChildJoin) {
			addDetachChildrenToDeque(table,
					pksToDetach,
					pendingOperations,
					context);
		}
	}

	protected final void synchronizeDeletes(
			SynchroTableOperation operation,
			SynchroTableMetadata table,
			List<List<Object>> pksToDelete,
			DaoFactoryImpl sourceDaoFactory,
			DaoFactoryImpl targetDaoFactory,
			SynchroContext context,
			SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations
			) throws SQLException {

		String tableName = table.getName();
		boolean checkPkNotUsed = isCheckPkNotUsedBeforeDelete(context);

		result.getProgressionModel().setMessage(t("adagio.synchro.synchronize.step3", tableName));

		SynchroTableDao sourceDao = null;
		if (sourceDaoFactory != null) {
			sourceDao = sourceDaoFactory.getSourceDao(table);
		}
		SynchroTableDao targetDao = targetDaoFactory.getTargetDao(table, sourceDao, operation);

		deleteRows(targetDao,
				pksToDelete,
				checkPkNotUsed,
				context,
				result,
				pendingOperations);
	}

	/**
	 * To update the content of the given {@code table} on the target db,
	 * from the given {@code incomingData} of the source db.
	 * <p/>
	 * The algorithm use remote_id : for each row of the {@code incomingData}, if exists on target table, then do an
	 * update, otherwise do a insert.
	 * <p/>
	 * 
	 * @param synchroContext
	 *            Synchronization context
	 * @param targetDao
	 *            connection on the target db
	 * @param incomingData
	 *            data to update from the source db
	 * @param result
	 *            where to store operation results
	 * @throws SQLException
	 *             if any sql errors
	 */
	protected final void updateTableWithUniqueConstraints(
			SynchroTableDao sourceDao,
			SynchroTableDao targetDao,
			ResultSet incomingData,
			Set<String> existingPks,
			boolean deleteExtraRows,
			Map<String, Timestamp> existingUpdateDates,
			SynchroContext context,
			SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations) throws SQLException {

		SynchroTableMetadata table = targetDao.getTable();
		Preconditions.checkArgument(MapUtils.isNotEmpty(table.getUniqueConstraints()));

		boolean hasChildTables = table.hasChildJoins();
		boolean checkUpdateDate = table.isWithUpdateDateColumn() && table.isRoot();
		boolean allowSkipRow = targetDao.getCurrentOperation().isEnableProgress();

		String tableName = table.getName();
		String tablePrefix = table.getTableLogPrefix() + " - " + result.getNbRows(tableName);
		Map<String, DuplicateKeyStrategy> duplicateKeyStrategies = table.getDuplicatKeyStrategies();

		boolean isTableEmpty = targetDao.countAll(true) == 0;

		List<List<Object>> processedSourcePks = null;
		if (hasChildTables) {
			processedSourcePks = Lists.newArrayList();
		}
		Set<String> targetPksStrToRemove = Sets.newHashSet();
		if (deleteExtraRows && existingPks != null) {
			targetPksStrToRemove.addAll(existingPks);
		}

		result.addTableName(tableName);
		int countR = 0;

		// For each incoming row
		while (incomingData.next()) {

			List<Object> pk = null;
			boolean skipRow = false;

			// Retrieve duplicate keys (if table is not empty)
			Map<String, List<Object>> duplicatedKeys = null;
			if (!isTableEmpty) {
				duplicatedKeys = targetDao.getPkFromUniqueConstraints(incomingData);
			}

			boolean hasDuplicateKey = MapUtils.isNotEmpty(duplicatedKeys);

			// Retrieve pk from duplicate keys
			if (hasDuplicateKey) {

				// Check each duplicate key found
				Map<String, List<Object>> duplicatedKeysRejected = Maps.newHashMap();
				for (Entry<String, List<Object>> entry : duplicatedKeys.entrySet()) {
					DuplicateKeyStrategy strategy = duplicateKeyStrategies.get(entry.getKey());
					List<Object> rowPk = entry.getValue();
					switch (strategy) {
					case WARN:
						log.warn(String.format("%s Duplicate key (%s): [%s]", tablePrefix, entry.getKey(), rowPk));
						break;
					case REPLACE:
						if (debug) {
							log.trace(String.format("%s Duplicate key (%s): [%s] - will try to update this row", tablePrefix, entry.getKey(), rowPk));
						}
						pk = rowPk;
						break;
					case REJECT:
						duplicatedKeysRejected.put(entry.getKey(), rowPk);
						break;
					}
				}

				// Check each duplicate key found, with the strategy REJECT
				for (Entry<String, List<Object>> entry : duplicatedKeysRejected.entrySet()) {
					List<Object> rowPk = entry.getValue();
					if (pk == null || !pk.equals(rowPk)) {
						if (debug) {
							log.trace(String.format("%s Duplicate key (%s): [%s] - will reject this row", tablePrefix, entry.getKey(), rowPk));
						}
						String rowPkStr = SynchroTableMetadata.toPkStr(rowPk);
						targetPksStrToRemove.remove(rowPkStr);
						skipRow = true;
						break;
					}
				}
			}

			// Duplicate key : add row to rejects
			if (skipRow) {
				rejectDuplicatedRow(tableName,
						sourceDao,
						incomingData,
						result);
				if (!allowSkipRow) {
					// Retrieve the original PK (not transformed)
					List<Object> sourcePk = targetDao.getPk(incomingData);
					throw new SynchroDuplicateRowException(tableName, sourcePk, pk);
				}
			}
			else {
				// Will update if a pk has been found
				boolean doUpdate = (pk != null);

				// apply the update, using the pk found
				if (doUpdate) {
					String pkStr = SynchroTableMetadata.toPkStr(pk);
					targetPksStrToRemove.remove(pkStr);

					// Try to lock
					if (!targetDao.lock(pk)) {
						// lock failed: add row to rejects, and skip
						rejectLockedRow(tableName,
								sourceDao,
								incomingData,
								result);
						skipRow = true;
					}

					else if (checkUpdateDate) {
						Timestamp transformIncomingUpdateDate = targetDao.getUpdateDate(incomingData, true);
						Timestamp existingUpdateDate = null;
						if (existingUpdateDates != null) {
							existingUpdateDate = existingUpdateDates.get(pkStr);
						}
						else {
							existingUpdateDate = targetDao.getUpdateDateByPk(pk);
						}
						int updateDateCompare = Daos.compareUpdateDates(existingUpdateDate, transformIncomingUpdateDate);

						// up to date : skip
						if (updateDateCompare == 0) {
							if (debug) {
								log.trace(String.format("%s row is up to date: [%s] - will skip this row", tablePrefix, pkStr));
							}
							skipRow = true;
						}

						else {
							Timestamp incomingUpdateDate = targetDao.getUpdateDate(incomingData, false);
							updateDateCompare = Daos.compareUpdateDates(existingUpdateDate, incomingUpdateDate);

							// existing date > incoming date: reject row (row has been updated !)
							if (updateDateCompare > 0) {
								rejectBadUpdateDateRow(tableName,
										sourceDao,
										incomingData,
										existingUpdateDate,
										pkStr,
										result);
								if (debug) {
									log.trace(String
											.format(
													"%s row has incorrect update_date: [%s] - will reject this row (expected update_date: '%s' but found: '%s')",
													tablePrefix, pkStr, existingUpdateDate, incomingUpdateDate));
								}
								skipRow = true;
							} else if (debug) {
								log.trace(String.format("%s row is older in target DB: [%s] - row will be updated (new update_date: '%s')",
										tablePrefix,
										pkStr, incomingUpdateDate));
							}
						}
					}

					if (!skipRow) {
						targetDao.executeUpdate(pk, incomingData);
					}
				}

				// Insert (if no pk found)
				else {

					try {
						targetDao.executeInsert(incomingData);
					} catch (SynchroRejectRowException re) {
						if (!allowSkipRow) {
							throw re;
						}
						// Skip the row, and continue
						rejectRow(re,
								result);
						skipRow = true;
					}
				}
			}

			if (!skipRow) {
				// Mark has updated
				if (hasChildTables) {
					// Retrieve the original PK (not transformed)
					List<Object> sourcePk = targetDao.getPk(incomingData);
					processedSourcePks.add(sourcePk);
				}
			}

			countR++;

			reportProgress(result, targetDao, countR, tablePrefix);
		}

		targetDao.flush();

		// Process row to delete
		if (CollectionUtils.isNotEmpty(targetPksStrToRemove)) {
			List<List<Object>> targetPksToRemove = SynchroTableMetadata.fromPksStr(targetPksStrToRemove);

			// If has children, add deletion to pending operations
			if (hasChildTables) {
				targetDao.getCurrentOperation().addAllMissingDelete(targetPksToRemove);
				addDeleteChildrenToDeque(table, targetPksToRemove, pendingOperations, context);
			}

			// If no children: do deletion
			else {
				deleteRows(targetDao,
						targetPksToRemove,
						isCheckPkNotUsedBeforeDelete(context),
						context,
						result,
						pendingOperations);
			}
		}

		// Add child table to pending operation
		if (hasChildTables && CollectionUtils.isNotEmpty(processedSourcePks)) {
			addChildrenToDeque(table, processedSourcePks, pendingOperations, context);
		}

		int insertCount = targetDao.getInsertCount();
		int updateCount = targetDao.getUpdateCount();
		int deleteCount = targetDao.getDeleteCount();

		result.addInserts(tableName, insertCount);
		result.addUpdates(tableName, updateCount);
		result.addDeletes(tableName, deleteCount);

		if (log.isInfoEnabled()) {
			log.info(String.format("%s done: %s (inserts: %s, updates: %s, delete: %s)", tablePrefix, insertCount + updateCount + deleteCount,
					insertCount, updateCount, deleteCount));
		}

		if (log.isDebugEnabled()) {
			log.debug(String.format("%s INSERT count: %s", tablePrefix, insertCount));
			log.debug(String.format("%s UPDATE count: %s", tablePrefix, updateCount));
			log.debug(String.format("%s DELETE count: %s", tablePrefix, deleteCount));
		}

		if (targetDao.getCurrentOperation().isEnableProgress()) {
			result.getProgressionModel().increments(countR % batchSize);
		}

	}

	/**
	 * To update the content of the given {@code table} on the target db,
	 * from the given {@code incomingData} of the source db.
	 * The algorithm use a standard update strategy, using primary key.
	 * 
	 * @param synchroContext
	 * @param targetDao
	 * @param incomingData
	 * @param result
	 * @throws SQLException
	 */
	protected final void updateTable(
			SynchroTableDao targetDao,
			ResultSet incomingData,
			Set<String> existingPks,
			boolean deleteExtraRows,
			Map<String, Timestamp> existingUpdateDates,
			SynchroContext context,
			SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations) throws SQLException {
		SynchroTableMetadata table = targetDao.getTable();
		Preconditions.checkArgument(MapUtils.isEmpty(table.getUniqueConstraints()));

		String tableName = table.getName();
		String tablePrefix = table.getTableLogPrefix() + " - " + result.getNbRows(tableName);

		boolean enableUpdateDateCheck = table.isWithUpdateDateColumn() && table.isRoot();
		boolean hasChildTables = table.hasChildJoins();

		List<List<Object>> processedTargetPks = Lists.newArrayList();

		long existingRowCount = existingPks != null ? existingPks.size() : targetDao.countAll(true);
		boolean isTableEmpty = existingRowCount == 0;
		if (log.isDebugEnabled()) {
			log.debug(tablePrefix + " existing rows: " + existingRowCount);
		}

		List<List<Object>> updatedPks = null;
		if (hasChildTables) {
			updatedPks = Lists.newArrayList();
		}
		Set<String> targetPksStrToRemove = Sets.newHashSet();
		if (deleteExtraRows && existingPks != null) {
			targetPksStrToRemove.addAll(existingPks);
		}

		result.addTableName(tableName);
		int countR = 0;

		// For each incoming row
		while (incomingData.next()) {
			List<Object> pk = targetDao.getPk(incomingData, true);
			String pkStr = SynchroTableMetadata.toPkStr(pk);

			boolean doUpdate = false;
			if (!isTableEmpty) {
				if (existingPks != null) {
					doUpdate = existingPks.contains(pkStr);
				}
				else {
					doUpdate = targetDao.exists(pk);
				}
			}
			boolean skipRow = false;

			if (doUpdate) {
				targetPksStrToRemove.remove(pkStr);

				if (enableUpdateDateCheck) {
					Timestamp transformedIncomingUpdateDate = targetDao.getUpdateDate(incomingData, true);
					Timestamp existingUpdateDate = null;
					if (existingUpdateDates != null) {
						existingUpdateDate = existingUpdateDates.get(pkStr);
					}
					else {
						existingUpdateDate = targetDao.getUpdateDateByPk(pk);
					}
					int updateDateCompare = Daos.compareUpdateDates(existingUpdateDate, transformedIncomingUpdateDate);

					// up to date : skip
					if (updateDateCompare == 0) {
						skipRow = true;
					}

					else {
						Timestamp incomingUpdateDate = targetDao.getUpdateDate(incomingData, false);
						updateDateCompare = Daos.compareUpdateDates(existingUpdateDate, incomingUpdateDate);

						// existing date > incoming date: reject row (row has been updated !)
						if (updateDateCompare > 0) {
							rejectBadUpdateDateRow(tableName,
									targetDao,
									incomingData,
									existingUpdateDate,
									pkStr,
									result);
							skipRow = true;
						}

						else if (debug) {
							log.trace(String.format("%s row is older in target DB: [%s] - row will be updated (new update_date: '%s')",
									tablePrefix,
									pkStr, incomingUpdateDate));
						}
					}
				}
				if (!skipRow) {
					targetDao.executeUpdate(pk, incomingData);
				}

			} else {

				targetDao.executeInsert(incomingData);
			}

			if (!skipRow) {
				if (hasChildTables) {
					// Retrieve the original PK (not transformed)
					List<Object> incomingPk = targetDao.getPk(incomingData);
					updatedPks.add(incomingPk);
				}

				if (!isTableEmpty && pk != null) {
					processedTargetPks.add(pk);
				}
			}

			countR++;

			reportProgress(result, targetDao, countR, tablePrefix);
		}

		// Process row to delete
		if (CollectionUtils.isNotEmpty(targetPksStrToRemove)) {
			List<List<Object>> targetPksToRemove = SynchroTableMetadata.fromPksStr(targetPksStrToRemove);

			// If has children, add deletion to pending operations
			if (hasChildTables) {
				targetDao.getCurrentOperation().addAllMissingDelete(targetPksToRemove);
				addDeleteChildrenToDeque(table, targetPksToRemove, pendingOperations, context);
			}

			// If no children: do deletion
			else {
				// For each pk to delete
				for (List<Object> pk : targetPksToRemove) {
					targetDao.executeDelete(
							pk,
							isCheckPkNotUsedBeforeDelete(context)
							);

					countR++;

					reportProgress(result, targetDao, countR, tablePrefix);
				}

			}
		}

		targetDao.flush();

		// Put in context (to be used by child join tables)
		if (hasChildTables && CollectionUtils.isNotEmpty(updatedPks)) {
			addChildrenToDeque(table, updatedPks, pendingOperations, context);
		}

		int insertCount = targetDao.getInsertCount();
		int updateCount = targetDao.getUpdateCount();
		int deleteCount = targetDao.getDeleteCount();

		result.addInserts(tableName, insertCount);
		result.addUpdates(tableName, updateCount);
		result.addDeletes(tableName, deleteCount);

		if (log.isInfoEnabled()) {
			log.info(String.format("%s done: %s (inserts: %s, updates: %s, delete: %s)", tablePrefix, insertCount + updateCount + deleteCount,
					insertCount, updateCount, deleteCount));
		}

		if (log.isDebugEnabled()) {
			log.debug(String.format("%s INSERT count: %s", tablePrefix, insertCount));
			log.debug(String.format("%s UPDATE count: %s", tablePrefix, updateCount));
			log.debug(String.format("%s DELETE count: %s", tablePrefix, deleteCount));
		}

		if (targetDao.getCurrentOperation().isEnableProgress()) {
			result.getProgressionModel().increments(countR % batchSize);
		}
	}

	/**
	 * To delete some rows of the given {@code table} on the target db,
	 * from the given {@code pks}.
	 * 
	 * @param synchroContext
	 * @param targetDao
	 * @param pks
	 *            PKs of the row to delete
	 * @param result
	 * @throws SQLException
	 */
	protected void deleteRows(
			SynchroTableDao targetDao,
			List<List<Object>> pks,
			boolean checkPkNotUsedBeforeDelete,
			SynchroContext context,
			SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations) throws SQLException {
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(pks));

		SynchroTableMetadata table = targetDao.getTable();
		String tableName = table.getName();
		String tablePrefix = table.getTableLogPrefix() + " - " + result.getNbRows(tableName);

		SynchroTableOperation operation = targetDao.getCurrentOperation();

		result.addTableName(tableName);

		// For each pk to delete
		int countR = 0;
		for (List<Object> pk : pks) {
			try {

				targetDao.executeDelete(
						pk,
						checkPkNotUsedBeforeDelete);
				countR++;
				reportProgress(result, targetDao, countR, tablePrefix);

			}

			// If check failed (e.g. PK still referenced by tables)
			catch (DataIntegrityViolationOnDeleteException e) {

				// If the deletion could be done later: add to missing delete
				// (this is allowed only once)
				boolean retryLater = operation.isAllowMissingDeletes()
						&& hasProcessedTable(e.getExistingFkTableNames(), context);

				// If missing deletes are NOT allowed: throw error
				if (!retryLater) {
					throw e;
				}
				// Else, do deletion later (mantis #23383)
				else {
					operation.setAllowMissingDeletes(false);
					operation.addMissingDelete(pk);
				}
			}
		}

		targetDao.flush();

		int deleteCount = targetDao.getDeleteCount();

		result.addDeletes(tableName, deleteCount);

		if (log.isInfoEnabled()) {
			log.info(String.format("%s done: %s (deletes: %s)", tablePrefix, deleteCount, deleteCount));
		}

		if (targetDao.getCurrentOperation().isEnableProgress()) {
			result.getProgressionModel().increments(countR % batchSize);
		}
	}

	protected void updateColumn(
			SynchroTableDao targetDao,
			String columnName,
			Map<String, Object> valuesByPkStr,
			SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations) throws SQLException {
		Preconditions.checkArgument(MapUtils.isNotEmpty(valuesByPkStr));

		SynchroTableMetadata table = targetDao.getTable();
		String tableName = table.getName();
		String tablePrefix = table.getTableLogPrefix() + " - " + result.getNbRows(tableName);

		result.addTableName(tableName);

		int countR = 0;

		// For each pk's row to update
		for (Entry<String, Object> entry : valuesByPkStr.entrySet()) {
			if (entry.getKey() != null) {

				List<Object> pk = SynchroTableMetadata.fromPkStr(entry.getKey());
				Object columnValue = entry.getValue();

				// Execute update on the pk's row
				targetDao.executeUpdateColumn(columnName, pk, columnValue);

				countR++;
				reportProgress(result, targetDao, countR, tablePrefix);
			}
		}
		targetDao.flush();

		int updateCount = targetDao.getUpdateColumnCount(columnName);
		result.addUpdates(tableName, updateCount);

		if (log.isInfoEnabled()) {
			log.info(String.format("%s done: %s (updates: %s)", tablePrefix, updateCount, updateCount));
		}

		if (targetDao.getCurrentOperation().isEnableProgress()) {
			result.getProgressionModel().increments(countR % batchSize);
		}
	}

	/**
	 * To detach some rows of the given {@code targetDao.getTable()} on the target db,
	 * from the given {@code pks}.<br/>
	 * 
	 * Should be override (e.g. to reset remote_id...)
	 * 
	 * @param targetDao
	 * @param pks
	 * @param result
	 * @param pendingOperations
	 * @throws SQLException
	 */
	protected void detachRows(
			SynchroTableDao targetDao,
			List<List<Object>> pks,
			SynchroContext context,
			SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations)
			throws SQLException {

		SynchroTableMetadata table = targetDao.getTable();
		String tableName = table.getName();
		String tablePrefix = table.getTableLogPrefix() + " - " + result.getNbRows(tableName);

		result.addTableName(tableName);

		// For each pk to detach
		int countR = 0;
		for (List<Object> pk : pks) {
			targetDao.executeDetach(pk);

			countR++;
			reportProgress(result, targetDao, countR, tablePrefix);
		}

		targetDao.flush();

		result.addUpdates(tableName, countR);

		if (log.isInfoEnabled()) {
			log.info(String.format("%s done: %s (detach: %s)", tablePrefix, countR, countR));
		}

		if (targetDao.getCurrentOperation().isEnableProgress()) {
			result.getProgressionModel().increments(countR % batchSize);
		}

	}

	protected final void rejectDuplicatedRow(
			String tableName,
			SynchroTableDao sourceDao,
			ResultSet incomingData,
			SynchroResult result
			) throws SQLException {

		List<Object> sourcePk = sourceDao.getPk(incomingData);
		String sourcePkStr = SynchroTableMetadata.toPkStr(sourcePk);
		result.addReject(tableName, sourcePkStr, RejectedRowStatus.DUPLICATE_KEY.name());
	}

	protected final void rejectLockedRow(
			String tableName,
			SynchroTableDao sourceDao,
			ResultSet incomingData,
			SynchroResult result
			) throws SQLException {

		List<Object> sourcePk = sourceDao.getPk(incomingData);
		String sourcePkStr = SynchroTableMetadata.toPkStr(sourcePk);
		result.addReject(tableName, sourcePkStr, RejectedRowStatus.LOCKED.name());
	}

	protected final void rejectBadUpdateDateRow(
			String tableName,
			SynchroTableDao sourceDao,
			ResultSet incomingData,
			Timestamp validUpdateDate,
			String targetPkStr,
			SynchroResult result
			) throws SQLException {

		// Retrieve source Pk
		List<Object> sourcePk = sourceDao.getPk(incomingData);
		String sourcePkStr = SynchroTableMetadata.toPkStr(sourcePk);

		rejectBadUpdateDateRow(
				tableName,
				sourcePkStr,
				validUpdateDate,
				targetPkStr,
				result);
	}

	protected final void rejectBadUpdateDateRow(
			String tableName,
			String sourcePkStr,
			Timestamp validUpdateDate,
			String targetPkStr,
			SynchroResult result
			) throws SQLException {

		result.addReject(tableName,
				sourcePkStr,
				RejectedRowStatus.BAD_UPDATE_DATE.name(),
				validUpdateDate.toString(),
				targetPkStr);
	}

	protected final void rejectRow(
			SynchroRejectRowException error,
			SynchroResult result
			) throws SQLException {
		Preconditions.checkNotNull(error);

		if (error instanceof SynchroDeletedRowException) {
			SynchroDeletedRowException e = (SynchroDeletedRowException) error;
			result.addReject(e.getTableName(),
					e.getSourcePkStr(),
					RejectedRowStatus.DELETED.name(),
					e.getTargetPkStr());
			return;
		}

		// Not known exception class
		throw new SynchroTechnicalException(String.format("Unknown reject exception class: %s", error.getClass().getName()));
	}

	protected List<SynchroTableOperation> getRootOperations(
			DaoFactoryImpl sourceDaoFactory,
			DaoFactoryImpl targetDaoFactory,
			SynchroDatabaseMetadata dbMeta,
			SynchroContext context) throws SQLException {
		Set<String> rootTableNames = dbMeta.getLoadedRootTableNames();

		List<SynchroTableOperation> result = Lists.newArrayListWithCapacity(rootTableNames.size());
		for (String rootTableName : rootTableNames) {
			SynchroTableOperation operation = new SynchroTableOperation(rootTableName, context);
			operation.setEnableProgress(true);
			result.add(operation);
		}
		return result;
	}

	protected void prepareRootTable(
			DaoFactoryImpl sourceDaoFactory,
			DaoFactoryImpl targetDaoFactory,
			SynchroTableMetadata table,
			SynchroContext context,
			SynchroResult result) throws SQLException {

		String tablePrefix = table.getTableLogPrefix();
		String tableName = table.getName();

		SynchroTableDao sourceDao = sourceDaoFactory.getSourceDao(table);
		SynchroTableDao targetDao = targetDaoFactory.getTargetDao(table, sourceDao, null);

		// Skip get last update if:
		// - target DB is a mirror database (=empty database)
		// - OR the table is too big (e.g. a data table on server)
		boolean skipGetLastUpdateDate = context.getTarget().isMirrorDatabase()
				|| targetDao.countAll(true) > MAX_ROW_COUNT_FOR_PK_PRELOADING;

		Timestamp updateDate = null;
		if (!skipGetLastUpdateDate) {
			// get last updateDate used by target db
			updateDate = targetDao.getLastUpdateDate();

			if (updateDate != null) {

				// just inscrements of 1 second to not having same
				// TODO BL : attention, a cause des transactions parfois longues sur le serveur Oracle, il faut
				// peut-etre justement prendre une date inférieure au max(update_date) ?? genre max(update_date) -
				// 2h ?
				// Ou mieux : stocker puis utiliser une date de dernière mise à jour (systimestamp côté serveur)
				updateDate = new Timestamp(DateUtils.setMilliseconds(updateDate, 0).getTime());
				updateDate = new Timestamp(DateUtils.addSeconds(updateDate, 1).getTime());
			}
		}
		result.setUpdateDate(tableName, updateDate);

		Map<String, Object> bindings = createSelectBindingsForTable(context, tableName);
		long countToUpdate = sourceDao.countData(bindings);
		result.addRows(tableName, (int) countToUpdate);
		if (log.isInfoEnabled()) {
			log.info(String.format("%s nb rows to update: %s", tablePrefix, countToUpdate));
		}
	}

	protected void releaseContext(SynchroContext context) {
		context.setSourceMeta(null);
		context.setTargetMeta(null);
	}

	protected void prepareSynch(Connection targetConnection, SynchroContext context) throws SQLException {
		if (disableIntegrityConstraints || context.getTarget().isMirrorDatabase()) {
			Daos.setIntegrityConstraints(targetConnection, false);
		}
	}

	protected void releaseSynch(Connection targetConnection, SynchroContext context) throws SQLException {
		if (disableIntegrityConstraints) {
			// If a last synchronization date has been set
			// do not try a enable integrity, because content is maybe partial
			if (context.getLastSynchronizationDate() != null && context.getTarget().isMirrorDatabase()) {
				return;
			}

			Daos.setIntegrityConstraints(targetConnection, true);
		}
	}

	protected final void addChildrenToDeque(
			SynchroTableMetadata parentTable,
			List<List<Object>> parentPks,
			Deque<SynchroTableOperation> pendingOperations,
			SynchroContext context) {

		Set<String> pkNames = parentTable.getPkNames();

		// More than one PK: not implement yet
		if (pkNames.size() > 1) {
			throw new UnsupportedOperationException("Not sure of this implementation: please check before comment out this exception !");
		}

		// Retrieve childs join
		for (SynchroJoinMetadata join : parentTable.getChildJoins()) {
			SynchroTableMetadata childTable = join.getTargetTable();
			SynchroColumnMetadata childTableColumn = join.getTargetColumn();

			// Add child to update into operation
			SynchroTableOperation operation = new SynchroTableOperation(parentTable.getName(), childTable.getName(), childTableColumn.getName(),
					parentPks, context);
			pendingOperations.addFirst(operation);
		}
	}

	protected final void addDeleteChildrenToDeque(SynchroTableMetadata parentTable,
			List<List<Object>> parentPks,
			Deque<SynchroTableOperation> pendingOperations,
			SynchroContext context) {

		Set<String> pkNames = parentTable.getPkNames();

		// More than one PK: not implemented yet
		if (pkNames.size() > 1) {
			throw new UnsupportedOperationException("Not sure of this implementation: please check before comment out this exception !");
		}

		// Retrieve childs join
		for (SynchroJoinMetadata join : parentTable.getChildJoins()) {
			SynchroTableMetadata childTable = join.getTargetTable();
			SynchroColumnMetadata childTableColumn = join.getTargetColumn();

			// Add child to update into operation
			SynchroTableOperation operation = new SynchroTableOperation(parentTable.getName(), context);
			operation.addChildrenToDeleteFromManyColumns(childTable.getName(), ImmutableSet.of(childTableColumn.getName()), parentPks);
			pendingOperations.addFirst(operation);
		}
	}

	protected final void addDetachChildrenToDeque(SynchroTableMetadata parentTable,
			List<List<Object>> parentPks,
			Deque<SynchroTableOperation> pendingOperations,
			SynchroContext context) {

		Set<String> pkNames = parentTable.getPkNames();

		// More than one PK: not implemented yet
		if (pkNames.size() > 1) {
			throw new UnsupportedOperationException("Not sure of this implementation: please check before comment out this exception !");
		}

		// Retrieve childs join
		for (SynchroJoinMetadata join : parentTable.getChildJoins()) {
			SynchroTableMetadata childTable = join.getTargetTable();
			SynchroColumnMetadata childTableColumn = join.getTargetColumn();

			// Add child to update into operation
			SynchroTableOperation operation = new SynchroTableOperation(parentTable.getName(), context);
			operation.addChildrenToDetachFromManyColumns(childTable.getName(), ImmutableSet.of(childTableColumn.getName()), parentPks);
			pendingOperations.addFirst(operation);
		}
	}

	protected List<SynchroTableOperation> getSourceMissingOperations(
			SynchroResult synchroResult,
			SynchroContext context) {

		List<SynchroTableOperation> result = Lists.newArrayList();

		// Add source missing updates
		if (MapUtils.isNotEmpty(synchroResult.getSourceMissingUpdates())) {
			for (Entry<String, Map<String, Map<String, Object>>> entry : synchroResult.getSourceMissingUpdates().entrySet()) {
				String tableName = entry.getKey();
				Map<String, Map<String, Object>> tableMissingUpdates = entry.getValue();

				SynchroTableOperation operation = new SynchroTableOperation(tableName, context);
				operation.addAllMissingColumnUpdates(tableMissingUpdates);
				result.add(operation);
			}
		}

		// Add source missing deletes
		if (synchroResult.getSourceMissingDeletes() != null
				&& !synchroResult.getSourceMissingDeletes().isEmpty()) {

			for (String tableName : synchroResult.getSourceMissingDeletes().keySet()) {
				Collection<String> pkStrs = synchroResult.getSourceMissingDeletes().get(tableName);

				List<List<Object>> pks = Lists.newArrayList();
				for (String pkStr : pkStrs) {
					List<Object> pk = SynchroTableMetadata.fromPkStr(pkStr);
					pks.add(pk);
				}
				SynchroTableOperation operation = new SynchroTableOperation(tableName, context);
				operation.addAllMissingDelete(pks);
				result.add(operation);
			}
		}

		return result;
	}

	/**
	 * Process rejects
	 * 
	 * @param synchroContext
	 *            Context of synchronization
	 * @param rejects
	 *            rejects to process
	 * @param rejectStrategies
	 *            strategies to use on rejected row, by reject status
	 * @throws SQLException
	 */
	protected void resolveRejects(
			Connection targetConnection,
			SynchroDatabaseMetadata dbMeta,
			DaoFactoryImpl targetDaoFactory,
			SynchroContext context,
			Map<String, String> rejects,
			Map<RejectedRowStatus, RejectedRowStrategy> rejectStrategies,
			Deque<SynchroTableOperation> pendingOperations) throws SQLException {
		Preconditions.checkNotNull(rejects);
		Preconditions.checkNotNull(rejectStrategies);

		if (log.isDebugEnabled()) {
			log.debug(String.format("Resolving rejects with strategies %s - %s", rejectStrategies, context.toString()));
		}

		SynchroResult result = context.getResult();

		// Process rejects
		for (Entry<String, String> entry : rejects.entrySet()) {
			String tableName = entry.getKey();
			String tableRejects = entry.getValue();

			SynchroTableMetadata table = dbMeta.getTable(tableName);

			// process the table
			resolveTableRejects(table,
					tableRejects,
					rejectStrategies,
					targetDaoFactory,
					context,
					result,
					pendingOperations);
		}
	}

	protected void resolveTableRejects(SynchroTableMetadata table,
			String rejectsRows,
			Map<RejectedRowStatus, RejectedRowStrategy> strategies,
			DaoFactoryImpl daoFactory,
			SynchroContext context,
			SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations
			) throws SQLException {

		String tableName = table.getName();
		ProgressionModel progressionModel = result.getProgressionModel();

		progressionModel.setMessage(t("adagio.synchro.data.finish.step1", tableName));
		if (log.isDebugEnabled()) {
			log.debug(t("adagio.synchro.data.finish.step1", tableName));
		}

		SynchroTableOperation operation = new SynchroTableOperation(tableName, context);

		for (String rejectRow : Splitter.on("\n").omitEmptyStrings().split(rejectsRows)) {
			String[] rejectInfos = rejectRow.split(";");

			String pkStr = rejectInfos[0];
			RejectedRowStatus rejectStatus = RejectedRowStatus.valueOf(rejectInfos[1]);
			RejectedRowStrategy rejectStrategy = strategies.get(rejectStatus);
			if (rejectStrategy == null) {
				rejectStrategy = RejectedRowStrategy.DO_NOTHING;
			}

			resolveTableRow(
					table,
					pkStr,
					rejectInfos,
					rejectStatus,
					rejectStrategy,
					operation,
					context,
					result);
		}

		// If operation is not empty: add it to queue
		addToPendingOperationsIfNotEmpty(operation, pendingOperations, context);
	}

	/**
	 * Process a reject. Could add some missing updates on the current operation
	 * 
	 * @param table
	 * @param pkStr
	 * @param rejectInfos
	 * @param rejectStatus
	 * @param rejectStrategy
	 * @param operation
	 * @param result
	 */
	protected void resolveTableRow(SynchroTableMetadata table,
			String pkStr,
			String[] rejectInfos,
			RejectedRowStatus rejectStatus,
			RejectedRowStrategy rejectStrategy,
			SynchroTableOperation operation,
			SynchroContext context,
			SynchroResult result) {
		String tableName = table.getName();

		switch (rejectStatus) {
		case BAD_UPDATE_DATE: {
			Timestamp updateDate = Timestamp.valueOf(rejectInfos[2]);

			switch (rejectStrategy) {
			case KEEP_LOCAL:
				if (debug) {
					log.debug(String.format("[%s] Bad update date [pk=%s]: need update_date=%s", tableName, pkStr, updateDate));
				}
				operation.addMissingColumnUpdate(context.getTarget().getColumnUpdateDate(), pkStr, updateDate);
				break;

			case UPDATE:
				String targetPkStr = rejectInfos.length < 4 ? pkStr : rejectInfos[3];
				if (debug) {
					log.debug(String.format("[%s] Bad update date [pk=%s]: will re-import remote pk [%s]", tableName, pkStr, targetPkStr));
				}
				result.addSourceMissingRevert(tableName, targetPkStr);
				break;

			case DO_NOTHING:
				result.addReject(tableName, pkStr, rejectStatus.name());

			default:
				break;
			}
			break;
		}
		case DELETED: {
			switch (rejectStrategy) {
			case KEEP_LOCAL:
				if (debug) {
					log.debug(String.format("[%s] Deleted row on server [pk=%s] will be keep locally", tableName, pkStr));
				}
				operation.addMissingDetach(pkStr);
				break;

			case UPDATE:
				if (debug) {
					log.debug(String.format("[%s] Deleted row on server [pk=%s] will deleted on local", tableName, pkStr));
				}
				operation.addMissingDelete(pkStr);
				break;

			case DO_NOTHING:
				result.addReject(tableName, pkStr, rejectStatus.name());

			default:
				break;
			}
			break;
		}
		case DUPLICATE_KEY:
			if (debug) {
				log.debug(String.format("[%s] Duplicate key [pk=%s]: row not exported", tableName, pkStr));
			}
			result.addReject(tableName, pkStr, rejectStatus.name());
			break;

		case LOCKED:
			if (debug) {
				log.debug(String.format("[%s] Unable to get server lock [pk=%s]: row not exported", tableName, pkStr));
			}
			result.addReject(tableName, pkStr, rejectStatus.name());
			break;

		default:
			break;
		}
	}

	/**
	 * Say if deletion must check that PK is not used anymore.
	 * => only if NO integrity constraints AND target DB is not a mirror
	 * constraint
	 */
	protected boolean isCheckPkNotUsedBeforeDelete(SynchroContext context) {
		return this.disableIntegrityConstraints
				&& !context.getTarget().isMirrorDatabase();
	}

	protected boolean hasProcessedTable(Set<String> tableNames, SynchroContext context) {
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(tableNames));
		Preconditions.checkNotNull(context);

		Set<String> expectedTableNames = context.getTableNames();
		for (String tableName : tableNames) {
			if (expectedTableNames.contains(tableName)) {
				return true;
			}
		}
		return false;
	}

	protected void updateResultOnSynchronizeError(Exception e, SynchroResult result) {
		// Clear result, but keep reject rows
		if (result.getRejectedRows().isEmpty()) {
			result.clear();
		}
		else {
			Map<String, String> copiedRejectedRows = ImmutableMap.copyOf(result.getRejectedRows());
			result.clear();
			result.getRejectedRows().putAll(copiedRejectedRows);
		}

		// Save the error
		result.setError(e);
	}
}
