package fr.ifremer.adagio.synchro.dao;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.io.IOException;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;

import fr.ifremer.adagio.synchro.SynchroTechnicalException;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata;
import fr.ifremer.adagio.synchro.service.SynchroDatabaseConfiguration;
import fr.ifremer.adagio.synchro.service.SynchroTableOperation;

public class DaoFactoryImpl implements DaoFactory {

	/** Logger. */
	private static final Log log =
			LogFactory.getLog(DaoFactoryImpl.class);

	private SynchroDatabaseMetadata dbMeta;
	private SynchroDatabaseConfiguration dbConfig;
	private Connection connection;
	private final LoadingCache<SynchroTableMetadata, SynchroTableDao> daoCache;
	private final SynchroBaseDao sharedBaseDao;
	private final DaoStats stats;

	public DaoFactoryImpl(SynchroDatabaseMetadata dbMeta) throws SQLException {
		this(dbMeta.getConnection(), dbMeta.getConfiguration(), dbMeta, DEFAULT_DAO_CACHE_SIZE, DEFAULT_STATEMENT_CACHE_SIZE);
	}

	public DaoFactoryImpl(SynchroDatabaseMetadata dbMeta, int daoCacheSize, int statementCacheSize) throws SQLException {
		this(dbMeta.getConnection(), dbMeta.getConfiguration(), dbMeta, daoCacheSize, statementCacheSize);
	}

	public DaoFactoryImpl(Connection connection,
			SynchroDatabaseConfiguration dbConfig,
			SynchroDatabaseMetadata dbMeta,
			int daoCacheSize,
			int statementCacheSize) {
		Preconditions.checkNotNull(dbMeta);
		Preconditions.checkNotNull(dbConfig);
		Preconditions.checkNotNull(connection);
		this.dbMeta = dbMeta;
		this.dbConfig = dbConfig;
		this.connection = connection;
		this.stats = new DaoStats();

		int validDaoCacheSize = daoCacheSize > 0 ? daoCacheSize : DEFAULT_DAO_CACHE_SIZE;
		int validStatementCacheSize = statementCacheSize > 0 ? statementCacheSize : DEFAULT_STATEMENT_CACHE_SIZE;

		if (log.isDebugEnabled()) {
			log.debug(String.format("Intializing a DaoFactory with [dao cache size: %s] and [statement cache size: %s]",
					validDaoCacheSize,
					validStatementCacheSize));
		}

		this.daoCache = initDaoCache(
				validDaoCacheSize,
				DEFAULT_DAO_CACHE_DURATION_SECONDS);

		this.sharedBaseDao = new SynchroBaseDaoImpl(connection, dbConfig, dbMeta, this,
				stats,
				validStatementCacheSize,
				DEFAULT_STATEMENT_CACHE_DURATION_SECONDS,
				DEFAULT_VALUE_CACHE_SIZE,
				DEFAULT_VALUE_CACHE_DURATION_SECONDS);
	}

	public SynchroTableDao getSourceDao(SynchroTableMetadata table) {
		Preconditions.checkNotNull(table);
		try {
			SynchroTableDao dao = daoCache.get(table);
			dao.setSourceDao(null);
			dao.setCurrentOperation(null);
			dao.prepare();
			return dao;
		} catch (ExecutionException e) {
			throw new SynchroTechnicalException(String.format("[%s] Error during dao loading (using cache): %s", table.getName(), e.getMessage()), e);
		}
	}

	public SynchroTableDao getTargetDao(SynchroTableMetadata table, SynchroTableDao sourceDao, SynchroTableOperation currentOperation) {
		SynchroTableDao targetDao = getSourceDao(table);
		targetDao.setSourceDao(sourceDao);
		targetDao.setCurrentOperation(currentOperation);
		return targetDao;
	}

	public SynchroTableDao getSourceDao(String tableName) {
		SynchroTableMetadata table = dbMeta.getTable(tableName);
		return getSourceDao(table);
	}

	public SynchroTableDao getTargetDao(String tableName, SynchroTableDao sourceDao, SynchroTableOperation currentOperation) {
		SynchroTableDao targetDao = getSourceDao(tableName);
		targetDao.setSourceDao(sourceDao);
		targetDao.setCurrentOperation(currentOperation);
		return targetDao;
	}

	@Override
	public SynchroBaseDao getDao() {
		return sharedBaseDao;
	}

	/* -- internal methods -- */

	protected LoadingCache<SynchroTableMetadata, SynchroTableDao> initDaoCache(final int maximumSize, int durationInSeconds) {
		return CacheBuilder.newBuilder()
				.maximumSize(maximumSize)
				.expireAfterAccess(durationInSeconds, TimeUnit.SECONDS)
				.removalListener(new RemovalListener<SynchroTableMetadata, SynchroTableDao>() {
					@Override
					public void onRemoval(RemovalNotification<SynchroTableMetadata, SynchroTableDao> notification) {
						// When removal, close the dao
						IOUtils.closeQuietly(notification.getValue());
					}
				})
				.build(
						new CacheLoader<SynchroTableMetadata, SynchroTableDao>() {
							public SynchroTableDao load(SynchroTableMetadata table) throws SQLException {
								return newSynchroTableDao(table);
							}
						});
	}

	protected SynchroTableDao newSynchroTableDao(SynchroTableMetadata table) throws SQLException {
		stats.incrementDao();
		return new SynchroTableDaoImpl(connection, dbConfig, table, sharedBaseDao);
	}

	@Override
	public void close() throws IOException {

		// Close base dao
		IOUtils.closeQuietly(sharedBaseDao);

		// Force the cache clean, to make sure onRemoval() will close all daos
		daoCache.invalidateAll();

		// Show statistics
		if (log.isDebugEnabled()) {
			log.debug(stats.toString());
		}

		dbMeta = null;
		dbConfig = null;
		connection = null;
	}
}
