package fr.ifremer.adagio.synchro.service.referential;

/*
 * #%L
 * Tutti :: Persistence
 * $Id: ReferentialSynchroServiceImpl.java 1573 2014-02-04 16:41:40Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/trunk/tutti-persistence/src/main/java/fr/ifremer/adagio/core/service/technical/synchro/ReferentialSynchroServiceImpl.java $
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import static org.nuiton.i18n.I18n.t;

import java.io.File;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.sql.DataSource;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.cfg.Environment;
import org.nuiton.util.TimeLog;
import org.springframework.jdbc.datasource.DataSourceUtils;

import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.Maps;

import fr.ifremer.adagio.synchro.config.SynchroConfiguration;
import fr.ifremer.adagio.synchro.dao.DaoUtils;
import fr.ifremer.adagio.synchro.dao.SynchroTableDao;
import fr.ifremer.adagio.synchro.dao.SynchroTableDaoImpl;
import fr.ifremer.adagio.synchro.intercept.SynchroInterceptor;
import fr.ifremer.adagio.synchro.meta.SynchroColumnMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroMetadataUtils;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata;
import fr.ifremer.adagio.synchro.service.SynchroBaseService;
import fr.ifremer.adagio.synchro.service.SynchroContext;
import fr.ifremer.adagio.synchro.service.SynchroSchemaValidationException;
import fr.ifremer.adagio.synchro.service.SynchroResult;
import fr.ifremer.adagio.synchro.service.SynchroServiceUtils;
import fr.ifremer.adagio.synchro.type.ProgressionModel;

/**
 * Created on 1/14/14.
 * 
 * @author Tony Chemit <chemit@codelutin.com>
 * @since 3.0
 */
public class ReferentialSynchroServiceImpl extends SynchroBaseService implements ReferentialSynchroService {

	/** Logger. */
	private static final Log log =
			LogFactory.getLog(ReferentialSynchroServiceImpl.class);

	private static final TimeLog TIME =
			new TimeLog(ReferentialSynchroServiceImpl.class);

	public ReferentialSynchroServiceImpl(DataSource dataSource, SynchroConfiguration config) {
		super(dataSource, config);
	}

	public ReferentialSynchroServiceImpl() {
		super();
	}

	@Override
	public SynchroContext createSynchroContext(File sourceDbDirectory) {

		String dbName = config.getDbName();
		Properties targetConnectionProperties = config.getConnectionProperties();

		Properties sourceConnectionProperties = new Properties(targetConnectionProperties);
		sourceConnectionProperties.setProperty(Environment.URL,
				DaoUtils.getJdbcUrl(sourceDbDirectory, dbName));

		Set<String> tableToIncludes = config.getImportReferentialTablesIncludes();

		SynchroContext context = SynchroContext.newContext(
				tableToIncludes,
				sourceConnectionProperties,
				targetConnectionProperties,
				new SynchroResult());
		return context;
	}

	@Override
	public SynchroContext createSynchroContext(Properties sourceConnectionProperties) {

		Properties targetConnectionProperties = config.getConnectionProperties();

		Set<String> tableToIncludes = config.getImportReferentialTablesIncludes();

		SynchroContext context = SynchroContext.newContext(
				tableToIncludes,
				sourceConnectionProperties,
				targetConnectionProperties,
				new SynchroResult());
		return context;
	}

	@Override
	public void prepare(SynchroContext synchroContext) {
		Preconditions.checkNotNull(synchroContext);

		Properties sourceConnectionProperties = synchroContext.getSourceConnectionProperties();
		Preconditions.checkNotNull(sourceConnectionProperties);

		Properties targetConnectionProperties = synchroContext.getTargetConnectionProperties();
		Preconditions.checkNotNull(targetConnectionProperties);

		Set<String> tableNames = synchroContext.getTableNames();
		Predicate<String> tableFilter = synchroContext.getTableFilter();
		if (CollectionUtils.isEmpty(tableNames) && tableFilter == null) {
			log.info(t("adagio.persistence.synchronizeReferential.prepare.noTableFilter"));
		}

		SynchroResult result = synchroContext.getResult();
		Preconditions.checkNotNull(result);

		result.setLocalUrl(DaoUtils.getUrl(targetConnectionProperties));
		result.setRemoteUrl(DaoUtils.getUrl(sourceConnectionProperties));

		Connection targetConnection = null;
		Connection sourceConnection = null;
		try {

			ProgressionModel progressionModel = result.getProgressionModel();
			progressionModel.setMessage(t("adagio.persistence.synchronizeReferential.prepare.step1"));

			// create target connection
			targetConnection = createConnection(targetConnectionProperties);

			progressionModel.setMessage(t("adagio.persistence.synchronizeReferential.prepare.step2"));

			// create source Connection
			sourceConnection = createConnection(sourceConnectionProperties);

			// load metas
			SynchroDatabaseMetadata targetMeta =
					SynchroDatabaseMetadata.loadDatabaseMetadata(
							targetConnection,
							DaoUtils.getDialect(targetConnectionProperties),
							DaoUtils.getConfiguration(targetConnectionProperties),
							synchroContext,
							tableNames,
							tableFilter,
							null /* no column filter */,
							false /* do not load joins */);

			SynchroDatabaseMetadata sourceMeta =
					SynchroDatabaseMetadata.loadDatabaseMetadata(
							sourceConnection,
							DaoUtils.getDialect(sourceConnectionProperties),
							DaoUtils.getConfiguration(sourceConnectionProperties),
							synchroContext,
							tableNames,
							tableFilter,
							null /* no column filter */,
							false /* do not load joins */);

			progressionModel.setMessage(t("adagio.persistence.synchronizeReferential.prepare.step3"));

			// check schema
			try {
				SynchroServiceUtils.checkSchemas(sourceMeta, targetMeta, true, true, result);
			} catch (SynchroSchemaValidationException e) {
				log.error(e.getMessage());
				result.setError(e);
				return;
			}

			if (result.isSuccess()) {

				// prepare model (compute update date, count rows to update,...)

				for (String tableName : targetMeta.getLoadedTableNames()) {

					long t0 = TimeLog.getTime();

					progressionModel.setMessage(t("adagio.persistence.synchronizeReferential.prepare.step4", tableName));

					SynchroTableMetadata sourceTable = sourceMeta.getTable(tableName);

					SynchroTableMetadata targetTable = targetMeta.getTable(tableName);

					prepareTable(
							sourceTable,
							targetTable,
							targetConnection,
							sourceConnection,
							synchroContext,
							result);

					TIME.log(t0, "prepare table " + tableName);
				}

				long totalRows = result.getTotalRows();
				if (log.isInfoEnabled()) {
					log.info("Total rows to update: " + totalRows);
				}
				targetConnection.rollback();
			}
		} catch (SQLException e) {
			try {
				if (targetConnection != null) {
					targetConnection.rollback();
				}
			} catch (SQLException e1) {

				// ignore the rollback error
			}
			result.setError(e);
		} finally {
			closeSilently(sourceConnection);
			closeSilently(targetConnection);
		}
	}

	@Override
	public void synchronize(SynchroContext synchroContext) {
		Preconditions.checkNotNull(synchroContext);

		Properties sourceConnectionProperties = synchroContext.getSourceConnectionProperties();
		Preconditions.checkNotNull(sourceConnectionProperties);

		Properties targetConnectionProperties = synchroContext.getTargetConnectionProperties();
		Preconditions.checkNotNull(targetConnectionProperties);

		Set<String> tableNames = synchroContext.getTableNames();
		Predicate<String> tableFilter = synchroContext.getTableFilter();

		SynchroResult result = synchroContext.getResult();
		Preconditions.checkNotNull(result);

		Connection targetConnection = null;
		Connection sourceConnection = null;
		try {

			// create target connection
			targetConnection = createConnection(targetConnectionProperties);

			// create source Connection
			sourceConnection = createConnection(sourceConnectionProperties);

			// Create column filter (exclude missing optional column)
			Predicate<SynchroColumnMetadata> columnFilter = null;
			if (!result.getMissingOptionalColumnNameMaps().isEmpty()) {
				columnFilter = SynchroMetadataUtils.newExcludeColumnPredicate(result.getMissingOptionalColumnNameMaps());
			}

			// load metas
			SynchroDatabaseMetadata dbMetas =
					SynchroDatabaseMetadata.loadDatabaseMetadata(
							targetConnection,
							DaoUtils.getDialect(targetConnectionProperties),
							DaoUtils.getConfiguration(targetConnectionProperties),
							synchroContext,
							tableNames,
							tableFilter,
							columnFilter,
							false /* do not load join metadata */);

			// set total in progression model
			ProgressionModel progressionModel = result.getProgressionModel();
			progressionModel.setTotal(result.getTotalRows());

			// prepare target (desactivate constraints)
			prepareSynch(targetConnection);

			try {

				for (String tableName : dbMetas.getLoadedTableNames()) {

					long t0 = TimeLog.getTime();

					progressionModel.setMessage(t("adagio.persistence.synchronizeReferential.synchronize.step1", tableName));

					SynchroTableMetadata table = dbMetas.getTable(tableName);

					if (log.isInfoEnabled()) {
						log.info("Synchronize table: " + tableName);
					}
					long countToUpdate = result.getNbRows(tableName);

					if (countToUpdate > 0) {

						synchronizeTable(
								table,
								targetConnection,
								sourceConnection,
								synchroContext,
								result);
					}

					TIME.log(t0, "synchronize table " + tableName);
				}
				if (log.isInfoEnabled()) {
					long totalInserts = result.getTotalInserts();
					long totalUpdates = result.getTotalUpdates();
					log.info("Total rows to treat: " + result.getTotalRows());
					log.info("Total rows inserted: " + totalInserts);
					log.info("Total rows  updated: " + totalUpdates);
					log.info("Total rows  treated: " + (totalInserts + totalUpdates));
				}
			} finally {
				releaseSynch(targetConnection);
			}

			progressionModel.setMessage(t("adagio.persistence.synchronizeReferential.synchronize.step2"));

			targetConnection.commit();

		} catch (SQLException e) {
			try {
				if (targetConnection != null) {
					targetConnection.rollback();
				}
			} catch (SQLException e1) {

				// ignore the rolback error
			}
			result.setError(e);
		} finally {
			closeSilently(sourceConnection);
			closeSilently(targetConnection);
		}
	}

	protected void prepareTable(
			SynchroTableMetadata sourceTable,
			SynchroTableMetadata targetTable,
			Connection targetConnection,
			Connection sourceConnection,
			SynchroContext context,
			SynchroResult result) throws SQLException {

		String tableName = sourceTable.getName();
		String tablePrefix = sourceTable.getTableLogPrefix();

		if (log.isDebugEnabled()) {
			log.debug("Prepare table: " + tableName);
		}

		SynchroTableDao targetDao = new SynchroTableDaoImpl(getTargetDialect(context), targetConnection, targetTable, false);
		SynchroTableDao sourceDao = new SynchroTableDaoImpl(getSourceDialect(context), sourceConnection, sourceTable, false);

		try {
			long targetCount = targetDao.count();

			// get last updateDate used by target db
			Timestamp updateDate = null;

			if (targetCount < 50000) {

				// only use the update date on small table, for big table we will re-insert all the table content
				updateDate = targetDao.getLastUpdateDate();

				if (updateDate != null) {

					// just inscrements of 1 milisecond to not having same
					// TODO BL : attention, a cause des transactions parfois longues sur le serveur Oracle, il faut
					// peut-etre justement prendre une date inférieure au max(update_date) ?? genre max(update_date) -
					// 2h ?
					// Ou mieux : stocker puis utiliser une date de dernière mise à jour (systimestamp côté serveur)
					updateDate = new Timestamp(DateUtils.setMilliseconds(updateDate, 0).getTime());
					updateDate = new Timestamp(DateUtils.addSeconds(updateDate, 1).getTime());
				}
			}

			long countToUpdate = sourceDao.countDataToUpdate(updateDate);

			if (log.isInfoEnabled()) {
				log.info(String.format("%s nb rows to update: %s", tablePrefix, countToUpdate));
			}

			result.setUpdateDate(tableName, updateDate);
			result.addRows(tableName, (int) countToUpdate);
		} finally {
			IOUtils.closeQuietly(targetDao);
			IOUtils.closeQuietly(sourceDao);
		}

	}

	protected void synchronizeTable(
			SynchroTableMetadata table,
			Connection targetConnection,
			Connection sourceConnection,
			SynchroContext context,
			SynchroResult result) throws SQLException {

		String tableName = table.getName();

		result.getProgressionModel().setMessage(t("adagio.persistence.synchronizeReferential.synchronizeTable", tableName));

		SynchroTableDao sourceDao = new SynchroTableDaoImpl(getSourceDialect(context), sourceConnection, table, false);
		SynchroTableDao targetDao = new SynchroTableDaoImpl(getTargetDialect(context), targetConnection, table, true);

		// get last updateDate used by target db
		Date updateDate = result.getUpdateDate(tableName);

		// get table count
		long count = targetDao.count();

		boolean bigTable = count > 50000;

		// get data to update from source db
		ResultSet dataToUpdate = sourceDao.getDataToUpdate(
				bigTable ? null : updateDate);

		try {

			if (bigTable) {

				// big table update strategy
				updateBigTable(
						targetDao,
						sourceDao,
						dataToUpdate,
						result);
			} else {

				// small table update strategy
				updateTable(targetDao,
						dataToUpdate,
						result);
			}
			dataToUpdate.close();
		} finally {

			IOUtils.closeQuietly(targetDao);
			IOUtils.closeQuietly(sourceDao);
			DaoUtils.closeSilently(dataToUpdate);
		}
	}

	/**
	 * To update the content of the given {@code table} on the target db,
	 * from the given {@code incomingData} of the source db.
	 * <p/>
	 * The algorithm is pretty simple, for each row of the {@code incomingData}, if exists on target table, then do an
	 * update, otherwise do a insert.
	 * <p/>
	 * As an update query is more expensive, we won't use this method for table with a lot of rows, we will prefer to
	 * use the {@code updateBigTable} method instead.
	 * 
	 * @param targetDao
	 *            connection on the target db
	 * @param incomingData
	 *            data to update from the source db
	 * @param result
	 *            where to store operation results
	 * @throws SQLException
	 *             if any sql errors
	 */
	protected void updateTable(SynchroTableDao targetDao,
			ResultSet incomingData,
			SynchroResult result) throws SQLException {

		SynchroTableMetadata table = targetDao.getTable();

		String tableName = table.getName();
		String tablePrefix = table.getTableLogPrefix() + " - " + result.getNbRows(tableName);

		// get existing ids in the target db
		Set<String> existingIds = targetDao.getExistingPrimaryKeys();

		if (log.isDebugEnabled()) {
			log.debug(tablePrefix + " existing rows: " + existingIds.size());
		}

		result.addTableName(tableName);

		int countR = 0;

		while (incomingData.next()) {

			List<Object> pk = table.getPk(incomingData);
			String pkStr = table.toPkStr(pk);

			boolean doUpdate = existingIds.contains(pkStr);

			if (doUpdate) {

				targetDao.executeUpdate(pk, incomingData);

			} else {

				targetDao.executeInsert(incomingData);
			}

			countR++;

			reportProgress(result, targetDao, countR, tablePrefix);
		}

		targetDao.flush();

		int insertCount = targetDao.getInsertCount();
		int updateCount = targetDao.getUpdateCount();

		result.addInserts(tableName, insertCount);
		result.addUpdates(tableName, updateCount);
		if (log.isInfoEnabled()) {
			log.info(String.format("%s done: %s (inserts: %s, updates: %s)", tablePrefix, countR, insertCount, updateCount));
		}

		if (log.isDebugEnabled()) {
			log.debug(String.format("%s INSERT count: %s", tablePrefix, result.getNbInserts(tableName)));
			log.debug(String.format("%s UPDATE count: %s", tablePrefix, result.getNbUpdates(tableName)));
		}

		result.getProgressionModel().increments(countR % 1000);
	}

	/**
	 * To update the content of the given {@code table} (with a lot of rows) on
	 * the target db, from the given {@code incomingData} of the source db.
	 * <p/>
	 * We can't use the simple algorithm, since update queries cost too much and is not acceptable when talking on huge
	 * numbers of rows.
	 * <p/>
	 * Here is what to do :
	 * <ul>
	 * <li>Get form the target db the data which are not in source db, keep them</li>
	 * <li>Delete target table content</li>
	 * <li>Insert source table in target table</li>
	 * <li>Insert the saved extra rows from original table</li>
	 * </ul>
	 * In that way we will only perform some insert queries.
	 * 
	 * @param dbMetas
	 * @param targetDao
	 *            connection on the target db
	 * @param sourceDao
	 *            connection on the target db
	 * @param incomingData
	 *            data to update from the source db
	 * @param interceptor
	 * @param result
	 *            where to store operation results @throws SQLException if any sql errors
	 */
	protected void updateBigTable(
			SynchroTableDao targetDao,
			SynchroTableDao sourceDao,
			ResultSet incomingData,
			SynchroResult result) throws SQLException {

		SynchroTableMetadata table = targetDao.getTable();
		String tableName = targetDao.getTable().getName();

		result.addTableName(tableName);

		String tablePrefix = table.getTableLogPrefix() + " - " + result.getNbRows(tableName);

		// get existing ids in the target db
		Set<String> existingIds = targetDao.getExistingPrimaryKeys();

		if (log.isDebugEnabled()) {
			log.debug(tablePrefix + " target existing rows: " + existingIds.size());
		}

		Set<String> sourceExistingIds = sourceDao.getExistingPrimaryKeys();

		if (log.isDebugEnabled()) {
			log.debug(tablePrefix + " source existing rows: " + sourceExistingIds.size());
		}

		existingIds.removeAll(sourceExistingIds);

		if (log.isDebugEnabled()) {
			log.debug(tablePrefix + " target existing rows not in source: " + existingIds.size());
		}
		if (log.isTraceEnabled()) {
			for (String existingId : existingIds) {
				log.trace("- " + existingId);
			}
		}

		// copy extra rows from target

		Map<List<Object>, Object[]> extraRows = Maps.newLinkedHashMap();

		for (String pkStr : existingIds) {

			List<Object> pk = table.fromPkStr(pkStr);

			Object[] extraRow = targetDao.findByPk(pk);

			extraRows.put(pk, extraRow);
		}

		// remove obsolete extra rows

		List<SynchroInterceptor> interceptors = table.getInterceptors();
		for (SynchroInterceptor interceptor : interceptors) {
			extraRows = interceptor.transformExtraLocalData(
					targetDao,
					sourceDao,
					extraRows);
			if (log.isDebugEnabled()) {
				log.debug(tablePrefix + " target data existingIds not in source (after apply task): " + extraRows.size());
			}
		}

		// delete table
		targetDao.deleteAll();

		int countR = 0;

		// add all data from source
		while (incomingData.next()) {

			targetDao.executeInsert(incomingData);

			countR++;

			reportProgress(result, targetDao, countR, tablePrefix);
		}

		// re-add extra target rows
		for (Map.Entry<List<Object>, Object[]> entry : extraRows.entrySet()) {

			Object[] row = entry.getValue();
			targetDao.executeInsert(row);

			countR++;

			reportProgress(result, targetDao, countR, tablePrefix);
		}

		targetDao.flush();

		int insertCount = targetDao.getInsertCount();
		result.addInserts(tableName, insertCount);
		if (log.isInfoEnabled()) {
			log.info(String.format("%s done: %s (inserts: %s)", tablePrefix, countR, insertCount));
		}

		if (log.isDebugEnabled()) {
			log.debug(String.format("%s INSERT count: %s", tablePrefix, result.getNbInserts(tableName)));
		}

		result.getProgressionModel().increments(countR % 1000);
	}

	protected void reportProgress(SynchroResult result, SynchroTableDao dao, int countR, String tablePrefix) {
		if (countR % 1000 == 0) {
			result.getProgressionModel().increments(1000);
		}

		if (countR % 10000 == 0) {
			if (log.isInfoEnabled()) {
				log.info(String.format("%s Done: %s (inserts: %s, updates: %s)", tablePrefix, countR, dao.getInsertCount(), dao.getUpdateCount()));
			}
		}
	}

	protected Connection createConnection(Properties connectionProperties) throws SQLException {
		return createConnection(
				connectionProperties.getProperty(Environment.URL),
				connectionProperties.getProperty(Environment.USER),
				connectionProperties.getProperty(Environment.PASS));
	}

	protected Connection createConnection(String jdbcUrl,
			String user,
			String password) throws SQLException {
		Preconditions.checkArgument(StringUtils.isNotBlank(jdbcUrl));

		// If same URL as datasource, use the dataSource
		if (jdbcUrl.equals(config.getJdbcURL()) && this.dataSource != null) {
			return DataSourceUtils.getConnection(this.dataSource);
		}

		Connection connection = DriverManager.getConnection(jdbcUrl,
				user,
				password);
		connection.setAutoCommit(false);
		return connection;
	}

	protected void closeSilently(Connection connection) {
		String jdbcUrl = null;
		if (connection == null) {
			return;
		}
		try {
			jdbcUrl = connection.getMetaData().getURL();
		} catch (SQLException e) {
			// TODO
		}
		// If same URL as datasource, use the dataSource
		if (jdbcUrl != null
				&& jdbcUrl.equals(config.getJdbcURL())
				&& this.dataSource != null) {
			DataSourceUtils.releaseConnection(connection, this.dataSource);
		}
		else {
			DaoUtils.closeSilently(connection);
		}
	}

	protected void prepareSynch(Connection connection) throws SQLException {
		PreparedStatement statement = connection.prepareStatement("SET REFERENTIAL_INTEGRITY FALSE;");
		statement.executeUpdate();
		statement.close();
	}

	protected void releaseSynch(Connection connection) throws SQLException {
		PreparedStatement statement = connection.prepareStatement("SET REFERENTIAL_INTEGRITY TRUE;");
		statement.executeUpdate();
		statement.close();
	}
}
