package fr.ifremer.adagio.synchro.service;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.collect.Sets;

import fr.ifremer.adagio.synchro.SynchroTechnicalException;
import fr.ifremer.adagio.synchro.meta.SynchroColumnMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroMetadataUtils;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata;

/**
 * Helper class to :
 * <ul>
 * <li>Check two database schemas.
 * </ul>
 * 
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since 3.5.3
 * 
 */
public class SynchroServiceUtils {
	/** Logger. */
	private static final Log log =
			LogFactory.getLog(SynchroServiceUtils.class);

	/**
	 * Check that the tow given shemas are compatible for a
	 * synchronize operation (same tables with same columns).
	 * <p/>
	 * If <code>allowMissingOptionalColumn=true</code> then missing columns are allowed. Missing columns will be added
	 * to the given result.
	 * <p/>
	 * If <code>allowAdditionalMandatoryColumnInSourceSchema=true</code> then additional mandatory columns in the source
	 * schema are allowed. It could be set to
	 * <code>false<code> for data synchronization, to avoid getting data from tables that could not be export later.  
	 * <p/>
	 * If schemas are incompatible, then a {@link SynchroTechnicalException} exception will be thrown.
	 * 
	 * @param targetSchema
	 *            schema 1 to check
	 * @param sourceSchema
	 *            schema 2 to check
	 * @param allowMissingOptionalColumn
	 *            Is missing optional columns are allowed (in source or target schema) ? If true, missing column will be
	 *            ignore in synchronization.
	 * @param allowAdditionalMandatoryColumnInSourceSchema
	 *            Is additional mandatory columns are allowed in source schema ? If true, source schema could have more
	 *            mandatory columns.
	 * @param result
	 *            Synchro result. Use to store missing column is any
	 * @throws SynchroSchemaValidationException
	 *             if schema are not compatible
	 */
	public static void checkSchemas(
			SynchroDatabaseMetadata sourceSchema,
			SynchroDatabaseMetadata targetSchema,
			boolean allowMissingOptionalColumn,
			boolean allowAdditionalMandatoryColumnInSourceSchema,
			SynchroResult result) throws SynchroSchemaValidationException {
		try {
			if (allowMissingOptionalColumn) {
				checkSchemasAllowMissingOptionalColumn(
						sourceSchema,
						targetSchema,
						allowAdditionalMandatoryColumnInSourceSchema,
						result);
			}
			else {
				checkSchemasStrict(
						sourceSchema,
						targetSchema);
			}
		} catch (SynchroTechnicalException e) {
			result.setError(e);
		}
	}

	/**
	 * Check that the tow given shemas are compatible for a synchronize operation (same tables with same columns). *
	 * <p/>
	 * This method allow missing columns (if define as nullable in the target schema)
	 * <p/>
	 * If schemas are incompatible, then a {@link SynchroTechnicalException} exception will be thrown.
	 * 
	 * @param synchroContext
	 *            Synchro context
	 * @param targetSchema
	 *            schema 1 to check
	 * @param sourceSchema
	 *            schema 2 to check
	 * @throws SynchroSchemaValidationException
	 *             if schema are not compatible
	 */
	protected static void checkSchemasAllowMissingOptionalColumn(
			SynchroDatabaseMetadata sourceSchema,
			SynchroDatabaseMetadata targetSchema,
			boolean allowAdditionalMandatoryColumnInSourceSchema,
			SynchroResult result) throws SynchroSchemaValidationException {
		Set<String> targetSchemaTableNames = targetSchema.getLoadedTableNames();
		Set<String> sourceSchemaTableNames = sourceSchema.getLoadedTableNames();

		// Check if table names are equals
		if (!targetSchemaTableNames.equals(sourceSchemaTableNames)) {
			Set<String> missingTargetTables = Sets.newHashSet();
			for (String targetTableName : targetSchemaTableNames) {
				if (!sourceSchemaTableNames.contains(targetTableName)) {
					missingTargetTables.add(targetTableName);
				}
			}
			Set<String> missingSourceTables = Sets.newHashSet();
			for (String sourceTableName : sourceSchemaTableNames) {
				if (!targetSchemaTableNames.contains(sourceTableName)) {
					missingSourceTables.add(sourceTableName);
				}
			}

			throw new SynchroSchemaValidationException(String.format(
					"Incompatible schemas.\nMissing tables in source database: %s\nMissing tables in target database: %s", missingTargetTables,
					missingSourceTables));
		}

		for (String tableName : targetSchemaTableNames) {
			SynchroTableMetadata targetTable = targetSchema.getTable(tableName);
			SynchroTableMetadata sourceTable = sourceSchema.getTable(tableName);
			Set<String> targetColumnNames = Sets.newHashSet(targetTable.getColumnNames());
			Set<String> sourceColumnNames = sourceTable.getColumnNames();

			// Check if columns names are equals
			if (!targetColumnNames.equals(sourceColumnNames)) {
				Set<String> missingMandatoryColumns = Sets.newTreeSet();
				Set<String> missingSourceColumnNames = Sets.newHashSet(sourceColumnNames);

				// Check if missing column (in source) are optional
				for (String targetColumnName : targetTable.getColumnNames()) {
					if (!sourceColumnNames.contains(targetColumnName)) {
						SynchroColumnMetadata targetColumn = targetTable.getColumnMetadata(targetColumnName);

						// Optional column: add it to the context (will be ignore in during synchronization)
						if (targetColumn.isNullable()) {
							log.debug(String.format("Optional column not found in source database: %s.%s. Will be ignore.",
									tableName, targetColumnName));
							result.addMissingOptionalColumnName(tableName, targetColumnName);
							targetColumnNames.remove(targetColumnName);
						}

						// Mandatory columns: add to list to check later
						else {
							log.warn(String.format("Column not found in source database: %s.%s", tableName, targetColumnName));
							missingMandatoryColumns.add(targetColumnName);
						}
					}

					missingSourceColumnNames.remove(targetColumnName);
				}

				// Check if missing column (in target) are optional
				for (String sourceColumnName : missingSourceColumnNames) {
					SynchroColumnMetadata sourceColumn = sourceTable.getColumnMetadata(sourceColumnName);
					if (allowAdditionalMandatoryColumnInSourceSchema || sourceColumn.isNullable()) {
						log.debug(String.format("Optional column not found in target database: %s.%s. Will be ignore.",
								tableName, sourceColumnName));
						result.addMissingOptionalColumnName(tableName, sourceColumnName);
					}
					else {
						log.warn(String.format("Column not found in target database: %s.%s. Will be ignore.", tableName, sourceColumnName));
						missingMandatoryColumns.add(sourceColumnName);
					}
				}

				// Throw an exception if exists any missing column
				if (CollectionUtils.isNotEmpty(missingMandatoryColumns)) {
					throw new SynchroSchemaValidationException(String.format("Incompatible schema of table: %s. Missing mandatory columns: %s",
							tableName, missingMandatoryColumns));
				}
			}

			// Check column types compatibility
			for (String columnName : targetColumnNames) {
				SynchroColumnMetadata targetColumn = targetTable.getColumnMetadata(columnName);
				SynchroColumnMetadata sourceColumn = sourceTable.getColumnMetadata(columnName);
				if (!targetColumn.isProtected()) {
					SynchroMetadataUtils.checkType(tableName, targetColumn, sourceColumn);
				}
			}
		}
	}

	/**
	 * Check that the tow given datasource shemas are compatible for a
	 * synchronize operation (same tables with same columns).
	 * <p/>
	 * If schemas are incompatible, then a {@link SynchroTechnicalException} exception will be thrown.
	 * 
	 * @param schema1
	 *            schema 1 to check
	 * @param schema2
	 *            schema 2 to check
	 * @throws SynchroSchemaValidationException
	 *             if schema are not compatible
	 */
	protected static void checkSchemasStrict(
			SynchroDatabaseMetadata sourceSchema,
			SynchroDatabaseMetadata targetSchema) throws SynchroSchemaValidationException {
		Set<String> sourceSchemaTableNames = sourceSchema.getLoadedTableNames();
		Set<String> targetSchemaTableNames = targetSchema.getLoadedTableNames();

		// Check if table names are equals
		if (!targetSchemaTableNames.equals(sourceSchemaTableNames)) {
			throw new SynchroSchemaValidationException("Incompatible schemas: missing tables");
		}

		for (String tableName : sourceSchemaTableNames) {
			SynchroTableMetadata sourceTable = sourceSchema.getTable(tableName);
			SynchroTableMetadata targetTable = targetSchema.getTable(tableName);
			Set<String> sourceColumnNames = sourceTable.getColumnNames();
			Set<String> targetColumnNames = targetTable.getColumnNames();

			// Check if columns names are equals
			if (!targetColumnNames.equals(sourceColumnNames)) {
				throw new SynchroSchemaValidationException("Incompatible schema of table: " + tableName);
			}

			// Check column types compatibility
			for (String columnName : targetColumnNames) {
				SynchroColumnMetadata sourceColumn = sourceTable.getColumnMetadata(columnName);
				SynchroColumnMetadata targetColumn = targetTable.getColumnMetadata(columnName);
				if (!targetColumn.isProtected()) {
					SynchroMetadataUtils.checkType(tableName, targetColumn, sourceColumn);
				}
			}
		}
	}

}
