package fr.ifremer.adagio.synchro.service;

/*
 * #%L
 * Tutti :: Persistence API
 * $Id: ReferentialSynchroResult.java 1486 2014-01-15 08:43:26Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/trunk/tutti-persistence/src/main/java/fr/ifremer/adagio/core/service/technical/synchro/ReferentialSynchroResult.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.sql.Timestamp;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import fr.ifremer.adagio.synchro.type.ProgressionModel;

/**
 * Result of a referential synchronize operation.
 * 
 * @author tchemit <chemit@codelutin.com>
 * @since 3.5.2
 */
public class SynchroResult {

	protected Exception error;

	/**
	 * Number of rows detected to update (per table).
	 * 
	 * @since 1.0
	 */
	protected final Map<String, Integer> rowHits = Maps.newTreeMap();

	/**
	 * Number of insert done (per table).
	 * 
	 * @since 1.0
	 */
	protected final Map<String, Integer> insertHits = Maps.newTreeMap();

	/**
	 * Number of update done (per table).
	 * 
	 * @since 1.0
	 */
	protected final Map<String, Integer> updateHits = Maps.newTreeMap();

	/**
	 * timestamp of last update date (per table).
	 * 
	 * @since 1.0
	 */
	protected final Map<String, Timestamp> updateDateHits = Maps.newTreeMap();

	/**
	 * All table treated.
	 * 
	 * @since 1.0
	 */
	protected final Set<String> tableNames = Sets.newHashSet();

	/**
	 * Columns not to synchronize (per table).
	 * 
	 * @since 3.5.3
	 */
	protected final Map<String, Set<String>> missingOptionalColumnNames = Maps.newTreeMap();

	protected String targetUrl;

	protected String sourceUrl;

	protected final ProgressionModel progressionModel = new ProgressionModel(this);

	public SynchroResult() {
	}

	public SynchroResult(String targetUrl, String sourceUrl) {
		this.targetUrl = targetUrl;
		this.sourceUrl = sourceUrl;
	}

	public void setLocalUrl(String targetUrl) {
		this.targetUrl = targetUrl;
	}

	public void setRemoteUrl(String sourceUrl) {
		this.sourceUrl = sourceUrl;
	}

	public boolean isSuccess() {
		return error == null;
	}

	public Exception getError() {
		return error;
	}

	public void setError(Exception error) {
		this.error = error;
	}

	public ProgressionModel getProgressionModel() {
		return progressionModel;
	}

	public Set<String> getTableNames() {
		return ImmutableSet.copyOf(tableNames);
	}

	public int getTotalRows() {
		int result = 0;
		for (Integer nb : rowHits.values()) {
			result += nb;
		}
		return result;
	}

	public int getTotalInserts() {
		int result = 0;
		for (Integer nb : insertHits.values()) {
			result += nb;
		}
		return result;
	}

	public int getTotalUpdates() {
		int result = 0;
		for (Integer nb : updateHits.values()) {
			result += nb;
		}
		return result;
	}

	public int getNbRows(String tableName) {
		Integer result = rowHits.get(tableName);
		if (result == null) {
			result = 0;
		}
		return result;
	}

	public int getNbInserts(String tableName) {
		Integer result = insertHits.get(tableName);
		if (result == null) {
			result = 0;
		}
		return result;
	}

	public int getNbUpdates(String tableName) {
		Integer result = updateHits.get(tableName);
		if (result == null) {
			result = 0;
		}
		return result;
	}

	public void addRows(String tableName, int nb) {
		if (nb > 0) {
			rowHits.put(tableName, getNbRows(tableName) + nb);
		}
	}

	public void addUpdates(String tableName, int nb) {
		if (nb > 0) {
			updateHits.put(tableName, getNbUpdates(tableName) + nb);
		}
	}

	public void addInserts(String tableName, int nb) {
		if (nb > 0) {
			insertHits.put(tableName, getNbInserts(tableName) + nb);
		}
	}

	public Timestamp getUpdateDate(String tableName) {
		return updateDateHits.get(tableName);
	}

	public void setUpdateDate(String tableName, Timestamp t) {
		updateDateHits.put(tableName, t);
	}

	public void addTableName(String tableName) {
		tableNames.add(tableName);
	}

	public String getLocalUrl() {
		return targetUrl;
	}

	public String getRemoteUrl() {
		return sourceUrl;
	}

	public void addMissingOptionalColumnName(String tableName, String columnName) {
		Set<String> columnNames = missingOptionalColumnNames.get(tableName);
		if (columnNames == null) {
			columnNames = Sets.newHashSet();
			missingOptionalColumnNames.put(tableName, columnNames);
		}
		columnNames.add(columnName);
	}

	public Set<String> getMissingOptionalColumnNames(String tableName) {
		return missingOptionalColumnNames.get(tableName);
	}

	public Map<String, Set<String>> getMissingOptionalColumnNameMaps() {
		return ImmutableMap.copyOf(missingOptionalColumnNames);
	}
}
