package fr.ifremer.adagio.synchro.meta;

/*
 * #%L
 * Tutti :: Persistence API
 * $Id: TuttiEntities.java 1578 2014-02-07 15:31:18Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/trunk/tutti-persistence/src/main/java/fr/ifremer/tutti/persistence/entities/TuttiEntities.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.sql.Types;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;

import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import fr.ifremer.adagio.synchro.SynchroTechnicalException;

/**
 * Useful method around DAO and entities.
 * 
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since 3.5.4
 */
public class SynchroMetadataUtils {

	public static final List<String> ORACLE_EXCLUDE_TABLE_PATTERNS = Lists.newArrayList(
			"BIN%", // Trash Oracle
			"MDR%" // Table metadata spatial
	);

	protected SynchroMetadataUtils() {
		// helper class does not instantiate
	}

	public static Predicate<String> newAllTablesOraclePredicate() {
		return newTablesOraclePredicate(null, null);
	}

	public static Predicate<String> newTablesOraclePredicate(Set<String> excludes, final Set<String> includes) {
		Set<String> excludesPatterns = Sets.newHashSet(ORACLE_EXCLUDE_TABLE_PATTERNS);

		if (CollectionUtils.isNotEmpty(excludes)) {
			excludesPatterns.addAll(excludes);
		}

		return newTablePredicate(excludesPatterns, includes);
	}

	public static Predicate<String> newTablePredicate(final Set<String> excludes, final Set<String> includes) {
		// If no filter
		if (CollectionUtils.isEmpty(excludes) && CollectionUtils.isEmpty(includes)) {
			return null;
		}

		return new Predicate<String>() {
			public boolean apply(String tableName) {

				if (CollectionUtils.isNotEmpty(excludes)) {
					for (String excludePattern : excludes) {
						if (tableName.matches(excludePattern.replaceAll("%", ".*"))) {
							return false;
						}
					}
				}
				if (CollectionUtils.isEmpty(includes)) {
					return true;
				}
				for (String includePattern : includes) {
					if (tableName.matches(includePattern.replaceAll("%", ".*"))) {
						return true;
					}
				}
				return false;
			}
		};
	}

	public static Predicate<SynchroColumnMetadata> newExcludeColumnPredicate(final Map<String, Set<String>> excludeColumnNamesMap) {
		Preconditions.checkNotNull(excludeColumnNamesMap);
		Preconditions.checkArgument(!excludeColumnNamesMap.isEmpty());

		return new Predicate<SynchroColumnMetadata>() {
			public boolean apply(SynchroColumnMetadata input) {
				Set<String> excludeColumnNames = excludeColumnNamesMap.get(input.getTableName());
				return CollectionUtils.isEmpty(excludeColumnNames) || !excludeColumnNames.contains(input.getName().toLowerCase());
			}
		};
	}

	/**
	 * Check if types are compatible, and return a DataRetrievalFailureException if not compatible.
	 * 
	 * @param tableName
	 * @param internalColumn
	 * @param externalColumn
	 */
	public static void checkType(String tableName, SynchroColumnMetadata internalColumn, SynchroColumnMetadata externalColumn) {

		// If numeric
		if (isNumericType(internalColumn) && isNumericType(externalColumn)) {
			int internalColumnSize = internalColumn.getColumnSize();
			int externalColumnSize = externalColumn.getColumnSize();
			if (internalColumnSize > 0 && externalColumnSize > 0 && internalColumnSize < externalColumnSize) {
				throw new SynchroTechnicalException(String.format("Incompatible column type of table / column: %s / %s", tableName,
						internalColumn.getName()));
			}
			int internalDecimalDigits = internalColumn.getDecimalDigits();
			int externalDecimalDigits = externalColumn.getDecimalDigits();
			if (internalDecimalDigits > 0 && externalDecimalDigits > 0 && internalDecimalDigits < externalDecimalDigits) {
				throw new SynchroTechnicalException(String.format("Incompatible column type of table / column: %s / %s", tableName,
						internalColumn.getName()));
			}
		}

		// If Date
		else if (isDateType(internalColumn) && isDateType(externalColumn)) {
			// OK
		}

		// If Boolean
		else if (isBooleanType(internalColumn) && isBooleanType(externalColumn)) {
			// OK
		}

		// Else : compare type code and name
		else {
			String internalColumnTypeName = internalColumn.getTypeName();
			String externalColumnTypeName = externalColumn.getTypeName();
			int internalColumnTypeCode = internalColumn.getTypeCode();
			int externalColumnTypeCode = externalColumn.getTypeCode();

			if (internalColumnTypeCode != externalColumnTypeCode && internalColumnTypeName.equals(externalColumnTypeName) == false) {
				throw new SynchroTechnicalException(String.format("Incompatible column type of table / column: %s / %s", tableName,
						internalColumn.getName()));
			}
		}
	}

	protected static boolean isNumericType(SynchroColumnMetadata column) {
		int typeCode = column.getTypeCode();
		if (typeCode == Types.BIGINT || typeCode == Types.INTEGER || typeCode == Types.NUMERIC
				|| typeCode == Types.DECIMAL || typeCode == Types.FLOAT || typeCode == Types.REAL
				|| typeCode == Types.SMALLINT || typeCode == Types.TINYINT || typeCode == Types.DOUBLE) {
			return true;
		}

		String columnTypeName = column.getTypeName();
		return columnTypeName.equals("NUMBER") || columnTypeName.equals("INTEGER") || columnTypeName.equals("SMALLINT");
	}

	protected static boolean isDateType(SynchroColumnMetadata column) {
		String columnTypeName = column.getTypeName();
		return columnTypeName.equals("TIMESTAMP") || columnTypeName.equals("DATE");
	}

	protected static boolean isBooleanType(SynchroColumnMetadata column) {
		String columnTypeName = column.getTypeName();
		int columnSize = column.getColumnSize();
		return columnTypeName.equals("BOOLEAN") ||
				(columnTypeName.equals("NUMBER") && columnSize == 1);
	}

}
