package fr.ifremer.adagio.synchro.meta;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import org.hibernate.tool.hbm2ddl.ColumnMetadata;

import com.google.common.base.Preconditions;

/**
 * Overrides of the {@link ColumnMetadata} with some improvements:
 * <ul>
 * <li>Obtains owner table name {@link #getTableName()};</li>
 * <li>Obtains linked column (for FK only) {@link #getParentJoin()} as a join metadata: used to synchronize data in
 * {@link DataSynchroService};</li>
 * <li>Obtains if a column is protected for data import {@link #isProtected()}: used by {@link SynchroServiceUtils} for
 * schemas check.</li>
 * </ul>
 * 
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since 3.5.2
 * 
 */
public class SynchroColumnMetadata {

	protected final ColumnMetadata delegate;

	protected final boolean isProtected;

	protected final String tableName;

	protected final int columnIndex;

	protected SynchroJoinMetadata parentJoin = null;

	public SynchroColumnMetadata(
			ColumnMetadata delegate,
			String tableName,
			int columnIndex,
			boolean isProtected) {

		Preconditions.checkNotNull(delegate);

		this.delegate = delegate;
		this.tableName = tableName;
		this.columnIndex = columnIndex;
		this.isProtected = isProtected;
	}

	public SynchroColumnMetadata(
			SynchroColumnMetadata beanToCopy) {

		Preconditions.checkNotNull(beanToCopy);

		this.delegate = beanToCopy.delegate;
		this.tableName = beanToCopy.tableName;
		this.columnIndex = beanToCopy.columnIndex;
		this.isProtected = beanToCopy.isProtected;
	}

	/**
	 * Obtains if a column is protected for data import.<br/>
	 * This is used by {@link SynchroServiceUtils} for schemas check.
	 * 
	 * @return true if the column is protected
	 */
	public boolean isProtected() {
		return isProtected;
	}

	/**
	 * Obtains owner table name
	 * 
	 * @return The owner table name
	 */
	public String getTableName() {
		return tableName;
	}

	/**
	 * Obtains linked column (for FK only)<br/>
	 * This is used to synchronize data in {@link DataSynchroService};
	 * 
	 * @return the parent join metadata
	 * @see SynchroJoinMetadata
	 */
	public SynchroJoinMetadata getParentJoin() {
		return parentJoin;
	}

	public void setParentJoin(SynchroJoinMetadata parentJoin) {
		this.parentJoin = parentJoin;
	}

	/**
	 * @see ColumnMetadata#hashCode()
	 */
	public int hashCode() {
		return delegate.hashCode();
	}

	/**
	 * @see ColumnMetadata#getName()
	 */
	public String getName() {
		return delegate.getName();
	}

	/**
	 * @see ColumnMetadata#getTypeName()
	 */
	public String getTypeName() {
		return delegate.getTypeName();
	}

	/**
	 * @see ColumnMetadata#getColumnSize()
	 */
	public int getColumnSize() {
		return delegate.getColumnSize();
	}

	/**
	 * @see ColumnMetadata#getDecimalDigits()
	 */
	public int getDecimalDigits() {
		return delegate.getDecimalDigits();
	}

	/**
	 * @see ColumnMetadata#getTypeCode()
	 */
	public String getNullable() {
		return delegate.getNullable();
	}

	public boolean isNullable() {
		return "YES".equalsIgnoreCase(getNullable());
	}

	/**
	 * @see ColumnMetadata#toString()
	 */
	public String toString() {
		return delegate.toString();
	}

	/**
	 * @see ColumnMetadata#getTypeCode()
	 */
	public int getTypeCode() {
		return delegate.getTypeCode();
	}

	/**
	 * @see ColumnMetadata#delegate(Object)
	 */
	public boolean equals(Object obj) {
		return delegate.equals(obj);
	}

}
