package fr.ifremer.adagio.synchro.intercept;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.io.IOException;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import com.google.common.base.Preconditions;

import fr.ifremer.adagio.synchro.SynchroTechnicalException;
import fr.ifremer.adagio.synchro.dao.SynchroTableDao;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroJoinMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata;
import fr.ifremer.adagio.synchro.service.SynchroContext;

public class SynchroInterceptorBase implements SynchroInterceptorChain {
	private SynchroInterceptor next;

	private SynchroContext context;

	private boolean enableOnRead = false;

	private boolean enableOnWrite = false;

	public SynchroInterceptorBase() {
		this.next = null;
		this.context = null;
	}

	public SynchroInterceptorBase(SynchroInterceptor next) {
		this.next = next;
		this.context = null;
	}

	@Override
	public SynchroInterceptorBase clone() {

		try {
			return this.getClass().newInstance();
		} catch (InstantiationException e) {
			throw new SynchroTechnicalException(String.format(
					"Could not clone the interceptor class %s. Make sure method clone() has been implemented.", this.getClass().getName()));
		} catch (IllegalAccessException e) {
			throw new SynchroTechnicalException(String.format(
					"Could not clone the interceptor class %s. Make sure method clone() has been implemented.", this.getClass().getName()));
		}
	}

	public boolean apply(SynchroDatabaseMetadata meta, TableMetadata table) {
		return false;
	}

	@Override
	public void setNext(SynchroInterceptor next) {
		this.next = next;
	}

	public SynchroInterceptor getNext() {
		return this.next;
	}

	@Override
	public final void close() throws IOException {
		doClose();

		if (next != null) {
			next.close();
		}
	}

	@Override
	public final void onRead(Object[] data, SynchroTableDao dao) throws SQLException {
		doOnRead(data, dao);

		if (next != null) {
			next.onRead(data, dao);
		}
	}

	@Override
	public final void onWrite(Object[] data, String pkStr, SynchroTableDao dao, SynchroWriteBuffer buffer) throws SQLException {
		doOnWrite(data, pkStr, dao, buffer);

		if (next != null) {
			next.onWrite(data, pkStr, dao, buffer);
		}
	}

	@Override
	public Map<List<Object>, Object[]> transformExtraLocalData(
			SynchroTableDao localDao, SynchroTableDao remoteDao,
			Map<List<Object>, Object[]> extraRows) throws SQLException {
		Map<List<Object>, Object[]> result = doTransformExtraLocalData(localDao, remoteDao, extraRows);

		if (next != null) {
			result = next.transformExtraLocalData(localDao, remoteDao, result);
		}
		return result;
	}

	@Override
	public final String onCreateSelectQuery(SynchroTableMetadata table, String queryName, String sql) {
		String newSql = doCreateSelectQuery(table, queryName, sql);

		if (next != null) {
			newSql = next.onCreateSelectQuery(
					table,
					queryName,
					newSql != null ? newSql : sql);
		}
		return newSql;
	}

	public final void onTableLoad(SynchroTableMetadata table) {
		doTableLoad(table);
		if (next != null) {
			next.onTableLoad(table);
		}
	}

	public final void onJoinLoad(SynchroTableMetadata table, SynchroJoinMetadata join) {
		doJoinLoad(table, join);
		if (next != null) {
			next.onJoinLoad(table, join);
		}
	}

	@Override
	public boolean enableOnRead() {
		return enableOnRead;
	}

	@Override
	public boolean enableOnWrite() {
		return enableOnWrite;
	}

	public void setEnableOnRead(boolean enableOnRead) {
		this.enableOnRead = enableOnRead;
	}

	public void setEnableOnWrite(boolean enableOnWrite) {
		this.enableOnWrite = enableOnWrite;
	}

	@Override
	public void setContext(SynchroContext context) {
		this.context = context;
	}

	protected SynchroContext getContext() {
		return this.context;
	}

	/* -- protected method -- */

	protected void doClose() throws IOException {
	}

	protected void doTableLoad(SynchroTableMetadata table) {
	}

	protected void doJoinLoad(SynchroTableMetadata table, SynchroJoinMetadata join) {
	}

	protected String doCreateSelectQuery(SynchroTableMetadata table, String queryName, String sql) {
		String[] clauses = sql.split("(FROM)|(WHERE)");
		Preconditions.checkArgument(
				clauses.length == 2 || clauses.length == 3,
				String.format("Bad sql query: [%s]. Expected: SELECT <...> FROM <...> WHERE <...>", sql));
		String selectClause = clauses[0];
		String fromClause = clauses[1];
		String whereClause = clauses.length == 3 ? clauses[2] : null;
		return doCreateSelectQuery(table, selectClause, fromClause, whereClause, queryName);
	}

	/**
	 * Default implementation, that return the same SQL query.
	 * This method should be override by subclasses
	 * 
	 * @param table
	 * @param selectClause
	 * @param fromClause
	 * @param whereClause
	 * @param queryName
	 * @return
	 */
	protected String doCreateSelectQuery(SynchroTableMetadata table,
			String selectClause,
			String fromClause,
			String whereClause,
			String queryName) {
		StringBuilder sb = new StringBuilder(selectClause).append(" FROM ").append(fromClause);
		if (StringUtils.isNotBlank(whereClause)) {
			sb.append(" WHERE ").append(whereClause);
		}
		return sb.toString();
	}

	protected void doOnRead(Object[] data, SynchroTableDao dao) throws SQLException {

	}

	protected void doOnWrite(Object[] data, String pkStr, SynchroTableDao dao, SynchroWriteBuffer buffer) throws SQLException {

	}

	protected Map<List<Object>, Object[]> doTransformExtraLocalData(
			SynchroTableDao localDao, SynchroTableDao remoteDao,
			Map<List<Object>, Object[]> extraRows) throws SQLException {
		return extraRows;
	}

}
