package fr.ifremer.adagio.synchro.dao;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.io.Closeable;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.hibernate.dialect.Dialect;

import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata;
import fr.ifremer.adagio.synchro.service.SynchroTableOperationBuffer;

public interface SynchroTableDao extends Closeable {

	Connection getConnection();

	SynchroTableMetadata getTable();

	void setPendingOperationBuffer(SynchroTableOperationBuffer pendingChangesBuffer);

	SynchroTableOperationBuffer getPendingOperationBuffer();

	/**
	 * Obtains the max update date found in table's rows
	 * 
	 * @return
	 * @throws SQLException
	 */
	Timestamp getLastUpdateDate() throws SQLException;

	/**
	 * Count the number of rows in the table
	 * 
	 * @return
	 * @throws SQLException
	 */
	long count() throws SQLException;

	/**
	 * Count the number of rows updated since the given date
	 * 
	 * @param fromDate
	 * @return
	 * @throws SQLException
	 */
	long countDataToUpdate(Date fromDate) throws SQLException;

	/**
	 * Retrieve a id by a remote id. Only used for data synchronization (not used in referential synchronization)
	 * 
	 * @param tableName
	 * @param remoteId
	 * @return the local id, or null if remote_id not found
	 */
	Integer getIdFromRemoteId(String tableName, Integer remoteId) throws SQLException;

	/**
	 * Obtains existing remoteIds
	 * 
	 * @return
	 * @throws SQLException
	 */
	Map<Integer, Integer> getExistingRemoteIdsMap() throws SQLException;

	/**
	 * Getting all row depending of column values
	 * 
	 * @param columnName
	 *            the FK column name
	 * @param values
	 *            values of the FK columns
	 * @return
	 * @throws SQLException
	 */
	ResultSet getDataByColumn(String columnName, List<Object> values) throws SQLException;

	ResultSet getDataByColumns(Set<String> columnNames, Set<List<Object>> columnsValues) throws SQLException;

	/**
	 * Obtains all data updated since the given date (or all data if the given date is null
	 * 
	 * @param fromDate
	 * @return A Resulset with expected data.
	 * @throws SQLException
	 */
	ResultSet getDataToUpdate(Date fromDate) throws SQLException;

	/**
	 * Obtains all primary keys of a table.
	 * If more than one key in tha table, use a serialization {@link SynchroTableMetadata#toPkStr}
	 * 
	 * @return
	 * @throws SQLException
	 */
	Set<String> getExistingPrimaryKeys() throws SQLException;

	/**
	 * Obtains a row, using values of each PK
	 * 
	 * @param pk
	 *            values of PK column
	 * @return the existing row in the table
	 * @throws SQLException
	 */
	Object[] findByPk(List<Object> pk) throws SQLException;

	/**
	 * Obtains a PK from the current ResultSet's row
	 * 
	 * @param incomingData
	 * @param tableBuffer
	 * @return
	 * @throws SQLException
	 */
	List<Object> getPk(ResultSet incomingData) throws SQLException;

	/**
	 * Delete all rows of a table
	 * 
	 * @throws SQLException
	 */
	void deleteAll() throws SQLException;

	/**
	 * Insert into a table the current row of the given {@link ResultSet}
	 * 
	 * @param pk
	 *            the PK of the current row (only need for logging)
	 * @param incomingData
	 *            a {@link ResultSet} with a row ready to be read ({@link ResultSet#next()} should have been call
	 *            before)
	 * @throws SQLException
	 */
	void executeInsert(ResultSet incomingData) throws SQLException;

	/**
	 * Insert into a table the current row of the given incoming data
	 * <p/>
	 * Same as {@link #executeInsert(List,ResultSet)}, but using a array instead of a ResultSet.
	 * 
	 * @param incomingData
	 *            a array of object
	 * @throws SQLException
	 */
	void executeInsert(Object[] row) throws SQLException;

	Integer executeInsertAndReturnId(ResultSet incomingData) throws SQLException;

	/**
	 * Update one table's row, using the current row of the given {@link ResultSet}
	 * 
	 * @param pk
	 *            the PK of the row to update
	 * @param incomingData
	 *            a {@link ResultSet} with a row ready to be read ({@link ResultSet#next()} should have been call
	 *            before)
	 * @throws SQLException
	 */
	void executeUpdate(List<Object> pk, ResultSet incomingData) throws SQLException;

	/**
	 * Update one table's row, using the current row of the given array of values
	 * <p/>
	 * Same as {@link #executeUpdate(List,ResultSet)}, but using a array instead of a ResultSet.
	 * 
	 * @param pk
	 *            the PK of the row to update
	 * @param row
	 *            a array of object
	 * @throws SQLException
	 */
	void executeUpdate(List<Object> pk, Object[] row) throws SQLException;

	void executeColumnUpdates(String columnName, Map<String, Object> valuesByPkStr) throws SQLException;

	/**
	 * Flush all pending updates (i.e. pending batch statement)
	 * 
	 * @throws SQLException
	 */
	void flush() throws SQLException;

	/**
	 * Obtains the number of updated rows
	 * 
	 * @return the number of updated rows
	 */
	int getUpdateCount();

	/**
	 * Obtains the number of inserted rows
	 * 
	 * @return the number of inserted rows
	 */
	int getInsertCount();

	/**
	 * Getting the database dialect
	 * 
	 * @return
	 */
	Dialect getDialect();
}
