package com.franciaflex.faxtomail;

/*
 * #%L
 * Extranet ENC-AHI :: Persistence
 * $Id: FaxToMailConfiguration.java 122 2014-06-02 18:45:22Z echatellier $
 * $HeadURL: http://svn.codelutin.com/faxtomail/tags/faxtomail-0.2/faxtomail-persistence/src/main/java/com/franciaflex/faxtomail/FaxToMailConfiguration.java $
 * %%
 * Copyright (C) 2013 Ministère des Affaires sociales et de la Santé
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Charsets;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.config.ApplicationConfig;
import org.nuiton.config.ApplicationConfigHelper;
import org.nuiton.config.ApplicationConfigProvider;
import org.nuiton.config.ArgumentsParserException;
import org.nuiton.jaxx.application.ApplicationConfiguration;
import org.nuiton.jaxx.application.ApplicationIOUtil;
import org.nuiton.jaxx.application.ApplicationTechnicalException;
import org.nuiton.util.Version;

import javax.swing.*;
import java.awt.*;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import static org.nuiton.i18n.I18n.t;

public class FaxToMailConfiguration extends ApplicationConfiguration {

    private static final Log log = LogFactory.getLog(FaxToMailConfiguration.class);

    private static FaxToMailConfiguration instance;

    public static FaxToMailConfiguration getInstance() {
        return instance;
    }

    public static void setInstance(FaxToMailConfiguration instance) {
        FaxToMailConfiguration.instance = instance;
    }

    protected final String[] optionKeyToNotSave;

    protected File configFile;

    public FaxToMailConfiguration(ApplicationConfig applicationConfig) {
        super(applicationConfig);
        optionKeyToNotSave = null;
    }

    public FaxToMailConfiguration(String file, String... args) {
        super(new ApplicationConfig());
        applicationConfig.setEncoding(Charsets.UTF_8.name());

        // get all config providers
        Set<ApplicationConfigProvider> providers =
                ApplicationConfigHelper.getProviders(null,
                                                     null,
                                                     null,
                                                     true);

        // load all default options
        ApplicationConfigHelper.loadAllDefaultOption(applicationConfig,
                                                     providers);

        // get all transient and final option keys
        Set<String> optionToSkip =
                ApplicationConfigHelper.getTransientOptionKeys(providers);

        if (log.isDebugEnabled()) {
            log.debug("Option that won't be saved: " + optionToSkip);
        }
        optionKeyToNotSave = optionToSkip.toArray(new String[optionToSkip.size()]);

        applicationConfig.setConfigFileName(file);

        try {
            applicationConfig.parse(args);

        } catch (ArgumentsParserException e) {
            throw new ApplicationTechnicalException(t("faxtomail.config.parse.error"), e);
        }

        //TODO Review this, this is very dirty to do this...
        File faxToMailBasedir = applicationConfig.getOptionAsFile(
                FaxToMailConfigurationOption.BASEDIR.getKey());

        if (faxToMailBasedir == null) {
            faxToMailBasedir = new File("");
        }
        if (!faxToMailBasedir.isAbsolute()) {
            faxToMailBasedir = new File(faxToMailBasedir.getAbsolutePath());
        }
        if (faxToMailBasedir.getName().equals("..")) {
            faxToMailBasedir = faxToMailBasedir.getParentFile().getParentFile();
        }
        if (faxToMailBasedir.getName().equals(".")) {
            faxToMailBasedir = faxToMailBasedir.getParentFile();
        }
        if (log.isInfoEnabled()) {
            log.info("Application basedir: " + faxToMailBasedir);
        }
        applicationConfig.setOption(
                FaxToMailConfigurationOption.BASEDIR.getKey(),
                faxToMailBasedir.getAbsolutePath());

//        if (log.isInfoEnabled()) {
//            StringBuilder builder = new StringBuilder();
//            List<FaxToMailConfigurationOption> options =
//                    Lists.newArrayList(FaxToMailConfigurationOption.values());
//            for (FaxToMailConfigurationOption option : options) {
//                String key = option.getKey();
//                Object value = applicationConfig.getOption(option);
//                builder.append(String.format("\n%1$-40s = %2$s",
//                                             key,
//                                             value));
//            }
//            log.info("faxtomail: " + builder.toString());
//        }
    }

    public void prepareDirectories() {

        File dataDirectory = getDataDirectory();
        ApplicationIOUtil.forceMkdir(
                dataDirectory,
                t("faxtomail.service.mkDir.error", dataDirectory));

        File tmpDirectory = getTmpDirectory();
        if (tmpDirectory.exists()) {

            ApplicationIOUtil.deleteDirectory(
                    tmpDirectory,
                    t("faxtomail.service.config.deleteTempDirectory.error", tmpDirectory));
        }
        ApplicationIOUtil.forceMkdir(
                tmpDirectory,
                t("faxtomail.service.mkDir.error", tmpDirectory));

    }

    public File getConfigFile() {
        if (configFile == null) {
            File dir = getBasedir();
            if (dir == null || !dir.exists() || !isFullLaunchMode()) {
                dir = new File(applicationConfig.getUserConfigDirectory());
            }
            configFile = new File(dir, applicationConfig.getConfigFileName());
        }
        return configFile;
    }

    public void save() {

        File file = getConfigFile();
        if (log.isInfoEnabled()) {
            log.info("Save configuration at: " + file);
        }
        try {
            applicationConfig.save(file, false, optionKeyToNotSave);
        } catch (IOException e) {
            throw new ApplicationTechnicalException(
                    t("faxtomail.config.save.error", file), e);
        }
    }

    @Override
    public String toString() {
        return ReflectionToStringBuilder.toString(this);
    }

    // OPTION SETTERS

    public void setCsvSeparator(char c) {
        applicationConfig.setOption(
                FaxToMailConfigurationOption.CSV_SEPARATOR.getKey(), c + "");
    }

    // OPTION GETTERS

    public boolean isDevMode() {
        boolean isDevMode = applicationConfig.getOptionAsBoolean(FaxToMailConfigurationOption.DEV_MODE.getKey());
        return isDevMode;
    }

    public Map<String, String> getTopiaProperties() {
        Map<String, String> jpaParameters = Maps.newHashMap();
        Properties hibernateProperties = applicationConfig.getOptionStartsWith("hibernate");
        jpaParameters.putAll((Map) hibernateProperties);
        return jpaParameters;
    }

    public String getSmtpHost() {
        return applicationConfig.getOption(FaxToMailConfigurationOption.SMTP_HOST.getKey());
    }

    public int getSmtpPort() {
        return applicationConfig.getOptionAsInt(FaxToMailConfigurationOption.SMTP_PORT.getKey());
    }
    
    public String getSmtpUser() {
        return applicationConfig.getOption(FaxToMailConfigurationOption.SMTP_USER.getKey());
    }
    
    public String getSmtpPassword() {
        return applicationConfig.getOption(FaxToMailConfigurationOption.SMTP_PASSWORD.getKey());
    }
    
    public boolean getSmtpUseSsl() {
        return applicationConfig.getOptionAsBoolean(FaxToMailConfigurationOption.SMTP_SSL.getKey());
    }

    public String getLdapHost() {
        return applicationConfig.getOption(FaxToMailConfigurationOption.LDAP_HOST.getKey());
    }
    
    public int getLdapPort() {
        return applicationConfig.getOptionAsInt(FaxToMailConfigurationOption.LDAP_PORT.getKey());
    }
    
    public String getLdapUser() {
        return applicationConfig.getOption(FaxToMailConfigurationOption.LDAP_USER.getKey());
    }
    
    public String getLdapPassword() {
        return applicationConfig.getOption(FaxToMailConfigurationOption.LDAP_PASSWORD.getKey());
    }

    public String getInstanceUrl() {
        return applicationConfig.getOption(FaxToMailConfigurationOption.INSTANCE_URL.getKey());
    }

    public String getInstanceDisclaimer() {
        return applicationConfig.getOption(FaxToMailConfigurationOption.INSTANCE_DISCLAIMER.getKey());
    }

    public File getStorageFile() {
        return applicationConfig.getOptionAsFile(FaxToMailConfigurationOption.STORAGE_FILE.getKey());
    }

    public boolean isLogConfigurationProvided() {
        boolean logConfigurationProvided =
                StringUtils.isNotBlank(applicationConfig.getOption(FaxToMailConfigurationOption.LOG_CONFIGURATION_FILE.getKey()));
        return logConfigurationProvided;
    }

    public File getLogConfigurationFile() {
        return applicationConfig.getOptionAsFile(FaxToMailConfigurationOption.LOG_CONFIGURATION_FILE.getKey());
    }

    @Override
    public String getApplicationName() {
        return "FaxToMail";
    }

    /** @return {@link FaxToMailConfigurationOption#VERSION} value */
    @Override
    public Version getVersion() {
        return applicationConfig.getOptionAsVersion(FaxToMailConfigurationOption.VERSION.getKey());
    }

    /** @return {@link FaxToMailConfigurationOption#SITE_URL} value */
    public URL getSiteUrl() {
        return applicationConfig.getOptionAsURL(FaxToMailConfigurationOption.SITE_URL.getKey());
    }

    /** @return {@link FaxToMailConfigurationOption#ORGANIZATION_NAME} value */
    public String getOrganizationName() {
        return applicationConfig.getOption(FaxToMailConfigurationOption.ORGANIZATION_NAME.getKey());
    }

    /** @return {@link FaxToMailConfigurationOption#INCEPTION_YEAR} value */
    public int getInceptionYear() {
        return applicationConfig.getOptionAsInt(FaxToMailConfigurationOption.INCEPTION_YEAR.getKey());
    }

    /** @return {@link FaxToMailConfigurationOption#BASEDIR} value */
    public File getBasedir() {
        File result = applicationConfig.getOptionAsFile(FaxToMailConfigurationOption.BASEDIR.getKey());
        return result;
    }
    
    /** @return {@link FaxToMailConfigurationOption#DEMO_DIRECTORY} value */
    public File getDemoDirectory() {
        File result = applicationConfig.getOptionAsFile(FaxToMailConfigurationOption.DEMO_DIRECTORY.getKey());
        return result;
    }

    /** @return {@link FaxToMailConfigurationOption#DATA_DIRECTORY} value */
    public File getDataDirectory() {
        File result = applicationConfig.getOptionAsFile(FaxToMailConfigurationOption.DATA_DIRECTORY.getKey());
        return result;
    }

    /** @return {@link FaxToMailConfigurationOption#EDI_DIRECTORY} value */
    public File getEdiDirectory() {
        File result = applicationConfig.getOptionAsFile(FaxToMailConfigurationOption.EDI_DIRECTORY.getKey());
        return result;
    }

    public File getTmpDirectory() {
        File result = applicationConfig.getOptionAsFile(FaxToMailConfigurationOption.TMP_DIRECTORY.getKey());
        return result;
    }

    public File newTempFile(String basename) {
        return new File(getTmpDirectory(), basename + "_" + System.nanoTime());
    }

    public char getCsvSeparator() {
        return applicationConfig.getOption(
                FaxToMailConfigurationOption.CSV_SEPARATOR.getKey()).charAt(0);
    }

    public boolean isFullLaunchMode() {
        return "full".equals(getFaxToMailLaunchMode());
    }

    public String getFaxToMailLaunchMode() {
        return applicationConfig.getOption(FaxToMailConfigurationOption.FAXTOMAIL_LAUNCH_MODE.getKey());
    }

    /** @return {@link FaxToMailConfigurationOption#UI_CONFIG_FILE} value */
    public File getUIConfigFile() {
        return applicationConfig.getOptionAsFile(FaxToMailConfigurationOption.UI_CONFIG_FILE.getKey());
    }

    /** @return {@link FaxToMailConfigurationOption#START_ACTION_FILE} value */
    public File getStartActionFile() {
        return applicationConfig.getOptionAsFile(FaxToMailConfigurationOption.START_ACTION_FILE.getKey());
    }

    public KeyStroke getShortCut(String actionName) {
        KeyStroke result = applicationConfig.getOptionAsKeyStroke(
                "faxtomail.ui." + actionName);
        return result;
    }

    public boolean isAutoPopupNumberEditor() {
        return applicationConfig.getOptionAsBoolean(FaxToMailConfigurationOption.AUTO_POPUP_NUMBER_EDITOR.getKey());
    }

    public boolean isShowNumberEditorButton() {
        return applicationConfig.getOptionAsBoolean(FaxToMailConfigurationOption.SHOW_NUMBER_EDITOR_BUTTON.getKey());
    }

    public Color getColorBlockingLayer() {
        return applicationConfig.getOptionAsColor(FaxToMailConfigurationOption.COLOR_BLOCKING_LAYER.getKey());
    }

    public Color getColorAlternateRow() {
        return applicationConfig.getOptionAsColor(FaxToMailConfigurationOption.COLOR_ALTERNATE_ROW.getKey());
    }

    public Color getColorSelectedRow() {
        return applicationConfig.getOptionAsColor(FaxToMailConfigurationOption.COLOR_SELECTED_ROW.getKey());
    }

    @Override
    public KeyStroke getShortcutClosePopup() {
        return applicationConfig.getOptionAsKeyStroke(FaxToMailConfigurationOption.SHORTCUT_CLOSE_POPUP.getKey());
    }

    public String getDateFormat() {
        return applicationConfig.getOption(FaxToMailConfigurationOption.DATE_FORMAT.getKey());
    }

    public File getNewTmpDirectory(String name) {
        return new File(getTmpDirectory(), name + "_" + System.nanoTime());
    }

    public File getI18nDirectory() {
        return applicationConfig.getOptionAsFile(
                FaxToMailConfigurationOption.FAXTOMAIL_I18N_DIRECTORY.getKey());
    }

    public Locale getI18nLocale() {
        return applicationConfig.getOptionAsLocale(
                FaxToMailConfigurationOption.FAXTOMAIL_I18N_LOCALE.getKey());
    }

    public void setI18nLocale(Locale locale) {
        applicationConfig.setOption(FaxToMailConfigurationOption.FAXTOMAIL_I18N_LOCALE.getKey(), locale.toString());
    }

}
