package jaxx.runtime.swing.editor.config.model;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;

import javax.swing.table.TableCellEditor;

/**
 * A builder of {@link jaxx.runtime.swing.editor.config.model.ConfigUIModel}
 * Created: 22 déc. 2009
 *
 * @author Tony Chemit <chemit@codelutin.com> Copyright Code Lutin
 * @version $Revision: 1694 $
 *          <p/>
 *          Mise a jour: $Date: 2009-12-22 18:34:34 +0100 (mar., 22 déc. 2009) $ par :
 *          $Author: tchemit $
 * @since 2.0.0
 */
public class ConfigUIModelBuilder {
    /**
     * Logger
     */
    private static final Log log = LogFactory.getLog(ConfigUIModelBuilder.class);

    /**
     * current model used
     */
    ConfigUIModel model;
    /**
     * current category used
     */
    CategoryModel category;
    /**
     * current option used
     */
    OptionModel option;

    /**
     * Create a new model and set it as current model.
     *
     * @param config the configuration used in model
     * @throws IllegalStateException if there is already a current model
     * @throws NullPointerException  if config is {@code null}
     */
    public void createModel(ApplicationConfig config) throws IllegalStateException, NullPointerException {
        checkNoCurrent(model, "model");
        checkNotNull(config, "createModel", "config");
        model = new ConfigUIModel(config);
        if (log.isDebugEnabled()) {
            log.debug("model created : " + model);
        }
    }

    /**
     * Set the {@code reloadApplicationCallback} of the current model.
     *
     * @param callback the call back to set
     * @throws IllegalStateException if there is not a current model
     * @throws NullPointerException  if any of parameter is {@code null}
     * @see jaxx.runtime.swing.editor.config.model.ConfigUIModel#setReloadApplicationCallback(Runnable)
     */
    public void setReloadApplicationCallback(Runnable callback) throws IllegalStateException, NullPointerException {
        checkCurrent(model, "model");
        checkNotNull(callback, "setReloadApplicationCallback", "callback");
        model.setReloadApplicationCallback(callback);
    }

    /**
     * Set the {@code reloadUICallback} of the current model.
     *
     * @param callback the call back to set
     * @throws IllegalStateException if there is not a current model
     * @throws NullPointerException  if any of parameter is {@code null}
     * @see jaxx.runtime.swing.editor.config.model.ConfigUIModel#setReloadUICallback(Runnable)
     */
    public void setReloadUICallback(Runnable callback) {
        checkCurrent(model, "model");
        checkNotNull(callback, "setReloadUICallback", "callback");
        model.setReloadUICallback(callback);
    }

    /**
     * Add a new category, and set it as current.
     * <p/>
     * <b>Note:</b> As side effets, if a previous category, then store it to the model.
     *
     * @param categoryName  the name of the new category (can not to be {@code null})
     * @param categoryLabel the label of the new category (can not to be {@code null})
     * @throws IllegalStateException if there is not a current model, nor category
     * @throws NullPointerException  if any of parameter is {@code null}
     */
    public void addCategory(String categoryName, String categoryLabel) throws IllegalStateException, NullPointerException {
        checkCurrent(model, "model");
        checkNotNull(categoryName, "addCategory", "categoryName");
        checkNotNull(categoryLabel, "addCategory", "categoryLabel");
        flushCategory();
        category = new CategoryModel(categoryName, categoryLabel);
        if (log.isDebugEnabled()) {
            log.debug("category created : " + category);
        }
    }

    /**
     * Add a new option, and set it as current.
     * <p/>
     * <b>Note:</b> As side effets, if a previous option, then store it to the model.
     *
     * @param def the def ot the new option
     * @throws IllegalStateException if there is not a current model, nor category
     * @throws NullPointerException  if any of parameter is {@code null}
     */
    public void addOption(ApplicationConfig.OptionDef def) throws IllegalStateException, NullPointerException {
        checkCurrent(model, "model");
        checkCurrent(category, "category");
        checkNotNull(def, "addOption", "def");
        flushOption();
        Object value = model.getConfig().getOption(def);
        option = new OptionModel(def, value);
        if (log.isDebugEnabled()) {
            log.debug("option created : " + option);
        }
    }

    /**
     * Add a new option with a propertyName, and set it as current.
     * <p/>
     * <b>Note:</b> As side effets, if a previous option, then store it to the model.
     * <p/>
     * <b>Note:</b> This method is a short-cut for
     * {@link #addOption(org.nuiton.util.ApplicationConfig.OptionDef)} then
     * {@link #setOptionPropertyName(String)}.
     *
     * @param def          the def ot the new option
     * @param propertyName the propertyName to set on the option
     * @throws IllegalStateException if there is not a current model, nor category
     * @throws NullPointerException  if any of parameter is {@code null}
     */
    public void addOption(ApplicationConfig.OptionDef def, String propertyName) throws IllegalStateException, NullPointerException {
        addOption(def);
        checkNotNull(propertyName, "setOptionPropertyName", "propertyName");
        option.setPropertyName(propertyName);
    }

    /**
     * Set the propertyName on the current option.
     *
     * @param propertyName the propertyName to set in the current option.
     * @throws IllegalStateException if there is not a current option set.
     * @throws NullPointerException  if any of parameter is {@code null}
     * @see jaxx.runtime.swing.editor.config.model.OptionModel#setPropertyName(String)
     */
    public void setOptionPropertyName(String propertyName) throws IllegalStateException, NullPointerException {
        checkCurrent(option, "option");
        checkNotNull(propertyName, "setOptionPropertyName", "propertyName");
        option.setPropertyName(propertyName);
    }

    /**
     * Set the editor on the current option.
     *
     * @param editor the editor to set in the current option.
     * @throws IllegalStateException if there is not a current option set.
     * @throws NullPointerException  if any of parameter is {@code null}
     * @see jaxx.runtime.swing.editor.config.model.OptionModel#setEditor(javax.swing.table.TableCellEditor)
     */
    public void setOptionEditor(TableCellEditor editor) throws IllegalStateException, NullPointerException {
        checkCurrent(option, "option");
        checkNotNull(editor, "setOptionEditor", "editor");
        option.setEditor(editor);
    }

    /**
     * Set the needReloadUI flag on the current option.
     *
     * @param needReload new value to set
     * @throws IllegalStateException if there is not a current option set.
     * @see jaxx.runtime.swing.editor.config.model.OptionModel#setNeedReloadUI(boolean)
     */
    public void setOptionNeedReloadUI(boolean needReload) throws IllegalStateException {
        checkCurrent(option, "option");
        option.setNeedReloadUI(needReload);
    }

    /**
     * Set the needReloadApplication flag on the current option.
     *
     * @param needReload new value to set
     * @throws IllegalStateException if there is not a current option set.
     * @see jaxx.runtime.swing.editor.config.model.OptionModel#setNeedReloadUI(boolean)
     */
    public void setOptionNeedReloadApplication(boolean needReload) throws IllegalStateException {
        checkCurrent(option, "option");
        option.setNeedReloadApplication(needReload);
    }

    /**
     * Flush the model and return it.
     * <p/>
     * <b>Note:</b> As a side effect, nothing is available in the builder after this operation.
     * To reuse the builder on a model, use the dedicated setter.
     *
     * @return the final model
     * @throws IllegalStateException if there is not a current model set.
     */
    public ConfigUIModel flushModel() throws IllegalStateException {
        checkCurrent(model, "model");
        flushCategory();
        ConfigUIModel result = model;
        model = null;
        return result;
    }

    /**
     * Set the given model as current model.
     * <p/>
     * <b>Note:</b> As side effets, il will clean current category and option.
     *
     * @param model the model to use
     * @throws IllegalStateException if there is already a current model
     */
    public void setModel(ConfigUIModel model) throws IllegalStateException {
        checkNoCurrent(model, "model");
        this.model = model;
        if (log.isDebugEnabled()) {
            log.debug("new current model : " + this.model);
        }
        category = null;
        option = null;
    }

    /**
     * Set the given category as current category.
     * <p/>
     * <b>Note:</b> As side effets, il will clean current option.
     *
     * @param categoryModel the category to use
     * @throws IllegalStateException if there is not a current model or a current category
     */
    public void setCategory(CategoryModel categoryModel) throws IllegalStateException {
        checkCurrent(model, "model");
        checkNoCurrent(category, "category");
        category = categoryModel;
        if (log.isDebugEnabled()) {
            log.debug("new current category : " + this.category);
        }
        option = null;
    }

    /**
     * Sets the given option as current option.
     *
     * @param optionModel the option to use
     * @throws IllegalStateException if there is not a current model, nor category, or a current option
     */
    public void setOption(OptionModel optionModel) throws IllegalStateException {
        checkCurrent(model, "model");
        checkCurrent(category, "category");
        checkNoCurrent(option, "option");
        option = optionModel;
        if (log.isDebugEnabled()) {
            log.debug("new current option : " + this.option);
        }
    }

    protected CategoryModel flushCategory() {
        CategoryModel result = category;
        if (category != null) {
            flushOption();
            // add the previous category to the model
            model.addCategory(category);
            category = null;
        }
        return result;
    }

    protected OptionModel flushOption() {
        OptionModel result = option;
        if (option != null) {
            // add the previous option to the category
            category.addOption(option);
            option = null;
        }
        return result;
    }

    protected void checkCurrent(Object o, String type) {
        if (o == null) {
            throw new IllegalStateException("no current " + type + "!");
        }
    }

    protected void checkNoCurrent(Object o, String type) {
        if (o != null) {
            throw new IllegalStateException("there is already a current " + type + "!");
        }
    }

    protected void checkNotNull(Object o, String method, String parameter) {
        if (o == null) {
            throw new NullPointerException("method " + method + " does not support null parameter " + parameter + "!");
        }
    }

}
