/*
 * #%L
 * JAXX :: Tutorial Config
 * 
 * $Id: RunDemo.java 2163 2011-01-25 11:58:22Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.3/jaxx-tutorial-validation/src/main/java/jaxx/demo/validation/RunDemo.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package jaxx.demo.validation;

import jaxx.demo.validation.ui.DemoUI;
import jaxx.demo.validation.ui.DemoUIHandler;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.context.DefaultApplicationContext;
import jaxx.runtime.context.JAXXInitialContext;
import jaxx.runtime.swing.ErrorDialogUI;
import jaxx.runtime.swing.editor.config.ConfigUIHelper;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;
import org.nuiton.i18n.init.DefaultI18nInitializer;
import org.nuiton.util.StringUtil;

import javax.swing.JOptionPane;
import java.util.Arrays;
import java.util.Date;

import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 2.0.2
 */
public class RunDemo {

    /** Logger */
    private static Log log = LogFactory.getLog(RunDemo.class);

    /** The singleton instance of the main context */
    protected static DefaultApplicationContext context;

    public static void main(String[] args) {

        long startingTime = System.nanoTime();
        log.info("JAXX Demo start at " + new Date() + " args: " + Arrays.toString(args));

        try {

            // init root context
            // to enable javassist on webstart, must remove any securityManager,
            // see if this can be dangerous (should not be since jnlp is signed ?)
            // moreover it speeds up the loading :)
            System.setSecurityManager(null);

            long t0 = System.nanoTime();

            I18n.init(new DefaultI18nInitializer("jaxx-tutorial-validation-i18n"), null);

            Runtime.getRuntime().addShutdownHook(new ShutdownHook());

            // init root context
            DefaultApplicationContext rootContext = init();

            // share a tree helper
//            rootContext.setContextValue(new DemoTreeHelper(new DemoDataProvider()));

            // init config
            DemoConfig config = new DemoConfig();
            config.parse(args);

            // share the config
            rootContext.setContextValue(config);

            // init decorator provider
//            DecoratorProvider p = new DemoDecoratorProvider();

            // share the decorator provider
//            rootContext.setContextValue(p);

            long t00 = System.nanoTime();

            // init i18n
            I18n.setDefaultLocale(config.getLocale());

            log.info("language : " + config.getLocale());

            if (log.isDebugEnabled()) {
                log.debug("i18n loading time : " + StringUtil.convertTime(t00, System.nanoTime()));
            }

            log.info(_("jaxxdemo.message.config.loaded", config.getVersion()));

            // prepare ui look&feel and load ui properties

            try {
                SwingUtil.initNimbusLoookAndFeel();
            } catch (Exception e) {
                // could not find nimbus look-and-feel
                log.warn(_("jaxxdemo.warning.nimbus.landf"));
            } catch (Throwable e) {
                log.warn(_("jaxxdemo.warning.no.ui"));

            }

            if (log.isDebugEnabled()) {
                log.debug("init done in " + StringUtil.convertTime(t0, System.nanoTime()));
            }

            log.info(_("jaxxdemo.init.context.done", StringUtil.convertTime(startingTime, System.nanoTime())));

            // on affiche l'ui principale
            DemoUIHandler handler = new DemoUIHandler();

            DemoUI ui = handler.initUI(rootContext, config);

            log.info(_("jaxxdemo.init.ui.done"));

            handler.displayUI(ui);

        } catch (Exception e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);
            System.exit(1);
        }
    }

    public static void buildConfigUI(ConfigUIHelper helper) {

        Runnable reloadUICallback = new Runnable() {

            @Override
            public void run() {
                if (log.isInfoEnabled()) {
                    log.info("will say Hello world");
                }
                JOptionPane.showMessageDialog(null, "Hello World!");
            }
        };

        Runnable reloadApplicationCallback = new Runnable() {

            @Override
            public void run() {
                if (log.isInfoEnabled()) {
                    log.info("will reload appplication");
                }
                if (log.isInfoEnabled()) {
                    log.info("will say Good Bye world");
                }
                JOptionPane.showMessageDialog(null, "Good Bye World!");
            }
        };

        helper.registerCallBack("ui",
                                n_("demo.action.reload.ui"),
                                SwingUtil.createActionIcon("reload-ui"),
                                reloadUICallback);

        helper.registerCallBack("application",
                                n_("demo.action.reload.application"),
                                SwingUtil.createActionIcon("reload-application"),
                                reloadApplicationCallback);

        // categorie repertoires

        helper.addCategory(n_("jaxxdemo.config.category.directories"),
                           n_("jaxxdemo.config.category.directories.description"));

        helper.addOption(DemoConfig.Option.CONFIG_FILE);

        // others
        helper.addCategory(n_("jaxxdemo.config.category.other"),
                           n_("jaxxdemo.config.category.other.description"));

        helper.addOption(DemoConfig.Option.FULL_SCREEN);
        helper.setOptionPropertyName(DemoConfig.PROPERTY_FULLSCREEN);
        helper.setOptionCallBack("ui");

        helper.addOption(DemoConfig.Option.FONT_SIZE);
        helper.setOptionPropertyName(DemoConfig.PROPERTY_FONT_SIZE);
        helper.setOptionCallBack("application");

        helper.addOption(DemoConfig.Option.LOCALE);
        helper.setOptionPropertyName(DemoConfig.PROPERTY_LOCALE);
        helper.setOptionCallBack("ui");

        helper.buildUI(new JAXXInitialContext(),
                       "jaxxdemo.config.category.other");

    }

    /**
     * @return <code>true</code> si le context a été initialisé via la méthode
     *         {@link #init()}, <ocde>false</code> autrement.
     */
    protected static boolean isInit() {
        return context != null;
    }

    /**
     * Permet l'initialisation du contexte applicatif et positionne
     * l'context partagée.
     * <p/>
     * Note : Cette méthode ne peut être appelée qu'une seule fois.
     *
     * @return l'context partagée
     * @throws IllegalStateException si un contexte applicatif a déja été positionné.
     */
    protected static synchronized DefaultApplicationContext init() throws IllegalStateException {
        if (isInit()) {
            throw new IllegalStateException("there is an already application context registred.");
        }
        context = new DefaultApplicationContext();
        return context;
    }

    /**
     * Récupération du contexte applicatif.
     *
     * @return l'context partagé du contexte.
     * @throws IllegalStateException si le contexte n'a pas été initialisé via
     *                               la méthode {@link #init()}
     */
    public static DefaultApplicationContext get() throws IllegalStateException {
        if (!isInit()) {
            throw new IllegalStateException("no application context registred.");
        }
        return context;
    }

    protected static class ShutdownHook extends Thread {

        public ShutdownHook() {
            super("shutdown JAXXDemo");
        }

        @Override
        public void run() {
            try {
                super.run();

                // force to kill main thread

                log.info(_("jaxxdemo.init.closed", new Date()));
                Runtime.getRuntime().halt(0);
            } catch (Exception ex) {
                log.error("error while closing " + ex.getMessage(), ex);
                Runtime.getRuntime().halt(1);
            }
        }
    }

}
