/*
 * *##% 
 * JAXX Runtime
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.runtime.context;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.Util;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

/**
 * The default {@link JAXXContext} to be used in a {@link JAXXObject} by delegation.
 * <p/>
 * The values are store in a {@link Map} but we can not use directly the values as key.
 * <p/>
 * Because, it does not work if we add for the same object multi entries (named and unamed)...
 * <p/>
 * We prefer use as entry the {@link JAXXContextEntryDef} associated with the value.
 *
 * @author chemit
 */
public class DefaultJAXXContext implements JAXXContext {

    /**
     * entry of the parent context
     */
    protected static final JAXXContextEntryDef<JAXXContext> PARENT_CONTEXT_ENTRY = Util.newContextEntryDef(JAXXContext.class);
    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(DefaultJAXXContext.class);
//    /**
//     * l'ui auquel est rattache le context
    //     */
//    protected JAXXObject ui;
    /**
     * le context parent
     */
    protected JAXXContext parentContext;
    /**
     * les données contenues dans le context
     */
    protected final Map<JAXXContextEntryDef<?>, Object> data;

    public DefaultJAXXContext() {
        data = new HashMap<JAXXContextEntryDef<?>, Object>();
    }

//    public DefaultJAXXContext(JAXXObject ui) {
//        this();
//        this.ui = ui;
//    }

    @Override
    public <T> void setContextValue(T o) {
        setContextValue(o, null);
    }

    @Override
    public <T> void setContextValue(T o, String name) {
        if (name == null && PARENT_CONTEXT_ENTRY.accept2(o.getClass(), null)) {
            setParentContext((JAXXContext) o);
            return;
        }
        JAXXContextEntryDef<?> entry = getKey(name, o.getClass());
        // first remove entry
        Object oldValue = remove0(o.getClass(), name);
        if (oldValue != null) {
            if (log.isDebugEnabled()) {
                log.debug("remove value " + oldValue.getClass() + " for " + entry);
            }
        }
        // then can put safely
        data.put(entry, o);
    }

    @Override
    public <T> T getContextValue(Class<T> clazz) {
        return getContextValue(clazz, null);
    }

    @SuppressWarnings({"unchecked"})
    @Override
    public <T> T getContextValue(Class<T> clazz, String name) {
        if (parentContext != null && parentContext.getClass() == clazz || PARENT_CONTEXT_ENTRY.accept(clazz, name)) {
            return (T) getParentContext();
        }
        for (Entry<JAXXContextEntryDef<?>, Object> entry : data.entrySet()) {
            if (entry.getKey().accept(clazz, name)) {
                return (T) entry.getValue();
            }
        }

        // no value found in this context, will try in the parent context
        if (JAXXContext.class == clazz) {
            // no seek in the parent context, since we are already looking for it
            return null;
        }

        JAXXContext parent = getParentContext();
        if (parent == null) {
            // no parent context, so no value find
            return null;
        }
        // seek in parent context
        return parent.getContextValue(clazz, name);
    }

    @Override
    public <T> void removeContextValue(Class<T> klazz) {
        removeContextValue(klazz, null);
    }

    @Override
    public <T> void removeContextValue(Class<T> klazz, String name) {
        remove0(klazz, name);
    }

//    @Override
//    public <O extends Container> O getParentContainer(Class<O> clazz) {
//        return this.getParentContainer(ui, clazz);
//    }
//
//    @SuppressWarnings({"unchecked"})
//    @Override
//    public <O extends Container> O getParentContainer(Object top, Class<O> clazz) {
//        if (ui == null) {
//            throw new IllegalStateException("no ui attached to this context");
//        }
//        if (top == null) {
//            throw new IllegalArgumentException("top parameter can not be null");
//        }
//        if (!Container.class.isAssignableFrom(top.getClass())) {
//            throw new IllegalArgumentException("top parameter " + top + " is not a " + Container.class);
//        }
//        Container parent = ((Container) top).getParent();
//        if (parent != null && !clazz.isAssignableFrom(parent.getClass())) {
//            parent = getParentContainer(parent, clazz);
//        }
//        return (O) parent;
//    }

    /**
     * Obtain all the keys of data for a given type.
     *
     * @param klass the type of searched keys
     * @return the array of all names of keys for the given type of data
     * @since 1.3
     */
    public String[] getKeys(Class<?> klass) {
        List<String> keys = new java.util.ArrayList<String>();
        for (JAXXContextEntryDef<?> key : data.keySet()) {
            if (key.getKlass() == klass) {
                keys.add(key.getName());
            }
        }
        return keys.toArray(new String[keys.size()]);

    }

    public void clear() {
        data.clear();
    }

//    protected JAXXObject getUi() {
//        return ui;
//    }
//
//    protected void setUi(JAXXObject ui) {
//        this.ui = ui;
//    }

    protected JAXXContextEntryDef<?> getKey(String name, Class<?> klass) {
        return Util.newContextEntryDef(name, klass);
    }

    @SuppressWarnings({"unchecked"})
    protected <T> T remove0(Class<T> klazz, String name) {
        if (PARENT_CONTEXT_ENTRY.accept(klazz, name)) {
            JAXXContext old = getParentContext();
            setParentContext(null);
            return (T) old;
        }
        JAXXContextEntryDef<?> entry = null;
        for (JAXXContextEntryDef<?> entryDef : data.keySet()) {
            if (entryDef.accept(klazz, name)) {
                entry = entryDef;
                break;
            }
        }
        if (entry != null) {
            return (T) data.remove(entry);
        }

        if (JAXXContext.class == klazz) {
            return null;
        }
        // try in parentContext
        JAXXContext parent = getParentContext();

        if (parent == null) {
            return null;
        }

        if (parent instanceof DefaultJAXXContext) {
            return ((DefaultJAXXContext) parent).remove0(klazz, name);
        }

        return null;
    }

    protected JAXXContext getParentContext() {
        return parentContext;
    }

    protected void setParentContext(JAXXContext parentContext) {
        if (parentContext instanceof JAXXObject) {
            // keep the real context, not the ui
            parentContext = ((JAXXObject) parentContext).getDelegateContext();
        }
        this.parentContext = parentContext;
    }
}
