/*
 * #%L
 * Lima :: Swing
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.ui.lettering;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.LimaSwingConfig;
import org.chorem.lima.beans.LetteringFilterImpl;
import org.chorem.lima.business.api.AccountService;
import org.chorem.lima.business.api.EntryBookService;
import org.chorem.lima.business.api.FinancialPeriodService;
import org.chorem.lima.business.api.FinancialTransactionService;
import org.chorem.lima.business.api.FiscalPeriodService;
import org.chorem.lima.business.exceptions.LockedEntryBookException;
import org.chorem.lima.business.exceptions.LockedFinancialPeriodException;
import org.chorem.lima.business.exceptions.UnbalancedEntriesException;
import org.chorem.lima.entity.Account;
import org.chorem.lima.entity.AccountImpl;
import org.chorem.lima.entity.Entry;
import org.chorem.lima.entity.FiscalPeriod;
import org.chorem.lima.business.LimaServiceFactory;
import org.chorem.lima.util.BigDecimalToString;
import org.chorem.lima.util.ErrorHelper;

import javax.swing.*;
import java.awt.event.ActionEvent;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.math.BigDecimal;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import static org.nuiton.i18n.I18n.t;


/**
 * Handler associated with financial transaction view.
 *
 * @author chatellier
 * @version $Revision$
 *          <p/>
 *          Last update : $Date$
 *          By : $Author$
 */
public class LetteringViewHandler{

    protected LetteringView view;
    protected LetteringTable table;

    /** Transaction service. */
    protected FiscalPeriodService fiscalPeriodService;
    protected FinancialPeriodService financialPeriodService;
    protected AccountService accountService;
    protected FinancialTransactionService financialTransactionService;
    protected EntryBookService entryBookService;

    protected LetteringFilterImpl filter;

    protected BigDecimal debit = BigDecimal.ZERO;
    protected BigDecimal credit = BigDecimal.ZERO;
    protected BigDecimal solde = BigDecimal.ZERO;
    protected LettringSelectionModel lettringSelectionModel;
    protected LetteringEditModel editModel;

    protected ErrorHelper errorHelper;

    protected enum ButtonMode {DELETTRED, LETTRED, EQUALIZED, ALL}
    private static final Log log = LogFactory.getLog(LetteringViewHandler.class);


    protected boolean initializationComplete;

    public LetteringViewHandler(LetteringView view) {
        initializationComplete = false;
        this.view = view;
        initShortCuts();
        financialPeriodService = LimaServiceFactory.getService(FinancialPeriodService.class);
        fiscalPeriodService = LimaServiceFactory.getService(FiscalPeriodService.class);
        accountService = LimaServiceFactory.getService(AccountService.class);
        financialTransactionService = LimaServiceFactory.getService(FinancialTransactionService.class);
        entryBookService = LimaServiceFactory.getService(EntryBookService.class);
        errorHelper = new ErrorHelper(LimaSwingConfig.getInstance());
    }

    /**
     * Init all combo box in view.
     */
    public void init() {
        filter = new LetteringFilterImpl();
        editModel = view.getEditModel();
        lettringSelectionModel = view.getLetteringSelectionModel();
        loadComboAndRows();

        editModel.addPropertyChangeListener(LetteringEditModel.PROPERTY_DEBIT, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                BigDecimal debit = (BigDecimal) evt.getNewValue();
                String text = BigDecimalToString.format(debit);
                view.getDebitTextField().setText(text);
            }
        });

        editModel.addPropertyChangeListener(LetteringEditModel.PROPERTY_CREDIT, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                BigDecimal credit = (BigDecimal) evt.getNewValue();
                String text = BigDecimalToString.format(credit);
                view.getCreditTextField().setText(text);
            }
        });

        editModel.addPropertyChangeListener(LetteringEditModel.PROPERTY_SOLD, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                BigDecimal sold = (BigDecimal) evt.getNewValue();
                String text = BigDecimalToString.format(sold);
                view.getSoldTextField().setText(text);
            }
        });

        initializationComplete = true;

        updateAllEntries();


    }

    protected void initShortCuts() {

        InputMap inputMap= view.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
        ActionMap actionMap = view.getActionMap();
        Object binding;

        //To block reaction of the dual key 'ctrl+a' (Selection of all lines)
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_A, InputEvent.CTRL_MASK), "none");

        // add action on Ctrl + L
        binding = "lettering";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_L, KeyEvent.CTRL_DOWN_MASK), binding);
        actionMap.put(binding, new AbstractAction() {
            private static final long serialVersionUID = 397305388204489988L;

            @Override
            public void actionPerformed(ActionEvent e) {
                addLetter();
            }
        });

        // add action on Ctrl + Delete
        binding = "un-lettering";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, 0), binding);
        actionMap.put(binding, new AbstractAction() {
            private static final long serialVersionUID = 6493175994438339351L;

            @Override
            public void actionPerformed(ActionEvent e) {
                removeLetter();
            }
        });

        // add action on Ctrl + B
        binding = "balance";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_B, KeyEvent.CTRL_DOWN_MASK), binding);
        actionMap.put(binding, new AbstractAction() {
            private static final long serialVersionUID = 5997811877503911744L;

            @Override
            public void actionPerformed(ActionEvent e) {
                roundAndCreateEntry();
            }
        });

        // refresh
        binding = "refresh";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_F5, 0), binding);
        actionMap.put(binding, new AbstractAction() {
            private static final long serialVersionUID = -7192846839712951680L;

            @Override
            public void actionPerformed(ActionEvent e) {
                updateAllEntries();
            }
        });
    }

    public void balanceAndActions() {
        if (log.isDebugEnabled()) {
            log.debug("balanceAndActions");
        }
        if (view.getTable().getSelectedRows().length == 0) {
            onButtonModeChanged(ButtonMode.ALL);
            onBalanceChanged(null);
        } else if (!letteringNotExist(view.getTable().getSelectedRow())) {

            //lettred entries
            onBalanceChanged(null);
            setValuesForSelectedEntries();

            //For U.I. buttons (Lettering and unlettering)
            onButtonModeChanged(ButtonMode.DELETTRED);
        } else {
            if (log.isDebugEnabled()) {
                log.debug("unlettred entries");
            }
            int[] selectedRows = view.getTable().getSelectedRows();
            if (selectedRows.length == 2) {
                if (log.isDebugEnabled()) {
                    log.debug("2 rows selected");
                }
                /*Treatment only if one of values contains decimals*/
                LetteringTableModel tableModel = view.getTableModel();
                Entry firstSelectedEntry = tableModel.get(selectedRows[0]);
                Entry secondSelectedEntry = tableModel.get(selectedRows[1]);

                /*Get decimals*/
                BigDecimal firstSelectedEntryAmount = firstSelectedEntry.getAmount();
                BigDecimal secondSelectedEntryAmount = secondSelectedEntry.getAmount();

                if ( secondSelectedEntry.isDebit() != firstSelectedEntry.isDebit()
                     && (firstSelectedEntryAmount.subtract(secondSelectedEntryAmount).abs().compareTo(BigDecimal.ZERO) >0
                         && firstSelectedEntryAmount.subtract(secondSelectedEntryAmount).abs().compareTo(BigDecimal.ONE) <0) ) {
                    onButtonModeChanged(ButtonMode.EQUALIZED);
                }
            }else {
                if (log.isDebugEnabled()) {
                    log.debug("!2 rows selected");
                }
                onButtonModeChanged(ButtonMode.ALL);
            }

            //Unlettred entries
            onBalanceChanged(null);
            //treatment unuseful if no rows are selected
            if (!view.getLetteringSelectionModel().isSelectionEmpty()) {
                if (log.isDebugEnabled()) {
                    log.debug("Rows selected");
                }
                setValuesForSelectedEntries();
                onButtonModeChanged(ButtonMode.LETTRED);
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("No Rows selected");
                }
                onButtonModeChanged(ButtonMode.ALL);
            }
        }
    }

    /**return true if lettering is null, or not null but empty
     * @param row index of the line to test
     * @return boolean
     * */
    public boolean letteringNotExist(int row){
        boolean emptyOrNull = false;
        if (row != -1) {
            Entry entry = view.getTableModel().get(row);
            String lettering = entry.getLettering();
            emptyOrNull = (lettering==null||lettering.isEmpty());
        }
        return emptyOrNull;
    }

    public void onButtonModeChanged(ButtonMode buttonMode) {

        switch (buttonMode) {
            case DELETTRED :
                editModel.setLettred(false);
                editModel.setUnLettred(true);
                break;
            case LETTRED:
                editModel.setUnLettred(false);
                editModel.setLettred(true);
                break;
            case EQUALIZED:
                editModel.setEqualized(true);
                break;
            default:
                editModel.setLettred(false);
                editModel.setUnLettred(false);
                editModel.setEqualized(false);
        }
    }

    public void setValuesForSelectedEntries() {
        Entry selectedEntry;
        LetteringTableModel tableModel = view.getTableModel();
        for (int i = 0; i < tableModel.getRowCount(); i ++){
            if (view.getLetteringSelectionModel().isSelectedIndex(i)){
                selectedEntry = tableModel.get(i);
                //Set values for calculation (By LetteringEditModel) of balance
                onBalanceChanged(selectedEntry);
            }
        }
    }

    public void onBalanceChanged(Entry balance) {
        if (balance == null) {
            editModel.setCredit(BigDecimal.ZERO);
            editModel.setDebit(BigDecimal.ZERO);
            editModel.setSolde(BigDecimal.ZERO, false);
        } else {
            balanceCalculation(balance.getAmount(), balance.isDebit());
        }
    }

    /**Allow to add / subtract credit / debit and balance
     * @param amount debit or credit
     * @param debit it indicate if amount is debit or not
     * */
    public void balanceCalculation(BigDecimal amount, boolean debit){

        BigDecimal debitVal = debit ? amount : BigDecimal.ZERO;
        BigDecimal creditVal = debit ? BigDecimal.ZERO : amount;

        if (log.isDebugEnabled()) {
            log.debug("balance calculation");
        }

        if (debitVal.equals(BigDecimal.ZERO)){

            if (!creditVal.equals(BigDecimal.ZERO)){

                editModel.setCredit(creditVal);
                editModel.setSolde(creditVal, true);
            }
        }else if (creditVal.equals(BigDecimal.ZERO)){
            editModel.setDebit(debitVal);
            editModel.setSolde(debitVal, false);
        }else{
            onBalanceChanged(null);
        }
    }

    public void loadComboAndRows(){

        List<Account> allAccounts = accountService.getAllAccounts();
        Account seeAllAccounts = new AccountImpl();
        seeAllAccounts.setAccountNumber("-");
        seeAllAccounts.setLabel(t("lima.lettering.account.aAll"));
        allAccounts.add(seeAllAccounts);

        view.getAccountComboBoxModel().setObjects(allAccounts);

        if (!allAccounts.isEmpty()) {

            view.getAccountComboBox().setSelectedItem(allAccounts.get(0));
        }

        //By default, we have the beginning of the fiscal period (Or of current
        //date if no fiscal period) and the end of the current date
        FiscalPeriod fiscalPeriod = fiscalPeriodService.getLastFiscalPeriod();
        Date defaultDateBegFiscalPeriod;

        Calendar calendar = Calendar.getInstance();
        int dernierJourMoisCourant = calendar.getActualMaximum(Calendar.DATE);
        int premierJourMoisCourant = calendar.getActualMinimum(Calendar.DATE);

        if (fiscalPeriod != null){
            defaultDateBegFiscalPeriod = fiscalPeriodService.getLastFiscalPeriod().getBeginDate();
        } else{
           defaultDateBegFiscalPeriod = DateUtils.setDays(new Date(), premierJourMoisCourant);
        }

        Date defaultDateEndCurrent = DateUtils.setDays(new Date(), dernierJourMoisCourant);


        view.getBeginPeriodPicker().setDate(defaultDateBegFiscalPeriod);
        view.getEndPeriodPicker().setDate(defaultDateEndCurrent);

        filter.setDateStart(defaultDateBegFiscalPeriod);
        filter.setDateEnd(defaultDateEndCurrent);

        TypeEntry type = view.getLetteredEntryComboBox().getSelectedItem();
        setTypeEntry(type);
    }

    public void setDateStart(Date date) {
        filter.setDateStart(date);
        updateAllEntries();
    }

    public void setDateEnd(Date date) {
        filter.setDateEnd(date);
        updateAllEntries();
    }

    public void setAccount(Account account) {
        filter.setAccount(account);
        updateAllEntries();
    }

    public void setTypeEntry(TypeEntry typeEntry) {
        filter.setDisplayLettered(typeEntry.isLettered());
        filter.setDisplayUnlettred(typeEntry.isNoLettered());
        updateAllEntries();
    }

    public void updateAllEntries() {

        if (initializationComplete
                && filter.getAccount() != null
                && filter.getDateStart() != null
                && filter.getDateEnd() != null) {

            List<Entry> entries = financialTransactionService.getAllEntrieByDatesAndAccountAndLettering(filter);

            view.getTableModel().setValues(entries);
        }

        onBalanceChanged(null);
    }

    /**To make the difference between two selected entries and
     * create a new entry with the result (debit or credit).
     * It allow to letter somme entries with different debit and credit
     * */
    public void roundAndCreateEntry() {

        LetteringTableModel tableModel = view.getTableModel();

        int[] selectedRows = view.getTable().getSelectedRows();
        if (editModel.isEqualized() && selectedRows.length == 2) {
            try {
                /*Treatment only if one of values contains decimals*/
                Entry firstSelectedEntry = tableModel.get(selectedRows[0]);
                Entry secondSelectedEntry = tableModel.get(selectedRows[1]);

                Entry[] newEntriesFormEqualizing = financialTransactionService.getEntriesFromEqualizing(firstSelectedEntry, secondSelectedEntry);

                /*Add new entries to the model and the table*/
                Entry newSameAccountEntry = newEntriesFormEqualizing[0];
                Entry newCostOrProductEntry = newEntriesFormEqualizing[1];
                tableModel.addValue(newSameAccountEntry);
                tableModel.addValue(newCostOrProductEntry);

                /*Re-select the two entries (firstSelectedEntry and secondSelectedEntry)
                * and the new sameAccountEntry
                * */
                view.getLetteringSelectionModel().selectRoundedAndNewEntries(selectedRows[0], selectedRows[1], newSameAccountEntry);
            } catch (LockedFinancialPeriodException e) {
                errorHelper.showErrorMessage(t("lima.lettering.roundAndCreateEntry.error.lockedFinancialPeriod",
                        e.getFinancialPeriod().getBeginDate(),
                        e.getFinancialPeriod().getEndDate()));
            } catch (LockedEntryBookException e) {
                errorHelper.showErrorMessage(t("lima.lettering.roundAndCreateEntry.error.lockedEntryBook",
                        e.getClosedPeriodicEntryBook().getEntryBook().getCode(),
                        e.getClosedPeriodicEntryBook().getEntryBook().getLabel(),
                        e.getClosedPeriodicEntryBook().getFinancialPeriod().getBeginDate(),
                        e.getClosedPeriodicEntryBook().getFinancialPeriod().getEndDate()));
            }
        }
    }

    /**
     * Select previous value in combo box.
     *
     * @param comboBox combo box
     */
    public void back(JComboBox comboBox) {
        int row = comboBox.getSelectedIndex();

        if (row > 0) {
            comboBox.setSelectedIndex(row - 1);
        }
        view.getLetteringSelectionModel().clearSelection();
    }

    /**
     * Select next value in combo box.
     *
     * @param comboBox combo box
     */
    public void next(JComboBox comboBox) {
        int size = comboBox.getModel().getSize();
        int row = comboBox.getSelectedIndex();

        if (row < size - 1) {
            comboBox.setSelectedIndex(row + 1);
        }
        view.getLetteringSelectionModel().clearSelection();
    }

    /**Add a group of three letters to n entries*/
    public void addLetter() {
        if (editModel.isLettred()) {
            int[] entrieSelected = view.getTable().getSelectedRows();

            LetteringTableModel tableModel = view.getTableModel();

            List<Entry> entries = Lists.newLinkedList();

            for (int indexEntry : entrieSelected) {

                Entry entry = tableModel.get(indexEntry);

                entries.add(entry);
            }

            try {

                if (!entries.isEmpty()) {

                    entries = financialTransactionService.addLetter(entries);

                    updateEntries(entries);
                }

            } catch (LockedEntryBookException e) {

                errorHelper.showErrorMessage(t("lima.entries.letter.closed.entryBook.error",
                        e.getClosedPeriodicEntryBook().getEntryBook().getCode(),
                        e.getClosedPeriodicEntryBook().getEntryBook().getLabel(),
                        e.getClosedPeriodicEntryBook().getFinancialPeriod().getBeginDate(),
                        e.getClosedPeriodicEntryBook().getFinancialPeriod().getEndDate()));

            } catch (UnbalancedEntriesException e) {

                errorHelper.showErrorMessage(t("lima.entries.letter.unbalanced.error"));

            }
            onButtonModeChanged(ButtonMode.DELETTRED);
        }
    }

    /**Remove a group of three letters to n entries*/
    public void removeLetter() {
        if (editModel.isUnLettred()) {

            int[] entrieSelected = view.getTable().getSelectedRows();

            LetteringTableModel tableModel = view.getTableModel();

            if (entrieSelected.length > 0) {

                Entry firstEntry = tableModel.get(entrieSelected[0]);

                String letter = firstEntry.getLettering();

                List<Entry> entries = financialTransactionService.removeLetter(letter);

                updateEntries(entries);
            }

            onButtonModeChanged(ButtonMode.LETTRED);
        }
    }


    protected void updateEntries(List<Entry> entries) {

        LetteringTableModel tableModel = view.getTableModel();

        for (final Entry entry : entries) {

            Entry oldEntry = Iterables.find(
                    tableModel.getValues(),
                    new Predicate<Entry>() {
                @Override
                public boolean apply(Entry input) {
                    return input.getTopiaId().equals(entry.getTopiaId());
                }
            },
                    null);

            if (oldEntry != null) {
                int indexEntry = tableModel.indexOf(oldEntry);
                tableModel.setValue(indexEntry, entry);
            }
        }
    }

}
