/*
 * #%L
 * Lima Swing
 * 
 * $Id: LimaConfig.java 3777 2014-04-14 07:29:08Z sbavencoff $
 * $HeadURL: https://svn.chorem.org/lima/tags/lima-0.7.3/lima-swing/src/main/java/org/chorem/lima/LimaConfig.java $
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima;

import com.google.common.collect.ImmutableSet;
import jaxx.runtime.JAXXUtil;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.actions.MiscAction;
import org.chorem.lima.business.api.OptionsService;
import org.chorem.lima.entity.LimaCallaoDAOHelper;
import org.chorem.lima.service.LimaServiceFactory;
import org.nuiton.config.ApplicationConfig;
import org.nuiton.config.ArgumentsParserException;
import org.nuiton.config.ConfigOptionDef;
import org.nuiton.topia.TopiaContextFactory;
import org.nuiton.util.Version;
import org.nuiton.util.VersionUtil;
import org.nuiton.util.converter.ConverterUtil;

import java.awt.*;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.IOException;
import java.util.Locale;
import java.util.Set;

import static org.nuiton.i18n.I18n.t;
import static org.nuiton.i18n.I18n.n;

/**
 * La configuration de l'application.
 *
 * @author chemit
 * @version $Revision: 3777 $
 *          <p/>
 *          Last update : $Date: 2014-04-14 09:29:08 +0200 (Mon, 14 Apr 2014) $
 *          By : $Author: sbavencoff $
 */
public class LimaConfig extends ApplicationConfig {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(LimaConfig.class);

    static public final Set<Character> NUMBER_SEPARATOR =  ImmutableSet.of(' ', ',', '.', ';');
    static public final Set<Integer> NUMBER_DECIMALS =  ImmutableSet.of(0, 1, 2, 3, 4, 5, 6);

    protected static LimaConfig instance;

    private static final String configFile = "lima-config.properties";

    protected OptionsService optionsService;

    /** La version du logiciel. */
    protected Version version;

    /**
     * Get the configuration file
     * @return name of the config. file
     * */
    /*public String getConfigFile() {
        return configFile;
    }*/

    /**
     * Get copyright text (include version).
     *
     * @return copyright text
     */
    public String getCopyrightText() {
        return "Version " + getVersion() + " Codelutin @ 2008-2012";
    }

    /**
     * Version as string.
     *
     * @return le nombre global ex: 3.2.0.0
     */
    public String getVersion() {
        return version.toString();
    }

    /**
     * Lima config constructor.
     * <p/>
     * Define all default options and action alias.
     */
    public LimaConfig() {

        // set defaut option (included configuration file name : important)
        loadDefaultOptions(Option.values());

        // set action alias
        for (Action a : Action.values()) {
            for (String alias : a.aliases) {
                addActionAlias(alias, a.action);
            }
        }

        // ajout des alias (can be set in option enum ?)
        addAlias("--disableui", "--launchui false");

        // initialisation des répertoires
        // TODO what is it for ?
        //FileUtil.setCurrentDirectory(getLimaUserDirectory());
        //getLimaUserDirectory().mkdirs();

    }

    public static LimaConfig getInstance() {
        if (instance == null) {
            instance = new LimaConfig();
            instance.loadConfiguration(configFile);
        }

        return instance;
    }

    protected void loadConfiguration(String configFileName) {

        instance.setConfigFileName(configFileName);
        try {
            instance.parse();
        } catch (ArgumentsParserException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't read configuration", ex);
            }
        }
        instance.setOption(TopiaContextFactory.CONFIG_PERSISTENCE_CLASSES,
                           LimaCallaoDAOHelper.getImplementationClassesAsString());
    }

    @Override
    public ApplicationConfig parse(String... args) throws ArgumentsParserException {

        // super parse, read config file etc...
        super.parse(args);

        // on ne peut pas connaitre la version avant la lecture du fichier
        // on supprime tout ce qui est apres - (-SNAPSHOT, -rc-1 ,...)
        String sVersion = VersionUtil.removeSnapshot(getOption("application.version"));
        version = VersionUtil.valueOf(sVersion);

        return this;

    }

    /**
     * Get application locale.
     *
     * @return configuration application locale
     */
    public Locale getLocale() {
        String local = getOption(Option.LOCALE.key);
        Locale result = ConverterUtil.convert(Locale.class, local);
        return result;
    }

    /**
     * Locale setter for command line parameters.
     *
     * @param locale new locale
     */
    public void setLocale(String locale) {
        setOption(Option.LOCALE.key, locale);
    }

    /**
     * Change locale (not command line version).
     * Save user file.
     *
     * @param newLocale new locale
     */
    public void setLocale(Locale newLocale) {
        setOption(Option.LOCALE.key, newLocale.toString());
        saveForUser();
        firePropertyChange("locale", null, newLocale);
    }

    /**
     * Get application decimal separator
     *
     * @return configuration application decimal separator
     */
    public char getDecimalSeparator() {
        char decimalSeparator = getOption(Option.DECIMAL_SEPARATOR.key).charAt(0);
        return decimalSeparator;
    }

    /**
     * Change decimal separator
     * Save user file.
     *
     * @param decimalSeparator new DecimalSeparator
     */
    public void setDecimalSeparator(String decimalSeparator) {
        setOption(Option.DECIMAL_SEPARATOR.key, decimalSeparator);
        saveForUser();
        firePropertyChange("decimalSeparator", null, decimalSeparator);
    }

    /**
     * Get application scale
     *
     * @return configuration application scale
     */
    public int getScale() {
        return getOptionAsInt(Option.SCALE.key);
    }

    /**
     * Change scale
     * Save user file.
     *
     * @param scale new Scale
     */
    public void setScale(String scale) {
        setOption(Option.SCALE.key, scale);
        saveForUser();
        firePropertyChange("scale", null, scale);
        if (log.isInfoEnabled()) {
            log.info("new scale" + scale);
        }
        optionsService.setScale(scale);
    }

    /**
     * Get application thousand separator
     *
     * @return configuration application thousand separator
     */
    public char getThousandSeparator() {
        return getOption(Option.THOUSAND_SEPARATOR.key).charAt(0);
    }

    /**
     * Change the thousand separator
     * Save user file.
     *
     * @param thousandSeparator new thousandSeparator
     */
    public void setThousandSeparator(String thousandSeparator) {
        setOption(Option.THOUSAND_SEPARATOR.key, thousandSeparator);
        saveForUser();
        firePropertyChange("thousandSeparator", null, thousandSeparator);
    }

    /**
     * currency configuration boolean
     *
     * @return {@code true} if the currency must be displayed
     */
    public boolean getCurrency() {
        return getOptionAsBoolean(Option.CURRENCY.key);
    }

    /**
     * Change the currency displaying
     *
     * @param currency the new currency to set in configuration
     */
    public void setCurrency(boolean currency) {
        setOption(Option.CURRENCY.key, Boolean.toString(currency));
        saveForUser();
        firePropertyChange("currency", null, currency);
    }

    /**
     * Launch ui configuration value.
     *
     * @return {@code true} if ui must be displayed
     */
    public boolean isLaunchui() {
        boolean launchUI = getOptionAsBoolean(Option.LAUNCH_UI.key);
        return launchUI;
    }

    /**
     * Launch ui setter for command line parameters.
     *
     * @param launchui new lauch ui value
     */
    public void setLaunchui(String launchui) {
        setOption(Option.LAUNCH_UI.key, launchui);
    }

    /**
     * Change full screen mode.
     * Save user file.
     *
     * @param fullscreen new fullscreen mode
     */
    public void setFullscreen(boolean fullscreen) {
        boolean oldValue = isFullScreen();
        setOption(Option.FULL_SCREEN.key, fullscreen + "");
        saveForUser();
        firePropertyChange("fullscreen", oldValue, fullscreen);
    }

    public boolean isFullScreen() {
        Boolean result = getOptionAsBoolean(Option.FULL_SCREEN.key);
        return result != null && result;
    }

    /**
     * Get support email address.
     *
     * @return support email
     */
    public String getSupportEmail() {
        return getOption(Option.SUPPORT_EMAIL.key);
    }

    /**
     * Return true if ejb mode is configured as remote.
     *
     * @return {@code true} if remote mode should be used
     */
    public boolean isEJBRemoteMode() {
        boolean result = getOptionAsBoolean(Option.OPENEJB_REMOTEMODE.key);
        return result;
    }

    public File getDataDirectory() {
        File result = getOptionAsFile(Option.DATA_DIR.key);
        return result;
    }

    public File getLimaStateFile() {
        File result = getOptionAsFile(Option.LIMA_STATE_FILE.key);
        return result;
    }

    public File getResourcesDirectory() {
        File result = getOptionAsFile(Option.RESOURCES_DIRECTORY.key);
        return result;
    }

    public File getI18nDirectory() {
        File result = getOptionAsFile(Option.I18N_DIRECTORY.key);
        return result;
    }

    public String getHostAdress() {
        return getOption(Option.LIMA_HOST_ADDRESS.key);
    }

    public void setColorSelectionFocus(String color) {
        setOption(Option.COLOR_SELECTION_FOCUS.key, color);
    }

    public Color getColorSelectionFocus() {
        return getOptionAsColor((Option.COLOR_SELECTION_FOCUS.key));
    }

    public void setSelectAllEditingCell(boolean selectAllEditingCell) {
        setOption(Option.SELECT_ALL_EDITING_CELL.key, Boolean.toString(selectAllEditingCell));
    }

    public boolean isSelectAllEditingCell() {
        return getOptionAsBoolean((Option.SELECT_ALL_EDITING_CELL.key));
    }

    /** Used in ???? */
    public static final String[] DEFAULT_JAXX_PCS = {"fullScreen", "locale", "decimalSeparator", "scale", "thousandSeparator", "currency"};

    /** Used in ???? */
    public void removeJaxxPropertyChangeListener() {
        PropertyChangeListener[] toRemove = JAXXUtil.findJaxxPropertyChangeListener(DEFAULT_JAXX_PCS, getPropertyChangeListeners());
        if (toRemove == null || toRemove.length == 0) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug("before remove : " + getPropertyChangeListeners().length);
            log.debug("toRemove : " + toRemove.length);

        }
        for (PropertyChangeListener listener : toRemove) {
            removePropertyChangeListener(listener);
        }
        if (log.isDebugEnabled()) {
            log.debug("after remove : " + getPropertyChangeListeners().length);
        }
    }

    /**
     * Lima option definition.
     * <p/>
     * Contains all lima configuration key, with defaut value and
     * information for jaxx configuration frame ({@link #type},
     * {@link #_transient}, {@link #_final}...)
     */
    public enum Option implements ConfigOptionDef {


        CONFIG_FILE(CONFIG_FILE_NAME,
                t("lima.config.configFileName.label"),
                n("lima.config.configFileName.description"),
                "lima-config.properties",
                String.class, true, true),

        DATA_DIR("lima.data.dir",
                t("lima.config.data.dir.label"),
                n("lima.config.data.dir.description"),
                "${user.home}/.lima",
                File.class, false, false),

        RESOURCES_DIRECTORY("lima.resources.dir",
                t("lima.config.resources.dir.label"),
                n("lima.config.resources.dir.description"),
                "${lima.data.dir}/resources-${application.version}",
                String.class, false, false),

        I18N_DIRECTORY("lima.i18n.dir",
                t("lima.config.i18n.dir.label"),
                n("lima.config.i18n.dir.description"),
                "${lima.resources.dir}/i18n",
                String.class, false, false),

        LOCALE("lima.ui.locale",
                t("lima.config.locale.label"),
                n("lima.config.locale.description"),
                "fr_FR",
                Locale.class, false, false),

        DECIMAL_SEPARATOR("lima.data.bigDecimal.decimalSeparator",
                t("lima.config.decimalseparator.label"),
                n("lima.config.decimalseparator.description"),
                ",",
                Character.class, false, false),

        SCALE("lima.data.bigDecimal.scale",
                t("lima.config.scale.label"),
                n("lima.config.scale.description"),
                "2",
                Integer.class, false, false),

        THOUSAND_SEPARATOR("lima.thousandSeparator",
                t("limma.config.thousandseparator.label"),
                n("limma.config.thousandseparator.description"),
                " ",
                Character.class, false, false),

        CURRENCY("lima.config.currency",
                t("lima.config.currency.label"),
                n("lima.config.currency.description"),
                "false",
                Boolean.class, false, false),

        FULL_SCREEN("lima.ui.fullscreen",
                t("lima.config.ui.fullscreen.label"),
                n("lima.config.ui.fullscreen.description"),
                "false",
                Boolean.class, false, false),

        LAUNCH_UI("lima.ui.launchui",
                t("lima.config.ui.flaunchui.label"),
                n("lima.config.ui.flaunchui.description"),
                "true", Boolean.class, true, true),

        SUPPORT_EMAIL("lima.misc.supportemail",
                t("lima.misc.supportemail.label"),
                n("lima.misc.supportemail.description"),
                "support@codelutin.com",
                String.class, false, false),

        OPENEJB_REMOTEMODE("lima.openejb.remotemode",
                t("lima.openejb.remotemode.label"),
                n("lima.openejb.remotemode.description"),
                "false",
                String.class, false, false),

        LIMA_HOST_ADDRESS("lima.host.address",
                t("lima.config.host.adress.label"),
                n("lima.config.host.adress.description"),
                "localhost",
                String.class, false, false),

        LIMA_STATE_FILE("lima.ui.state.file",
                t("lima.config.state.file.label"),
                n("lima.config.state.file.description"),
                "${lima.data.dir}/limaState.xml",
                String.class, false, false),

        COLOR_SELECTION_FOCUS("lima.ui.table.cell.colorSelectionFocus",
                t("lima.config.colorselectionfocus.label"),
                n("lima.config.colorselectionfocus.description"),
                "#000000",
                Color.class, false, false),

        SELECT_ALL_EDITING_CELL("lima.ui.table.cell.selectAllEditingCell",
                t("lima.config.selectAllEditingCell.label"),
                n("lima.config.selectAllEditingCell.description"),
                "true",
                Boolean.class, false, false),

        TABLE_CELL_BACKGROUND("lima.ui.table.cell.background",
                t("lima.config.cell.background.label"),
                n("lima.config.cell.background.description"),
                "#FFFFFF",
                Color.class, false, false),

        TABLE_CELL_FOREGROUND("lima.ui.table.cell.foreground",
                t("lima.config.cell.foreground.label"),
                n("lima.config.cell.foreground.description"),
                "#000000",
                Color.class, false, false),

        TABLE_CELL_PAIR_BACKGROUND("lima.ui.table.cell.pair.background",
                t("lima.config.cell.pair.background.label"),
                n("lima.config.cell.pair.background.description"),
                "#EEEEEE",
                Color.class, false, false),

        TABLE_CELL_PAIR_FOREGROUND("lima.ui.table.cell.pair.foreground",
                t("lima.config.cell.pair.foreground.label"),
                n("lima.config.cell.pair.foreground.description"),
                "#000000",
                Color.class, false, false),

        TABLE_CELL_SELECTED_BACKGROUND("lima.ui.table.cell.selected.background",
                t("lima.config.cell.selected.background.label"),
                n("lima.config.cell.selected.background.description"),
                "#0066CC",
                Color.class, false, false),

        TABLE_CELL_SELECTED_FOREGROUND("lima.ui.table.cell.selected.foreground",
                t("lima.config.cell.selected.foreground.label"),
                n("lima.config.cell.selected.foreground.description"),
                "#FFFFFF",
                Color.class, false, false),

        TABLE_CELL_PAIR_SELECTED_BACKGROUND("lima.ui.table.cell.pair.selected.background",
                t("lima.config.cell.pair.selected.background.label"),
                n("lima.config.cell.pair.selected.background.description"),
                "#006699",
                Color.class, false, false),

        TABLE_CELL_PAIR_SELECTED_FOREGROUND("lima.ui.table.cell.pair.selected.foreground",
                t("lima.config.cell.pair.selected.foreground.label"),
                n("lima.config.cell.pair.selected.foreground.description"),
                "#FFFFFF",
                Color.class, false, false),

        TABLE_CELL_ERROR_BACKGROUND("lima.ui.table.cell.error.background",
                t("lima.config.cell.error.background.label"),
                n("lima.config.cell.error.background.description"),
                "#FFFFFF",
                Color.class, false, false),

        TABLE_CELL_ERROR_FOREGROUND("lima.ui.table.cell.error.foreground",
                t("lima.config.cell.error.foreground.label"),
                n("lima.config.cell.error.foreground.description"),
                "#FF0936",
                Color.class, false, false),

        TABLE_CELL_PAIR_ERROR_BACKGROUND("lima.ui.table.cell.pair.error.background",
                t("lima.config.cell.pair.error.background.label"),
                n("lima.config.cell.pair.error.background.description"),
                "#EEEEEE",
                Color.class, false, false),

        TABLE_CELL_PAIR_ERROR_FOREGROUND("lima.ui.table.cell.pair.error.foreground",
                t("lima.config.cell.pair.error.foreground.label"),
                n("lima.config.cell.pair.error.foreground.description"),
                "#FC0625",
                Color.class, false, false),

        TABLE_CELL_SELECTED_ERROR_BACKGROUND("lima.ui.table.cell.selected.error.background",
                t("lima.config.cell.selected.error.background.label"),
                n("lima.config.cell.selected.error.background.description"),
                "#0066CC",
                Color.class, false, false),

        TABLE_CELL_SELECTED_ERROR_FOREGROUND("lima.ui.table.cell.selected.error.foreground",
                t("lima.config.cell.selected.error.foreground.label"),
                n("lima.config.cell.selected.error.foreground.description"),
                "#C998C1",
                Color.class, false, false),

        TABLE_CELL_PAIR_SELECTED_ERROR_BACKGROUND("lima.ui.table.cell.pair.selected.error.background",
                t("lima.config.cell.pair.selected.error.background.label"),
                n("lima.config.cell.pair.selected.error.background.description"),
                "#006699",
                Color.class, false, false),

        TABLE_CELL_PAIR_SELECTED_ERROR_FOREGROUND("lima.ui.table.cell.pair.selected.error.foreground",
                t("lima.config.cell.pair.selected.error.foreground.label"),
                n("lima.config.cell.pair.selected.error.foreground.description"),
                "#C96678",
                Color.class, false, false),

        TABLE_CELL_MANDATORY_BACKGROUND("lima.ui.table.cell.mandatory.background",
                t("lima.config.cell.mandatory.background.label"),
                n("lima.config.cell.mandatory.background.description"),
                "#FFCCCC",
                Color.class, false, false),

        TABLE_CELL_MANDATORY_FOREGROUND("lima.ui.table.cell.mandatory.foreground",
                t("lima.config.cell.mandatory.foreground.label"),
                n("lima.config.cell.mandatory.foreground.description"),
                "#000000",
                Color.class, false, false),

        TABLE_CELL_PAIR_MANDATORY_BACKGROUND("lima.ui.table.cell.pair.mandatory.background",
                t("lima.config.cell.pair.mandatory.background.label"),
                n("lima.config.cell.pair.mandatory.background.description"),
                "#FF99CC",
                Color.class, false, false),

        TABLE_CELL_PAIR_MANDATORY_FOREGROUND("lima.ui.table.cell.pair.mandatory.foreground",
                t("lima.config.cell.pair.mandatory.foreground.label"),
                n("lima.config.cell.pair.mandatory.foreground.description"),
                "#000000",
                Color.class, false, false),

        TABLE_CELL_SELECTED_MANDATORY_BACKGROUND("lima.ui.table.cell.selected.mandatory.background",
                t("lima.config.cell.selected.mandatory.background.label"),
                n("lima.config.cell.selected.mandatory.background.description"),
                "#FF0000",
                Color.class, false, false),

        TABLE_CELL_SELECTED_MANDATORY_FOREGROUND("lima.ui.table.cell.selected.mandatory.foreground",
                t("lima.config.cell.selected.mandatory.foreground.label"),
                n("lima.config.cell.selected.mandatory.foreground.description"),
                "#FFFFFF",
                Color.class, false, false),

        TABLE_CELL_PAIR_SELECTED_MANDATORY_BACKGROUND("lima.ui.table.cell.pair.selected.mandatory.background",
                t("lima.config.cell.pair.selected.mandatory.background.label"),
                n("lima.config.cell.pair.selected.mandatory.background.description"),
                "#990000",
                Color.class, false, false),

        TABLE_CELL_PAIR_SELECTED_MANDATORY_FOREGROUND("lima.ui.table.cell.pair.selected.mandatory.foreground",
                t("lima.config.cell.pair.selected.mandatory.foreground.label"),
                n("lima.config.cell.pair.selected.mandatory.foreground.description"),
                "#000000",
                Color.class, false, false);

        public enum ComportmentEditingCellEnum {ALL, NOTHING}

        public final String key;

        public final String label;

        public final String description;

        public String defaultValue;

        public final Class<?> type;

        public boolean _transient;

        public boolean _final;

        Option(String key, String label, String description, String defaultValue,
               Class<?> type, boolean _transient, boolean _final) {
            this.key = key;
            this.label = label;
            this.description = description;
            this.defaultValue = defaultValue;
            this.type = type;
            this._final = _final;
            this._transient = _transient;
        }

        @Override
        public boolean isFinal() {
            return _final;
        }

        @Override
        public void setFinal(boolean _final) {
            this._final = _final;
        }

        @Override
        public boolean isTransient() {
            return _transient;
        }

        @Override
        public void setTransient(boolean _transient) {
            this._transient = _transient;
        }

        @Override
        public String getDefaultValue() {
            return defaultValue;
        }

        @Override
        public void setDefaultValue(String defaultValue) {
            this.defaultValue = defaultValue;
        }

        public String getLabel () {
            return label;
        }

        @Override
        public String getDescription() {
            return t(description);
        }

        @Override
        public String getKey() {
            return key;
        }

        @Override
        public Class<?> getType() {
            return type;
        }
    }

    /** Lima action definition. */
    public enum Action {

        HELP(n("lima.action.commandline.help"), MiscAction.class.getName() + "#help", "-h", "--help");

        /** Before init action step. */
        public static final int BEFORE_EXIT_STEP = 0;

        /** After init action step. */
        public static final int AFTER_INIT_STEP = 1;

        public String description;

        public String action;

        public String[] aliases;

        Action(String description, String action, String... aliases) {
            this.description = description;
            this.action = action;
            this.aliases = aliases;
        }

        public String getDescription() {
            return t(description);
        }
    }

    /**
     * Override save action to propagate some option to server.
     */
    @Override
    public void save(File file,
                     boolean forceAll,
                     String... excludeKeys) throws IOException {

        super.save(file, forceAll, excludeKeys);

        // propagate scale option to serveur option
        optionsService = LimaServiceFactory.getService(OptionsService.class);

        // scale server option
        String scaleOption = getOption("scale");
        if (StringUtils.isNotBlank(scaleOption)) {
            optionsService.setScale(scaleOption);
        }
    }
}
