/*
 * #%L
 * Lima Swing
 * *
 * $Id: LetteringViewHandler.java 3783 2014-05-06 16:47:18Z dcosse $
 * $HeadURL: https://svn.chorem.org/lima/tags/lima-0.7.2/lima-swing/src/main/java/org/chorem/lima/ui/lettering/LetteringViewHandler.java $
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.ui.lettering;

import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.beans.LetteringFilterImpl;
import org.chorem.lima.business.api.AccountService;
import org.chorem.lima.business.api.EntryBookService;
import org.chorem.lima.business.api.FinancialPeriodService;
import org.chorem.lima.business.api.FinancialTransactionService;
import org.chorem.lima.business.api.FiscalPeriodService;
import org.chorem.lima.entity.Account;
import org.chorem.lima.entity.AccountImpl;
import org.chorem.lima.entity.Entry;
import org.chorem.lima.entity.FiscalPeriod;
import org.chorem.lima.service.LimaServiceFactory;

import javax.swing.*;
import java.awt.event.ActionEvent;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.math.BigDecimal;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import static org.nuiton.i18n.I18n.t;


/**
 * Handler associated with financial transaction view.
 *
 * @author chatellier
 * @version $Revision: 3783 $
 *          <p/>
 *          Last update : $Date: 2014-05-06 18:47:18 +0200 (Tue, 06 May 2014) $
 *          By : $Author: dcosse $
 */
public class LetteringViewHandler{

    protected LetteringView view;
    protected LetteringTable table;

    /** Transaction service. */
    protected FiscalPeriodService fiscalPeriodService;
    protected FinancialPeriodService financialPeriodService;
    protected AccountService accountService;
    protected FinancialTransactionService financialTransactionService;
    protected EntryBookService entryBookService;

    protected LetteringFilterImpl filter;

    protected BigDecimal debit = BigDecimal.ZERO;
    protected BigDecimal credit = BigDecimal.ZERO;
    protected BigDecimal solde = BigDecimal.ZERO;
    protected LettringSelectionModel lettringSelectionModel;
    protected LetteringEditModel editModel;

    protected enum ButtonMode {DELETTRED, LETTRED, EQUALIZED, ALL}
    private static final Log log = LogFactory.getLog(LetteringViewHandler.class);

    public LetteringViewHandler(LetteringView view) {
        this.view = view;
        initShortCuts();
        financialPeriodService = LimaServiceFactory.getService(FinancialPeriodService.class);
        fiscalPeriodService = LimaServiceFactory.getService(FiscalPeriodService.class);
        accountService = LimaServiceFactory.getService(AccountService.class);
        financialTransactionService = LimaServiceFactory.getService(FinancialTransactionService.class);
        entryBookService = LimaServiceFactory.getService(EntryBookService.class);
    }

    /**
     * Init all combo box in view.
     */
    public void init() {
        filter = new LetteringFilterImpl();
        editModel = view.getEditModel();
        lettringSelectionModel = view.getLettringSelectionModel();
        loadComboAndRows();
    }

    protected void initShortCuts() {

        InputMap inputMap= view.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
        ActionMap actionMap = view.getActionMap();
        Object binding;

        //To block reaction of the dual key 'ctrl+a' (Selection of all lines)
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_A, InputEvent.CTRL_MASK), "none");

        // add action on Ctrl + L
        binding = "lettering";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_L, KeyEvent.CTRL_DOWN_MASK), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                addLetter();
            }
        });

        // add action on Ctrl + Delete
        binding = "un-lettering";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, 0), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                removeLetter();
            }
        });

        // add action on Ctrl + B
        binding = "balance";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_B, KeyEvent.CTRL_DOWN_MASK), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                roundAndCreateEntry();
            }
        });

        // refresh
        binding = "refresh";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_F5, 0), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                updateAllEntries();
            }
        });
    }

    public void balanceAndActions() {
        if (log.isDebugEnabled()) {
            log.debug("balanceAndActions");
        }
        if (view.getTable().getSelectedRows().length == 0) {
            onButtonModeChanged(ButtonMode.ALL);
            onBalanceChanged(null);
        } else if (!letteringNotExist(view.getTable().getSelectedRow())) {

            //lettred entries
            onBalanceChanged(null);
            setValuesForSelectedEntries();

            //For U.I. buttons (Lettering and unlettering)
            onButtonModeChanged(ButtonMode.DELETTRED);
        } else {
            if (log.isDebugEnabled()) {
                log.debug("unlettred entries");
            }
            int[] selectedRows = view.getTable().getSelectedRows();
            if (selectedRows.length == 2) {
                if (log.isDebugEnabled()) {
                    log.debug("2 rows selected");
                }
                /*Treatment only if one of values contains decimals*/
                LetteringTableModel tableModel = view.getTableModel();
                Entry firstSelectedEntry = tableModel.get(selectedRows[0]);
                Entry secondSelectedEntry = tableModel.get(selectedRows[1]);

                /*Get decimals*/
                BigDecimal firstSelectedEntryAmount = firstSelectedEntry.getAmount();
                BigDecimal secondSelectedEntryAmount = secondSelectedEntry.getAmount();

                if ( secondSelectedEntry.getDebit() != firstSelectedEntry.getDebit()
                     && (firstSelectedEntryAmount.subtract(secondSelectedEntryAmount).abs().compareTo(BigDecimal.ZERO) >0
                         && firstSelectedEntryAmount.subtract(secondSelectedEntryAmount).abs().compareTo(BigDecimal.ONE) <0) ) {
                    onButtonModeChanged(ButtonMode.EQUALIZED);
                }
            }else {
                if (log.isDebugEnabled()) {
                    log.debug("!2 rows selected");
                }
                onButtonModeChanged(ButtonMode.ALL);
            }

            //Unlettred entries
            onBalanceChanged(null);
            //treatment unuseful if no rows are selected
            if (!view.getLettringSelectionModel().isSelectionEmpty()) {
                if (log.isDebugEnabled()) {
                    log.debug("Rows selected");
                }
                setValuesForSelectedEntries();
                onButtonModeChanged(ButtonMode.LETTRED);
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("No Rows selected");
                }
                onButtonModeChanged(ButtonMode.ALL);
            }
        }
    }

    /**return true if lettering is null, or not null but empty
     * @param row index of the line to test
     * @return boolean
     * */
    public boolean letteringNotExist(int row){
        boolean emptyOrNull = false;
        if (row != -1) {
            Entry entry = view.getTableModel().get(row);
            String lettering = entry.getLettering();
            emptyOrNull = (lettering==null||lettering.isEmpty());
        }
        return emptyOrNull;
    }

    public void onButtonModeChanged(ButtonMode buttonMode) {

        switch (buttonMode) {
            case DELETTRED :
                editModel.setLettred(false);
                editModel.setUnLettred(true);
                break;
            case LETTRED:
                editModel.setUnLettred(false);
                editModel.setLettred(true);
                break;
            case EQUALIZED:
                editModel.setEqualized(true);
                break;
            default:
                editModel.setLettred(false);
                editModel.setUnLettred(false);
                editModel.setEqualized(false);
        }
    }

    public void setValuesForSelectedEntries() {
        Entry selectedEntry;
        LetteringTableModel tableModel = view.getTableModel();
        for (int i = 0; i < tableModel.getRowCount(); i ++){
            if (view.getLettringSelectionModel().isSelectedIndex(i)){
                selectedEntry = tableModel.get(i);
                //Set values for calculation (By LetteringEditModel) of balance
                onBalanceChanged(selectedEntry);
            }
        }
    }

    public void onBalanceChanged(Entry balance) {
        if (balance == null) {
            editModel.setCredit(BigDecimal.ZERO);
            editModel.setDebit(BigDecimal.ZERO);
            editModel.setSolde(BigDecimal.ZERO, false);
        } else {
            balanceCalculation(balance.getAmount(), balance.getDebit());
        }
    }

    /**Allow to add / subtract credit / debit and balance
     * @param amount debit or credit
     * @param debit it indicate if amount is debit or not
     * */
    public void balanceCalculation(BigDecimal amount, boolean debit){

        BigDecimal debitVal = debit ? amount : BigDecimal.ZERO;
        BigDecimal creditVal = debit ? BigDecimal.ZERO : amount;

        if (log.isDebugEnabled()) {
            log.debug("Balance calculation");
        }

        if (debitVal.equals(BigDecimal.ZERO)){

            if (!creditVal.equals(BigDecimal.ZERO)){

                editModel.setCredit(creditVal);
                editModel.setSolde(creditVal, true);
            }
        }else if (creditVal.equals(BigDecimal.ZERO)){
            editModel.setDebit(debitVal);
            editModel.setSolde(debitVal, false);
        }else{
            onBalanceChanged(null);
        }
    }

    public void loadComboAndRows(){

        List<Account> allAccounts = accountService.getAllAccounts();
        Account seeAllAccounts = new AccountImpl();
        seeAllAccounts.setAccountNumber("-");
        seeAllAccounts.setLabel(t("lima.ui.list.seeAll"));
        allAccounts.add(seeAllAccounts);

        view.getAccountComboBoxModel().setObjects(allAccounts);

        if (!allAccounts.isEmpty()) {

            view.getAccountComboBox().setSelectedItem(allAccounts.get(0));
        }

        //By default, we have the beginning of the fiscal period (Or of current
        //date if no fiscal period) and the end of the current date
        FiscalPeriod fiscalPeriod = fiscalPeriodService.getLastFiscalPeriod();
        Date defaultDateBegFiscalPeriod;

        Calendar calendar = Calendar.getInstance();
        int dernierJourMoisCourant = calendar.getActualMaximum(Calendar.DATE);
        int premierJourMoisCourant = calendar.getActualMinimum(Calendar.DATE);

        if (fiscalPeriod != null){
            defaultDateBegFiscalPeriod = fiscalPeriodService.getLastFiscalPeriod().getBeginDate();
        } else{
           defaultDateBegFiscalPeriod = DateUtils.setDays(new Date(), premierJourMoisCourant);
        }

        Date defaultDateEndCurrent = DateUtils.setDays(new Date(), dernierJourMoisCourant);


        view.getPickerDebut().setDate(defaultDateBegFiscalPeriod);
        view.getPickerFin().setDate(defaultDateEndCurrent);

        filter.setDateStart(defaultDateBegFiscalPeriod);
        filter.setDateEnd(defaultDateEndCurrent);

        TypeEntry type = view.getLettredEntryComboBox().getSelectedItem();
        setTypeEntry(type);

        updateAllEntries();
    }

    protected List<Entry> findAllEntries(LetteringFilterImpl filter){
        if (filter != null) {
            List<Entry> entries =
                    financialTransactionService.getAllEntrieByDatesAndAccountAndLettering(filter);
            return  entries;
        }
        return null;
    }

    public void setDateStart(Date date) {
        filter.setDateStart(date);
        updateAllEntries();
    }

    public void setDateEnd(Date date) {
        filter.setDateEnd(date);
        updateAllEntries();
    }

    public void setAccount(Account account) {
        filter.setAccount(account);
        updateAllEntries();
    }

    public void setTypeEntry(TypeEntry typeEntry) {
        filter.setDisplayLettered(typeEntry.isLettered());
        filter.setDisplayUnlettred(typeEntry.isNoLettered());
        updateAllEntries();
    }

    public void updateAllEntries() {

        if (filter.getAccount() != null && filter.getDateStart() != null && filter.getDateEnd() != null) {

            List<Entry> entries = findAllEntries(filter);

            view.getTableModel().setValues(entries);
        }
        onBalanceChanged(null);
    }

    /**To make the difference between two selected entries and
     * create a new entry with the result (debit or credit).
     * It allow to letter somme entries with different debit and credit
     * */
    public void roundAndCreateEntry() {

        LetteringTableModel tableModel = view.getTableModel();

        int[] selectedRows = view.getTable().getSelectedRows();
        if (editModel.isEqualized() && selectedRows.length == 2) {
            /*Treatment only if one of values contains decimals*/
            Entry firstSelectedEntry = tableModel.get(selectedRows[0]);
            Entry secondSelectedEntry = tableModel.get(selectedRows[1]);

            Entry[] newEntriesFormEqualizing = financialTransactionService.getEntriesFromEqualizing(firstSelectedEntry, secondSelectedEntry);

            /*Add new entries to the model and the table*/
            Entry newSameAccountEntry = newEntriesFormEqualizing[0];
            Entry newCostOrProductEntry = newEntriesFormEqualizing[1];
            tableModel.addValue(newSameAccountEntry);
            tableModel.addValue(newCostOrProductEntry);

            /*Re-select the two entries (firstSelectedEntry and secondSelectedEntry)
            * and the new sameAccountEntry
            * */
            view.getLettringSelectionModel().selectRoundedAndNewEntries(selectedRows[0], selectedRows[1], newSameAccountEntry);
        }
    }

    /**
     * Select previous value in combo box.
     *
     * @param comboBox combo box
     */
    public void back(JComboBox comboBox) {
        int row = comboBox.getSelectedIndex();

        if (row > 0) {
            comboBox.setSelectedIndex(row - 1);
        }
        view.getLettringSelectionModel().clearSelection();
    }

    /**
     * Select next value in combo box.
     *
     * @param comboBox combo box
     */
    public void next(JComboBox comboBox) {
        int size = comboBox.getModel().getSize();
        int row = comboBox.getSelectedIndex();

        if (row < size - 1) {
            comboBox.setSelectedIndex(row + 1);
        }
        view.getLettringSelectionModel().clearSelection();
    }

    /**Add a group of three letters to n entries*/
    public void addLetter() {
        if (editModel.isLettred()) {
            String newLetters = financialTransactionService.getNextLetters();
            changeLetter(newLetters);
            onButtonModeChanged(ButtonMode.DELETTRED);
        }
    }

    /**Remove a group of three letters to n entries*/
    public void removeLetter() {
        if (editModel.isUnLettred()) {
            changeLetter(null);
            onButtonModeChanged(ButtonMode.LETTRED);
        }
    }

    /**Add or remove a group of three letters to n entries*/
    protected void changeLetter(String newLetters) {

        int[] entrieSelected = view.getTable().getSelectedRows();

        LetteringTableModel tableModel = view.getTableModel();

        for (int indexEntry : entrieSelected){
            Entry entry = tableModel.get(indexEntry);
            entry.setLettering(newLetters);
            financialTransactionService.updateEntry(entry);
            tableModel.fireTableRowsUpdated(indexEntry, indexEntry);

        }
    }

}
