/*
 * #%L
 * Lima Swing
 * 
 * $Id: FinancialTransactionUnbalancedTableModel.java 3184 2011-06-17 15:32:15Z vsalaun $
 * $HeadURL: http://svn.chorem.org/svn/lima/tags/lima-0.5/lima-swing/src/main/java/org/chorem/lima/ui/financialtransactionunbalanced/FinancialTransactionUnbalancedTableModel.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
/* *##% Lima Swing
 * Copyright (C) 2008 - 2010 CodeLutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * ##%*/

package org.chorem.lima.ui.financialtransactionunbalanced;

import static org.nuiton.i18n.I18n._;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import javax.swing.table.AbstractTableModel;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.business.FinancialTransactionServiceMonitorable;
import org.chorem.lima.business.LimaException;
import org.chorem.lima.business.ServiceListener;
import org.chorem.lima.business.utils.EntryComparator;
import org.chorem.lima.entity.Account;
import org.chorem.lima.entity.Entry;
import org.chorem.lima.entity.EntryBook;
import org.chorem.lima.entity.EntryImpl;
import org.chorem.lima.entity.FinancialTransaction;
import org.chorem.lima.entity.FiscalPeriod;
import org.chorem.lima.entity.Letter;
import org.chorem.lima.service.LimaServiceFactory;
import org.chorem.lima.util.DialogHelper;
import org.chorem.lima.util.ErrorHelper;

/**
 * Basic transaction table model.
 * 
 * Le modele est filtré sur {@link #selectedFiscalPeriod}(montée en charge !).
 * 
 * @author ore
 * @author chatellier
 * @version $Revision: 3184 $
 * 
 * Last update : $Date: 2011-06-17 17:32:15 +0200 (Fri, 17 Jun 2011) $
 * By : $Author: vsalaun $
 */
public class FinancialTransactionUnbalancedTableModel extends AbstractTableModel implements ServiceListener{

    /** serialVersionUID. */
    private static final long serialVersionUID = 3914954536809622358L;

    /** log. */
    private static final Log log = LogFactory
            .getLog(FinancialTransactionUnbalancedTableModel.class);

    /** Transaction service. */
    protected final FinancialTransactionServiceMonitorable financialTransactionService;
    
    /** selected financial period */
    protected FiscalPeriod selectedFiscalPeriod;
    
    /** data cache */
    protected List<Object> cacheDataList;
    
    /** collection

    
    /**
     * Model constructor.
     * 
     * Just init service proxies.
     */
    public FinancialTransactionUnbalancedTableModel() {
        /* Services */
        financialTransactionService =
            LimaServiceFactory.getInstance().getService(
                    FinancialTransactionServiceMonitorable.class);
        financialTransactionService.addListener(this);
    }

    /**
     * Le model est une combinaison de Transaction/Entries.
     * 
     * 
     * @return
     */
    protected List<Object> getDataList() {
        List<Object> results = new ArrayList<Object>();
        if(selectedFiscalPeriod != null){
            try {
                List<FinancialTransaction> financialtransactions =
                    financialTransactionService.getAllInexactFinancialTransactions(selectedFiscalPeriod);
                for (FinancialTransaction financialtransaction : financialtransactions) {
                    results.add(financialtransaction);
                    List<Entry> entries = (List<Entry>) financialtransaction.getEntry();
                    Collections.sort(entries, new EntryComparator());
                    results.addAll(entries);
                }
            }
            catch (LimaException eee) {
                if (log.isErrorEnabled()) {
                    log.debug("Can't update model", eee);
                }
               
                ErrorHelper.showErrorDialog("Can't get transaction list", eee);
            }    
        }
        
        return results;
    }
    
    public void refresh(){
        cacheDataList = getDataList();
        fireTableDataChanged();
    }

    @Override
    public int getColumnCount() {
        return 9;
    }

    @Override
    public Class<?> getColumnClass(int column) {

        Class<?> result = null;

        switch (column) {
        case 0:
            result = Date.class;
            break;
        case 1:
            result = EntryBook.class;
            break;
        case 2:
            result = String.class;
            break;
        case 3:
            result = Account.class;
            break;
        case 4:
            result = String.class;
            break;
        case 5:
            result = BigDecimal.class;
            break;
        case 6:
            result = BigDecimal.class;
            break;
        case 7:
            result = BigDecimal.class;
            break;
        case 8:
            result = Letter.class;
            break;
        }

        return result;
    }

    @Override
    public String getColumnName(int column) {
        String result = "n/a";

        switch (column) {
        case 0:
            result = _("lima.table.date");
            break;
        case 1:
            result = _("lima.table.entrybook");
            break;
        case 2:
            result = _("lima.table.voucher");
            break;
        case 3:
            result = _("lima.table.account");
            break;
        case 4:
            result = _("lima.table.description");
            break;
        case 5:
            result = _("lima.table.debit");
            break;
        case 6:
            result = _("lima.table.credit");
            break;
        case 7:
            result = _("lima.table.balance");
            break;
        case 8:
            result = _("lima.table.letter");
            break;
        }

        return result;
    }

    @Override
    public int getRowCount() {
        int result = 0;
        
        // just prevent too much result
        if (cacheDataList != null) {
            result = cacheDataList.size();
        }

        return result;
    }

    @Override
    public Object getValueAt(int row, int column) {
        Object result = null;
        
        // just prevent too much result
        if (cacheDataList != null) {
            result = cacheDataList.get(row);
            
            if (result instanceof FinancialTransaction) {
                FinancialTransaction currentRow = (FinancialTransaction)result;
                BigDecimal amountDebit = currentRow.getAmountDebit();
                BigDecimal amountCredit = currentRow.getAmountCredit();
                
                switch (column) {
                case 0:
                    result = currentRow.getTransactionDate();
                    break;
                case 1:
                    if (currentRow.getEntryBook() != null){
                        result = currentRow.getEntryBook().getCode();
                    }
                    else {
                        result = null;
                    }
                    break;
                case 2:
                    result = null; //entrybook
                    break;
                case 3:
                    result = null; // account
                    break;
                case 4:
                    result = null; // description
                    break;
                case 5 :
                    result = amountDebit;
                    break;
                case 6:
                    result = amountCredit;
                    break;
                case 7:
                    result = amountDebit.subtract(amountCredit);
                    break;
                case 8:
                    result = null;
                    break;
                }
            }
            else if (result instanceof Entry) {
                Entry currentEntry = (Entry)result;
                switch (column) {
                case 0:
                    result = null; // date
                    break;
                case 1 : // entry book   
                    result = null;
                    break;
                case 2:
                    result = currentEntry.getVoucher();
                    break;
                case 3: // account
                    if (currentEntry.getAccount() != null){
                        result = currentEntry.getAccount().getAccountNumber();
                    }
                    else {
                        result = null;
                    }
                    break;
                case 4:
                    result = currentEntry.getDescription();
                    break;
                case 5 :
                    result = currentEntry.getDebit() ? currentEntry.getAmount() : BigDecimal.ZERO;
                    break;
                case 6:
                    result = currentEntry.getDebit() ? BigDecimal.ZERO : currentEntry.getAmount();
                    break;
                case 7:
                    result = null;
                    break;
                case 8:
                    if (currentEntry.getLetter() != null){
                        result = currentEntry.getLetter().getCode();
                    }
                    else {
                        result = null;
                    }
                    break;
                }
                
            }
        }
        
        return result;
    }
        
    public void setFiscalPeriod(FiscalPeriod fiscalPeriod){               
        selectedFiscalPeriod = fiscalPeriod;
    }
    
    /**
     * To set cells editable or not
     * different condition for entry or financial transaction
     */
    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
        boolean editableCell=false;
        Object currentRow = cacheDataList.get(rowIndex);
        // cells editable for the entry row, all cells exclude the date 
        if ((currentRow instanceof Entry) && !((columnIndex==0) || (columnIndex==1))) {
            editableCell=true;
        }
        // cells editable for the financialtransaction row, no cells exclude the date
        if ((currentRow instanceof FinancialTransaction) && ((columnIndex==0) || (columnIndex==1))){
            editableCell=true;
        }
        return editableCell;
    }
    
    

    /**
     * 
     * @param value
     * @param description
     * @param row
     * @return int: indexOf new Entry
     * @throws LimaException
     */
    public int addEmptyEntry(Object value, String description, int row) throws LimaException {
        FinancialTransaction currentTransaction = null;
        Object currentRow = cacheDataList.get(row);
        Entry entry = new EntryImpl();
        entry.setAmount(BigDecimal.ZERO);
        if (description != null) {
            entry.setDescription(description);
        }
        //check if current row is a transaction or an entry
        if (currentRow instanceof FinancialTransaction) {
            currentTransaction = (FinancialTransaction)currentRow;
        }
        else if (currentRow instanceof Entry) {
            Entry currentEntry = (Entry)currentRow;
            //get back the parent transaction of the entry
            currentTransaction = currentEntry.getFinancialTransaction();
        }
        //create it
        entry.setFinancialTransaction(currentTransaction);
        Entry newEntry = financialTransactionService.createEntry(entry);
        //on recharge la liste
        int newrow = cacheDataList.indexOf(newEntry);
        fireTableRowsInserted(row, row);
        return newrow;
    }
    
    /**
     * to modifiy financialtransaction or entry
     */
    @Override
    public void setValueAt(Object value, int row, int column) {
        int financialTransactionRow=0;
        // just prevent too much result
        if (selectedFiscalPeriod != null) {
            Object currentRow = cacheDataList.get(row);
            if (currentRow instanceof FinancialTransaction) {
                FinancialTransaction currentFinancialTransaction =
                    (FinancialTransaction)currentRow;
                switch (column) {
                case 0:
                    //update
                    currentFinancialTransaction.setTransactionDate((Date)value);
                    break;
                case 1 :
                    currentFinancialTransaction.setEntryBook((EntryBook)value);
                    break;
                }
                // notify service for modification
                try {
                    financialTransactionService.
                        updateFinancialTransaction(currentFinancialTransaction);
                } catch (LimaException eee) {
                    if (log.isDebugEnabled()){
                        log.debug("Can't update financial transaction", eee);
                    }
                    DialogHelper.showMessageDialog(eee.getMessage());
                }
              //update the financial transaction in entire
                financialTransactionRow = 
                    cacheDataList.indexOf(((FinancialTransaction) currentRow));
            }
            else if (currentRow instanceof Entry) {
                Entry currentEntry = (Entry)currentRow;
                switch (column) {
                case 2 :
                    currentEntry.setVoucher((String)value);
                    break;
                case 3:
                    currentEntry.setAccount((Account)value);
                    break;
                case 4:
                    currentEntry.setDescription((String)value);
                    break;
                case 5 :
                    currentEntry.setAmount((BigDecimal) value);
                    currentEntry.setDebit(true);
                    break;
                case 6:
                    currentEntry.setAmount((BigDecimal) value);
                    currentEntry.setDebit(false);
                    break;
                case 8:
                    currentEntry.setLetter((Letter)value);
                    break;
                }
                try {
                    financialTransactionService.updateEntry(currentEntry);
                } catch (LimaException eee) {
                    if (log.isDebugEnabled()){
                        log.debug("Can't update entry", eee);
                    }
                    DialogHelper.showMessageDialog(eee.getMessage());
                }
                //update the financial transaction in entire
                financialTransactionRow = 
                    cacheDataList.indexOf(((Entry) currentRow).
                            getFinancialTransaction());
            }
            //on recharge la liste
            cacheDataList = getDataList();
            fireTableDataChanged();
        }
    }
    
    public Object getElementAt(int row){
        
        Object currentRow = cacheDataList.get(row);
        return currentRow;
    }
    

    /**
     * Delete selected row in table (could be transaction or entry).
     * 
     * Called by model.
     * @param Object, int
     * @throws LimaException 
     */
    public void removeObject(Object object, int row) throws LimaException {
        Object currentRow = cacheDataList.get(row);
        if (currentRow instanceof FinancialTransaction) {
            FinancialTransaction currentTransaction =
                (FinancialTransaction)currentRow;
            financialTransactionService.removeFinancialTransaction(currentTransaction);
        }
        else if (currentRow instanceof Entry) {
            Entry currentEntry = (Entry)currentRow;
            financialTransactionService.removeEntry(currentEntry);
        }
        //on recharge la liste
        refresh();
    }
     
    @Override
    public void notifyMethod(String serviceName, String methodeName) {
        if (serviceName.contains("FinancialTransaction") || methodeName.contains("importEntries") || methodeName.contains("importAll")){
            refresh();
        }
    } 
}
